<?php

namespace Modules\ProductControl\Controllers;

use Module;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Modules\ProductControl\Models\Click;
use Modules\ProductControl\Models\Product;
use Modules\ProductControl\Models\OrderItem;
use Modules\ProductControl\Models\SystemLog;
use Modules\ProductControl\Models\CategoryAverageConversion;

class HomeController extends Controller
{
    public function __construct()
    {        
        Module::onView("content", function() {
            return "This is content view from ProductControl Module HomeController";
        }, 5);
    }
    public function index(Request $request)
    {
        $message = config("product-control::app.message");
        return view('product-control::home.welcome', [
            'message' => $message,
        ]);
    }

    public function downloadProductImage(Request $request){
        set_time_limit(120);
        $productId = $request->get("product_id",0);
        $products = DB::table('product_sku')->where('product_id',$productId)->whereNotNull('image_url')->where('image_url','!=','')->get(['image_url','id']);
        $zipname = 'images_'.env('APP_LOCALE','us').$productId.'.zip';
        $zip = new \ZipArchive();
        $zip->open($zipname, \ZipArchive::CREATE);
        $files = [];
        $ids = [$productId];
        $hasFile = false;
        foreach ($products as $item) {
            if (!in_array($item->id,$ids)){
                $ids[] = $item->id;
            }
            if(!empty($item->image_url)) {
                $this->zipFile($item->image_url,$files,$zip,$hasFile);
            }
        }

        $gallery = DB::table('product_gallery')->whereIn('product_id',$ids)->whereNotNull('image_url')->where('image_url','!=','')->get(['image_url']);
        foreach ($gallery as $item) {
            if(!empty($item->image_url)) {
                $this->zipFile($item->image_url,$files,$zip,$hasFile);
            }
        }
        $product = DB::table('product')->where('id',$productId)->whereNotNull('image_url')->where('image_url','!=','')->first(['image_url','id']);
        if(!empty($product->image_url)) {
            $this->zipFile($product->image_url,$files,$zip,$hasFile);
        }
        $zip->close();
        if ($hasFile){
            header('Content-Type: application/zip');
            header('Content-disposition: attachment; filename='.$zipname);
            header('Content-Length: ' . filesize($zipname));
            readfile($zipname);
            unlink(public_path('/') .$zipname);
            foreach ($files as $item){
                unlink($item);
            }
        }else{
            echo "Not found image";
        }


    }

    public function viewPage($params){
        if ($this->isValid() && isset($params['data']['id']) && false) {
            \DB::table('product')
                ->where('id','=', $params['data']['id'])
                ->increment('view_count',1);

        }
    }

    private function isValid() {
        $isBot = $this->isBot();
        $isStaff = $this->isStaff();
        $retVal = true;
        if ($isBot || $isStaff) {
            $retVal = false;
        }
        return $retVal;
    }


    private function isBot()
    {
        return preg_match("/rambler|abacho|acoi|accona|aspseek|altavista|estyle|scrubby|lycos|geona|ia_archiver|alexa|sogou|skype|facebook|twitter|pinterest|linkedin|naver|bing|google|yahoo|duckduckgo|yandex|baidu|teoma|xing|java\/1.7.0_45|bot|crawl|slurp|spider|mediapartners|\sask\s|\saol\s/i", isset($_SERVER["HTTP_USER_AGENT"]) ? $_SERVER["HTTP_USER_AGENT"] : "");
    }

    private function isStaff() {
        try {
            return \Auth::guard('web')->check();
        } catch (\Exception $e) {
            \Log::info('Module ProductControl error:' . $e->getMessage());
            return 0;
        }

    }

    public function getProductStatistics(Request $request) {
        $response = [
            "status" => "fail"
        ];
        
        $productAdvertisingStatus = null;

        if ($request->has("product_id")) {
            $productId = $request->input("product_id");
            $product = Product::where("id", $productId)->first();
            $timestamp = time();
            $day_of_the_week = date('N', $timestamp); // N returns mon-sun as digits 1 - 7. 
    
            $sunday_ts = $timestamp + ( 7 - $day_of_the_week) * 24 * 60 * 60;
            $monday_ts = $timestamp - ( $day_of_the_week - 1) * 24 * 60 * 60;
    
            $dates = [];
            $lastDate = null;
            for($i = 0; $i < 10; $i++) {
                $dates_key = $i;
                $dates[$dates_key] = [
                    'start' => date('Y-m-d', $monday_ts - $i * 7 * 24 * 60 * 60),
                    'end' => date('Y-m-d', $sunday_ts - $i * 7 * 24 * 60 * 60)
                ];
                $lastDate = date('Y-m-d', $sunday_ts - $i * 7 * 24 * 60 * 60);
            }
            $result = [];
            $currentPrice = $product->price;
            $lastPrice = $currentPrice;
            $lastDate = new \DateTime($lastDate);
            $createLog = SystemLog::where("target_id", $productId)
                ->where("type", "INSERT")
                ->where("table", "=", 'sb_product')
                ->first();

            if ($createLog) {
                $createLogData = json_decode($createLog->data);
                $lastPrice = $createLogData->price;
            }

            $productCreateTime = new \DateTime($product->created_at);
            if ($lastDate >= $productCreateTime) {
                $lastPriceLog = SystemLog::where("target_id", $productId)
                        ->where("data", "like", '%"price"%')
                        ->where("table", "=", 'sb_product')
                        ->where("type", "UPDATE")
                        ->where("created_at", "<=", $lastDate)
                        ->orderBy("created_at", "DESC")
                        ->first();
                if ($lastPriceLog) {
                    $lastPrice = json_decode($lastPriceLog->data)->price;
                }
            }

            $productAdvertisingStatus = DB::table('merchant_product')
                ->where('product_id', $productId)
                ->select(['id'])
                ->first();
                
            if ($productAdvertisingStatus) $productAdvertisingStatus = 1; else $productAdvertisingStatus = 0;

            foreach ($dates as $key => $value) {
                $startTime = \DateTime::createFromFormat("Y-m-d", $value["start"])->setTime(0, 0, 0);
                $endTime = \DateTime::createFromFormat("Y-m-d", $value["end"])->setTime(23, 59, 59);
                $click = Click::where("product_id", $productId)
                            ->where("date", ">=", $startTime)
                            ->where("date", "<=", $endTime)
                            ->groupBy("product_id")
                            ->sum("clicks");
                $quantity = OrderItem::where("product_id", $productId)
                            ->join('order', 'order.id', '=', 'order_item.order_id')
                            ->where("order.payment_status", "=", "PAID")
                            ->where("order.created_at", ">=", $startTime)
                            ->where("order.created_at", "<=", $endTime)
                            ->groupBy("order_item.order_id")
                            ->get();
                $result[$key] = [
                    "clicks" => $click,
                    "quantity" => count($quantity),
                ];
            }
            foreach (array_reverse($dates) as $key => $value) {
                if ((float) $lastPrice != (float) $currentPrice && $key == 9) {
                    $lastPrice = $currentPrice;
                }
                $avgPrice = $lastPrice;
                $startTime = \DateTime::createFromFormat("Y-m-d", $value["start"])->setTime(0, 0, 0);
                $endTime = \DateTime::createFromFormat("Y-m-d", $value["end"])->setTime(23, 59, 59);
                if ($endTime < $productCreateTime) {
                    $avgPrice = 0;
                }
                $changeLogs = SystemLog::where("target_id", $productId)
                                ->where("data", "like", '%"price"%')
                                ->where("table", "=", 'sb_product')
                                ->where("type", "UPDATE")
                                ->where("created_at", ">=", $startTime)
                                ->where("created_at", "<=", $endTime)
                                ->get();
                if (count($changeLogs) > 0) {
                    $totalPrice = 0; 
                    foreach ($changeLogs as $index => $item) {
                        $dataItem = json_decode($item->data);
                        $totalPrice += $dataItem->price;
                    }
                    $avgPrice = round($totalPrice / count($changeLogs), 2);
                }
                $result[9 - $key]["price"] = $avgPrice;
            }

            $avgConversion = Cache::get(decorCacheKey("product-avg-conversion", $productId));
            if (!$avgConversion || $request->get('clear_cache')) {
                $avgConversion = $this->getProductAvgConversion($productId, 30);
                Cache::put(decorCacheKey("product-avg-conversion", $productId), $avgConversion, 1440); //24 hours
            }

            $categoryId = $this->getCategory($productId);

            $response = [
                "status" => "successful",
                "result" => $result,
                "advertising_status" => $productAdvertisingStatus,
                'avg_conversion' => $avgConversion,
                'category_avg_conversion' => $this->getCategoryAvgConversion($categoryId, 30),
            ];

        }

        return response()->json($response);
    }

    private function getCategory($productId){
        $pnt = DB::table('product_n_category')->where('product_id', $productId)->where('is_parent', 0)->first();
        if ($pnt) {
            return $pnt->category_id;
        }

        return 0;
    }

    private function getProductAvgConversion($productId, $days = 30){
        $startTime = date('Y-m-d 00:00:00', strtotime('-' . $days . ' days'));
        $endTime = date('Y-m-d 23:59:59', strtotime('-1 day'));
        $click = Click::where("product_id", $productId)
            ->where("date", ">=", $startTime)
            ->where("date", "<=", $endTime)
            ->groupBy("product_id")
            ->sum("clicks");
        if ($click == 0) {
            return 0;
        }
        $orderCount = DB::table('order')
            ->join('order_item', 'order.id', '=', 'order_item.order_id')
            ->where('order.payment_status', 'PAID')
            ->where("order.created_at", ">=", $startTime)
            ->where("order.created_at", "<=", $endTime)
            ->where('order.status', '!=', 'CANCELLED')
            ->where('order_item.product_id', $productId)
            ->get([DB::raw('distinct sb_order.id')])
            ->count();

        return $orderCount / $click;
    }

    private function getCategoryAvgConversion($categoryId, $days = 30){
        $result = CategoryAverageConversion::where('category_id', $categoryId)
            ->first();

        if($result) {
            return $result->value;
        }

        return 0;
    }

    private function downloadImageFromUrl($url){
        $retVal = '';
        $fileName = basename($url);
        $path = public_path('/files/zip/');
        if (!file_exists($path)) {
            mkdir($path, 0777, true);
        }
        $filePath = $path.$fileName;
        try{
            if (file_put_contents( $filePath,file_get_contents($url))){
                $retVal['path'] =  $filePath;
                $retVal['fileName'] =  $fileName;
            }
        }catch(\Exception $e){
            \Log::error('ProductControl DownloadFile ERROR:' . $e->getTraceAsString());
        }

        return $retVal;
    }
    private function zipFile($imageUrl, &$files, &$zip, &$hasFile){
        $file = $this->downloadImageFromUrl($imageUrl);
        if (!empty($file)){
            $path = $file['path'];
            $fileName = $file['fileName'];
            if (!in_array($path, $files)) {
                if (file_exists($path)) {
                    $hasFile = true;
                    $zip->addFile($path,$fileName);
                }
                $files[] = $path;
            }
        }
    }



    public function getProductsStatistics(Request $request) {
        $response = [
            "status" => "fail"
        ];
        if ($request->has("product_ids")) {
            $productIds = $request->input("product_ids");
            $productIds = explode(',', $productIds);
            $dates = [];
            for($i = 0; $i < 7; $i++) {
                $dates_key = date('Y-m-d', strtotime('-' . $i . ' days'));
                $dates[$dates_key] = [
                    'start' => $dates_key,
                    'end' => $dates_key
                ];
            }
            $result = [];
            foreach ($productIds as $productId) {
                foreach ($dates as $key => $value) {
                    $startTime = \DateTime::createFromFormat("Y-m-d", $value["start"])->setTime(0, 0, 0);
                    $endTime = \DateTime::createFromFormat("Y-m-d", $value["end"])->setTime(23, 59, 59);
                    $click = Click::where("product_id", $productId)
                        ->where("date", ">=", $startTime)
                        ->where("date", "<=", $endTime)
                        ->groupBy("product_id")
                        ->sum("clicks");
                    $quantity = OrderItem::where("product_id", $productId)
                        ->join('order', 'order.id', '=', 'order_item.order_id')
                        ->where("order.payment_status", "=", "PAID")
                        ->where("order.created_at", ">=", $startTime)
                        ->where("order.created_at", "<=", $endTime)
                        ->groupBy("order_item.order_id")
                        ->get();
                    $result[$productId][$key] = [
                        "clicks" => intval($click),
                        "quantity" => count($quantity),
                    ];
                }
            }
            $response = [
                "status" => "successful",
                "result" => $result,
            ];
        }
        return response()->json($response);
    }
}
