<?php

namespace Modules\Reports\Controllers;

use Illuminate\Http\Request;
use Modules\Reports\Controllers\Controller;
use Module;
use DB;

class HomeController extends Controller
{
    public function __construct()
    {        
        Module::onView("content", function() {
            return "This is content view from Reports Module HomeController";
        }, 5);
    }
    public function index(Request $request)
    {
        $message = config("reports::app.message");
        return view('reports::home.welcome', [
            'message' => $message,
        ]);
    }

    public function buildSummaryCountProducts (Request $request) {
        set_time_limit(2 * 60 * 60);
        $fromDate = $request->get('from_date', '-1');
        $toDate = $request->get('to_date', '-1');
        $daysNeedCountProducts = $this->getDaysNeedCountProducts($fromDate, $toDate);
        foreach ($daysNeedCountProducts as $date) {
            $countProducts = $this->countProductsInDay($date);
            foreach ($countProducts as $sourceId => $countProductsByCategory) {
                foreach ($countProductsByCategory as $categoryId => $countProductByActor) {
                    foreach ($countProductByActor as $actorId => $countProduct) {
                        $this->storeOrUpdateSummaryCountProduct($date, $sourceId, $categoryId, $actorId, $countProduct);
                    }
                }
            }
        }
        return $daysNeedCountProducts;
    }

    public function buildSummaryCountContentProducts (Request $request) {
        set_time_limit(2 * 60 * 60);
        $fromDate = $request->get('from_date', '-1');
        $toDate = $request->get('to_date', '-1');
        $daysNeedCountProducts = $this->getDaysNeedCountProducts($fromDate, $toDate);
        foreach ($daysNeedCountProducts as $date) {
            $countContentProducts = $this->countContentProductInDay($date);
            foreach ($countContentProducts as $actorId => $countProduct) {
                $this->storeOrUpdateCountProduct($date, $actorId, $countProduct);
            }
        }
        return $daysNeedCountProducts;
    }

    private function getDaysNeedCountProducts($fromDate, $toDate)
    {
        $retVal = [];
        $fromDay = date('Y-m-d 00:00:00', strtotime($fromDate . " days"));
        $toDay = date('Y-m-d 23:59:59', strtotime($toDate ." days"));
        // $products = DB::table('product')
        //     ->where('updated_at', '>=', $fromDay)
        //     ->where('updated_at', '<=', $toDay)
        //     ->select('id', 'created_at')
        //     ->groupBy(DB::raw('date(sb_product.created_at)'))
        //     ->orderBy('updated_at', 'desc')
        //     ->get();
        $orders = DB::table('order')
            ->where('created_at', '>=', $fromDay)
            ->where('created_at', '<=', $toDay)
            ->select('id', 'created_at')
            ->groupBy(DB::raw('date(sb_order.created_at)'))
            ->orderBy('created_at', 'desc')
            ->get();
        foreach ($orders as $order) {
            $date = date('Y-m-d', strtotime($order->created_at));
            if (!in_array($date, $retVal)) {
                $retVal[] = $date;
            }
        }
        return $retVal;
    }

    private function countProductsInDay($date)
    {
        $filter = [];
        $fromDate = \DateTime::createFromFormat('Y-m-d', $date);
        $toDate = \DateTime::createFromFormat('Y-m-d', $date);
        $filter['query_from'] = $fromDate->setTime(0, 0, 0);
        $filter['query_to'] = $toDate->setTime(23,59,59);
        $products = DB::table('product as p')
            ->from(DB::raw("`sb_product` FORCE INDEX(`createdAt`)"))
            ->join('product_n_category as pnc', 'pnc.product_id', '=', 'product.id')
            ->join('product_actor_source as pas', 'pas.product_id', '=', 'product.id')
            ->where('pnc.is_parent', 0)
            ->where('product.created_at', '>=', $filter['query_from'])
            ->where('product.created_at', '<=', $filter['query_to'])
            ->where('product.status', 'ACTIVE')
            ->select(
                'product.id',
                'pnc.category_id',
                'pas.source_id',
                'pas.actor_id',
                DB::raw('COUNT(sb_product.id) as count_products')
            )
            ->groupBy('pas.source_id')
            ->groupBy('pnc.category_id')
            ->groupBy('pas.actor_id')
            ->get();
        $countProducts= [];
        foreach ($products as $product) {
            $sourceId = $product->source_id;
            $categoryId = $product->category_id;
            $actorId = $product->actor_id;
            if (!array_key_exists($sourceId, $countProducts)) {
                $countProducts[$sourceId] = [];
            }
            if (!array_key_exists($categoryId, $countProducts[$sourceId])) {
                $countProducts[$sourceId][$categoryId] = [];
            }
            if (!array_key_exists($actorId, $countProducts[$sourceId][$categoryId])) {
                $countProducts[$sourceId][$categoryId][$actorId] = 0;
            }
            $countProducts[$sourceId][$categoryId][$actorId] = $product->count_products;
        }
        return $countProducts;
    }

    private function countContentProductInDay($date)
    {
        $filter = [];
        $fromDate = \DateTime::createFromFormat('Y-m-d', $date);
        $toDate = \DateTime::createFromFormat('Y-m-d', $date);
        $filter['query_from'] = $fromDate->setTime(0, 0, 0);
        $filter['query_to'] = $toDate->setTime(23,59,59);
        $products = DB::table('product as p')
            ->from(DB::raw("`sb_product` FORCE INDEX(`createdAt`)"))
            ->where('product.created_at', '>=', $filter['query_from'])
            ->where('product.created_at', '<=', $filter['query_to'])
            ->where('product.status', 'ACTIVE')
            ->select(
                'product.actor_id',
                DB::raw('COUNT(sb_product.id) as count_products')
            )
            ->groupBy('product.actor_id')
            ->get();
        $countProducts= [];
        foreach ($products as $product) {
            $actorId = $product->actor_id;
            if (!array_key_exists($actorId, $countProducts)) {
                $countProducts[$actorId] = [];
            }
            $countProducts[$actorId] = $product->count_products;
        }
        return $countProducts;
    }

    private function storeOrUpdateSummaryCountProduct($date, $sourceId, $categoryId, $actorId, $countProduct)
    {
        $data = DB::table('summary_product_date')
            ->where('date', $date)
            ->where('source_id', $sourceId)
            ->where('category_id', $categoryId)
            ->where('actor_id', $actorId)
            ->first();
        if ($data) {
            DB::table('summary_product_date')
                ->where('id', $data->id)
                ->update([
                    'count_products' => $countProduct,
                    'updated_at' => new \DateTime(),
                ]);
        } else {
            DB::table('summary_product_date')
                ->insert([
                    'date' =>  $date,
                    'count_products' => $countProduct,
                    'created_at' => new \DateTime(),
                    'updated_at' => new \DateTime(),
                    'source_id' => $sourceId,
                    'category_id' => $categoryId,
                    'actor_id' => $actorId,
                ]);
        }
    }

    private function storeOrUpdateCountProduct($date, $actorId, $countProduct)
    {
        $data = DB::table('count_content_product_date')
            ->where('date', $date)
            ->where('actor_id', $actorId)
            ->first();
        if ($data) {
            DB::table('count_content_product_date')
                ->where('id', $data->id)
                ->update([
                    'count_products' => $countProduct,
                    'updated_at' => new \DateTime(),
                ]);
        } else {
            DB::table('count_content_product_date')
                ->insert([
                    'date' =>  $date,
                    'count_products' => $countProduct,
                    'created_at' => new \DateTime(),
                    'updated_at' => new \DateTime(),
                    'actor_id' => $actorId,
                ]);
        }
    }

    public function buildSummaryRevenueAndCost(Request $request)
    {
        set_time_limit(2*60*60);
        $fromDate = $request->get('from_date', '-1');
        $toDate = $request->get('to_date', '-1');
        $daysNeedCalculateRevenueAndCost = $this->getDaysNeedCalculateRevenueAndCost($fromDate, $toDate);
        foreach ($daysNeedCalculateRevenueAndCost as $date) {
            $revenueAndCostInDay = $this->calculateRevenueAndCostInDay($date);
            foreach ($revenueAndCostInDay as $sourceId => $revenueAndCostByCategory) {
                foreach ($revenueAndCostByCategory as $categoryId => $revenueAndCost) {
                    $this->storeOrUpdateSummaryRevenueAndCost(
                        $date,
                        $sourceId,
                        $categoryId,
                        $revenueAndCost
                    );
                }

            }
        }
        return $daysNeedCalculateRevenueAndCost;
    }

    public function getDaysNeedCalculateRevenueAndCost($fromDate, $toDate)
    {
        $retVal = [];
        $fromDay = date('Y-m-d 00:00:00', strtotime($fromDate . " days"));
        $toDay = date('Y-m-d 23:59:59', strtotime($toDate ." days"));
        $orders = DB::table('order')
                ->where('created_at', '>=', $fromDay)
                ->where('created_at', '<=', $toDay)
                ->groupBy(DB::raw('date(sb_order.created_at)'))
                ->select('created_at', 'id')
                ->get();
        foreach ($orders as $order) {
            $date = date('Y-m-d', strtotime($order->created_at));
            if (!in_array($date, $retVal)) {
                $retVal[] = $date;
            }
        }
        return $retVal;
    }

    private function calculateRevenueAndCostInDay($date)
    {
        $filter = [];
        $fromDate = \DateTime::createFromFormat('Y-m-d', $date);
        $toDate = \DateTime::createFromFormat('Y-m-d', $date);
        $filter['query_from'] = $fromDate->setTime(0, 0, 0);
        $filter['query_to'] = $toDate->setTime(23,59,59);
        $orders = DB::table('order as o')
                    ->from(DB::raw("`sb_order` FORCE INDEX(`sa_inoutput_created_at`)"))
                    ->join('order_item as oi', 'oi.order_id', '=', 'order.id')
                    ->join('product_n_category as pnc', 'pnc.product_id', '=', 'oi.product_id')
                    ->join('product_actor_source as pas', 'pas.product_id', '=', 'oi.product_id')
                    ->where('order.created_at', '>=', $filter['query_from'])
                    ->where('order.created_at', '<=', $filter['query_to'])
                    ->whereNotIn('order.status', ['CANCELED','RETURNED','REQUEST_RETURN'])
                    ->where('order.payment_status', 'PAID')
                    ->where('pnc.is_parent', 0)
                    ->select('order.id',
                            'oi.price',
                            'pnc.category_id',
                            'pas.source_id',
                            'oi.shipping_cost',
                            'oi.cost',
                            'oi.quantity',
                            'oi.product_id',
                            'order.country_id',
                            'order.shipping_configurations',
                            'oi.product_sku_id',
                            'oi.id as order_item_id'
                    )
                    ->get();
        $data = [];
        $orderIds = [];
        foreach ($orders as $order) {
            $sourceId = $order->source_id;
            $categoryId = $order->category_id;
            if(!array_key_exists($sourceId, $data)) {
                $data[$sourceId] = [];
                $orderIds[$sourceId] = [];
            }
            if(!array_key_exists($categoryId, $data[$sourceId])) {
                $data[$sourceId][$categoryId] = [
                    'revenue' => 0,
                    'cost' => 0,
                    'quantity' => 0,
                    'ads_cost' => 0,
                ];
                $data[$sourceId][$categoryId]['ads_cost'] = $this->getAdsCostInDay($date, $sourceId, $categoryId);
                $orderIds[$sourceId][$categoryId] = [];
            }
            if (!in_array($order->id, $orderIds[$sourceId][$categoryId])) {
                $orderIds[$sourceId][$categoryId][] = $order->id;
            }
            $warehouseId = $this->getWarehouseId($order->order_item_id, $order->shipping_configurations);
            if ($order->product_sku_id) {
                $order->cost = $this->fetchCost($order->product_sku_id, $order->country_id, $warehouseId) * $order->quantity;
            }
            $data[$sourceId][$categoryId]['cost'] += $order->shipping_cost + $order->cost;
            $data[$sourceId][$categoryId]['quantity'] += $order->quantity;
            $data[$sourceId][$categoryId]['revenue'] += $order->quantity * $order->price;
        }
        return $data;
    }

    private function getAdsCostInDay($date, $sourceId, $categoryId)
    {
        $cost = 0;
        $adsCost = DB::table('product_cost_ads as pca')
            ->join('product_n_category as pnc', 'pnc.product_id', '=', 'pca.product_id')
            ->join('product_actor_source as pas', 'pas.product_id', '=', 'pca.product_id')
            ->where('pca.date', $date)
            ->where('pas.source_id', $sourceId)
            ->where('pnc.category_id', $categoryId)
            ->where('pnc.is_parent', 0)
            ->select(DB::raw('SUM(sb_pca.cost) as cost'))
            ->groupBy('pca.date')
            ->first();
        if ($adsCost) {
            $cost = $adsCost->cost;
        }
        return $cost;
    }

    private function storeOrUpdateSummaryRevenueAndCost($date, $sourceId, $categoryId, $revenueAndCost)
    {
        $revenue = $revenueAndCost['revenue'];
        $cost = $revenueAndCost['cost'];
        $quantity = $revenueAndCost['quantity'];
        $adsCost = $revenueAndCost['ads_cost'];
        $data = DB::table('summary_order_date')
            ->where('date', $date)
            ->where('source_id', $sourceId)
            ->where('category_id', $categoryId)
            ->first();
        if ($data) {
            DB::table('summary_order_date')
                ->where('id', $data->id)
                ->update([
                    'revenue' => $revenue,
                    'cost' => $cost,
                    'quantity' => $quantity,
                    'ads_cost' => $adsCost,
                    'updated_at' => new \DateTime(),
                ]);
        } else {
            DB::table('summary_order_date')
                ->insert([
                    'date' =>  $date,
                    'revenue' => $revenue,
                    'cost' => $cost,
                    'quantity' => $quantity,
                    'ads_cost' => $adsCost,
                    'created_at' => new \DateTime(),
                    'updated_at' => new \DateTime(),
                    'source_id' => $sourceId,
                    'category_id' => $categoryId
                ]);
        }
    }

    public function buildSummaryCreatorProductRevenue (Request $request)
    {
        set_time_limit(2*60*60);
        $fromDate = $request->get('from_date', '-1');
        $toDate = $request->get('to_date', '-1');
        $dates = $this->getDaysNeedCalculateCreatorRevenue($fromDate, $toDate);
        foreach ($dates as $date) {
            $creators = $this->calculateCreatorRevenueAndCostInDay($date);
            foreach ($creators as $creatorId => $data) {
                $this->storeOrUpdateSummaryCreatorRevenueAndCost($date, $creatorId, $data);
            }
        }
        return $dates;
    }

    public function getDaysNeedCalculateCreatorRevenue($fromDate, $toDate)
    {
        $retVal = [];
        $fromDay = date('Y-m-d 00:00:00', strtotime($fromDate . " days"));
        $toDay = date('Y-m-d 23:59:59', strtotime($toDate ." days"));
        $orders = DB::table('order')
            ->where('updated_at', '>=', $fromDay)
            ->where('updated_at', '<=', $toDay)
            ->select('created_at', 'id')
            ->groupBy(DB::raw('date(sb_order.created_at)'))
            ->get();
        foreach ($orders as $order) {
            $date = date('Y-m-d', strtotime($order->created_at));
            if (!in_array($date, $retVal)) {
                $retVal[] = $date;
            }
        }
        return $retVal;
    }

    private function calculateCreatorRevenueAndCostInDay($date)
    {
        $filter = [];
        $fromDate = \DateTime::createFromFormat('Y-m-d', $date);
        $toDate = \DateTime::createFromFormat('Y-m-d', $date);
        $filter['query_from'] = $fromDate->setTime(0, 0, 0);
        $filter['query_to'] = $toDate->setTime(23,59,59);
        $orders = DB::table('order as o')
            ->from(DB::raw("`sb_order` FORCE INDEX(`sa_inoutput_created_at`)"))
            ->join('order_item as oi', 'oi.order_id', '=', 'order.id')
            ->join('product_actor_source as pas', 'pas.product_id', '=', 'oi.product_id')
            ->where('order.created_at', '>=', $filter['query_from'])
            ->where('order.created_at', '<=', $filter['query_to'])
            ->whereNotIn('order.status', ['CANCELED','RETURNED','REQUEST_RETURN'])
            ->where('order.payment_status', 'PAID')
            ->select(
                'order.id',
                'oi.price',
                'pas.actor_id',
                'oi.shipping_cost',
                'oi.cost',
                'oi.quantity',
                'oi.product_id',
                'order.return_fee',
                'order.country_id',
                'order.shipping_configurations',
                'oi.product_sku_id',
                'oi.id as order_item_id'
            )
            ->get();
        $data = [];
        $orderIds = [];
        $productIds = [];
        foreach ($orders as $order) {
            $actorId = $order->actor_id ? $order->actor_id : 1;
            $productId = $order->product_id;
            if(!array_key_exists($actorId, $data)) {
                $data[$actorId] = [
                    'revenue' => 0,
                    'cost' => 0,
                    'quantity' => 0,
                    'ads_cost' => 0,
                    'return_fee' => 0,
                    'date' => $date
                ];
                $orderIds[$actorId] = [];
                $productIds[$actorId] = [];
            }
            if (!in_array($productId, $productIds[$actorId])) {
                $data[$actorId]['ads_cost'] += $this->getProductAdsCostInDay($date, $productId);
                $productIds[$actorId][] = $productId;
            }

            if (!in_array($order->id, $orderIds[$actorId])) {
                $data[$actorId]['return_fee'] += $order->return_fee;
                $orderIds[$actorId][] = $order->id;
            }
            $warehouseId = $this->getWarehouseId($order->order_item_id, $order->shipping_configurations);
            if ($order->product_sku_id) {
                $order->cost = $this->fetchCost($order->product_sku_id, $order->country_id, $warehouseId) * $order->quantity;
            }
            $data[$actorId]['cost'] += $order->shipping_cost + $order->cost;
            $data[$actorId]['quantity'] += $order->quantity;
            $data[$actorId]['revenue'] += $order->quantity * $order->price;
        }
        return $data;
    }

    private function storeOrUpdateSummaryCreatorRevenueAndCost($date, $actorId, $revenueAndCost)
    {
        $revenue = $revenueAndCost['revenue'];
        $cost = $revenueAndCost['cost'];
        $quantity = $revenueAndCost['quantity'];
        $adsCost = $revenueAndCost['ads_cost'];
        $returnFee = $revenueAndCost['return_fee'];
        $data = DB::table('summary_actor_revenue_date')
            ->where('date', $date)
            ->where('actor_id', $actorId)
            ->first();
        if ($data) {
            DB::table('summary_actor_revenue_date')
                ->where('id', $data->id)
                ->update([
                    'revenue' => $revenue,
                    'cost' => $cost,
                    'quantity' => $quantity,
                    'ads_cost' => $adsCost,
                    'return_fee' => $returnFee,
                    'updated_at' => new \DateTime(),
                ]);
        } else {
            DB::table('summary_actor_revenue_date')
                ->insert([
                    'date' =>  $date,
                    'revenue' => $revenue,
                    'cost' => $cost,
                    'quantity' => $quantity,
                    'ads_cost' => $adsCost,
                    'created_at' => new \DateTime(),
                    'updated_at' => new \DateTime(),
                    'return_fee' => $returnFee,
                    'actor_id' => $actorId,
                ]);
        }
    }

    private function getProductAdsCostInDay($date, $productId)
    {
        $cost = 0;
        $adsCost = DB::table('product_cost_ads as pca')
            ->where('pca.date', $date)
            ->where('product_id', $productId)
            ->select(DB::raw('SUM(sb_pca.cost) as cost'))
            ->groupBy('pca.date')
            ->first();
        if ($adsCost) {
            $cost = $adsCost->cost;
        }
        return $cost;
    }


}
