<?php

namespace Modules\Reports\Controllers;

use App\Helpers\DBConnect;
use App\Models\ProductNCategoryTop;
use Illuminate\Http\Request;
use Modules\Reports\Models\Category;
use Modules\Reports\Models\Order;
use Modules\Reports\Models\OrderItem;
use Modules\Reports\Models\OrderStatusDuration;
use Modules\Reports\Models\Printing;

class PrintingShopController extends Controller
{
    static $connection;
    protected $locales;

    public static function table($table)
    {
        if (self::$connection) {
            return self::$connection->table($table);
        } else {
            return DB::table($table);
        }
    }

    public static function setConnection($locale)
    {
        $connection = DBconnect::connect($locale);
        self::$connection = $connection;
    }

    public function index()
    {
        $locates = getModuleLocale();
        return view('reports::printing-shop.index', compact('locates'));
    }

    public function getCategory()
    {
        $data = Category::query()
            ->where('type', 'PRODUCT')
            ->orderBy('name', 'ASC')
            ->get();

        $response = [
            "status" => "successful",
            "data" => $data
        ];

        return response()->json($response);
    }

    public function getPrinting()
    {
        $data = Printing::query()
            ->orderBy('name', 'ASC')
            ->get();

        foreach ($data as $key => $val) {
            $data[$key]->name = $val->name . ' - ' . $val->code;
        }

        $response = [
            "status" => "successful",
            "data" => $data
        ];

        return response()->json($response);
    }

    public function findChart(Request $request)
    {
        $start_date = date('Y-m-d', strtotime(str_replace('/', '-', $request->start_date))) . ' 00:00:00';
        $end_date = date('Y-m-d H:i:s', strtotime(str_replace('/', '-', $request->end_date))) . '23:59:59';

        $dataChart = $this->getDataChart($start_date, $end_date);
        $response = [
            "status" => "successful",
            "result" => [
                "dataChart" => $dataChart,
            ]
        ];
        return response()->json($response);
    }

    public function findAverage(Request $request)
    {
        $start_date = date('Y-m-d', strtotime(str_replace('/', '-', $request->start_date))) . ' 00:00:00';
        $end_date = date('Y-m-d H:i:s', strtotime(str_replace('/', '-', $request->end_date))) . '23:59:59';
        $category_id = $request->category_id;
        $printing_code = $request->printing_code;

        $dataAverageByPrinting = $this->getDataAverageByPrinting($start_date, $end_date, $printing_code, $category_id);

        $response = [
            "status" => "successful",
            "result" => [
                "dataAverage" => $dataAverageByPrinting
            ]
        ];
        return response()->json($response);
    }

    public function findPercentStatus(Request $request)
    {
        $start_date = date('Y-m-d', strtotime(str_replace('/', '-', $request->start_date))) . ' 00:00:00';
        $end_date = date('Y-m-d H:i:s', strtotime(str_replace('/', '-', $request->end_date))) . '23:59:59';
        $printing_code = $request->printing_code;

        $percentInStatus = $this->getDataPercentInStatus($start_date, $end_date, $printing_code);
        $totalPercentInStatus = [];
        foreach ($percentInStatus ?? [] as $value) {
            if (array_key_exists($value->status, $totalPercentInStatus)) {
                $totalPercentInStatus[$value->status] += $value->total_products;
            } else {
                $totalPercentInStatus[$value->status] = $value->total_products;
            }
        }

        $response = [
            "status" => "successful",
            "result" => [
                "percentInStatus" => $percentInStatus,
                "totalPercentInStatus" => $totalPercentInStatus,
            ]
        ];
        return response()->json($response);
    }

    public function findIssue(Request $request)
    {
        $start_date = date('Y-m-d', strtotime(str_replace('/', '-', $request->start_date))) . ' 00:00:00';
        $end_date = date('Y-m-d H:i:s', strtotime(str_replace('/', '-', $request->end_date))) . '23:59:59';
        $category_id = $request->category_id;
        $printing_code = $request->printing_code;

        $dataIssue = $this->getDataIssue($start_date, $end_date, $printing_code, $category_id);
        $totalPercentIssue = [];
        foreach ($dataIssue ?? [] as $item) {
            $printing_code = $item->printing_code != null ? $item->printing_code : 'unknow';
            if (array_key_exists($printing_code, $totalPercentIssue)) {
                $totalPercentIssue[$printing_code] += $item->total_order;
            } else {
                $totalPercentIssue[$printing_code] = $item->total_order;
            }
        }

        $response = [
            "status" => "successful",
            "result" => [
                "dataIssue" => $dataIssue,
                "totalPercentIssue" => $totalPercentIssue
            ]
        ];
        return response()->json($response);
    }

    protected function getDataIssue($start_date, $end_date, $printing_code, $category_id)
    {
        $query = OrderItem::query()
            ->join('order', 'order.id', '=', 'order_item.order_id')
            ->select('order_item.printing_code',
                'order_item.feedback_issue',
                \DB::raw('COUNT(DISTINCT order_id) as total_order'),
                \DB::raw('COUNT(DISTINCT CASE WHEN feedback_issue IS NOT NULL THEN order_id ELSE NULL END) as total_feedback_issue')
            );
        if ($category_id) {
            $query->addSelect('category.id as category_id', 'category.name as category_name')
                ->join('product_n_category_top', 'order_item.product_id', '=', 'product_n_category_top.product_id')
                ->join('category', 'product_n_category_top.category_id', '=', 'category.id')
                ->where('product_n_category_top.category_id', $category_id);
        }
        $query->whereBetween('order.created_at', array($start_date, $end_date));
        if ($printing_code) {
            $query->where('printing_code', $printing_code);
        }

        $data = $query->groupBy('printing_code', 'feedback_issue')
            ->get();

        return $data;
    }

    protected function getDataChart($start_date, $end_date)
    {
        try {
            $dataChart = Order::query()
                ->join('order_item', 'order_item.order_id', '=', 'order.id')
                ->select('order_item.printing_code',
                    \DB::raw("SUM(sb_order.cost) as total_cost"),
                    \DB::raw("SUM(sb_order.shipping_cost) as total_shipping_cost"),
                    \DB::raw("SUM(sb_order.tax_cost) as total_tax_cost"))
                ->whereBetween('order.created_at', array($start_date, $end_date))
                ->groupBy('order_item.printing_code')
                ->get();

            return $dataChart;
        } catch (\Exception $exception) {
            dd($exception->getMessage());
        }
    }

    protected function getDataAverageByPrinting($start_date, $end_date, $printing_code, $category_id)
    {
        $query = OrderItem::query()
            ->select(
                'order_item.printing_code',
                \DB::raw('SUM(CASE WHEN sb_order_status_duration.type = "issued_time" THEN sb_order_status_duration.duration ELSE 0 END) as issued_time_sum'),
                \DB::raw('COUNT(CASE WHEN sb_order_status_duration.type = "issued_time" THEN sb_order_status_duration.duration ELSE NULL END) as issued_time_count'),
                \DB::raw('SUM(CASE WHEN sb_order_status_duration.type = "delivering_time" THEN sb_order_status_duration.duration ELSE 0 END) as delivering_time_sum'),
                \DB::raw('COUNT(CASE WHEN sb_order_status_duration.type = "delivering_time" THEN sb_order_status_duration.duration ELSE NULL END) as delivering_time_count')
            )
            ->join('order', 'order.id', '=', 'order_item.order_id')
            ->join('order_status_duration', 'order_item.order_id', '=', 'order_status_duration.order_id');
        if ($category_id) {
            $query->addSelect('category.id as category_id', 'category.name as category_name')
                ->join('product_n_category_top', 'order_item.product_id', '=', 'product_n_category_top.product_id')
                ->join('category', 'product_n_category_top.category_id', '=', 'category.id')
                ->where('product_n_category_top.category_id', $category_id);
        }

        $query->whereBetween('order.created_at', array($start_date, $end_date));

        if ($printing_code != null && $printing_code != 'null') {
            $query->where('order_item.printing_code', $printing_code);
        }
        $dataAverage = $query->groupBy('order_item.printing_code')
            ->get();

        return $dataAverage;
    }

    protected function getDataPercentInStatus($start_date, $end_date, $printing_code)
    {
        $arrayStatus = ['PENDING', 'READY_TO_SHIP', 'PROCESSING', 'DELIVERING', 'FINISHED'];

        $query = OrderItem::query()
            ->join('order', 'order_item.order_id', '=', 'order.id')
            ->select('order_item.printing_code', 'order.status', \DB::raw('count(sb_order_item.product_id) as total_products'))
            ->whereBetween('order.created_at', array($start_date, $end_date));
        if ($printing_code) {
            $query->where('order_item.printing_code', $printing_code);
        }
        $data = $query->whereIn('order.status', $arrayStatus)
            ->groupBy('order_item.printing_code', 'order.status')
            ->get();

        return $data;
    }
}