<?php

namespace Modules\Reports\Controllers;


use Illuminate\Http\Request;
use Modules\Reports\Controllers\Controller;
use App\Helpers\DBConnect;
use DB;


class ProductIdeaController extends Controller
{
    protected $pageSize = 40;
    protected $locales = [];
    protected $users = [];
    protected $userByLocates = [];
    protected $ignoreEmails = ['mactrantung@gmail.com', 'info@megaads.vn'];
    static $connection;

    public function __construct() {
        $this->initCl();
    }

    public static function table($table) {
        if (self::$connection) {
            return self::$connection->table($table);
        } else {
            return DB::table($table);
        }
    }

    public static function setConnection($locale) {
        $connection = DBconnect::connect($locale);
        self::$connection = $connection;
    }

    private function initCl() {
        $locates = getModuleLocale();
        $marketEnableds = [];
        foreach ($locates as $locale) {
            if ($locale['enable'] && $locale['locale'] != 'central') {
                $marketEnableds[] = $locale['locale'];
                $this->locales[$locale['locale']] = $locale['name'];
            }
        }
        foreach ($marketEnableds as $market) {
            $this->userByLocates[$market] = [];
            $connection = DBconnect::connect($market);
            $roleNPermissions = $connection->table('role_n_permission as rnp')
                                            ->leftJoin('permission as p', 'p.id', '=', 'rnp.permission_id')
                                            ->where('p.value', '=', 'admin/products*')
                                            ->get(['rnp.role_id', 'rnp.permission_id']);
            $roleIds = [];
            foreach ($roleNPermissions as $item) {
                $roleIds[] = $item->role_id;
            }

            if (!empty($roleIds)) {
                $roleNUsers = $connection->table('role_n_user as rnu')
                                    ->leftJoin('users as u', 'u.id', '=', 'rnu.user_id')
                                    ->whereIn('rnu.role_id', $roleIds)
                                    ->whereNotIn('u.email', $this->ignoreEmails)
                                    ->where('status', '=', 'ACTIVE')
                                    ->get(['u.email', 'u.name']);

                foreach ($roleNUsers as $user) {
                    if ($user->email != null) {
                        $this->userByLocates[$market][] = $user->email;
                        if (!array_key_exists($user->email, $this->users)) {
                            $this->users[] = $user;
                        }
                    }
                }
            }
        }
    }



    public function find(Request $request) {
        set_time_limit(3600);
        ini_set('memory_limit','4096M');
        $filter = $this->buildFilter($request);
        $markets = $this->buildMarket($filter);
        $result = $this->getResult($filter, $markets);
        $pageId = $filter['page_id'];
        $size = count($result);
        $pagesCount = ceil($size / $this->pageSize);
        $start = $pageId * $this->pageSize;
        $end = $start + $this->pageSize;
        if ($end > $size) {
            $end = $size;
        }
        $items = [];
        for ($i = $start; $i < $end; $i++) {
            $items[] = $result[$i];
        }
        $response = [
            'status' => 'successful',
            'pagesCount' => $pagesCount,
            'pageId' => $pageId,
            'result' => $items,
        ];
        return response()->json($response);
    }
    
    private function getResult($filter, $markets) {
        $tmp = [];
        foreach ($markets as $market => $users) {
            $connection = DBconnect::connect($market);
            $itemNews = $this->getProduct($connection, $filter, $users);
            $this->countItems($connection, $filter, $tmp, $itemNews);
        }
        $retVal = [];
        foreach ($tmp as $key => $value) {
            $newRetVal = [
                'email' => $key,
            ];
            if (array_key_exists('product', $value)) {
                $newRetVal['product'] = $value['product'];
            } else {
                $newRetVal['product'] = 0;
            }
            if (array_key_exists('order', $value)) {
                $newRetVal['order'] = $value['order'];
            } else {
                $newRetVal['order'] = 0;
            }
            if (array_key_exists('amount', $value)) {
                $newRetVal['amount'] = $value['amount'];
            } else {
                $newRetVal['amount'] = 0;
            }
            if (array_key_exists('cost', $value)) {
                $newRetVal['cost'] = $value['cost'];
            } else {
                $newRetVal['cost'] = 0;
            }
            if (array_key_exists('refund', $value)) {
                $newRetVal['refund'] = $value['refund'];
            } else {
                $newRetVal['refund'] = 0;
            }
            if (array_key_exists('ads_cost', $value)) {
                $newRetVal['ads_cost'] = $value['ads_cost'];
            } else {
                $newRetVal['ads_cost'] = 0;
            }
            $newRetVal['profit'] = $newRetVal['amount'] - $newRetVal['cost'] - $newRetVal['refund'] - $newRetVal['ads_cost'];
            $retVal[] = $newRetVal;
        }
        if (isset($filter['orders']) && !empty($filter['orders'])) {
            foreach ($filter['orders'] as $key => $val) {
                $retVal = $this->sort($retVal, $key, $val);
            }
        }
        return $retVal;
    }

    public function sort($items, $col = null, $sorter) {
        usort($items, function ($itemA, $itemB) use ($col, $sorter){
            $aValue = isset($itemA[$col]) ? $itemA[$col] : '';
            $bValue = isset($itemB[$col]) ? $itemB[$col] : '';
            $result = 0;
            if ($sorter == 'desc') {
                if ($aValue > $bValue) {
                    $result = -1;
                } else if ($aValue < $bValue) {
                    $result = 1;
                }
            } else {
                if ($aValue < $bValue) {
                    $result = -1;
                } else if ($aValue > $bValue) {
                    $result = 1;
                }
            }
            return $result;
        });
        return $items;
    }

    private function countItems($connection, $filter, &$retVal, $data) {
        $items = $connection->table('report_product_sale')
                    ->where('date', '>=', $filter['query_from'])
                    ->where('date', '<=', $filter['query_to'])
                    ->groupBy('product_id')
                    ->select('product_id', 
                            \DB::raw("sum(sale) as sale"), 
                            \DB::raw("sum(quantity) as quantity"),
                            \DB::raw("sum(revenue) as revenue"),
                            \DB::raw("sum(cost) as cost"),
                            \DB::raw("sum(refund) as refund")
                            )
                    ->get();
        foreach ($data as $key => $values) {        
            $itemByIds = [];
            $totalSale = 0;
            $totalRevenue = 0;
            $totalCost = 0;
            $totalRefund = 0;
            foreach ($items as $item) {
                if (in_array($item->product_id, $values)) {
                    $totalSale += $item->sale;
                    $totalRevenue += $item->revenue;
                    $totalCost += $item->cost;
                    $totalRefund += $item->refund;
                }
            }

            $productCount = count($values);
            if (!array_key_exists($key, $retVal)) {
                $retVal[$key] = [];
            }
            if (!array_key_exists('product', $retVal[$key])) {
                $retVal[$key]['product'] = 0;
            }
            if (!array_key_exists('order', $retVal[$key])) {
                $retVal[$key]['order'] = 0;
            }
            if (!array_key_exists('amount', $retVal[$key])) {
                $retVal[$key]['amount'] = 0;
            }
            if (!array_key_exists('cost', $retVal[$key])) {
                $retVal[$key]['cost'] = 0;
            }
            if (!array_key_exists('refund', $retVal[$key])) {
                $retVal[$key]['refund'] = 0;
            }
            if (!array_key_exists('ads_cost', $retVal[$key])) {
                $retVal[$key]['ads_cost'] = 0;
            }
            $adsCost = 0;
            if (!empty($values)) {
                $adsCosts = $connection->table('product_cost_ads')
                                ->whereIn('product_id', $values)
                                ->where('date', '>=', $filter['query_from'])
                                ->where('date', '<=', $filter['query_to'])
                                ->select(\DB::raw("sum(cost) as cost"))
                                ->get();
                if (count($adsCosts) > 0 && !empty($adsCosts[0]->cost)) {
                    $adsCost = $adsCosts[0]->cost;
                }
            }
            
            $retVal[$key]['product'] = $retVal[$key]['product'] + $productCount;
            $retVal[$key]['order'] = $retVal[$key]['order'] + $totalSale;
            $retVal[$key]['amount'] = $retVal[$key]['amount'] + $totalRevenue;
            $retVal[$key]['cost'] = $retVal[$key]['cost'] + $totalCost;
            $retVal[$key]['refund'] = $retVal[$key]['refund'] + $totalRefund;
            $retVal[$key]['ads_cost'] = $retVal[$key]['ads_cost'] + $adsCost;
        }
    }

    private function getProduct($connection, $filter, $users) {
        $query = $connection->table('report_product_author as rpa')
                            ->where('rpa.type', '=', $filter['type']);
        $query->join('users as u', 'u.id', '=', 'rpa.product_author_id');
        $query->whereBetween('rpa.product_change_log', [$filter['query_from'], $filter['query_to']]);
        $query->whereIn('u.email', $users);
        $query->whereNotIn('u.email', $this->ignoreEmails);
        $columns = ['u.email', 'rpa.product_id', 'rpa.product_change_log', 'rpa.product_author_id'];
        $items = $query->get($columns);
        $productByEmails = [];
        foreach ($items as $item) {
            if (isset($item->email)) {
                if (!array_key_exists($item->email, $productByEmails)) {
                    $productByEmails[$item->email] = [];
                }
                $productByEmails[$item->email][] = $item->product_id;
            }
        }
        return $productByEmails;
    }

    private function buildMarket($filter) {
        $market = $this->userByLocates;
        if (array_key_exists('locate', $filter)) {
            $market = [];
            $market[$filter['locate']] = $this->userByLocates[$filter['locate']];
        }
        if (array_key_exists('email', $filter)) {
            $userLoops = $market;
            $market = [];
            foreach ($userLoops as $key => $values) {
                foreach($values as $email) {
                    if ($email == $filter['email']) {
                        $market[$key] = [$email];
                    }
                }
            }
        }
        return $market;
    }

    private function buildFilter($request) {
        $params = ['email', 'locate', 'type'];
        $retVal = [];
        foreach ($params as $param) {
            if ($request->has($param)) {
                $retVal[$param] = $request->input($param);
            }
        }
        $sort = $request->has('sort') ? $request->input('sort') : '';
        if ($sort != '') {
            $retVal['sort'] = $sort;
            $arrOrders = explode('-', $retVal['sort']);
            $retVal['orders'] = [$arrOrders[0] => $arrOrders[1]];
        }
        $currentDate = new \DateTime();
        $fromDate = $request->has('dateFrom') ? \DateTime::createFromFormat('d/m/Y', $request->input('dateFrom')) : $currentDate;
        $toDate = $request->has('dateTo') ? \DateTime::createFromFormat('d/m/Y', $request->input('dateTo')) : $currentDate;
        $retVal['query_from'] = $fromDate->setTime(0, 0, 0);
        $retVal['query_to'] = $toDate->setTime(23,59,59);
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function reportProductsSource (Request $request)
    {
        $retVal = [
            'status' => 'fail',
            'result' => []
        ];
        $locale = $request->get('locale', 'us');
        $this->setConnection($locale);
        $filter = $this->buildFilter($request);
        $sources = self::table('report_product_author as rpa')
            ->join('product as p', 'p.id', '=', 'rpa.product_id')
            ->join('report_product_sourcing as rps', 'rps.id', '=', 'rpa.source_id')
            ->where('status', "ACTIVE")
            ->whereNull('deleted_at')
            ->where('p.created_at', '>=', $filter['query_from'])
            ->where('p.created_at', '<=', $filter['query_to'])
            ->select(
                    DB::raw('COUNT(sb_rpa.product_id) as count_products'),
                    'rpa.source_id',
                    'rps.name as source_name'
            )
            ->groupBy('rpa.source_id')
            ->get();
        $countProductsSource = self::table('order as o')
            ->join('order_item as oi', 'oi.order_id', '=', 'o.id')
            ->join('report_product_author as rpa', 'oi.product_id', '=', 'rpa.product_id')
            ->where('o.payment_status', 'PAID')
            ->where('o.created_at', '>=', $filter['query_from'])
            ->where('o.created_at', '<=', $filter['query_to'])
            ->select(
                DB::raw('COUNT(sb_oi.quantity) as sale_products'),
                DB::raw('SUM(sb_oi.quantity * sb_oi.price) as revenue'),
                'rpa.source_id'
            )
            ->groupBy('rpa.source_id')
            ->get();
        $countProductsSourceKeyBySourceId = [];
        foreach ($countProductsSource as $item) {
            $countProductsSourceKeyBySourceId[$item->source_id] = [
                'sale_products' => $item->sale_products,
                'revenue' => $item->revenue,
            ];
        }
        foreach ($sources as $source) {
            $source->sale_products = $countProductsSourceKeyBySourceId[$source->source_id]['sale_products'];
            $source->revenue = $countProductsSourceKeyBySourceId[$source->source_id]['revenue'];
        }
        $retVal['status'] = 'successful';
        $retVal['result'] = $sources;
        return $retVal;
    }
}
