<?php

namespace Modules\Reports\Controllers;

use DateInterval;
use DatePeriod;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Reports\Models\DesignJob;
use Modules\Reports\Models\OrderItem;

class ReportDesignController extends Controller
{
    public function indexReportDesign()
    {
        return view('reports::design.report-design.index');
    }

    public function getCountDesign(Request $request)
    {
        list($locale, $start, $end, $startInput, $endInput) = $this->getInputCommon($request);
        $count = $this->countOrderHasDesign($start, $end);
        $url = env('APP_URL') .
            '/central/report/performance/design/need-design?locale=' . $locale . '&start=' . $startInput . '&end=' . $endInput;
        $responseCentral = $this->triggerAsyncRequest($url);
        $reDesign = 0;
        $needDesign = 0;
        if (isset($responseCentral['status']) && $responseCentral['status'] == 'successful' && isset($responseCentral['reDesign']) && isset($responseCentral['needDesign'])) {
            $reDesign = $responseCentral['reDesign'];
            $needDesign = $responseCentral['needDesign'];
        }

        return response()->json([
            'status' => 'successful',
            'hasDesign' => $count,
            'reDesign' => $reDesign,
            'needDesign' => $needDesign,
            'start' => $start,
            'end' => $end
        ]);
    }

    public function getCountNeedDesign(Request $request)
    {
        list($locale, $start, $end, $startInput, $endInput) = $this->getInputCommon($request);
        $filters = [
            'start' => $start,
            'end' => $end,
            'locale' => $locale
        ];
        $data = $this->getDesignJobsByFilters($filters);
        $countReDesign = 0;
        $countNeedDesign = 0;
        foreach ($data as $item) {
            if ($item->creator_id) {
                $countReDesign++;
            } else {
                $countNeedDesign++;
            }
        }

        return response()->json([
            'status' => 'successful',
            'reDesign' => $countReDesign,
            'needDesign' => $countNeedDesign,
            'total' => count($data)
        ]);
    }

    public function buildConversionRate(Request $request)
    {
        list($locale, $start, $end, $startInput, $endInput) = $this->getInputCommon($request);
        $filters = [
            'start' => $start,
            'end' => $end
        ];
        $ordersCollection = $this->getOrdersByFilters($filters);
        $result = [
            'availableDesign' => [],
            'needDesign' => []
        ];
        $arrayKeysDateForHasDesign = $this->getKeysDate($start, $end);
        $arrayKeysDateForNoDesign = $this->getKeysDate($start, $end);
        foreach ($ordersCollection as $day => $orders) {
            $orderIds = [];
            foreach ($orders as $order) {
                $orderIds[] = $order->id;
            }
            $numberOrder = $this->getNumberOrderHasDesign($orderIds);
            $clicks = $this->getNumberClickOrderHasDesign($orderIds, $filters);
            $result['availableDesign'][$day] = [
                'number_order' => $numberOrder,
                'number_click' => $clicks
            ];
            $arrayKeysDateForHasDesign = array_filter($arrayKeysDateForHasDesign, function ($item) use ($day) {
                return $item != $day;
            });
        }
        if (count($arrayKeysDateForHasDesign) > 0) {
            foreach ($arrayKeysDateForHasDesign as $ele) {
                $result['availableDesign'][$ele] = [
                    'number_order' => 0,
                    'number_click' => 0
                ];
            }
        }
        uksort($result['availableDesign'], function($a, $b) {
            $dateA = DateTime::createFromFormat('d-m-Y', $a);
            $dateB = DateTime::createFromFormat('d-m-Y', $b);

            if ($dateA == $dateB) {
                return 0;
            }
            return $dateA < $dateB ? -1 : 1;
        });

        $urlGetOrderNeedDesign = env('APP_URL') .
            '/central/report/performance/design/no-need-design/conversion-rate?locale=' . $locale . '&start=' . $startInput . '&end=' . $endInput;
        $response = $this->triggerAsyncRequest($urlGetOrderNeedDesign);
        if (isset($response['status']) && $response['status'] == 'successful' && isset($response['result'])) {
            $ordersByDateNeedDesign = $response['result'];
            foreach ($ordersByDateNeedDesign as $day => $orders) {
                $orderIds = [];
                foreach ($orders as $ele) {
                    if (!in_array($ele['order_id'], $orderIds)) {
                        $orderIds[] = $ele['order_id'];
                    }
                }
                $clicks = $this->getNumberClickOrderNoDesign($orderIds, $filters);
                $result['needDesign'][$day] = [
                    'number_order' => count($orderIds),
                    'number_click' => $clicks
                ];
                $arrayKeysDateForNoDesign = array_filter($arrayKeysDateForNoDesign, function ($item) use ($day) {
                    return $item != $day;
                });
            }
            if (count($arrayKeysDateForNoDesign) > 0) {
                foreach ($arrayKeysDateForNoDesign as $ele) {
                    $result['needDesign'][$ele] = [
                        'number_order' => 0,
                        'number_click' => 0
                    ];
                }
            }
        }
        uksort($result['needDesign'], function($a, $b) {
            $dateA = DateTime::createFromFormat('d-m-Y', $a);
            $dateB = DateTime::createFromFormat('d-m-Y', $b);

            if ($dateA == $dateB) {
                return 0;
            }
            return $dateA < $dateB ? -1 : 1;
        });

        return response()->json([
            'status' => 'successful',
            'inputs' => [
                'start' => $start,
                'end' => $end,
                'locale' => $locale
            ],
            'result' => $result
        ]);
    }

    private function getInputCommon($request, $isAds = false)
    {
        $locale = $request->get('locale', 'us');
        $startInput = $request->has('start') ? $request->get('start') : date('d/m/Y', time() - 7 * 86400);
        $endInput = $request->has('start') ? $request->get('end') : date('d/m/Y', time());
        $dateFrom = \DateTime::createFromFormat('d/m/Y', $startInput);
        $dateTo = \DateTime::createFromFormat('d/m/Y', $endInput);
        if ($isAds) {
            $start = $dateFrom->format('Y-m-d');
            $end = $dateTo->format('Y-m-d');
        } else {
            $start = $dateFrom->format('Y-m-d 00:00:00');
            $end = $dateTo->format('Y-m-d 23:59:59');
        }

        return [$locale, $start, $end, $startInput, $endInput];
    }

    private function getDesignJobsByFilters($filters)
    {
        $start = $filters['start'];
        $end = $filters['end'];
        $locale = $filters['locale'];

        return DesignJob::query()->where('order_created_at', '>=', $start)
            ->where('order_created_at', '<=', $end)
            ->where('local_code', $locale)
            ->get();
    }

    private function getOrdersByFilters($filters)
    {
        $start = $filters['start'];
        $end = $filters['end'];

        return DB::table('order')
            ->select(DB::raw('DATE_FORMAT(created_at, "%d-%m-%Y") as order_date'), 'order.id')
            ->where('payment_status', 'PAID')
            ->where('created_at', '>=', $start)
            ->where('created_at', '<=', $end)
            ->get()
            ->groupBy('order_date');
    }

    private function getKeysDate($start, $end, $isAds = false)
    {
        $start = new DateTime($start);
        $end = new DateTime($end);
        $interval = new DateInterval('P1D');
        $period = new DatePeriod($start, $interval, $end);
        $arrKeysDate = [];
        foreach ($period as $date) {
            if ($isAds) {
                $arrKeysDate[] = $date->format('Y-m-d');
            } else {
                $arrKeysDate[] = $date->format('d-m-Y');
            }
        }

        return $arrKeysDate;
    }

    private function getNumberOrderHasDesign($orderIds)
    {
        $orderIdsOutput = OrderItem::query()->whereIn('order_id', $orderIds)
            ->join('product_n_design as pnd', 'order_item.product_id', '=', 'pnd.product_id')
            ->pluck('order_item.order_id')->toArray();

        return count(array_unique($orderIdsOutput));
    }

    private function getNumberClickOrderHasDesign($orderIds, $filters)
    {
        $start = $filters['start'];
        $end = $filters['end'];
        $productIds = OrderItem::query()->whereIn('order_id', $orderIds)
            ->join('product_n_design as pnd', 'order_item.product_id', '=', 'pnd.product_id')
            ->pluck('order_item.product_id')->toArray();

        return DB::table('user_viewed')
            ->whereIn('target', $productIds)->where('type', 'product')
            ->where('created_at', '>=', $start)
            ->where('created_at', '<=', $end)
            ->count();
    }

    private function getNumberClickOrderNoDesign($orderIds, $filters)
    {
        $start = $filters['start'];
        $end = $filters['end'];

        $productIds = OrderItem::query()->whereIn('order_id', $orderIds)
            ->leftJoin('product_n_design', 'order_item.product_id', '=', 'product_n_design.design_id')
            ->whereNull('product_n_design.design_id')
            ->pluck('order_item.product_id')->toArray();

        return DB::table('user_viewed')
            ->whereIn('target', $productIds)->where('type', 'product')
            ->where('created_at', '>=', $start)
            ->where('created_at', '<=', $end)
            ->count();
    }

    public function getConversionRateNoNeedDesign(Request $request)
    {
        list($locale, $start, $end, $startInput, $endInput) = $this->getInputCommon($request);
        $data = DB::table('design_job')
            ->select(DB::raw('DATE_FORMAT(order_created_at, "%d-%m-%Y") as order_date'), 'design_job.*')
            ->where('order_created_at', '>=', $start)
            ->where('order_created_at', '<=', $end)
            ->where('local_code', $locale)
            ->get()
            ->groupBy('order_date');

        return response()->json([
            'status' => 'successful',
            'result' => $data
        ]);
    }

    protected function triggerAsyncRequest($url, $method = "GET", $params = [], $headers = [], $timeout = 120) {
        $channel = curl_init();
        curl_setopt($channel, CURLOPT_URL, $url);
        // curl_setopt($channel, CURLOPT_NOSIGNAL, 1);
        curl_setopt($channel, CURLOPT_TIMEOUT, $timeout);
        curl_setopt($channel, CURLOPT_RETURNTRANSFER, 1);
        if($method == "post" || $method == "POST") {
            curl_setopt($channel, CURLOPT_POST, true);
            curl_setopt($channel, CURLOPT_POSTFIELDS, json_encode($params));
        }
        if ($method == "put" || $method == "PUT") {
            curl_setopt($channel, CURLOPT_CUSTOMREQUEST, "PUT");
            curl_setopt($channel, CURLOPT_POSTFIELDS, json_encode($params));
        }
        if ($headers) {
            curl_setopt($channel, CURLOPT_HTTPHEADER, $headers);
        }
        curl_setopt($channel, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($channel, CURLOPT_SSL_VERIFYPEER, 0);
        $data = curl_exec($channel);
        if (curl_errno($channel)) {
            \Log::error('triggerAsyncRequest', [curl_error($channel)]);
        }
        curl_close($channel);
        return json_decode($data, true);
    }

    private function countOrderHasDesign($start, $end)
    {
        $orderIds = DB::table('design')->whereNotIn('source', ['design', 'auto_fulfill'])
            ->join('product_n_design as pnd', 'design.id', '=', 'pnd.design_id')
            ->join('order_item as ot', 'pnd.product_id', '=', 'ot.product_id')
            ->join('order', 'order.id', '=', 'ot.order_id')
            ->where('order.created_at', '>=', $start)
            ->where('order.updated_at', '<=', $end)
            ->where('order.payment_status', 'PAID')
            ->pluck('order.id')->toArray();

        return count(array_unique($orderIds));
    }

    public function countAdsPerformance(Request $request)
    {
        list($locale, $start, $end, $startInput, $endInput) = $this->getInputCommon($request, true);
        $clickOfProductHasDesign = $this->getClicksProduct($start, $end);
        $clickOfProductOfSeller = $this->getClicksProduct($start, $end, true);
        $arrayKeysDateForHasDesign = $this->getKeysDate($start . " 00:00:00", $end . " 23:59:59", true);
        $arrayKeysDateForSeller = $this->getKeysDate($start . " 00:00:00", $end . " 23:59:59", true);
        $clicksProductHasDesign = $this->formatDataByDate($clickOfProductHasDesign, $arrayKeysDateForHasDesign);
        $clicksProductOfSeller = $this->formatDataByDate($clickOfProductOfSeller, $arrayKeysDateForSeller);
        uksort($clicksProductHasDesign, function($a, $b) {
            $dateA = DateTime::createFromFormat('Y-m-d', $a);
            $dateB = DateTime::createFromFormat('Y-m-d', $b);

            if ($dateA == $dateB) {
                return 0;
            }
            return $dateA < $dateB ? -1 : 1;
        });
        uksort($clicksProductOfSeller, function($a, $b) {
            $dateA = DateTime::createFromFormat('Y-m-d', $a);
            $dateB = DateTime::createFromFormat('Y-m-d', $b);

            if ($dateA == $dateB) {
                return 0;
            }
            return $dateA < $dateB ? -1 : 1;
        });

        return response()->json([
            'status' => 'successful',
            'result' => [
                'has_design' => $clicksProductHasDesign,
                'seller' => $clicksProductOfSeller
            ]
        ]);
    }

    private function getClicksProduct($start, $end, $isSeller = false)
    {
        $query = DB::table('click')
            ->select('click.date', DB::raw('sum(clicks) as clicks'))
            ->where('date', '>=', $start)
            ->where('date', '<=', $end);
        if ($isSeller) {
            $query->join('product_n_user as pnu', 'click.product_id', '=', 'pnu.product_id')
                ->join('users', 'pnu.user_id', '=', 'users.id')
                ->where('users.status', 'ACTIVE')
                ->where('users.role', 'SELLER')
                ->whereNotNull('users.seller_token');
        } else {
            $query->leftJoin('product_n_user as pnu', 'click.product_id', '=', 'pnu.product_id')
                ->whereNull('pnu.product_id');
        }

        return $query->groupBy('click.date')
            ->get()
            ->toArray();
    }

    private function formatDataByDate($clicks, $arrayKeysDate)
    {
        $clicksResult = [];
        foreach ($clicks as $item) {
            $date = $item->date;
            if (in_array($date, $arrayKeysDate)) {
                $clicksResult[$date] = $item;
                $arrayKeysDate = array_filter($arrayKeysDate, function ($item) use ($date) {
                    return $item != $date;
                });
            }
        }
        if (count($arrayKeysDate) > 0) {
            foreach ($arrayKeysDate as $ele) {
                $obj = new \stdClass();
                $obj->date = $ele;
                $obj->clicks = 0;
                $clicksResult[$ele] = $obj;
            }
        }

        return $clicksResult;
    }
}
