<?php

namespace Modules\Reviews\Controllers;

use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redis;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Reviews\Controllers\Controller;
use Module;
use Modules\Reviews\Models\Category;
use Modules\Reviews\Models\Comment;
use Modules\Reviews\Models\Option;
use Modules\Reviews\Models\Post;
use Modules\Reviews\Models\Product;
use Modules\Reviews\Models\ProductNCategory;
use Illuminate\Support\Facades\Route;
use Modules\Reviews\Models\CommentLib;

class CommentController extends Controller
{
    public function __construct()
    {
    }

    public function storeComment(Request $request)
    {
        $responseData = [
            'status' => 'fail',
            'result' => []
        ];

        $comments = $request->input('comments');
        if ($comments) {
            foreach ($comments as $valueComment) {
                if (!empty($valueComment['images']) && $valueComment['images'] == '[]') {
                    $valueComment['images'] = null;
                }

                if (isset($valueComment['id']) && $valueComment['id']) {
                    $comment = Comment::find($valueComment['id']);
                    if ($comment) {
                        $comment->fill($valueComment);
                    }
                } else {
                    $comment = new Comment();
                    $comment->fill($valueComment);
                }

                if ($comment->save()) {
                    if (isset($valueComment['target_id']))
                        $this->updateCountComment($valueComment['target_id']);

                    $this->buildCommentBySeller($comment);
                    $responseData['status'] = 'successful';
                    $responseData['result'][] = $comment;
                }
            }
        }
        return $responseData;
    }

    public function getComment($commentIds)
    {
        $responseData = [
            'status' => 'fail',
            'result' => []
        ];

        $commentIds = explode(',', $commentIds);
        $comments = Comment::whereIn('id', $commentIds)->get();

        if ($comments && count($comments))
            $responseData['status'] = 'successful';

        foreach ($comments as $comment) {
            $target = null;
            if ($comment->target_type == 'PRODUCT') {
                $target = Product::select(['id', 'sku', 'name', 'slug'])
                    ->find($comment->target_id);
            } else if ($comment->target_type == 'POST') {
                $target = Post::select(['id', 'name', 'slug'])
                    ->find($comment->target_id);
            }
            if ($target) {
                $responseData['result'][$comment->id] = $target;
            }
        }
        return $responseData;
    }

    public function find(Request $request)
    {
        $responseData = [
            'status' => 'fail',
            'result' => []
        ];

        $filters = $request->input('filters');
        $pageId = $request->input('page_id');
        $filters = json_decode($filters, true);

        $query = Comment::query();
        if (isset($filters['isParent']) && $filters['isParent']) {
            $query->whereNull('parent_id');
        }
        if (isset($filters['has_image'])) {
            if ($filters['has_image'] == 'EXISTS') {
                $query->whereNotNull('images')
                    ->where('images', '!=', '[]');
            } else {
                $query->whereNull('images');
            }
        }
        if (isset($filters['targetId'])) {
            $query->where('target_id', $filters['targetId']);
        }
        if (isset($filters['rating']) && $filters['rating']) {
            if (is_array($filters['rating'])) {
                $query->whereIn('rating', $filters['rating']);
            } else {
                $query->where('rating', $filters['rating']);
            }
        }
        if (isset($filters['target_type']) && $filters['target_type']) {
            $query->where('target_type', $filters['target_type']);
        }
        if (isset($filters['status']) && $filters['status']) {
            $query->where('status', $filters['status']);
        }
        if (isset($filters['isReplied'])) {
            $query->where('is_replied', $filters['isReplied']);
        }
        if (isset($filters['is_verified'])) {
            $query->where('is_verified', $filters['is_verified']);
        }
        if (isset($filters['is_real']) && $filters['is_real'] == 1) {
            $listEmailFake = CommentLib::whereNotNull('email')
                ->where('email', '!=', '')
                ->groupBy('email')
                ->pluck('email')
                ->toArray();

            $query->whereNotIn('email', $listEmailFake)
                ->where(function ($q) {
                    $q->whereNotNull("email");
                    $q->Where("email", '<>', "");
                });
        }
        if (isset($filters['withChildren'])) {
            $query->with('children');
        }
        if (isset($filters['order'])) {
            foreach ($filters['order'] as $key => $value) {
                $query->orderBy($key, $value);
            }
        } else {
            $query->orderBy('id', 'desc');
        }

        if (isset($filters['created_from'])) {
            $query->where('created_at', '>=', $filters['created_from']);
        }
        if (isset($filters['created_to'])) {
            $d = new \DateTime($filters['created_to']);
            $d->setTime(23, 59, 59);
            $query->where('created_at', '<=', $d->format('Y-m-d H:i:s'));
        }

        if (isset($filters['search']) && $filters['search']) {

            preg_match('/^ID:([0-9]+)$/i', $filters['search'], $matches);
            if (isset($matches[1])) {
                $query->where('id', $matches[1]);
            } else {
                $query->where(function ($query) use ($filters) {
                    $query->where('full_name', 'like', '%' . $filters['search'] . '%')
                        ->orWhere('email', 'like', '%' . $filters['search'] . '%')
                        ->orWhere('title', 'like', '%' . $filters['search'] . '%')
                        ->orWhere('content', 'like', '%' . $filters['search'] . '%');
                });
            }
        }

        /* if (isset($filters['author']) && $filters['author']) {
            $query->where(function ($query) use ($filters) {
                $query->where('full_name', 'like', '%' . $filters['author'] . '%')
                    ->orWhere('email', 'like', '%' . $filters['author'] . '%')
                    ->orWhere('title', 'like', '%' . $filters['author'] . '%');
            });
        } */

        $total = $query->count();
        if (!isset($filters['pageSize']) || $filters['pageSize'] <= 0) {
            $filters['pageSize'] = 100;
        }
        $query->limit($filters['pageSize']);
        if (isset($pageId)) {
            $query->offset($pageId * $filters['pageSize']);
        }
        $query->with('parent');

        $result = $query->get();
        foreach ($result as $comment) {
            $count = 0;
            $target = null;
            $target = Product::select(['id', 'name', 'slug', 'image_url'])
                ->find($comment->target_id);
            if ($target) {
                $comment->target = $target;
                $count = Comment::where('target_id', $target->id)->where('status', 'ACTIVE')->count();
            }

            $comment->count = $count;
            if ($comment->user_id) {
                $comment->user = DB::table('users')->select('id', 'name', 'email')->where('id', $comment->user_id);
            }
        }

        $meta = $this->getMetaData($filters, $total);
        if ($result) {
            $responseData = [
                'status' => 'successful',
                'result' => $result,
                'meta' => $meta,
            ];
        }
        return $responseData;
    }

    private function formatData($comments)
    {
        if (!empty($comments)) {
            foreach ($comments as $item) {
                if (!empty($item->images)) {
                    $item->images = json_decode($item->images, true);
                }
            }
        }
        return $comments;
    }

    public function getMetaData($filters, $total)
    {
        $pageId = 0;
        $pageSize = 20;
        if (!empty($filters['pageSize'])) {
            $pageSize = $filters['pageSize'];
        }
        if (!empty($filters['pageId'])) {
            $pageId = $filters['pageId'];
        }
        $meta = [
            'page_id' => $pageId,
            'page_size' => $pageSize,
            'page_count' => 0,
            'has_next' => false,
            'total_count' => $total,
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function deleteComment($commentIds)
    {
        $responseData = [
            'status' => 'fail',
            'result' => []
        ];

        $commentIds = explode(',', $commentIds);
        $targetIds = [];
        foreach ($commentIds as $commentId) {
            $target = Comment::where('id', $commentId)->first(['target_id']);
            if ($target) $targetIds[] = $target->target_id;
        }

        $deleteComments = Comment::whereIn('id', $commentIds)->delete();
        if ($deleteComments) {
            foreach ($targetIds as $targetId) {
                $this->updateCountComment($targetId);
            }

            $responseData['status'] = 'successful';
            $responseData['result'] = $deleteComments;
        }

        return $responseData;
    }

    public function LoadCommentTarget($commentIds)
    {
        $routeParameters = Route::current()->parameters();
        $commentIds = $routeParameters['commentIds'];

        $responseData = [
            'status' => 'fail',
            'result' => []
        ];

        $commentIds = explode(',', $commentIds);
        $comments = Comment::whereIn('id', $commentIds)->get();

        if ($comments && count($comments))
            $responseData['status'] = 'successful';

        foreach ($comments as $comment) {
            $target = null;
            if ($comment->target_type == 'PRODUCT') {
                $target = Product::select(['id', 'sku', 'name', 'slug', 'image_url'])
                    ->find($comment->target_id);
            } else if ($comment->target_type == 'POST') {
                $target = Post::select(['id', 'name', 'slug'])
                    ->find($comment->target_id);
            }

            if ($target) {
                $responseData['result'][$comment->id] = $target;
            }
        }

        return $responseData;
    }

    public function fakeComments(Request $request)
    {
        // Tim san pham co view trong ngay hom truoc
        $yesterday = new DateTime('yesterday');

        $mapCategories = $this->mapCategories();

        $viewsReviews = [
            10 => 5,
            20 => 9,
            200 => 15,
            500 => 20
        ];

        $option = DB::table('option')->where('key', 'reviews::faker::config')->first();
        if ($option) {
            $config = json_decode($option->value, true);
            if (gettype($config) == 'array' && count($config)) {
                $viewsReviews = $config;
            }
        }

        $minClick = 1000000;
        $maxClick = 0;
        foreach (array_keys($viewsReviews) as $value) {
            if($value < $minClick) $minClick = $value;
            if($value > $maxClick) $maxClick = $value;
        }

        /* $productClicks = DB::table('payment_stats')->where('date', $yesterday->format('Y-m-d'))
            ->where('type', 'click')
            ->where('click', '>=', $minClick)
            ->join('product', 'product.id', 'payment_stats.product_id')
            ->select(['product_id', 'click', 'rating_count']);

        $productIdsHasClick = $productClicks->pluck('product_id')->toArray();
        $productClicks = $productClicks->pluck('click', 'product_id')->toArray(); */

        $commentsInLib = CommentLib::where('category_id', '>', 0)->get();
        $commentsInLibAllCategory = CommentLib::where('category_id', -1)->get();

        $commentsFeature = [];
        $commentsFeatureCategories = [];
        $groupCategories = [];

        $commentsFeatureAllCategory = [];
        foreach ($commentsInLibAllCategory as $row) {
            $commentsFeatureAllCategory[] = [
                'full_name' => $row->fullname,
                'title' => $row->title,
                'content' => $row->content,
                'rating' => $row->rating,
                'images' => $row->images
            ];
        }

        foreach ($commentsInLib as $row) {
            // SLUG của review này: str_slug($row->category_slug)
            $thisRowCategoryKey = $row->category_id;
            $commentsFeature[$thisRowCategoryKey][] = [
                'full_name' => $row->fullname,
                'title' => $row->title,
                'content' => $row->content,
                'rating' => $row->rating,
                'images' => $row->images
            ];
        }

        foreach (array_keys($commentsFeature) as $categoryId) {

            $commentsFeature[$categoryId] = array_merge($commentsFeature[$categoryId], $commentsFeatureAllCategory);

            // Tìm category con của row này
            $categoryIds = [$categoryId];
            if (isset($mapCategories[$categoryIds[0]]['descendants']) && count($mapCategories[$categoryIds[0]]['descendants'])) {
                foreach ($mapCategories[$categoryIds[0]]['descendants'] as $descendant) {
                    $categoryIds[] = $descendant['id'];
                }
            }

            foreach ($categoryIds as $id) {
                if (!isset($commentsFeatureCategories[$id])) {
                    $commentsFeatureCategories[$id] = $categoryId;
                }
                if (!in_array($id, $groupCategories)) {
                    $groupCategories[] = $id;
                }
            }
        }

        $productsNCategories = ProductNCategory::select([
                'product_id', 'category.name as category_name', 'category_id', 'product.status', 'product.sold',
                'view_count', 'rating_count'
            ])
            ->whereIn('category_id', $groupCategories)
            ->where('status', 'ACTIVE')
            ->join('product', 'product_id', '=', 'product.id')
            ->join('category', 'category_id', '=', 'category.id')
            ->get();

        $affectUpdateCount = 0;
        $affectCreateCount = 0;
        foreach ($productsNCategories as $product) {
            if (!$product['view_count']) continue;
            if($maxClick < $product['view_count']) {
                $selectedViews = $maxClick;
            } else {
                // Tìm số nhỏ nhất lớn hơn $productClicks[$product->product_id]
                $selectedViews = $minClick;
                foreach ($viewsReviews as $view => $reviews) {
                    if($product['view_count'] >= $view && !$selectedViews) $selectedViews = $view;
                    if($selectedViews && $selectedViews > $view) {
                        $selectedViews = $view;
                    }
                }
            }

            $needFakeReviewsCount = $viewsReviews[$selectedViews];
            // check xem hôm nay có bao nhiêu review
            $todayReviewCount = Comment::query()->where('created_at', '>=', date("Y-m-d", time()) . ' 00:00:00')
                    ->where('target_id', $product->product_id)
                    ->count();

            if($needFakeReviewsCount > $todayReviewCount) {
                $needFakeReviewsCount = $needFakeReviewsCount - $todayReviewCount;

                if($product['rating_count'] < 50) {
                    $keyArray = $commentsFeatureCategories[$product->category_id];
                    $maxComments = count($commentsFeature[$keyArray]);
                    $randCommentsKeys = array_rand($commentsFeature[$keyArray], ($needFakeReviewsCount <= $maxComments) ? $needFakeReviewsCount : $maxComments);
    
                    if (!(gettype($randCommentsKeys) == 'array')) {
                        $randCommentsKeys = [$randCommentsKeys];
                    }
    
                    $commentData = [];
                    foreach ($randCommentsKeys as $key) {
                        $row = $commentsFeature[$keyArray][$key];
                        $stringTime = date("Y-m-d H:i:s", time());
    
                        $commentData[] = [
                            'full_name' => $row['full_name'],
                            'title' => $row['title'],
                            'email' => $row['email'],
                            'content' => $row['content'],
                            'rating' => $row['rating'],
                            'target_type' => 'PRODUCT',
                            'status' => 'ACTIVE',
                            'created_at' => $stringTime,
                            'updated_at' => $stringTime,
                            'target_id' => $product->product_id,
                            'images' => $row['images']
                        ];
                    }
    
                    DB::table('comment')->insert($commentData);
                    // update count in product
                    $this->updateCountComment($product->product_id);
                    $affectCreateCount += count($commentData);
                } else {
                    $latestComments = Comment::orderBy('created_at', 'asc')
                        ->where('target_id', $product->product_id)
                        ->limit($needFakeReviewsCount)
                        ->pluck('id')->toArray();

                    DB::table('comment')
                        ->whereIn('id', $latestComments)
                        ->update([
                            'created_at' => date("Y-m-d H:i:s", time()),
                            'updated_at' => date("Y-m-d H:i:s", time())
                        ]);

                    $affectUpdateCount += count($latestComments);
                }

            }
        }

        return [
            'status' => 'successful',
            'result' => [
                'updated' => $affectUpdateCount,
                'created' => $affectCreateCount
            ]
        ];
    }


    /* public function fakeProductCommentV2(Request $request)
    {
        $responsePayload = [
            'status' => 'successful'
        ];

        $productId = $request->input('id');
    
        if (!$productId) return $responsePayload;


    } */

    protected function getRandomFeature($detalTime = 60) {

        $dayFeature = time() + ($detalTime * 60);
        $timeRand = rand($dayFeature, time());
        
        return date("Y-m-d H:i:s", $timeRand);
    }

    protected function getFeatureComments($productId)
    {
        $ignoreCategoryIds = $this->loadIgnoreSeedingSetting();

        $productCategoryIds = ProductNCategory::query()
            ->where('product_id', $productId)
            ->pluck('category_id')
            ->toArray();

        $commentsFeatureAllCategory = [];
        if(!count(array_intersect($productCategoryIds, $ignoreCategoryIds))) {
            $commentsInLibAllCategory = CommentLib::query()->where('category_id', -1)->get();
            foreach ($commentsInLibAllCategory as $row) {
                $commentsFeatureAllCategory[] = [
                    'full_name' => $row->fullname,
                    'email' => $row->email,
                    'title' => $row->title,
                    'content' => $row->content,
                    'rating' => $row->rating,
                    'images' => $row->images
                ];
            }
        }

        $commentsFeature = [];
        $commentsInLib = CommentLib::query()->whereIn('category_id', $productCategoryIds)->get();
        foreach ($commentsInLib as $row) {
            $commentsFeature[] = [
                'full_name' => $row->fullname,
                'email' => $row->email,
                'title' => $row->title,
                'content' => $row->content,
                'rating' => $row->rating,
                'images' => $row->images
            ];
        }

        return array_merge($commentsFeature, $commentsFeatureAllCategory);
    }

    protected function getProductSale($productId, $period = 30) 
    {
        // $period = 400;
        // $dayAgo = date('Y-m-d H:i:s', strtotime("-$period days", time()));

        // $orderIds = DB::table('order_item')
        //     ->where('product_id', $productId)
        //     ->where('created_at', '>=', $dayAgo)
        //     ->select('order_id')
        //     ->pluck('order_id')
        //     ->toArray();

        // $orderIds = array_unique($orderIds);

        $soldCount = 0;

        $soldProduct = DB::table('sold_product')->where('product_id', $productId)->first();
        if ($soldProduct) {
            $soldCount = $soldProduct->sold_count;
        }

        return $soldCount;
    }

    public function fakeProductComments(Request $request)
    {
        $affectUpdateCount = 0;
        $affectCreateCount = 0;

        $productId = $request->input('id');
        
        $isFakeRecently = $this->checkRecentlyFaked($productId);

        // if product have been gen fake comment recently, dont run script
        if($productId && !$isFakeRecently) { 

            // some categories dont need or must disable seeding reviews
            $ignoreCategoryIds = $this->loadIgnoreSeedingSetting();

            // get product categories
            $productCategoryIds = ProductNCategory::query()
                ->where('product_id', $productId)
                ->pluck('category_id')->toArray();

            $commentsFeature = []; // array comment compatible w product categories

            /* 
            STEP 1:
            get comments in libary can apply to any product categories */

            $commentsFeatureAllCategory = [];

            // check product categories not in ignore categories
            if(!count(array_intersect($productCategoryIds, $ignoreCategoryIds))) { 

                // in comment lib category_id equal -1 is represent for any categories
                $commentsInLibAllCategory = CommentLib::query()->where('category_id', -1)->get();
                foreach ($commentsInLibAllCategory as $row) {

                    // convert to standard form
                    $commentsFeatureAllCategory[] = [
                        'full_name' => $row->fullname,
                        'email' => $row->email,
                        'title' => $row->title,
                        'content' => $row->content,
                        'rating' => $row->rating,
                        'images' => $row->images
                    ];
                }
            }

            /* 
            STEP 2:
            get comment in lib compatible by product cetegoryIds */

            $commentsInLib = CommentLib::query()->whereIn('category_id', $productCategoryIds)->get();
            foreach ($commentsInLib as $row) {
                $commentsFeature[] = [
                    'full_name' => $row->fullname,
                    'email' => $row->email,
                    'title' => $row->title,
                    'content' => $row->content,
                    'rating' => $row->rating,
                    'images' => $row->images
                ];
            }

            $commentsFeature = array_merge($commentsFeature, $commentsFeatureAllCategory);

            /* 
            STEP 3:
            checking product and make insert new comments */

            $product = Product::query()->select([
                    'id', 'status', 'sold',
                    'view_count', 'rating_count'
                ])
                ->where('id', $productId)
                ->where('status', 'ACTIVE')
                ->first();

            if($product) {

                // get product score
                $productScore = DB::table('score_product')->where('id', $productId)->first();
                $productScore = $productScore ? $productScore->score : 0;
    
                $needFakeReviewsCount = $this->getNeedGenerateByScore($productScore);

                // check xem hôm nay có bao nhiêu review
                $todayReviewCount = Comment::where('created_at', '>=', date("Y-m-d", time()) . ' 00:00:00')
                    ->where('target_id', $product->id)
                    ->count();

                /* 
                    if number of need fake greater than comments today 
                    get minus with today count to get new number of need fake.
                */
                if($needFakeReviewsCount && $needFakeReviewsCount > $todayReviewCount) {
                    
                    $needFakeReviewsCount = $needFakeReviewsCount - $todayReviewCount;

                    /* 
                        Nếu lấy được comment fake từ thư viện và sản phẩm cần fake comment (needFakeReviewsCount > 0)
                        thì tạo comment fake như bình thường,
                        nhưng nếu không tìm được comment fake nào phù hợp trong thư viện mà sản phẩm vẫn cần fake
                        thì điều chỉnh lại ngày comment của các comment cũ thành comment mới
                    */
                    if(count($commentsFeature) > 0 && $needFakeReviewsCount > 0) {

                        $maxComments = count($commentsFeature);
                        $randCommentsKeys = array_rand($commentsFeature, ($needFakeReviewsCount <= $maxComments) ? $needFakeReviewsCount : $maxComments);

                        if (!is_array($randCommentsKeys)) {
                            $randCommentsKeys = [$randCommentsKeys];
                        }
        
                        $commentData = [];

                        $_dayAgo = date('Y-m-d', strtotime('-5 days', time()));

                        if ($needFakeReviewsCount > 40) {
                            $_dayAgo = date('Y-m-d', strtotime('-20 days', time()));
                        } else if ($needFakeReviewsCount > 20) {
                            $_dayAgo = date('Y-m-d', strtotime('-10 days', time()));
                        }

                        $dayAgo = new DateTime($_dayAgo);
                        $dayAgo = $dayAgo->format('U');

                        foreach ($randCommentsKeys as $key) {

                            $row = $commentsFeature[$key];

                            $timeRand = rand($dayAgo, time());
                            $stringTime = date("Y-m-d H:i:s", $timeRand);

                            $newData = [
                                'full_name' => $row['full_name'],
                                'title' => $row['title'],
                                'content' => $row['content'],
                                'email' => $row['email'],
                                'rating' => $row['rating'],
                                'target_type' => 'PRODUCT',
                                'status' => 'ACTIVE',
                                'created_at' => $stringTime,
                                'updated_at' => $stringTime,
                                'target_id' => $product->id,
                                'images' => $row['images'],
                                'user_id' => -1 // -1 is fake comment
                            ];

                            $newData["is_verified"] = 0;
                            $rand = (rand(0, 100) < 85 ? 1 : 0);
                            if ($rand == 1) {
                                $newData["is_verified"] = 1;
                            }
                            $commentData[] = $newData;
                        }

                        try {
                            
                            DB::table('comment')->insert($commentData);

                            // update count in product
                            $this->updateCountComment($product->id);
                            $this->buildFakeCommentByUser($product->id, count($commentData));
                            $affectCreateCount += count($commentData);

                        } catch (\Exception $ex) {
                            //throw $th;
                        }
    
                    } else if ($needFakeReviewsCount > 0) {

                        $_dayAgo = date('Y-m-d', strtotime('-5 days', time()));
                        
                        if ($needFakeReviewsCount > 40) {
                            $_dayAgo = date('Y-m-d', strtotime('-20 days', time()));
                        } else if ($needFakeReviewsCount > 20) {
                            $_dayAgo = date('Y-m-d', strtotime('-10 days', time()));
                        }

                        $dayAgo = new DateTime($_dayAgo);
                        $dayAgo = $dayAgo->format('U');

                        $latestComments = Comment::orderBy('created_at', 'asc')
                            ->where('target_id', $product->id)
                            ->limit($needFakeReviewsCount)
                            ->get();

                        foreach ($latestComments as $comment) {

                            $timeRand = rand($dayAgo, time());
                            $stringTime = date("Y-m-d H:i:s", $timeRand);

                            $comment->created_at = $stringTime;
                            $comment->updated_at = $stringTime;

                            $comment->save();
                        }

                        $affectUpdateCount += count($latestComments);

                    }
    
                }
            }

        }
        
        return [
            'status' => 'successful',
            'result' => [
                'updated' => $affectUpdateCount,
                'created' => $affectCreateCount
            ]
        ];
    }

    private function getNeedGenerateByScore($score) {
        if ($score < 10) {
            return 1;
        } elseif ($score >= 10 && $score < 40) {
            return 2;
        } elseif ($score >= 40 && $score < 70) {
            return 3;
        } elseif ($score >= 70) {
            return 4;
        }
    }

    protected function buildFakeCommentByUser($targetId, $limit = 1) {
        $newCommentIds = \DB::table('comment')->where('target_id', $targetId)
                            ->where('target_type', 'PRODUCT')
                            ->orderBy('id', 'DESC')
                            ->limit($limit)
                            ->pluck('id');
        $seller = \DB::table('product_n_user')->where('product_id', $targetId)->first();
        if (!empty($seller) && !empty($newCommentIds)) {
            $insertData = [];
            foreach ($newCommentIds as $commentId) {
                $insertData[] = [
                    'user_id' => $seller->user_id,
                    'product_id' => $targetId,
                    'comment_id' => $commentId,
                    'created_at' => date('Y-m-d H:i:s')
                ];
            }
            if (!empty($insertData)) {
                \DB::table('comment_n_user')->insert($insertData);
            }
        }
    }

    protected function mapCategories()
    {
        $cacheKey = env('APP_LOCALE') . '_map_category_descendants';

        if (Cache::has($cacheKey)) {
            $mapCategories = Cache::get($cacheKey);
        } else {
            $categories = Category::select(['id', 'name', 'sorder', 'image_url', 'parent_id', '_lft', '_rgt', 'depth'])
                ->where('type', 'PRODUCT')
                ->get();

            $mapCategories = [];
            foreach ($categories as $category) {
                $category->append('descendants');
                $mapCategories[$category->id] = $category->toArray();
            }
            unset($categories);

            Cache::put($cacheKey, $mapCategories, 86400);
        }

        return $mapCategories;
    }

    public function importToLib(Request $request)
    {
        // comment_lib
        set_time_limit(0);
        ini_set('memory_limit', '2048M');

        $countRow = 0;

        $file = $request->file('comments');
        if ($file) {
            $baseName = $file->getClientOriginalName();
            $filename = date("Y-m-d_H-i-s", time()) . '_' . $baseName;
            $uploadFolder = public_path('files/import');

            if (!is_dir($uploadFolder)) {
                mkdir($uploadFolder, 0777, true);
            }

            $path = $file->move(public_path('files/import'), $filename);
            if ($path) {
                $categories = Category::select(['id', 'name', 'sorder', 'image_url', 'parent_id', '_lft', '_rgt', 'depth'])
                    ->where('type', 'PRODUCT')
                    ->get();

                $mapCategories = [];
                foreach ($categories as $category) {
                    $category->append('descendants');
                    $slug = str_slug($category->name);

                    if (!$slug) 
                        $slug = mb_strtolower($category->name);

                    $mapCategories[$slug] = $category->toArray();
                }
                unset($categories);

                Excel::selectSheetsByIndex(0)->load($path, function ($reader) use ($mapCategories, &$countRow) {

                    $results = $reader->get();
                    $commentsFeature = [];
                    /*if (get_class($results) == "Maatwebsite\Excel\Collections\SheetCollection" && isset($results[0])) {
                        $results = $results[0];
                    }*/

                    foreach ($results as $row) {
                        $categorySlug = str_slug($row->category);

                        if (!$categorySlug) 
                            $categorySlug = mb_strtolower($row->category);

                        $rowCateId = -1; // default all category
                        if (isset($mapCategories[$categorySlug]['id'])) {
                            $rowCateId = $mapCategories[$categorySlug]['id'];
                        }

                        if ($row->rating) {
                            $commentsFeature[] = [
                                'fullname' => $row->fullname,
                                'title' => $row->title,
                                'email' => $row->email,
                                'content' => $row->content,
                                'rating' => $row->rating,
                                'category_slug' => $categorySlug,
                                'category_id' => $rowCateId
                            ];
                        }
                        
                    }

                    $countRow += count($commentsFeature);
                    DB::table('comment_lib')->insert($commentsFeature);
                });
            }
        }

        return [
            'count_row' => $countRow,
            'status' => 'successful'
        ];
    }

    public function importProductCommentsV2(Request $request)
    {
        set_time_limit(0);
        ini_set('memory_limit', '2048M');

        $commentCount = 0;
        $file = $request->file('comments');
        if ($file) {
            $baseName = $file->getClientOriginalName();
            $filename = date("Y-m-d_H-i-s", time()) . '_' . $baseName;
            $uploadFolder = public_path('files/import');

            if (!is_dir($uploadFolder)) {
                mkdir($uploadFolder, 0777, true);
            }

            $mapCategories = $this->mapCategories();

            $path = $file->move(public_path('files/import'), $filename);
            if ($path) {
                Excel::selectSheetsByIndex(0)->load($path, function ($reader) use ($mapCategories, &$commentCount) {

                    $results = $reader->get();
                    $commentsFeature = [];
                    $groupCategories = [];

                    /*if (get_class($results) == "Maatwebsite\Excel\Collections\SheetCollection" && isset($results[0])) {
                        $results = $results[0];
                    }*/

                    foreach ($results as $row) {
                        $categorySlug = str_slug($row->category);

                        if (!isset($commentsFeature[$categorySlug])) {
                            $commentsFeature[$categorySlug] = [];
                        }

                        $commentsFeature[$categorySlug][] = [
                            'full_name' => $row->fullname,
                            'title' => $row->title,
                            'email' => $row->email,
                            'content' => $row->content,
                            'rating' => $row->rating,
                            'category' => str_slug($row->category)
                        ];

                        if (!in_array($row->category, $groupCategories)) {
                            $groupCategories[] = $categorySlug;
                        }
                    }

                    $categoriesIdsHasComment = [];
                    for ($i = 0; $i < count($groupCategories); $i++) {
                        if (isset($mapCategories[$groupCategories[$i]])) {
                            if (!in_array($mapCategories[$groupCategories[$i]]['id'], $categoriesIdsHasComment)) {
                                $categoriesIdsHasComment[] = $mapCategories[$groupCategories[$i]]['id'];
                            }
                        }
                    }

                    $productsNCategories = ProductNCategory::select(['product_id', 'category.name as category_name', 'category_id', 'product.status', 'product.sold'])
                        ->whereIn('category_id', $categoriesIdsHasComment)
                        ->where('status', 'ACTIVE')
                        ->join('product', 'product_id', '=', 'product.id')
                        ->join('category', 'category_id', '=', 'category.id')
                        // ->limit(100)
                        ->get();

                    foreach ($productsNCategories as $product) {
                        $productCategory = str_slug($product->category_name);
                        $commentsInCate = array_values($commentsFeature[$productCategory]);
                        $maxComments = count($commentsInCate);

                        $randCommentsKeys = [0];
                        if ($maxComments > 1) {
                            if ($product->sold >= 10) {
                                $randCommentsKeys = array_rand($commentsInCate, (20 <= $maxComments) ? rand(15, 20) : $maxComments);
                            } else if ($product->sold >= 1) {
                                $randCommentsKeys = array_rand($commentsInCate, (15 <= $maxComments) ? rand(10, 15) : $maxComments);
                            } else {
                                $randCommentsKeys = array_rand($commentsInCate, (5 <= $maxComments) ? rand(3, 5) : $maxComments);
                            }
                        }

                        $commentData = [];
                        foreach ($randCommentsKeys as $key) {
                            $row = $commentsFeature[$productCategory][$key];

                            $d = new DateTime('first day of this month');
                            $d->modify('-90 day');
                            $d = $d->format('U');

                            $int = rand($d, time());
                            $stringTime = date("Y-m-d H:i:s", $int);

                            $newData = [
                                'full_name' => $row['full_name'],
                                'title' => $row['title'],
                                'content' => $row['content'],
                                'email' => $row['email'],
                                'rating' => $row['rating'],
                                'target_type' => 'PRODUCT',
                                'status' => 'ACTIVE',
                                'created_at' => $stringTime,
                                'updated_at' => $stringTime,
                                'target_id' => $product->product_id
                            ];

                            $newData["is_verified"] = 0;
                            $rand = (rand(0, 100) < 85 ? 1 : 0);
                            if ($rand == 1) {
                                $newData["is_verified"] = 1;
                                if (mt_rand(0,1)) {
                                    $newData["is_verified"] = 2;
                                }
                            }

                            $commentData[] = $newData;
                        }

                        DB::table('comment')->insert($commentData);
                        $commentCount += count($commentData);
                        // update count in product
                        $this->updateCountComment($product->product_id);
                    }

                });

            }
        }

        return [
            'comment_count' => $commentCount,
            'status' => 'successful'
        ];
    }

    public function importProductComments(Request $request)
    {
        $response = [
            'status' => 'fail',
            'result' => []
        ];

        $file = $request->file('comments');
        if ($file) {
            $baseName = $file->getClientOriginalName();
            $filename = date("Y-m-d_H-i-s", time()) . '_' . $baseName;
            $uploadFolder = public_path('files/import');

            if (!is_dir($uploadFolder)) {
                mkdir($uploadFolder, 0777, true);
            }

            $path = $file->move(public_path('files/import'), $filename);
            if ($path) {
                $comments = Comment::select(['full_name', 'email', 'content', 'title'])->get()->toArray();
                $categories = Category::select(['id', 'name', 'sorder', 'image_url', 'parent_id', '_lft', '_rgt', 'depth'])
                    ->where('type', 'PRODUCT')
                    ->get();

                $productsNCategories = ProductNCategory::select(['product_id', 'category_id'])
                    // ->limit(100)
                    ->get();

                $groupProductsNCategories = [];
                $arrayCategories = $categories->toArray();

                $mapCategories = $this->mapCategories();

                foreach ($productsNCategories as $value) {
                    $categoryFiltered = array_filter($arrayCategories, function ($category) use ($value) {
                        return $category['id'] == $value->category_id;
                    });

                    $groupProductsNCategories[$value->product_id][] = array_shift($categoryFiltered);
                }

                $insertData = [];

                Excel::load($path, function ($reader) use (
                    $comments,
                    $groupProductsNCategories,
                    $insertData,
                    $mapCategories,
                    &$response
                ) {
                    $results = $reader->get();
                    $productCountInsertComments = [];
                    $maxCount = 0;

                    foreach ($results as $key => $row) {

                        // if ($key > 5) break;
                        $d = new DateTime('first day of this month');
                        $d = $d->format('U');

                        $int = rand($d, time());
                        $stringTime = date("Y-m-d H:i:s", $int);

                        $commentFeature = [
                            'full_name' => $row->fullname,
                            'title' => $row->title,
                            'email' => $row->email,
                            'content' => $row->content,
                            'rating' => $row->rating,
                            'target_type' => 'PRODUCT',
                            'status' => 'ACTIVE',
                            'created_at' => $stringTime,
                            'updated_at' => $stringTime
                        ];
                        $featureCategory = strtolower($row->category);

                        // Kiểm tra trùng lặp bình luận theo tiêu chí fullname + email + content
                        $hasDuplicate = false;
                        $countComments = count($comments);
                        for ($i = 0; $i < $countComments; $i++) {
                            $commentPointer = $comments[$i];
                            if (
                                $commentPointer['title'] == $commentFeature['title'] &&
                                $commentPointer['email'] == $commentFeature['email'] &&
                                strlen($commentPointer['content']) == strlen($commentFeature['content'])
                            ) {
                                if ($commentPointer['content'] == $commentFeature['content']) {
                                    $hasDuplicate = true;
                                    break;
                                }
                            }
                        }

                        if ($hasDuplicate) continue; // bỏ qua

                        // Tìm trong danh sách sản phẩm nếu có danh mục trùng hoặc là con của danh mục bình luận!
                        $foundValidProduct = $this->findValidProduct($commentFeature, $groupProductsNCategories, $featureCategory, $productCountInsertComments, $maxCount, $mapCategories);
                        if ($foundValidProduct) {
                            $insertData[] = $foundValidProduct;
                        } else {

                            $maxCount++;
                            $foundValidProduct = $this->findValidProduct($commentFeature, $groupProductsNCategories, $featureCategory, $productCountInsertComments, $maxCount, $mapCategories);
                            if ($foundValidProduct) {
                                $insertData[] = $foundValidProduct;
                            }
                        }
                    }

                    DB::table('comment')->insert($insertData);
                    $response = [
                        'status' => 'successful',
                        'result' => $insertData
                    ];
                });
            }
        }

        return $response;
    }

    function findValidProduct($commentFeature, $groupProductsNCategories, $featureCategory, &$productCountInsertComments, &$maxCount, $mapCategories)
    {
        $insertData = null;
        foreach ($groupProductsNCategories as $productId => $categories) {

            $commentFeature['target_id'] = $productId;
            if (!isset($productCountInsertComments[$productId]))
                $productCountInsertComments[$productId] = 0;
            else if ($productCountInsertComments[$productId] + 1 > $maxCount) {
                $maxCount = $productCountInsertComments[$productId];
                continue;
            }

            $isBreak = false;
            for ($i = 0; $i < count($categories); $i++) {
                $category = $categories[$i];

                if (strtolower($category['name']) == $featureCategory) {
                    $insertData = $commentFeature;
                    $productCountInsertComments[$productId]++;
                    $isBreak = true;
                    break;
                } else if (isset($mapCategories[$featureCategory])) {

                    $descendantsCatgories = $mapCategories[strtolower($category['name'])]['descendants'];
                    foreach ($descendantsCatgories as $desCategory) {
                        if (strtolower($desCategory['name']) == $featureCategory) {
                            $insertData = $commentFeature;
                            $productCountInsertComments[$productId]++;
                            $isBreak = true;
                            break;
                        }
                    }
                }
            }

            if ($isBreak) break;
        }

        return $insertData;
    }

    public function updateCountComment($target_id)
    {
        $countComments = Comment::where('target_id', $target_id)
            ->whereNull('parent_id')
            ->where('status', 'ACTIVE')
            ->count();

        if ($countComments) {
            $sumCommentsScore = Comment::where('target_id', $target_id)
                ->whereNull('parent_id')
                ->where('status', 'ACTIVE')
                ->sum('rating');

            Product::where('id', $target_id)->update([
                'rating_count' => $countComments,
                'rating_value' => round($sumCommentsScore / $countComments, 1)
            ]);

        } else {
            Product::where('id', $target_id)->update([
                'rating_count' => 0,
                'rating_value' => 0
            ]);
        }

        return $countComments;
    }

    public function reCalculateRating()
    {
        $products = Product::select(['id'])->get();
        $count = 0;

        foreach ($products as $value) {
            $commentCount = $this->updateCountComment($value->id);
            $count += $commentCount;
        }

        $response = [
            'status' => 'successful',
            'result' => $count
        ];

        return $response;
    }

    public function reCalculateRatingByIds(Request $request)
    {
        $ids = $request->input('ids');

        foreach ($ids as $id) {
            $this->updateCountComment(trim($id));
        }

        return [
            'status' => 'successful'
        ];
    }

    public function loadBestReviews() {
        $bestReviews = Comment::where('status', 'ACTIVE')
            ->select(['full_name', 'title', 'content', 'target_id'])
            ->where('rating', '>=', 4)
            ->where('target_type', '=', 'PRODUCT')
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        $products = Product::whereIn('id', $bestReviews->pluck('target_id')->toArray())
            ->select(['id', 'name', 'slug'])
            ->get()->toArray();

        $cuttingLimit = 60;

        foreach ($bestReviews as &$value) {
            $foundProduct = array_filter($products, function($product) use($value) {
                return $value->target_id == $product['id'];
            });
            if($foundProduct && count($foundProduct)) {
                $target = current($foundProduct);
                if(strlen($target['name']) > $cuttingLimit) {
                    $s = substr($target['name'], 0, $cuttingLimit);
                    $target['name'] = substr($s, 0, strrpos($s, ' ')) . '...';
                }

                $value->target = $target;
            }

            if(strlen($value->content) > $cuttingLimit) {
                $s = substr($value->content, 0, $cuttingLimit);
                $value->content = substr($s, 0, strrpos($s, ' ')) . '...';
            }

            if(strlen($value->title) > $cuttingLimit) {
                $s = substr($value->title, 0, $cuttingLimit);
                $value->title = substr($s, 0, strrpos($s, ' ')) . '...';
            }
        }

        $response = [
            'status' => 'successful',
            'result' => $bestReviews
        ];

        return $response;
    }

    public function mapFullname()
    {
        $comments = Comment::where('full_name', '')
            ->orWhereNull('full_name')
            ->where('status', 'ACTIVE')
            ->select(['id', 'title'])
            ->get();

        $commentLibs = CommentLib::select(['id', 'title', 'fullname'])->get()->toArray();
        $commentLibsFullname = CommentLib::select(['fullname'])->pluck('fullname')->toArray();

        foreach ($comments as $comment) {
            $fullname = '';
            foreach ($commentLibs as $commentLib) {
                if($commentLib['title'] == $comment->title) {
                    $fullname = $commentLib['fullname'];
                    break;
                }
            }
            if (!$fullname) {
                $fullname = $commentLibsFullname[array_rand($commentLibsFullname)];
            }

            if($fullname) {
                Comment::where('id', $comment->id)->update([
                    'full_name' => $fullname
                ]);
            }
            
        }

        return $comments;
    }

    public function buildVerifiedComment() {
        set_time_limit(0);
        ini_set('memory_limit', "4096M");
        $total = Comment::where("target_type", "PRODUCT")->count();
        $count = 0;
        $lastId = 0;
        while ($total > $count) {
            $comments = Comment::where("target_type", "PRODUCT")
                ->select(["id", "target_id", "target_type"])
                ->where("id", ">", $lastId)
                ->limit(1000)
                ->get();
            $listUpdateId = [];
            foreach ($comments as $comment) {
                $rand = (rand(0, 100) < 65 ? 1 : 0);
                if ($rand == 1) {
                    $listUpdateId[] = $comment->id;
                }
                $lastId = $comment->id;
            }
            DB::table('comment')
                ->whereIn('id', $listUpdateId)
                ->update([
                    'is_verified' => 1,
                    'updated_at' => date("Y-m-d H:i:s", time())
                ]);
            $count += 1000;
        }
        return response()->json(["status" => "successful"]);
    }

    private function checkRecentlyFaked($productId)
    {
        $locate = 'US';
        if(env('APP_LOCALE')) $locate = strtoupper(env('APP_LOCALE'));

        $today = date('dmY');
        $yesterday = date('dmY',strtotime("-1 days"));
        $beforeYesterday = date('dmY',strtotime("-2 days"));

        $expiration = 86400 * 4;

        $key = 'FAKE_COMMENT::' . $locate . '_' . $today . '_' . $productId;
        $key2 = 'FAKE_COMMENT::' . $locate . '_' . $yesterday . '_' . $productId;
        $key3 = 'FAKE_COMMENT::' . $locate . '_' . $beforeYesterday . '_' . $productId;

        if (Redis::exists($key) || Redis::exists($key2) || Redis::exists($key3)) {
            return true;
        } else {
            Redis::set($key, 1);
            Redis::expire($key, $expiration);
            return false;
        }
    }

    public function loadIgnoreSeedingSetting()
    {
        $categoryIds = [];
        $option = Option::query()->where('key', 'ignore.seeding.shared-review')->first(['value']);
        if ($option) {
            $string = str_replace(' ', '', $option->value);
            $setting = explode(',', $string);
            if ($setting && count($setting)) {
                $categoryIds = $setting;
            }
        }
        return $categoryIds;
    }

    public function changeCommentPosition(Request $request)
    {
        $responseData = [
            'status' => 'fail'
        ];

        $action = $request->input('action', 'up');
        $commentId = $request->input('id');

        if ($commentId) {
            $comment = Comment::query()->find($commentId);
            if ($comment) {

                $sortId = $comment->sort_id;
                if ($action == 'up') $sortId++; else if ($action == 'down') $sortId--;

                $attrValue = $comment->is_pin;
                if ($action == 'pin') $attrValue = 1; else if ($action == 'unpin') $attrValue = 0;

                if ($sortId > 0) {
                    DB::table('comment')->where('id', $commentId)
                        ->update([
                            'sort_id' => $sortId,
                            'is_pin' => $attrValue
                        ]);
                }

                if ($comment->save()) {
                    $responseData['status'] = 'successful';
                    $responseData['result'] = $comment;
                }
            }
        }

        return $responseData;
    }

    private function reIndexCommentSortId($targetId)
    {
        $commentIds = Comment::query()
            ->where('target_id', $targetId)
            ->orderBy('sort_id', 'desc')
            ->pluck('id')
            ->toArray();

        foreach ($commentIds as $index => $id) {
            Comment::query()
                ->where('id', $id)
                ->update([
                    'sort_id' => $index + 1
                ]);
        }

    }

    private function getReviewLimit($productId, $seller) {
    
        $dataQuery = DB::table('comment')
            ->select('comment.*')
            ->orderBy('comment.is_pin', 'desc')
            ->orderBy('comment.sort_id', 'desc')
            ->orderBy('comment.created_at', 'desc')
            ->where('comment.status', 'ACTIVE')
            ->whereNull('comment.parent_id');

        if ($seller && $seller->user_id) {
            $dataQuery->join('comment_n_user as cnu', 'cnu.comment_id', '=', 'comment.id')
                        ->where('cnu.user_id', $seller->user_id);
        } else {
            $dataQuery->where('comment.target_id', $productId);
        }

        $count = $dataQuery->count();
        $countLimit = 0;

        if ($count > 50000) {
            $countLimit = $count * 0.15;
        } else if ($count > 100000) {
            $countLimit = $count * 0.1;
        }

        return $countLimit;
    }

    public function getShopReviews(Request $request) 
    {
        $responseData = [
            'status' => 'fail'
        ];

        $productId = $request->input('productId');
        $sellerId = $request->input('seller_id');
        $pageId = intval($request->input('pageId', 0));
        $pageSize = intval($request->input('pageSize', 3));

        if ($productId) $productId = intval($productId);

        if ($productId || !empty($sellerId))
        {
            if (!empty($sellerId)) {
                $seller = DB::table('users')->where('id', $sellerId)->first(['id', 'slug']);
                if ($seller) {
                    $seller->user_id = $seller->id;
                }
            } else {
                $seller = DB::table('product_n_user')->where('product_id', $productId)->select(['user_id'])->first();
            }
            $reviewLimit = $this->getReviewLimit($productId, $seller);

            $dataQuery = DB::table('comment')
                ->select('comment.*')
                ->orderBy('comment.is_pin', 'desc')
                ->orderBy('comment.sort_id', 'desc')
                ->orderBy('comment.created_at', 'desc')
                ->where('comment.status', 'ACTIVE')
                ->whereNull('comment.parent_id');

            /* $cacheKey =  '1fake_email_' . env('APP_LOCALE', 'us'); 
            if (Cache::has($cacheKey)) {
                $fakeEmails = Cache::get($cacheKey);
            } else {
                $fakeEmails = DB::table('comment_lib')
                    ->whereNotNull('email')
                    ->groupBy('email')
                    ->pluck('email')
                    ->toArray();
                Cache::put($cacheKey, $fakeEmails, 24 * 60);
            } */

            /* $dataQuery->whereNotIn('comment.email', $fakeEmails)
                ->where('comment.title', '!=', '')
                ->where('comment.email', '!=', '')
                ->whereNotNull('comment.email'); */

            if ($seller && $seller->user_id) {
                $dataQuery->join('comment_n_user as cnu', 'cnu.comment_id', '=', 'comment.id')
                        ->where('cnu.user_id', $seller->user_id);
            } else {
                $dataQuery->where('comment.target_id', $productId);
            }

            $comments = [];
            $metas = [];
            $ratingCount = [];
            if ($request->has('get_rating')) {
                $ratingCount = $this->getRating($dataQuery);
            }
            if ($reviewLimit) {

                $pageOffset = $pageId * $pageSize;
                $dataQuery = $dataQuery->limit($pageSize)->offset($pageOffset);
                $pageCount = ceil($reviewLimit / $pageSize);

                $comments = $dataQuery->get();
                $metas = [
                    'page_id' => $pageId,
                    'page_size' => $pageSize,
                    'page_count' => $pageCount,
                    'total_count' => $reviewLimit
                ];

            } else {

                $dataQuery = $dataQuery->paginate($pageSize);
                $comments = $dataQuery->items();

                $metas = [
                    'page_id' => $pageId,
                    'page_size' => $pageSize,
                    'page_count' => $dataQuery->lastPage(),
                    'total_count' => $dataQuery->total()
                ];
            }

            $responseData = [
                'status' => 'successful',
                'result' => $comments,
                'meta' => $metas,
                'ratingCount' => $ratingCount
            ];
        }

        return $responseData;
    }

    public function getRating($query)
    {
        $items = [];
        $newQuery = clone $query;
        $result = $newQuery->select('comment.id', 'comment.rating')->get();
        $data = $result->groupBy('rating')->toArray();
        $total = $result->count();
        $totalRate = 0;
        $totalPercent = 100;
        for ($i=5; $i > 0 ; $i--) { 
            $percent = 0;
            if (!empty($data[$i])) {
                $percent = round((count($data[$i]) / $total * 100));
                if ($percent > $totalPercent) {
                    $percent = $totalPercent;
                }
                $totalPercent -= $percent;
                $totalRate += count($data[$i]) * $i;
            }
            $items[] = [
                'rating' => $i,
                'count' => !empty($data[$i]) ? count($data[$i]) : 0,
                'percent' => $percent
            ];
        }
        return [
            'items' => $items,
            'ratingAvg' => !empty($total) ? round(($totalRate / $total), 1) : 0
        ];
    }

    public function ratingShopCount(Request $request)
    {   
        echo 'Not allow';
        exit();
        
        $responseData = [
            'status' => 'fail'
        ];
        $productId = $request->input('productId');

        if ($productId) 
        {
            $ratingCount  = null;
            $cacheKey = decorCacheKey('p::rating::count::' . $productId);

            if (\Cache::has($cacheKey)) {
                
                $ratingCount = \Cache::get($cacheKey);

            } else {
                
                $seller = DB::table('product_n_user')->where('product_id', $productId)->select(['user_id'])->first();
          
                $query = DB::table('comment');

                /* $cacheKey =  '1fake_email_' . env('APP_LOCALE', 'us'); 
                if (Cache::has($cacheKey)) {
                    $fakeEmails = Cache::get($cacheKey);
                } else {
                    $fakeEmails = DB::table('comment_lib')
                        ->whereNotNull('email')
                        ->groupBy('email')
                        ->pluck('email')
                        ->toArray();
                    Cache::put($cacheKey, $fakeEmails, 24 * 60);
                }

                $query->whereNotIn('comment.email', $fakeEmails)
                    ->where('comment.title', '!=', '')
                    ->where('comment.email', '!=', '')
                    ->whereNotNull('comment.email'); */
                
                $query->select('comment.rating', \DB::raw('count(*) as count'))
                    ->where('comment.status', 'ACTIVE')
                    ->where('comment.rating', '>', 0)
                    ->where('comment.status', 'ACTIVE')
                    ->whereNull('comment.parent_id')
                    ->groupBy('comment.rating');
    
                if ($seller && $seller->user_id) {
                    $query->join('product_n_user', 'product_n_user.product_id', '=', 'comment.target_id');
                    $query->where('product_n_user.user_id', $seller->user_id);
                } else {
                    $query->where('comment.target_id', $productId);
                }
        
                $ratingCount = $query->get()->toArray();
                \Cache::put($cacheKey, $ratingCount, 86400);

            }
    
            $rates = [];
            $total = 0;

            foreach ($ratingCount as $item) {
                $total += $item->count;
            }
            
            $curentPercent = 0;
            for ($i = 5; $i >= 1; $i--) {
                $rate = [
                    'rating' => $i,
                    'count' => 0,
                    'percent' => 0
                ];
                if ($total > 0) {
                    foreach ($ratingCount as $rating) {
                        if ($i == $rating->rating) {
                            $rate['count'] = $rating->count;
                            $rate['percent'] = ceil($rate['count'] / $total * 100);
                            if ($curentPercent + $rate['percent'] > 100) {
                                $rate['percent'] = floor($rate['count'] / $total * 100);
                            }
                            $curentPercent += $rate['percent'];
                        }
                    }
                }
                $rates[] = $rate;
            }
    
            $responseData = [
                'status' => 'successful',
                'result' => $rates
            ];
        }

        return $responseData;
    }

    public function hideFakeComments()
    {
        
    }

    protected function buildCommentBySeller($comment)
    {
        if (!empty($comment->target_id) && !empty($comment->id)) {
            $sellerId = \DB::table('product_n_user')->where('product_id', $comment->target_id)->value('user_id');
            if (!empty($sellerId)) {
                $insertData = [
                    'user_id' => $sellerId,
                    'comment_id' => $comment->id,
                    'product_id' => $comment->target_id,
                    'created_at' => date('Y-m-d H:i:s')
                ];
                \DB::table('comment_n_user')->insert($insertData);
            }
        }
    }

}
