<?php

namespace Modules\Seller\Controllers\Service;

use DB;
use Illuminate\Http\Request;
use Modules\Seller\Models\Product;
use Modules\Seller\Models\OrderItem;
use Modules\Seller\Models\ProductSku;
use Modules\Seller\Controllers\Controller;
use Modules\Seller\Models\ProductSkuValue;
use Modules\Seller\Models\ProductNCategory;

class RevenueController extends Controller
{
    public function getOrderItems($request) {
        $query = OrderItem::join('order', 'order.id', '=', 'order_item.order_id')
            ->join('product', 'order_item.product_id', 'product.id')
            ->join('users', 'product.actor_id', 'users.id')
            ->where('order.status', '=', 'FINISHED')
            ->where('order.payment_status', '=', 'PAID')
            ->where('order.amount', '>', 0)
            ->where('users.role', 'SELLER')
            ->where('users.status', 'ACTIVE')
            ->whereNotNull('users.seller_token');

        if ($request->has('codes')) {
            $query->whereIn('order.code', explode(',', $request->get('codes')));
        } else {
            if($request->has('from_date')) {
                $from = \DateTime::createFromFormat('Y-m-d', $request->get('from_date'));
            } else {
                $from = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time()));
            }
    
            if($request->has('to_date')) {
                $to = \DateTime::createFromFormat('Y-m-d', $request->get('to_date'));
            } else {
                $to = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time()));
            }
            if ($request->get('use_created_at')) {
                $query->where('order.created_at', '>=', $from->format('Y-m-d 00:00:00'))
                    ->where('order.created_at', '<=', $to->format('Y-m-d 23:59:59'));
            } else {
                $query->where('order.updated_at', '>=', $from->format('Y-m-d 00:00:00'))
                    ->where('order.updated_at', '<=', $to->format('Y-m-d 23:59:59'));
            }
        }

        if ($request->get('seller_token')) {
            $query->where('users.seller_token', $request->get('seller_token'));
        }

        $result = $query->get([
            'order.code', 'order.discount', 'order_item.id', 'order_item.order_id', 'order_item.product_id', 
            'order_item.quantity', 'order_item.price', 'order.created_at', 
            'order.seller_id', 'users.seller_token', 'order_item.configurations'
        ]);
        $ids = $result->pluck('order_id');
        $sources = DB::table('order_meta')->whereIn('order_id', $ids)->where('key', 'from')->get()->pluck('value', 'order_id')->toArray();

        foreach ($result as &$item) {
            $item->source = '';
            if (isset($sources[$item->order_id])) {
                $item->source = $sources[$item->order_id];
            }
        }

        return $result;
    }

    public function getOrderItemsV2($request) {
        $query = OrderItem::join('order', 'order.id', '=', 'order_item.order_id')
            ->join('product', 'order_item.product_id', 'product.id')
            ->join('users', 'product.actor_id', 'users.id')
            ->join('order_status_time as ost', 'ost.order_id', 'order_item.order_id')
            ->where('order.status', '=', 'FINISHED')
            ->where('ost.status', '=', 'FINISHED')
            ->where('order.payment_status', '=', 'PAID')
            ->where('order.amount', '>', 0)
            ->where('users.role', 'SELLER')
            ->where('users.status', 'ACTIVE')
            ->whereNotNull('users.seller_token');

        if ($request->has('codes')) {
            $query->whereIn('order.code', explode(',', $request->get('codes')));
        } else {
            if($request->has('from_date')) {
                $from = \DateTime::createFromFormat('Y-m-d', $request->get('from_date'));
            } else {
                $from = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time()));
            }
    
            if($request->has('to_date')) {
                $to = \DateTime::createFromFormat('Y-m-d', $request->get('to_date'));
            } else {
                $to = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time()));
            }
            $query->where('ost.created_at', '>=', $from->format('Y-m-d 00:00:00'))
                    ->where('ost.created_at', '<=', $to->format('Y-m-d 23:59:59'));
        }

        if ($request->get('seller_token')) {
            $query->where('users.seller_token', $request->get('seller_token'));
        }

        $result = $query->get([
            'order.code', 'order.discount', 'order_item.id', 'order_item.order_id', 'order_item.product_id', 
            'order_item.quantity', 'order_item.price', 'order.created_at', 'ost.created_at as pinished_at', 
            'order.seller_id', 'users.seller_token', 'order_item.configurations'
        ]);
        $ids = $result->pluck('order_id');
        $sources = DB::table('order_meta')->whereIn('order_id', $ids)->where('key', 'from')->get()->pluck('value', 'order_id')->toArray();

        foreach ($result as &$item) {
            $item->source = '';
            if (isset($sources[$item->order_id])) {
                $item->source = $sources[$item->order_id];
            }
        }

        return $result;
    }

    public function getAllOrder($request) {
        $query = OrderItem::join('order', 'order.id', '=', 'order_item.order_id')
            ->join('product', 'order_item.product_id', 'product.id')
            ->join('users', 'product.actor_id', 'users.id')
            ->where('order.payment_status', '=', 'PAID')
            ->where('order.amount', '>', 0)
            ->where('users.role', 'SELLER')
            ->where('users.status', 'ACTIVE')
            ->whereNotNull('users.seller_token');

        if ($request->has('codes')) {
            $query->whereIn('order.code', explode(',', $request->get('codes')));
        } else {
            if($request->has('from_date')) {
                $from = \DateTime::createFromFormat('Y-m-d', $request->get('from_date'));
            } else {
                $from = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time()));
            }
    
            if($request->has('to_date')) {
                $to = \DateTime::createFromFormat('Y-m-d', $request->get('to_date'));
            } else {
                $to = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time()));
            }
            $query->where('order.updated_at', '>=', $from->format('Y-m-d 00:00:00'))
                    ->where('order.updated_at', '<=', $to->format('Y-m-d 23:59:59'));
        }

        if (!empty($request['except_status'])) {
            $query->whereNotIn('order.status', explode(',', $request->get('except_status')));
        }
        if ($request->get('seller_token')) {
            $query->where('users.seller_token', $request->get('seller_token'));
        }

        $result = $query->get([
            'order.code', 'order.discount', 'order_item.id', 'order_item.order_id', 'order_item.product_id', 
            'order_item.quantity', 'order_item.price', 'order.created_at', 'order.updated_at', 
            'order.seller_id', 'users.seller_token', 'order_item.configurations', 'order.status', 'order.return_fee', 'order.return_reason'
        ]);
        $ids = $result->pluck('order_id');
        $sources = DB::table('order_meta')->whereIn('order_id', $ids)->where('key', 'from')->get()->pluck('value', 'order_id')->toArray();

        foreach ($result as &$item) {
            $item->source = '';
            if (isset($sources[$item->order_id])) {
                $item->source = $sources[$item->order_id];
            }
        }

        return $result;
    }

    public function getProductNUser($productIds) {
        $result = [];
        foreach (array_chunk($productIds, 100) as $partIds) {
            $diffIds = array_diff($partIds, array_keys($result));
            $pnd = DB::table('product_n_user')
                ->whereIn('product_id', $diffIds)
                ->get(['product_n_user.*']);
            foreach ($pnd as $item) {
                $result[$item->product_id] = $item->user_id;
            }
        }

        return $result;
    }

    public function getRevenue(Request $request) {
        set_time_limit(3600);
        
        $response = [
            'status' => 'successful',
            'result' => []
        ];
        $items = $this->getOrderItems($request);
        $result = [];
        $productIds = [];
        foreach ($items as $item) {
            $productIds[] = $item['product_id'];
        }
        
        foreach($items as $item) {
            $date = \DateTime::createFromFormat('Y-m-d H:i:s', $item['created_at'])->format('Y-m-d');
            if (!array_key_exists($item->seller_token, $result)) {
                $result[$item->seller_token] = [];
            }
            if (!isset($result[$item->seller_token][$item->product_id])) {
                $result[$item->seller_token][$item->product_id] = [];
            }
            if (!isset($result[$item->seller_token][$item->product_id][$date])) {
                $result[$item->seller_token][$item->product_id][$date] = [
                    'quantity' => 0,
                    'revenue' => 0,
                    'payment' => 0,
                    'sub_payment' => 0,
                    'source' => $item->source,
                ];
            }
            $result[$item->seller_token][$item->product_id][$date]['quantity'] += $item['quantity'];
            $result[$item->seller_token][$item->product_id][$date]['revenue'] += $item['quantity'] * $item['price'];
            $discountEachItem = 0;
            if (!empty(doubleval($item['discount']))) {
                $discountRate = $this->getDiscountRate($item);
                $discountEachItem = $discountRate * $item['discount'] / $item['quantity'];
            }
            $paymentAmount = $this->calculatePayment($item, $discountEachItem);
            $result[$item->seller_token][$item->product_id][$date]['payment'] += (float) number_format($paymentAmount, 2, '.', '');
            $result[$item->seller_token][$item->product_id][$date]['sub_payment'] += $this->calculatePayment($item);
        }
        $this->formatResult($result);
        $response['status'] = 'successful';
        $response['result'] = $result;

        return response()->json($response);
    }

    public function getRevenueV2(Request $request) {
        set_time_limit(3600);
        
        $response = [
            'status' => 'successful',
            'result' => []
        ];
        $items = $this->getOrderItemsV2($request);
        $result = [];
        $productIds = [];
        foreach ($items as $item) {
            $productIds[] = $item['product_id'];
        }
        
        foreach($items as $item) {
            $date = date('Y-m-d', strtotime($item['pinished_at']));
            if (!array_key_exists($item->seller_token, $result)) {
                $result[$item->seller_token] = [];
            }
            if (!isset($result[$item->seller_token][$item->product_id])) {
                $result[$item->seller_token][$item->product_id] = [];
            }
            if (!isset($result[$item->seller_token][$item->product_id][$date])) {
                $result[$item->seller_token][$item->product_id][$date] = [
                    'quantity' => 0,
                    'revenue' => 0,
                    'payment' => 0,
                    'sub_payment' => 0,
                    'source' => $item->source,
                ];
            }
            $result[$item->seller_token][$item->product_id][$date]['quantity'] += $item['quantity'];
            $result[$item->seller_token][$item->product_id][$date]['revenue'] += $item['quantity'] * $item['price'];
            $discountEachItem = 0;
            if (!empty(doubleval($item['discount']))) {
                $discountRate = $this->getDiscountRate($item);
                $discountEachItem = $discountRate * $item['discount'] / $item['quantity'];
            }
            $paymentAmount = $this->calculatePayment($item, $discountEachItem);
            $result[$item->seller_token][$item->product_id][$date]['payment'] += (float) number_format($paymentAmount, 2, '.', '');
            $result[$item->seller_token][$item->product_id][$date]['sub_payment'] += $this->calculatePayment($item);
        }
        $this->formatResult($result);
        $response['status'] = 'successful';
        $response['result'] = $result;

        return response()->json($response);
    }

    public function getRevenueByOrder(Request $request) {
        set_time_limit(3600);
        
        $response = [
            'status' => 'successful',
            'result' => []
        ];

        $items = $this->getAllOrder($request);
        $result = [];
        
        foreach($items as $item) {
            
            $insertItem = [
                'quantity' => $item['quantity'],
                'revenue' => $item['quantity'] * $item['price'],
                'payment' => 0,
                'sub_payment' => 0,
                'source' => !empty($item['source']) ? $item['source'] : '',
                'seller_token' => $item['seller_token'],
                'product_id' => $item['product_id'],
                'order_id' => $item['order_id'],
                'date' => date('Y-m-d', strtotime($item['created_at'])),
                'created_at' => date('Y-m-d H:i:s', strtotime($item['created_at'])),
                'status' => $item['status'],
                'is_return' => (!empty($item['return_fee']) && $item['return_fee'] > 0 && !empty($item['return_reason'])) ? 1 : 0,
            ];
            $discountEachItem = 0;
            if (!empty(doubleval($item['discount']))) {
                $discountRate = $this->getDiscountRate($item);
                $discountEachItem = $discountRate * $item['discount'] / $item['quantity'];
            }

            $paymentAmount = $this->calculatePayment($item, $discountEachItem);
            $insertItem['payment'] += (float) number_format($paymentAmount, 2, '.', '');
            $insertItem['sub_payment'] += $this->calculatePayment($item);
            $result[] = $insertItem;
        }
        $response['status'] = 'successful';
        $response['result'] = $result;

        return response()->json($response);
    }


    public function getDiscountRate($orderItem) {
        $sameOrderItems = DB::table('order_item')
            ->where('order_id', $orderItem['order_id'])
            ->get();
        
        $subTotal = 0;
        $itemSubTotal = 0;
        foreach ($sameOrderItems as $item) {
            $tmp = $item->price * $item->quantity;
            if ($item->id == $orderItem['id']) {
                $itemSubTotal = $tmp;
            }

            $subTotal += $tmp;
        }

        return (float) number_format($itemSubTotal / $subTotal, 2, '.', '');
    }

    public function formatResult(&$result) {
        foreach ($result as $sellerToken => $products) {
            foreach ($products as $productId => $revenues) {
                foreach ($revenues as $date => $item) {
                    $result[$sellerToken][$productId][$date]['revenue'] = (float) number_format($item['revenue'], 2, '.', '');
                    $result[$sellerToken][$productId][$date]['payment'] = (float) number_format($item['payment'], 2, '.', '');
                }
            }
        }
    }

    public function getCommissionRate($item, $type = 'regular') {
        $retVal = 0.1;
        $default = [
            "regular" => 15,
            "sale" => 10
        ];
        $categoryIds = ProductNCategory::where('product_id', $item['product_id'])
            ->get(['category_id'])->pluck('category_id');
        $config = null;
        foreach ($categoryIds as $categoryId) {
            if (config('seller::commission.' . $categoryId)) {
                $config = config('seller::commission.' . $categoryId, ['config' => ['default' => $default]]);
                break;
            }
        }

        if (!$config) {
            $config = config('seller::commission.default', ['config' => ['default' => $default]]);
        }
        $optionIds = [];
        if ($item['product_sku_id']) {
            $optionIds = ProductSkuValue::where('sku_id', $item['product_sku_id'])
                ->orderByRaw('field(variant_id, "5","7","2","1")', 'asc')
                ->get(['variant_option_id'])
                ->pluck('variant_option_id')
                ->toArray();
        }
        $commissionConfig = !empty($config['config']['default']) ? $config['config']['default'] : $default;
        if (count($optionIds)) {
            $variantKey = implode('-', $optionIds);
            foreach ($config['config'] as $key => $value) {
                if (strpos($variantKey, $key) === 0) {
                    $commissionConfig = $value;
                    break;
                }
            }
        }

        $retVal = $commissionConfig[$type] / 100;

        return $retVal;
    }

    public function calculatePayment($orderItem, $discountEachItem = 0) {
        $retVal = 0;
        $price = $orderItem['price'];
        $highPrice = 0;
        if ($orderItem['product_sku_id']) {
            $sku = ProductSku::find($orderItem['product_sku_id']);
            if ($sku) {
                $highPrice = $sku->high_price;
            }
        } else {
            $product = Product::find($orderItem['product_id']);
            if ($product) {
                $highPrice = $product->high_price;
            }
        }
        //sp sale
        $commissionType = 'regular';
        if ($price < $highPrice) {
            $commissionType = 'sale';
        }
        $commissionRate = $this->getCommissionRate($orderItem, $commissionType);
        // $sellerHasWholeCommissiontRate = [
        //     'facebook' => [
        //         '3669c39b4740ca215b331668cb6e94f7'                
        //     ],
        //     'facebook.com' => [
        //         '3669c39b4740ca215b331668cb6e94f7'
        //     ],
        //     'facebook_ads' => [
        //         '3669c39b4740ca215b331668cb6e94f7'
        //     ],
        // ];
        // if (
        //     !empty($item['seller_token']) && 
        //     !empty($item['source']) && 
        //     !empty($sellerHasWholeCommissiontRate[$item['source']]) && 
        //     in_array($item['seller_token'], $sellerHasWholeCommissiontRate[$item['source']])
        // ) {
        //     $commissionRate = 1;
        // }
        $retVal = round(($price - $discountEachItem) * $orderItem['quantity'] * $commissionRate, 2);

        return $retVal;
    }

    public function getCommisionEachOrder(Request $request) {
        set_time_limit(3600);
        
        $items = $this->getOrderItems($request);

        foreach ($items as &$item) {
            $item->payment_amount = $this->calculatePayment($item);
        }

        return [
            'status' => 'successful',
            'result' => $items
        ];
    }

    public function getRemakeOrder(Request $request) {
        $remakeQuery = \DB::table('order')
                            ->join('order_meta', 'order_meta.order_id', '=', 'order.id')
                            ->where('order_meta.key', 'remake_origin_code')
                            ->whereNotNull('order.parent_id')
                            ->where('order.payment_status', '=', 'PAID');
        if($request->has('from_date')) {
            $from = \DateTime::createFromFormat('Y-m-d', $request->get('from_date'));
        } else {
            $from = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time()));
        }

        if($request->has('to_date')) {
            $to = \DateTime::createFromFormat('Y-m-d', $request->get('to_date'));
        } else {
            $to = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time()));
        }
        $remakeQuery->where('order.updated_at', '>=', $from->format('Y-m-d 00:00:00'))
                ->where('order.updated_at', '<=', $to->format('Y-m-d 23:59:59'));

        $remakeOrdes = $remakeQuery->pluck('order.parent_id');
      
        return response()->json([
            'status' => 'successful',
            'result' => $remakeOrdes
        ]);                
    }
}

