<?php

namespace Modules\Seller\Controllers\System;

use Carbon\Carbon;
use DateInterval;
use DatePeriod;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Seller\Controllers\Controller;
use Modules\Seller\Controllers\Service\RevenueController;
use Modules\Seller\Models\Order;
use Modules\Seller\Models\User;

class DashboardController extends Controller
{
    protected $revenueController;

    public function __construct(RevenueController $revenueController)
    {
        $this->revenueController = $revenueController;
    }

    public function getSellerGrowthRate(Request $request)
    {
        $filters = $this->buildFilterTime($request);
        $dates = $this->getDatesFromRange($filters['start_date'], $filters['end_date']);
        $totalSeller = DB::table('users')
            ->select(DB::raw('COUNT(*) as count_seller'))
            ->where('created_at', '<', $filters['start_date'])
            ->where('role', User::SELLER_ROLE)
            ->whereNotNull('seller_token')
            ->first()->count_seller;

        $statisticTotalSeller = DB::table('users')
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count_seller'))
            ->where('created_at', '<=', $filters['end_date'])
            ->where('created_at', '>=', $filters['start_date'])
            ->where('role', User::SELLER_ROLE)
            ->where('status', User::ACTIVE_STATUS)
            ->whereNotNull('seller_token')
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->get();
        $arrayStatisticSeller = [];
        foreach ($statisticTotalSeller as $item) {
            $arrayStatisticSeller[$item->date] = $item->count_seller;
        }

        $arrayStatisticTotalSeller = [];
        $totalSellerYesterday = $totalSeller;
        foreach ($dates as $date) {
            $arrayStatisticTotalSeller[$date] = [
                'date' => $date,
                'count_seller' => $totalSellerYesterday,
                'growth_rate' => 0
            ];

            foreach ($statisticTotalSeller->toArray() as $itemStatistic) {
                if ($itemStatistic->date == $date) {
                    $arrayStatisticTotalSeller[$date]['count_seller'] =  $totalSellerYesterday + $itemStatistic->count_seller;
                    $arrayStatisticTotalSeller[$date]['growth_rate'] = round((($arrayStatisticTotalSeller[$date]['count_seller'] - $totalSellerYesterday) / $totalSellerYesterday) * 100, 4);
                }
                $totalSellerYesterday = $arrayStatisticTotalSeller[$date]['count_seller'];
            }
        }

        $retval = [
            'status' => 'successful',
            'result' => $arrayStatisticTotalSeller
        ];

        return response()->json($retval);
    }

    public function getProductSubmittedBySeller(Request $request)
    {
        // get data approved
        $filters = $this->buildFilterTime($request);
        $dates = $this->getDatesFromRange($filters['start_date'], $filters['end_date']);

        $totalFeed = DB::table('seller_advertised_product')
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as total_feed'))
            ->where('created_at', '>=', $filters['start_date'])
            ->where('created_at', '<=', $filters['end_date'])
            ->where('status', '=', 1)
            ->whereExists(function ($query) {
                    $query->select(DB::raw(1))
                            ->from('merchant_product as smp')
                            ->whereColumn('smp.product_id', 'seller_advertised_product.product_id');
                })
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->get();

        // get data pending
        $totalPending = DB::table('seller_advertised_product')
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as total_pending'))
            ->where('created_at', '>=', $filters['start_date'])
            ->where('created_at', '<=', $filters['end_date'])
            ->where('status', '=', 0)
            ->groupBy('date')
            ->orderBy('date', 'desc')
            ->get();

        // get data feed

        $totalApproved = DB::table('seller_advertised_product')
                        ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as total_approved'))
                        ->where('created_at', '>=', $filters['start_date'])
                        ->where('created_at', '<=', $filters['end_date'])
                        ->where('status', '=', 1)
                        ->groupBy('date')
                        ->orderBy('date', 'desc')
                        ->get();
        $dataPending = [];
        foreach ($totalPending as $item) {
            $dataPending[$item->date] = $item->total_pending;
        }

        $dataApproved = [];
        foreach ($totalApproved as $item) {
            $dataApproved[$item->date] = $item->total_approved;
        }

        $dataFeed = [];
        foreach ($totalFeed as $item) {
            $dataFeed[$item->date] = $item->total_feed;
        }

        $arrayStatistic = [];
        foreach ($dates as $date) {
            $arrayStatistic[$date] = [
                'date' => $date,
                'number_approved' => isset($dataApproved[$date]) ? $dataApproved[$date] : 0,
                'number_pending' => isset($dataPending[$date]) ? $dataPending[$date] : 0,
                'number_feed' => isset($dataFeed[$date]) ? $dataFeed[$date] : 0
            ];
        }

        $retval = [
            'status' => 'successful',
            'result' => $arrayStatistic
        ];

        return response()->json($retval);
    }

    public function getTotalOrdersBySeller(Request $request)
    {
        $filters = $this->buildFilterTime($request);
        $dates = $this->getDatesFromRange($filters['start_date'], $filters['end_date']);
        $statisticTotalOrder = Order::join('order_item as oi', 'oi.order_id', '=', 'order.id')
            ->join('users as u', 'u.id', '=', 'oi.seller_id')
            ->where('u.role', 'SELLER')
            ->whereNotNull('u.seller_token')
            ->where('order.paid_at', '<=', $filters['end_date'])
            ->where('order.paid_at', '>=', $filters['start_date'])
            ->where('order.payment_status', 'PAID')
            ->select(
                DB::raw('DATE(sb_order.paid_at) as date'),
                'order.id',
                DB::raw('sb_oi.id as order_item_id'),
                'oi.price',
                'oi.product_sku_id',
                'oi.product_id',
                'oi.quantity'
            )
            ->orderBy('date', 'asc')->get();

        foreach ($statisticTotalOrder as &$item) {
            // calculation commission
            $item->commission = $this->revenueController->calculatePayment($item);
            // calculation revenue
            $item->revenue = $item->price * $item->quantity;
        }

        $dates = $this->getDatesFromRange($filters['start_date'], $filters['end_date']);
        $dataTotalOrder = [];
        $dataTotalCommission = [];
        $dataTotalRevenue = [];
        foreach ($statisticTotalOrder as $item) {
            if (!isset($dataTotalOrder[$item->date])) {
                $dataTotalOrder[$item->date] = 0;
                $dataTotalCommission[$item->date] = 0;
                $dataTotalCommission[$item->date] = 0;
                $dataTotalRevenue[$item->date] = 0;
            }
            $dataTotalOrder[$item->date] += 1;
            $dataTotalCommission[$item->date] += $item->commission;
            $dataTotalRevenue[$item->date] += $item->revenue;
        }

        $arrayStatistic = [];
        foreach ($dates as $date) {
            $arrayStatistic[$date] = [
                'date' => $date,
                'count_order' => isset($dataTotalOrder[$date]) ? $dataTotalOrder[$date] : 0,
                'commission' => isset($dataTotalCommission[$date]) ? $dataTotalCommission[$date] : 0,
                'revenue' => isset($dataTotalRevenue[$date]) ? $dataTotalRevenue[$date] : 0,
            ];
        }

        $retval = [
            'status' => 'successful',
            'result' => $arrayStatistic
        ];

        return response()->json($retval);
    }

    public function buildDates()
    {
        $dates = [];
        for ($i = 0; $i < 7; $i++) {
            $dates[] = Carbon::today()->subDays($i)->toDateString();
        }

        return array_reverse($dates);
    }

    public function getRevenueByCategory(Request $request)
    {
        $retval = [
            'status' => 'successful',
            'result' => []
        ];

        $filters = $this->buildFilterTime($request);
        $query = \DB::table('order_item as ot')
                    ->join('order as o', 'o.id', '=', 'ot.order_id')
                    ->join('product_n_category as pnc', 'pnc.product_id', '=', 'ot.product_id')
                    ->join('users as u', 'u.id', '=', 'ot.seller_id')
                    ->where('u.role', 'SELLER')
                    ->where('o.payment_status', 'PAID')
                    ->whereNotNull('u.seller_token')
                    ->where('pnc.is_parent', 0)
                    ->where('o.paid_at', '<=', $filters['end_date'])
                    ->where('o.paid_at', '>=', $filters['start_date'])
                    ->orderBy('paid_at', 'DESC');
        $results = $query->get(['ot.price', 'ot.quantity', 'pnc.category_id', 'ot.seller_id', 'o.paid_at']);
        $days = $this->getDatesFromRange($filters['start_date'], $filters['end_date']);
        if (!empty($results)) {
           
            $results = $results->groupBy('category_id');

            foreach ($results as $key => $value) {
                $data = $this->groupByCategory($value, $days);
                $color = $this->getRandomColor();
                $category = \DB::table('category')->where('id', $key)->first();
                if (empty($category)) {
                    continue;
                }

                if (!empty($data['revenue'])) {
                    $item = $this->buildItem($category, $data['revenue'], $color);
                    $item['quantity'] = $data['quantity'];
                    $retval['result'][] = $item;
                }

                if (!empty($data['quantity'])) {
                    $retval['result'][] = $this->buildItem($category, $data['quantity'], $color, true);
                }
            }
        }
        $retval['days'] = $days;

        return $retval;
    }

    protected function buildItem($category, $data, $color, $isQuantity = false) {
        $item = [
            'color' => $color,
            'data' => $data,
            'name' => $category->name,
            'dataSorting' => [
                'enabled' => true,
                'sortKey' => 'poin.y'
            ]
        ];
      
        if (!$isQuantity) {
            $item['stack'] = '1';
            $item['yAxis'] = 0;
            $item['stack_name'] = 'Doanh thu';
            $item['unit'] = '$';
        } else {
            $item['stack'] = '2';
            $item['yAxis'] = 1;
            $item['showInLegend'] = false;
            $item['stack_name'] = 'Số lượng';
        }
        return $item;
    }

    protected function groupByCategory($items, $days)
    {
        $category = [];
        foreach ($items as $item) {
            if (empty($item->paid_at)) {
                continue;
            }
            $paidAt = date('Y-m-d', strtotime($item->paid_at));

            if (!empty($category[$paidAt])) {
                $category[$paidAt]['quantity'] += $item->quantity;
                $category[$paidAt]['amount'] += ($item->quantity * $item->price);
            } else {
                $category[$paidAt]['quantity'] = $item->quantity;
                $category[$paidAt]['amount'] = ($item->quantity * $item->price);
            }
        }

        $result = [
            'revenue' => [],
            'quantity' => []
        ];
        if (!empty($category)) {
            foreach ($days as $day) {
                if (!empty($category[$day])) {
                    $result['revenue'][] = !empty($category[$day]['amount']) ? $category[$day]['amount'] : 0;
                    $result['quantity'][] = !empty($category[$day]['quantity']) ? $category[$day]['quantity'] : 0;
                } else {
                    $result['revenue'][] = 0;
                    $result['quantity'][] = 0;
                }
            }
        }
        return $result;
    }

    function getRandomColor() {
        // Generate random values for the red, green, and blue components
        $red = mt_rand(0, 255); // Random value between 0 and 255
        $green = mt_rand(0, 255);
        $blue = mt_rand(0, 255);
    
        // Convert the decimal values to hexadecimal
        $redHex = dechex($red);
        $greenHex = dechex($green);
        $blueHex = dechex($blue);
    
        // Ensure that the hexadecimal values are two characters long
        $redHex = str_pad($redHex, 2, "0", STR_PAD_LEFT);
        $greenHex = str_pad($greenHex, 2, "0", STR_PAD_LEFT);
        $blueHex = str_pad($blueHex, 2, "0", STR_PAD_LEFT);
    
        // Concatenate the components to form the color code
        $colorCode = "#" . $redHex . $greenHex . $blueHex;
    
        return $colorCode;
    }

    protected function buildFilterTime($request) {
        $filter = [];
        if (!empty($request['dateFrom']) && !empty($request['dateTo'])) {
            $filter['start_date'] = date('Y-m-d 00:00:00', $request['dateFrom'] + (7 * 3600));
            $filter['end_date'] = date('Y-m-d 23:59:59', $request['dateTo']);
        } else {
            $filter['start_date'] = date('Y-m-d 00:00:00', (time() - 7 * 86400));
            $filter['end_date'] = date('Y-m-d 23:59:59', time());
        }
        return $filter;
    }

    function getDatesFromRange($start, $end, $format = 'Y-m-d') {
        $array = array();
        $interval = new DateInterval('P1D');
    
        $realEnd = new DateTime($end);
        
        $period = new DatePeriod(new DateTime($start), $interval, $realEnd);
        foreach($period as $date) { 
            $array[] = $date->format($format); 
        }
        return $array;
    }

    public function statisticAll(Request $request) {
        $filter = $this->buildFilterTime($request);
        $this->getTotalOrder($filter);

        $result = [
            'total_order' => number_format($this->getTotalOrder($filter), 0, '.', ','),
            'total_revenue' => number_format($this->getTotalRevenue($filter), 2, '.', ','),
            'total_seller' => 0,
            'total_new_seller' => 0
        ];
        $sellerUrl = config('sa.seller_url', '') . '/api/site/get-statistic-seller?auth_key=' . config('sa.seller_auth_key', '');
        foreach ($filter as $field => $value) {
            $sellerUrl .= '&' . $field . '=' . str_replace(' ', '%20', $value);
        }
        
        $response = $this->triggerSyncRequest($sellerUrl, 'GET', $filter);
        if (!empty($response['result'])) {
            if (!empty($response['result']['total_seller'])) {
                $result['total_seller'] = number_format($response['result']['total_seller'], 0, '.', ',');
            }

            if (!empty($response['result']['total_new_seller'])) {
                $result['total_new_seller'] = number_format($response['result']['total_new_seller'], 0, '.', ',');
            }
        }
        
        $retval = [
            'status' => 'successful',
            'result' => $result
        ];

        return response()->json($retval);
    }

    protected function getTotalOrder($filter) {
        return \DB::table('order as o')
                    ->leftJoin('order_item as oi', 'oi.order_id', '=', 'o.id')
                    ->join('users as u', 'u.id', '=', 'oi.seller_id')
                    ->where('o.payment_status', 'PAID')
                    ->whereNotNull('oi.seller_id')
                    ->where('u.role', 'SELLER')
                    ->whereNotNull('u.seller_token')
                    ->where('o.paid_at', '<=', $filter['end_date'])
                    ->where('o.paid_at', '>=', $filter['start_date'])
                    ->groupBy('o.id')
                    ->get(['o.id'])->count();
    }

    protected function getTotalRevenue($filter) {
        return \DB::table('order_item as oi')
                    ->join('order as o', 'oi.order_id', '=', 'o.id')
                    ->join('users as u', 'u.id', '=', 'oi.seller_id')
                    ->where('o.payment_status', 'PAID')
                    ->whereNotNull('oi.seller_id')
                    ->where('u.role', 'SELLER')
                    ->whereNotNull('u.seller_token')
                    ->where('o.paid_at', '<=', $filter['end_date'])
                    ->where('o.paid_at', '>=', $filter['start_date'])
                    ->select(\DB::raw('SUM(sb_oi.quantity * sb_oi.price) as revenue'))
                    ->value('revenue');
    }
}
