<?php

namespace Modules\Seller\Controllers\System;

use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Modules\Seller\Controllers\Controller;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Modules\Seller\Controllers\Frontend\HomeController;
use Modules\Seller\Models\ProductSku;
use Modules\Seller\Models\Product;

class SellerAdvertisedProductsController extends Controller
{
    public function list()
    {
        $sellers = $this->getSellers();

        return view('seller::system.advertised-products.index', ['sellers' => $sellers]);
    }

    public function configCommission()
    {
        return view('seller::system.advertised-products.commission');
    }

    public function getCommissionConfig()
    {
        $categories = $this->getCategories();
        $configs = DB::table('seller_category_commission')
            ->get()
            ->toArray();

        array_map(function($config) use($categories) {
            $category = array_filter($categories, function($category) use ($config) {
                return $config->category_id == $category['id'];
            });

            if (count($category)) {
                $config->name = end($category)['name'];
            }

        }, $configs);

        return $configs;
    }

    protected function getCategories()
    {
        $endpoint = config("sa.api_url") . '/categories';          
        $client = new Client();

        $categories = [];
        try {
            $response = $client->request('GET', $endpoint);
            $jsonResponse = json_decode($response->getBody(), true);

            if (isset($jsonResponse['result'])) {
                $categories = $jsonResponse['result'];
            }

        } catch (ClientException $e) {}

        return $categories;
    }

    public function saveCommission(Request $request)
    {
        $categoryIds = $request->input('category_ids');
        $commission = $request->input('commission', 0);
        $table = 'seller_category_commission';
        $date = date('Y-m-d H:i:s');

        foreach ($categoryIds as $id) {
            if(DB::table($table)->where('category_id', $id)->count()) {
                DB::table($table)
                    ->where('category_id', $id)
                    ->update([
                        'commission' => $commission,
                        'updated_at' => $date,
                    ]);
            } else {
                DB::table($table)->insert([
                    'category_id' => $id,
                    'commission' => $commission,
                    'created_at' => $date,
                    'updated_at' => $date,
                ]);
            }
        }
    }

    public function getSellerHasApproveRequest(Request $request)
    {
        $status = $request->input('status');

        $query = \DB::table('users')
            ->whereNotNull('users.seller_token')
            ->join('seller_advertised_product', 'users.id', 'seller_advertised_product.seller_id')
            ->select(['users.id', 'users.name', 'users.slug'])
            ->groupBy('users.id');

        if ($status) {
            $query->where('seller_advertised_product.status', $status);
        }

        return $query->get(); 
    }

    public function countSellerProducts(Request $request)
    {
        $countResult = [
            'approved' => 0,
            'waiting' => 0
        ];
        $sellerId = $request->input('sellerId', null);

        if ($sellerId) {

            $data = DB::query()
                ->select(
                    'product.approve_advertising',
                    'product.id',
                    'seller_advertised_product.status',
                    'users.id as seller_id'
                )
                ->from(DB::raw("`sb_product` FORCE INDEX(`primary`)"))
                ->where('product.status', 'ACTIVE')
                ->join('seller_advertised_product', 'product.id', 'seller_advertised_product.product_id')
                ->join('users', 'seller_advertised_product.seller_id', 'users.id')
                ->where('seller_advertised_product.seller_id', $sellerId)
                ->get();

            if ($data) {
                for ($i = 0; $i < count($data); $i++) { 
                    $product = $data[$i];
                    if (!$product->status) {
                        $countResult['waiting']++;
                    } else {
                        $countResult['approved']++;
                    }
                }
            }
        }

        return $countResult;
    }

    public function getProducts(Request $request)
    {
        $sellerId = $request->input('sellerId', null);
        $status = $request->input('status', null);

        $query = DB::query()
            ->select([
                'product.id', 
                'product.name', 
                'product.slug', 
                'product.image_url', 
                'product.approve_advertising', 
                'users.name as seller_name', 
                'users.slug as seller_slug', 
                'users.id as seller_id',
            ])
            ->from(DB::raw("`sb_product` FORCE INDEX(`primary`)"))
            ->where('product.status', 'ACTIVE')
            ->join('seller_advertised_product', 'product.id', 'seller_advertised_product.product_id')
            ->join('users', 'seller_advertised_product.seller_id', 'users.id');

            if ($status !== null) {
                $query->where('product.approve_advertising', $status);
            }

        if ($sellerId) {
            $query->where('seller_advertised_product.seller_id', $sellerId);
        }

        $products = $query->paginate(100);

        return $products;
    }

    protected function getSellers()
    {
        return \DB::table('users')
            /* ->where('users.role', 'SELLER')
            ->where('users.status', 'ACTIVE') */
            ->whereNotNull('users.seller_token')
            ->join('seller_advertised_product', 'users.id', 'seller_advertised_product.seller_id')
            ->select(['users.id', 'users.name', 'users.slug'])
            ->groupBy('users.id')
            ->get();
    }

    protected function updateProductAdStatus($productId, $status = 1)
    {
        $endpoint = config("sa.api_url") . '/product/approve-advertising?token=' . \Auth::user()->token;          
        $client = new Client();
        try {
            $response = $client->request('POST', $endpoint, [
                'json' => [
                    'product_id' => $productId,
                    'status' => $status
                ]
            ]);
        } catch (ClientException $e) {
            $response = $e->getResponse();
        }
        return json_decode($response->getBody(), true);
    }

    public function approveProduct(Request $request)
    {
        $productIds = $request->input('product_ids');
        return $this->updateProductAdStatus(implode(',', $productIds), 1);
    }

    public function rejectProduct(Request $request)
    {
        $productIds = $request->input('product_ids');

        foreach ($productIds as $id) {
            DB::table('seller_advertised_product')
                ->where('product_id', $id)
                ->update([
                    'status' => 0
                ]);
        }

        return $this->updateProductAdStatus(implode(',', $productIds), 0);
    }

    public function getAdvertiseInfo(Request $request)
    {
        $products = $request->input('products', []);
        $statusCode = 400;
        $responsePayload = [];

        $homeController = new HomeController($request);
        $client = new Client();
        
        if (count($products)) {
            
            $sellerPercentConfigs = [];
            $productIds = array_filter(array_map(function($product) {
                if (isset($product['id'])) return $product['id'];
            }, $products));

            $sellerIds = DB::table('product')
                ->whereIn('product.id', $productIds)
                ->join('users', 'users.id', 'product.actor_id')
                ->pluck('users.seller_token', 'product.id')
                ->toArray();

            foreach ($products as $product) {

                if (!isset($product['id'])) continue;
                if (empty($product['product_sku_id'])) $product['product_sku_id'] = '';

                $sellerToken = $sellerIds[$product['id']] ?? null;

                $price = $product['price'] ?? 0;
                $highPrice = 0;

                if ($product['product_sku_id'] != '') {
                    $sku = ProductSku::find($product['product_sku_id']);
                    if ($sku) {
                        if (!$price) $price = $sku->price;
                        $highPrice = $sku->high_price;
                    }
                } else {
                    $product = Product::find($product['id']);
                    if ($product) {
                        if (!$price) $price = $product->price;
                        $highPrice = $product->high_price;
                    }
                }
                //sp sale
                $commissionType = 'regular';
                if ($price < $highPrice) {
                    $commissionType = 'sale';
                }

                $data = [
                    'product_id' => $product['id'],
                    'product_sku_id' => $product['product_sku_id'],
                    'seller_token' => $sellerToken,
                    'commission_rate' => $homeController->getCommissionRate([
                        'product_id' => $product['id'],
                        'product_sku_id' => $product['product_sku_id'],
                    ], $commissionType),
                    'budget_percent' => $this->getBudgetPercent($client, $sellerPercentConfigs, $sellerToken)
                ];

                $responsePayload[$product['id'] . '-' .$product['product_sku_id']] = $data;

                if ($statusCode != 200) $statusCode = 200;
            }
        }

        return response($responsePayload, $statusCode);
    }

    protected function getBudgetPercent($client, $sellerPercentConfigs, $sellerId)
    {
        $value = null;

        if (isset($sellerPercentConfigs[$sellerId])) {
            $value = $sellerPercentConfigs[$sellerId];
        } else {
            $sellerUrl = env('SELLER_URL');
            $endpoint = $sellerUrl . '/marketing/budget-config?seller_token=' . $sellerId;    
            try {

                $response = $client->request('GET', $endpoint);
                $data = json_decode($response->getBody(), true);

                if (isset($data['value'])) {
                    $sellerPercentConfigs[$sellerId] = $data['value'];
                    $value = intval($data['value']) / 100;
                }

            } catch (ClientException $e) {}

        }
        
        return $value;
    }
}