<?php 

namespace Modules\Seller\Controllers\System;

use App\Utils\Email;
use DB;
use Auth;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB as FacadesDB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Modules\Seller\Controllers\Controller;
use Modules\Seller\Models\User;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Seller\Models\Product;
use Modules\Seller\Models\SellerSuspend;
use Modules\Seller\Requests\UpdateStatusRequest;

class SellerController extends Controller {
    public function indexFake(Request $request) {
        return view('seller::system.fake-invoice.index');
    }

    public function find(Request $request) {
        $query = User::query();
        $retVal = [
            'status' => 'successful',
            'meta' => [],
            'result' => []
        ];
        if ($request->has('date_from')) {
            $query->where('created_at', '>=', $request->get('date_from') . ' 00:00:00');  
        }

        if ($request->has('date_to')) {
            $query->where('created_at', '<=', $request->get('date_to') . ' 23:59:59');  
        }

        if ($request->has('search')) {
            $query->where(function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->get('search') . '%')
                    ->orWhere('email', 'like', '%' . $request->get('search') . '%');
            });
        }

        if ($request->has('status')) {
            $query->where('status', $request->get('status'));
        }
        $query->whereNotIn('role', ['STAFF', 'ADMIN']);
        $query->orderBy('seller_token', 'DESC');
        $query->whereNotNull('seller_token');

        $total = $query->count();

        $pageId = $request->get('page_id', 0);
        $pageSize = $request->get('page_size', 20);

        $query->limit($pageSize)->offset($pageId * $pageSize);

        $retVal['meta'] = [
            'total_count' => $total,
            'page_id' => intVal($pageId),
            'page_size' => intVal($pageSize),
            'page_count' => ceil($total / $pageSize),
            'has_next' => $pageId < ceil($total / $pageSize) - 1
        ];

        $retVal['result'] = $query->get(['id', 'name', 'email', 'slug', 'status', 'created_at']);
        
        return $retVal;
    }

    public function updateStatus(Request $request, $id) {
        try {
            set_time_limit(3 * 3600);
            FacadesDB::beginTransaction();

            $retval = ['status' => 'failed'];
            $validator = $this->validateInputUpdateStatus($request->all());
            
            if ($validator->fails()) {
                $retval['errors'] = $validator->errors();
    
                return response()->json($retval);
            }
    
            $user = User::where('id', $id)
                ->where('role', 'SELLER')
                ->whereNotNull('seller_token')
                ->first();
            $seller = $user;
    
            if ($user) {
                User::where('id', $id)->update(['status' => $request->get('status')]);
                
                if ($request->get('status') == 'SUSPENDED') {
                    Product::where('actor_id', $id)
                        ->where('status', 'ACTIVE')
                        ->update([
                            'status' => 'PENDING',
                            'note' => DB::raw("CONCAT(note, ' sp bị ẩn do seller bị SUSPENDED')"),
                            'updated_at' => date('Y-m-d H:i:s', time())
                        ]);
                    
                    $dataSuspend = [
                        'seller_id' => $seller->id,
                        'note' => $request->note,
                    ];
    
                    $this->createNoteSuspend($dataSuspend);
                    $this->mailToSeller($seller->email, $seller->name);
                }
                $this->asyncToSeller($user, $request->get('status'));          
                FacadesDB::commit();
                return ['status' => 'successful'];
            }
    
            return [
                'status' => 'fail', 
                'message' => 'invalid seller'
            ];
        } catch (Exception $ex) {
            FacadesDB::rollBack();
            Log::error('Error Update Status:' . $ex->getMessage());
        }
    }

    protected function asyncToSeller($seller, $status) {
        if (!empty($seller->seller_token) && !empty($status)) {
            $url = config('seller::sa.seller_url') . '/api/seller/change-status';
            $params = [
                'seller_token' => $seller->seller_token,
                'status' => $status,
                'auth_key' => config('sa.seller_auth_key', '')
            ];
            $this->triggerSyncRequest($url, 'POST', $params);
        }
    }

    public function validateInputUpdateStatus($data) {
        $listStatus = [
            User::ACTIVE_STATUS,
            User::PENDING_STATUS,
            User::SUSPENDED_STATUS,
        ];

        return Validator::make($data, [
            'status' => [
                'required',
                Rule::in($listStatus)
            ],
            'note' =>  [
                'required_if:status,' . User::SUSPENDED_STATUS,
                'min:5',
                'max:255',
            ],
        ],[
            'status.required' => 'Status cannot be empty',
            'status.in' => 'Invalid status',
            'note.required_if' => 'Reason for suspension cannot be empty',
            'note.string' => 'Reason for suspension must be string',
            'note.min' => 'Reason for suspension must be at least 5 characters',
            'note.max' => 'Reason for suspension must have a maximum of 255 characters',
        ]);
    }

    public function createNoteSuspend($data) {
        SellerSuspend::create($data);
    }

    public function mailToSeller($email, $sellerName) {
        $emailDefault = [
            'driver' => 'ticket',
            'from' => '"Printerval Support" <support@printerval.com>'
        ];

        $emailConfig = config('seller::sa.send_email', $emailDefault);
        $param = [
            'driver' => $emailConfig['driver'],
            'subject' => sprintf(__('URGENT: Seller Account Suspension - Action Required - Check More Details')),
            'from' => $emailConfig['from'],
            'to' => $email,
            'html' => utf8_encode(
                view('seller::system/email/template-email-suspend',[
                        'data' => [
                            'policy_link' => config('seller::sa.policy_link', 'https://printerval.com/creator-license-agreement-n631.html'),
                            'seller_name' => $sellerName
                        ]
                    ]
                )
            ),
            'piority' => 1
        ];

        Email::sendToGearman($param);
    }

    public function getInvoice(Request $request) {
        $retVal = [
            'status' => 'fail',
            'result' => [],
        ];

        $filter = $request->all();
        
        $queryInvoices = \DB::table('seller_invoice')
            ->join('users', 'users.id', '=', 'seller_invoice.seller_id');
        if ($request->has('payment_status')) {
            $queryInvoices->where('payment_status', $request->get('payment_status'));
        }

        if ($request->has('search')) {
            $queryInvoices->where(function ($query) use ($request) {
                $query->where('name', 'like', '%' . $request->get('search') . '%')
                ->orWhere('code', 'like', '%' . $request->get('search') . '%');
            });  
        }
        if ($request->has('date_from')) {
            $queryInvoices->where('seller_invoice.created_at', '>=', $request->get('date_from'));  
        }

        if ($request->has('date_to')) {
            $queryInvoices->where('seller_invoice.created_at', '<=', $request->get('date_to'));  
        }
        $queryInvoices->orderBy('seller_invoice.created_at', 'desc');
        $total = $queryInvoices->count();
        if (isset($filter['page_size']) && $filter['page_size'] > 0) {
            $queryInvoices->limit($filter['page_size']);
            if (isset($filter['page_id'])) {
                $queryInvoices->offset($filter['page_id'] * $filter['page_size']);
            }
        }
        $retVal['meta'] = $this->getMetaData($filter, $total);
        $invoices = $queryInvoices->get(['seller_invoice.*', 'users.name', 'users.email', 'users.payment_info']);
        if($request->has('export_excel') && $request->get('export_excel') == 1) {
            $dataExport = [];

            $dataExport[] = [
                '#', 'Code', 'Shop Name', 'Period', 'Amount', 'Payment Email', 'Payment Type', 'Payment Date', 'Payment Status'
            ];
            foreach ($invoices as $key => $invoice) {
                $paymentInfo = json_decode($invoice->payment_info, true);
                $paymentEmail = !empty($paymentInfo['email']) ? $paymentInfo['email'] : (!empty($paymentInfo['pingpong_email']) ? $paymentInfo['pingpong_email'] : $invoice->email);
                $paymentType = !empty($paymentInfo['email']) ? 'Paypal' : (!empty($paymentInfo['pingpong_email']) ? 'PingPong' : '');
                $dataExport[] = [
                    '#' => $key + 1, 
                    'Code' => $invoice->code, 
                    'Shop Name' => $invoice->name, 
                    'Period' => $invoice->period_from . ' to ' . $invoice->period_to,
                    'Amount' => $invoice->amount, 
                    'Payment Email' => $paymentEmail,
                    'Payment Type' => $paymentType,
                    'Payment Date' => $invoice->payment_date ?  $invoice->payment_date : 'N/A',
                    'Payment Status' => $invoice->payment_status
                ];
            }
            return Excel::create("export-invoice-" . time(), function ($excel) use ($dataExport) {
                $excel->setTitle('export_innvoices')->setCreator('Megaads')->setCompany('Megaads');
                $excel->sheet('Invoice', function ($sheet) use ($dataExport) {
                    foreach ($dataExport as $value) {
                        $sheet->appendRow($value);
                    }
                });
            })->download('xlsx');
        }
        if ($invoices) {
            $retVal['status'] = 'successful';
            $retVal['result'] = $invoices;
        }
        return response()->json($retVal);
    }

    public function paymentOnDue(Request $request) {
        $subDay = $request->get('sub_day', 0);
        $paymentDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-15', time() - $subDay * 86400));
        $today = new \DateTime();
        $fromDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time() - $subDay * 86400));
        $toDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time() - $subDay * 86400));
        if ($today <= $paymentDate) {
            $fromDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time() - $subDay * 86400));
            $fromDate->modify('-1 month');
            $toDate = \DateTime::createFromFormat('Y-m-d', $fromDate->format('Y-m-t'));
        }
        $payment = DB::table('seller_revenue')
            ->where('seller_id', $request->get('seller_id'))
            ->where('date', '<=', $toDate->format('Y-m-d'))
            ->where('status', '=', 'PENDING')
            ->sum('payment');
        
        $total = DB::table('seller_revenue')
            ->where('seller_id', $request->get('seller_id'))
            ->where('status', '=', 'PENDING')
            ->sum('payment');
            
        return [
            'status' => 'successful',
            'result' => [
                'fromDateFormated' => $fromDate->format('d/m/Y'),
                'fromDate' => $fromDate->format('Y-m-d'),
                'toDateFormated' => $toDate->format('d/m/Y'),
                'toDate' => $toDate->format('Y-m-d'),
                'paymentDateFormated' => $paymentDate->format('d/m/Y'),
                'paymentDate' => $paymentDate->format('Y-m-d'),
                'payment' => $payment,
                'total' => $total
            ]
        ];
    }

    public function buildInvoice(Request $request) {
        set_time_limit(3 * 3600);

        $sellers = DB::table('users')
            ->get(['seller_token', 'id', 'status']);
        $sellerStatuses = $sellers->pluck('status', 'id');
        $sellerTokens = $sellers
            ->pluck('seller_token', 'id');
        $buildInvoiceItems = [];
        $subDay = $request->get('sub_day', 0);
        $fromDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time() - $subDay * 86400))->format('Y-m-d');
        $toDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time() - $subDay * 86400))->format('Y-m-d');
        $paymentDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-15', time() - $subDay * 86400))->format('Y-m-d');
        $paidRevenues = [];
        $prepareRevenues = [];
        
        $sumPayments = DB::table('seller_revenue')
                ->select(DB::raw('seller_id, sum(payment) as payment'))
                ->where('status', '=', 'PENDING')
                ->where('date', '<=', $toDate)
                ->groupBy('seller_id')
                ->get();
        foreach ($sumPayments as $item) {
            if ($item->payment >= config('payment.minimum_payout', 15)) {
                $sellerId = $item->seller_id;
                $sellerToken = isset($sellerTokens[$sellerId]) ? $sellerTokens[$sellerId] : null;
                if (($subDay > 30 || (isset($sellerStatuses[$sellerId]) && $sellerStatuses[$sellerId] == 'ACTIVE'))) {
                    $paidRevenues[] = $sellerId;
                } else {
                    $prepareRevenues[] = $sellerId;
                }
                $buildInvoiceItems[] = [
                    'code' => 'PS' . str_pad(time() . rand(1, 1000), 15, 0, STR_PAD_LEFT),
                    'seller_id' => $sellerId,
                    'seller_token' => $sellerToken,
                    'period_from' => $fromDate,
                    'period_to' => $toDate,
                    'amount' => $item->payment,
                    'payment_date' => $paymentDate . ' ' . str_pad(rand(8, 15), 2, '0', STR_PAD_LEFT) . ':' . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT) . ':' . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
                    'payment_status' => ($subDay > 30 || (isset($sellerStatuses[$sellerId]) && $sellerStatuses[$sellerId] == 'ACTIVE')) ? 'PAID' : 'ONHOLD',
                    'created_at' => $paymentDate . ' ' . str_pad(rand(8, 15), 2, '0', STR_PAD_LEFT) . ':' . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT) . ':' . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
                    'updated_at' => $paymentDate . ' ' . str_pad(rand(8, 15), 2, '0', STR_PAD_LEFT) . ':' . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT) . ':' . str_pad(rand(0, 59), 2, '0', STR_PAD_LEFT),
                ];
            }
        }

        foreach (array_chunk($paidRevenues, 500) as $partIds) {
            DB::table('seller_revenue')
                ->whereIn('seller_id', $partIds)
                ->where('date', '<=', $toDate)
                ->where('status', 'PENDING')
                ->update([
                    'status' => 'PAID'
                ]);
        }

        foreach (array_chunk($prepareRevenues, 500) as $partIds) {
            DB::table('seller_revenue')
                ->whereIn('seller_id', $partIds)
                ->where('date', '<=', $toDate)
                ->where('status', 'PENDING')
                ->update([
                    'status' => 'PREPARE'
                ]);
        }

        foreach (array_chunk($buildInvoiceItems, 500) as $partData) {
            DB::table('seller_invoice')->insert($partData); 
        }

        return $buildInvoiceItems;
    }

    public function getMetaData($data, $total)
    {
        $pageId = 0;
        $pageSize = 20;
        if (!empty($data['page_size'])) {
            $pageSize = $data['page_size'];
        }
        if (!empty($data['page_id'])) {
            $pageId = $data['page_id'];
        }
        $meta = [
            'page_id' => intval($pageId),
            'page_size' => intval($pageSize),
            'page_count' => 0,
            'has_next' => false,
            'total_count' => intval($total),
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function getRevenueLastPeriod(Request $request) {
        set_time_limit(3 * 3600);

        $response = ['status' => 'successful', 'result' => [], 'message' => ''];
        $locale = $request->get('locale', 'us');
        $syncUrl = '/seller/revenue';
        $cronUrl = config('app.url') . ($locale !== 'us' ? "/$locale" : "") . $syncUrl;

        $subDay = $request->get('sub_day', 0);
        $paymentDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-15', time() - $subDay * 86400));
        $today = new \DateTime();
        $fromDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time() - $subDay * 86400));
        $toDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-t', time() - $subDay * 86400));
        if ($today <= $paymentDate) {
            $fromDate = \DateTime::createFromFormat('Y-m-d', date('Y-m-01', time() - $subDay * 86400));
            $fromDate->modify('-1 month');
            $toDate = \DateTime::createFromFormat('Y-m-d', $fromDate->format('Y-m-t'));
        }
        $newRequest = new Request();
        $newRequest->merge([
            'from_date' => $fromDate->format('Y-m-d'),
            'to_date' => $toDate->format('Y-m-d'),
            'role' => 'STAFF,SELLER'
        ]);
        $output = call('App\Modules\Seller\Controllers\Service\RevenueController@getRevenue', $newRequest);
        try {
            if ($output && $output['status'] == 'successful') {
                $result = $output['result'];
                $productNUser = DB::table('product_n_user')
                    ->whereIn('product_id', array_keys($result))
                    ->get(['product_id', 'user_id'])
                    ->pluck('user_id', 'product_id')
                    ->toArray();
                $productNDesign = DB::table('product_n_design')
                    ->whereIn('product_id', array_keys($result))
                    ->where('is_primary', 1)
                    ->get(['product_id', 'design_id'])
                    ->pluck('design_id', 'product_id')
                    ->toArray();
                $sellerTokens = DB::table('users')->get(['id', 'seller_token'])->pluck('seller_token', 'id')->toArray();
                $saveData = [];
                foreach ($result as $productId => $dates) {
                    if(isset($productNUser[$productId])) {
                        $sellerId = $productNUser[$productId];
                        foreach ($dates as $date => $sale) {
                            $data = [];
                            $data['seller_id'] = $sellerId;
                            $data['seller_token'] = $sellerTokens[$sellerId];
                            $data['date'] = $date;
                            $data['locale'] = $locale;
                            $data['sale'] = $sale['quantity'];
                            $data['revenue'] = $sale['revenue'];
                            $data['payment'] = $sale['payment'];
                            if (isset($productNDesign[$productId])) {
                                $data['design_id'] = $productNDesign[$productId];
                            }
                            $data['updated_at'] = date('Y-m-d H:i:s', time());
                            $data['created_at'] = date('Y-m-d H:i:s', time());
                            $saveData[] = $data;
                        }
                    }
                }
                $this->saveRevenue($saveData);
                $response['result'] = $saveData;
            } else {
                \Log::error('getRevenueLastPeriod', [$cronUrl]);
                $response['status'] = 'fail';
                $response['message'] = $output;
            } 
        } catch (\Exception $ex) {
            \Log::error('getRevenueLastPeriod', [$ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine()]);
            $response['status'] = 'fail';
            $response['message'] = $ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine();
        }

        return $response;
    }

    public function saveRevenue($data) {
        foreach ($data as $item) {
            $query = DB::table('seller_revenue')
                ->where('seller_id', $item['seller_id'])
                ->where('date', '=', $item['date']);
            if (!empty($item['design_id'])) {
                $query->where('design_id', $item['design_id']);
            }
            $first = $query->first();
            if (!$first) {
                DB::table('seller_revenue')->insert($item);
            } else {
                DB::table('seller_revenue')
                    ->where('id', $first->id)
                    ->update($item);
            }
        }
    }

    public function changeInvoiceStatus(Request $request, $id) {
        $userEmail = Auth::user() ? Auth::user()->email : '';
        $status = $request->get('status');
        $data = [
            'payment_status' => $status, 
            'actor_email' => $userEmail
        ];
        if ($status != 'PAID') {
            $data['payment_date'] = null;
        }
        DB::table('seller_invoice')
            ->where('id', $id)
            ->update($data);
        
        return [
            'status' => 'successful',
        ];
    }

    public function getRevenueBySeller(Request $request) {
        $retval = [
            'status' => 'fail',
            'result' => []
        ];
        if (!empty($request['email'])) {
            $url = config('seller::sa.seller_url') . '/api/seller/get-revenue';
            $params = [
                'email' => $request['email'],
            ];
            if (!empty($request['order_ids'])) {
                $params['order_ids'] = $request['order_ids'];
            }
            $response = $this->triggerSyncRequest($url, 'POST', $params);
            $retval = $response;
        }
        return $retval;
    }

    public function statisticRevenueBySeller(Request $request) {
        $retval = [
            'status' => 'fail',
            'result' => []
        ];
        if (!empty($request['email'])) {
            $seller = User::where('email', $request['email'])->whereNotNull('seller_token')->first();
            $url = config('seller::sa.seller_url') . '/api/seller/statistic-revenue';
            $params = [
                'seller_token' => $seller->seller_token,
            ];
            $response = $this->triggerSyncRequest($url, 'POST', $params);
            $retval = $response;
        }
        return $retval;
    }
}
