global.__dir = __dirname;
const mysql = require('mysql');
const MySQLEvents = require('@rodrigogs/mysql-events');
const readConfig = require('read-config');
const logConfigsObj = readConfig(__dirname + '/config/log.json', { override: false });
const logConfigs = Object.keys(logConfigsObj).map((key) => logConfigsObj[key]);
const Queue = require(__dir + '/helper/queue');
const QueueHandleInsert = new Queue(7, 'Handle Insert');
const logByDB = [];
const knex = [];
const connections = [];
const instances = [];

for (var i = 0; i < logConfigs.length; i++) {
    var logConfig = logConfigs[i];
    var dbName = logConfig.db.database;
    knex[dbName] = require('knex')({
        client: 'mysql',
        connection: logConfig.db
    });
    logByDB[dbName] = logConfig;
}
const SimpleNodeLogger = require('simple-node-logger'),
    opts = {
        logFilePath:'debug.log',
        timestampFormat:'YYYY-MM-DD HH:mm:ss.SSS'
    },
    log = SimpleNodeLogger.createSimpleLogger( opts );


async function clearConnect() {
    if (Object.keys(connections).length > 0) {
        for (let dbName in connections) {
            try {
                await connections[dbName].destroy();
                await connections[dbName].end();
            } catch (e) {
                console.log("Error on clear connections!!!", e);
            }
        }
    }

    if (Object.keys(instances).length > 0) {
        for (let dbName in instances) {
            try {
                await instances[dbName].stop();
            } catch (e) {
                console.log("Error on clear instances connections!!!");
            }
        }
    }

}

const program = async () => {
    await clearConnect();
    for (var i = 0; i < logConfigs.length; i++) {
        var logConfig = logConfigs[i];
        var dbName = logConfig.db.database;
        connections[dbName] = mysql.createConnection(logConfig.db);
        connections[dbName].on('error', function() {
            console.log("Error connection...!!!");
        });
        var includeSchema = {};
        includeSchema[dbName] = logConfig.tables;
        instances[dbName] = new MySQLEvents(connections[dbName], {
            startAtEnd: true, // to record only the new binary logs, if set to false or you didn'y provide it all the events will be console.logged after you start the app
            serverId: i + 3,
            includeSchema: includeSchema
        });
        const instance = instances[dbName];

        await instance.start()
            .then(() => console.log('I\'m running!'))
            .catch(function(err) {
                console.error('Something bad happened', err);
                instance.stop();
                program()
            });

        await instance.addTrigger({
            name: 'monitoring all statments',
            expression: dbName + '.*', // listen to database !!!
            statement: MySQLEvents.STATEMENTS.ALL, // you can choose only insert for example MySQLEvents.STATEMENTS.INSERT, but here we are choosing everything
            onEvent: e => {
                handleAction(e);
            }
        });

        instance.on(MySQLEvents.EVENTS.CONNECTION_ERROR, function(err) {
            log.error("CONNECTION_ERROR");
            instance.stop();
            program()
            log.info(MySQLEvents.EVENTS.CONNECTION_ERROR);
            log.info(err);
        });
        instance.on(MySQLEvents.EVENTS.ZONGJI_ERROR, function(err) {
            log.error("ZONGJI_ERROR");
            instance.stop();
            program()
            log.info(MySQLEvents.EVENTS.ZONGJI_ERROR);
            log.info(err);
        });
        instance.on(MySQLEvents.EVENTS.TRIGGER_ERROR, function(err) {
            log.info(MySQLEvents.EVENTS.TRIGGER_ERROR);
            log.info(err);
        });


    }
};

function handleAction(event) {
    if (typeof logByDB[event.schema] != 'undefined' && logByDB[event.schema]) {
        var listEvent = ['INSERT', 'UPDATE', 'DELETE'];
        var logConfig = logByDB[event.schema];
        var dbName = logConfig.db.database;
        if (listEvent.includes(event.type)) {
            var affectedColumns = event.affectedColumns;
            for (var j in event.affectedRows) {
                var affectedRows = null;
                if (event.type == 'DELETE') {
                    affectedRows = event.affectedRows[j].before;
                } else {
                    affectedRows = event.affectedRows[j].after;
                }
                var affectedData = {};
                for (var i in affectedColumns) {
                    var column = affectedColumns[i];
                    if (typeof affectedRows[column] != 'undefined') {
                        if (typeof affectedRows[column] == 'number' && affectedRows[column] < 0.0000001) {
                            affectedRows[column] = 0;
                        }
                        affectedData[column] = affectedRows[column];
                    }
                }
                let nz_date_string = new Date().toLocaleString("en-US", {
                    timeZone: "Asia/Ho_Chi_Minh"
                });
                let date_nz = new Date(nz_date_string);
                let year = date_nz.getFullYear();
                let month = ("0" + (date_nz.getMonth() + 1)).slice(-2);
                let date = ("0" + date_nz.getDate()).slice(-2);
                let hours = ("0" + date_nz.getHours()).slice(-2);
                let minutes = ("0" + date_nz.getMinutes()).slice(-2);
                let seconds = ("0" + date_nz.getSeconds()).slice(-2);
                let date_time = year + "-" + month + "-" + date + " " + hours + ":" + minutes + ":" + seconds;

                var dataLog = {
                    type: event.type,
                    table: event.table,
                    target_id: typeof affectedRows['id'] != 'undefined' ? affectedRows['id'] : null,
                    data: JSON.stringify(affectedData),
                    created_at: date_time
                }

                pushToInsertQueue(dbName, logConfig.log_table, dataLog);
            }
        }
    }
}


function pushToInsertQueue(dbName, table, data) {
    let task = () => {
        return handleInsertLog(dbName, table, data).then(function (res) {
        }).catch((e) => {
            console.log('pushOdds err', e)
            Promise.resolve();
        });

    };
    QueueHandleInsert.pushTask(task);
}

function handleInsertLog(dbName, table, data) {
    return knex[dbName](table).insert(data);
}

program()
    .then()
    .catch(console.error);
