<?php

namespace Modules\Ticket\Controllers\Services;

use Modules\Ticket\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Modules\Ticket\Models\Ticket;
use Modules\Ticket\Models\TicketItem;
use Modules\Ticket\Models\TicketAssign;
use Modules\Ticket\Helpers\DBConnect;

class TicketAssignService extends Controller
{

    public function store(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('user_id') && $request->has('locate')) {
            $id = $request->input('id');
            $data = [
                'user_id' => $request->input('user_id'),
                'locate' => $request->input('locate'),
                'status' => $request->has('status') ? $request->input('status') : 'active'
            ];
            $columns = ['note', 'from_hours', 'to_hours'];
            foreach ($columns as $column) {
                if ($request->has($column) && $request->input($column) != '') {
                    $data[$column] = $request->input($column);
                }
            }
            if ($request->has('days') && !empty($request->input('days'))) {
                $data['days'] = implode(',', $request->input('days'));
            }
            if ($id != null) {
                $data['id'] = $id;
            }
            if ($id == null) {
                $checkExists = $this->checkExists($data);
                if ($checkExists) {
                    $response['message'] = 'Nhân viên này đã được cấu hình cho ngôn ngữ ' . $data['locate'];
                    return response()->json($response);
                }
                $result = new TicketAssign;
                $result->fill($data);
                $result->save();
            } else {
                $result = TicketAssign::find($id);
                $result->fill($data);
                $result->save();
            }
            $response['status'] = 'successful';
            $response['result'] = $result;
        }
        return response()->json($response);
    }

    private function checkExists($data) {
        $query = TicketAssign::where('user_id', '=', $data['user_id'])
                                ->where('locate', '=', $data['locate']);
        if (isset($data['id'])) {
            $query->where('id', '!=', $data['id']);
        }
        return $query->exists();                              
    }

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $columns = $filter['columns'];
        $query = TicketAssign::buildTicketQuery($filter);
        //Get count
        $queryCount = clone $query;
        $queryCount->select($columns);
        $count = $queryCount->count();
        //Get items
        $filter["page_id"] = $request->input("page_id", 0);
        $filter["page_size"] = $request->input("page_size", 20);
        $pageCount = $this->recordsCountToPagesCount($count, $filter['page_size']);
        
        $items = $this->getItems($query, $filter);
        $items = $items->toArray();
        $result = [];
        foreach ($items as $item) {
            if (!empty($item['days'])) {
                $item['days'] = explode(',', $item['days']);
                $item['from_hours'] = (string) $item['from_hours'];
                $item['to_hours'] = (string) $item['to_hours'];
            }
            $result[] = $item;
        }
        $meta = [
            "has_next" => $filter["page_size"] + 1 < $pageCount,
            "off_set" => (int) $filter["page_size"] * $filter["page_id"],
            "page_count" => (int) $pageCount,
            "page_id" => (int) $filter["page_id"],
            "total_count" => (int) $count,
            "page_size" => (int) $filter["page_size"],
        ];
        $response = array(
            "status" => 'successful',
            "result" => $result,
            "meta" => $meta,
        );
        return response()->json($response);
    }

    private function buildFilter($request){
        $retVal = [];
        $columns = ['user_id', 'locate', 'status'];
        foreach($columns as $column) {
            if ($request->has($column)){
                $retVal[$column] = $request->input($column);
            }
        }
        
        $retVal['columns'] = [
            "*"
        ];
        return $retVal;
    }

    private function getItems($query, $filter)
    {
        $pageId = $filter["page_id"] + 1;
        $pageSize = $filter["page_size"];
        if (array_key_exists('order', $filter)) {
            $order = $filter['order'];
            $query->orderBy($order['column'], $order['direction']);
        } else {
            $query->orderBy('created_at', 'DESC');
        }
        if ($pageSize > 0) {
            $query->forPage(($filter["page_id"] + 1), $filter["page_size"]);
        }
        return $query->get($filter['columns']);
    }

    protected function recordsCountToPagesCount($recordsCount, $pageSize) {
        $retVal = (int) ($recordsCount / $pageSize);
        if ($recordsCount % $pageSize > 0) {
            $retVal++;
        }
        return $retVal;
    }

    public function assign(Request $request) {
        if (!$request->has('ticket_id')) {
            $this->reCalculateNumberTicketAssignment();
        }
        $queryTime = date('Y-m-d H:i:s', strtotime('-30 days'));
        $query = Ticket::whereIn('status', ['open', 'waiting_third_party', 'waiting_customer'])
                        ->where('updated_at', '>=', $queryTime)
                        ->where('is_reply', 0);
        $isOldCustomer = false;
        if ($request->has('ticket_id')) {
            $query->where('id', '=', $request->input('ticket_id'));
            $isOldCustomer = $this->assignWithOldCustomer($request->input('ticket_id'));
        } else {
            $cloneQuery = clone $query;
            $cloneQuery->groupBy('email');
            $cloneQuery->havingRaw('count(*) = 1');
            $ids = $cloneQuery->pluck('id')->toArray();
            if (!empty($ids)) {
                $query->whereIn('id', $ids);
            } else {
                return response()->json(['status' => 'successful']);
            }
        }
        $items = $query->get(['id', 'locate', 'user_id']);
        $itemGroupByLocates = [];
        if (!empty($items)) {
            foreach ($items as $item) {
                if (!isset($itemGroupByLocates[$item->locate])) {
                    $itemGroupByLocates[$item->locate] = [];
                }
                if (!$request->has('ticket_id')) {
                    $itemGroupByLocates[$item->locate][] = $item->id;
                } else {
                    if (!$isOldCustomer) {
                        $itemGroupByLocates[$item->locate][] = $item->id;
                    }
                }
            }
        }
        foreach ($itemGroupByLocates as $locate => $values) {
            if (!empty($values)) {
                $staffAvailables = $this->buildStaffAvailables($locate);
                if (!empty($staffAvailables)) {
                    $staffDisableds = $this->buildStaffDisabled($locate);
                    $staffIds = [];
                    foreach ($staffAvailables as $id => $number) {
                        $staffIds[] = $id;
                    }
                    $assigment = [];
                    $ticketReAssignments = Ticket::whereIn('id', $values)
                                                    ->where(function($q) use ($staffIds, $staffDisableds){
                                                        $q->whereNotIn('user_id', $staffIds);
                                                        $q->orWhereNull('user_id');
                                                        if (!empty($staffDisableds)) {
                                                            $q->orWhereIn('user_id', $staffDisableds);
                                                        }
                                                    })
                                                    ->pluck('id')->toArray();
                    $date = getDate();
                    if (!$request->has('ticket_id') && $date['hours'] == 3) {
                        $ticketNews = Ticket::with(['ticketItems'])
                                                ->whereIn('id', $values)
                                                ->get()->toArray();
                        foreach ($ticketNews as $ticket) {
                            if ((empty($ticket['ticket_items']) || (!empty($ticket['ticket_items']) && count($ticket['ticket_items']) == 1 && $ticket['ticket_items'][0]['type'] == 'customer')) 
                                && !in_array($ticket['id'], $ticketReAssignments)) {
                                    $ticketReAssignments[] = $ticket['id'];
                            }
                        }
                    }
                    foreach ($ticketReAssignments  as $ticketId) {
                        $i = 0;
                        $newTicketAssigns = [];
                        foreach ($staffAvailables as $key => $val) {
                            if ($i == 0) {
                                if (!isset($assigment[$key])) {
                                    $assigment[$key] = [];
                                }
                                $assigment[$key][] = $ticketId;
                                $newTicketAssigns[$key] = $val + 1;
                            } else {
                                $newTicketAssigns[$key] = $val;
                            }
                            $i++;
                        }
                        $staffAvailables = $newTicketAssigns;
                        uasort($staffAvailables, function($a, $b) {
                            if ($a == $b) {
                                return 0;
                            }
                            return ($a < $b) ? -1 : 1;
                        });
                    }
                    foreach ($assigment as $userId => $tickets) {
                        foreach ($tickets as $ticketId) {
                            Ticket::where('id', '=', $ticketId)
                                    ->update(['user_id' => $userId]);
                        }
                    }
                    foreach ($staffAvailables as $userId => $numberTicketOpen) {
                        TicketAssign::where('user_id', '=', $userId)
                                        ->where('locate', '=', $locate)
                                        ->update(['number_ticket_open' => $numberTicketOpen]);
                    }
                }
            }
        }
        if (!$request->has('ticket_id')) {
            $this->reAssignTicketForOldStaff($queryTime);
        }
        $this->reCalculateNumberTicketAssignment();
        return response()->json(['status' => 'successful']);
    }

    private function assignWithOldCustomer($ticketId) {
        $retVal = false;
        $ticket = Ticket::find($ticketId);
        if ($ticket) {
            $ticketOld = Ticket::where('email', '=', $ticket->email)
                                ->whereNotNull('user_id')
                                ->orderBy('id', 'DESC')
                                ->first();
            if ($ticketOld) {
                $checkAvaliable = TicketAssign::where('user_id', '=', $ticketOld->user_id)
                                                ->where('status', '!=', 'quit_job')
                                                ->exists();
                if ($checkAvaliable) {
                    $retVal = true;
                    $dataUpdate = [
                        'user_id' => $ticketOld->user_id
                    ];
                    if ($ticket->deadline == null) {
                        $timeNow = time();
                        $date = date('Y-m-d H:i:s', strtotime('+1 day', $timeNow));
                        $deadline = \DateTime::createFromFormat('Y-m-d H:i:s', $date);
                        $dataUpdate['deadline'] = $deadline;
                    }
                    Ticket::where('id', '=', $ticketId)
                            ->update($dataUpdate);
                }
                
            }
        }
        return $retVal;
    }

    private function buildStaffDisabled($locate) {
        $retVal = TicketAssign::where('locate', '=', $locate)
                            ->where('status', '!=', 'active')
                            ->pluck('user_id')->toArray();
        return $retVal;
    }

    private function reAssignTicketForOldStaff($queryTime) {
        $items = Ticket::with(['ticketItems' => function($query) {
                    $query->where('type', '=', 'staff');
                    $query->orderBy('created_at', 'DESC');
                }])
                ->where('created_at', '>=', $queryTime)
                ->where('ticket.is_reply', '=', 0)
                ->get()
                ->toArray();
        $itemByLocates = [];
        foreach ($items as $item) {
            if (!array_key_exists($item['locate'], $itemByLocates)) {
                $itemByLocates[$item['locate']] = [];
            }
            $itemByLocates[$item['locate']][] = $item;
        }
        foreach ($itemByLocates as $key => $values) {
            $staffs = $this->buildStaffAvailables($key);
            $staffAvailables = [];
            foreach ($staffs as $k => $v) {
                $staffAvailables[] = $k;
            }
            foreach($values as $value) {
                if (!empty($value['ticket_items'])) {
                    if ($value['user_id'] != $value['ticket_items'][0]['user_id'] && in_array($value['ticket_items'][0]['user_id'], $staffAvailables)) {
                        Ticket::where('id', '=', $value['id'])
                                ->update(['user_id' => $value['ticket_items'][0]['user_id']]);
                    }
                }
            }
        }
    }

    private function buildStaffAvailables($locate) {
        $date = getDate();
        $dayOfWeek = $date['wday'];
        $hour = $date['hours'];
        $from = 0;
        $to = 23;
        if ($hour >= 0 && $hour <= 16) {
            $to = 17;
        } elseif ($hour >= 17 && $hour <= 23) {
            $from = 17;
        }
        $ticketAssigns = TicketAssign::where('locate', '=', $locate)
                                        ->where('status', '=', 'active')
                                        ->where('from_hours', '>=', $from)
                                        ->where('to_hours', '<=', $to)
                                        ->orderBy('number_ticket_open', 'ASC')
                                        ->get(['number_ticket_open', 'user_id', 'days']);
        $retVal = [];
        if (!empty($ticketAssigns)) {
            foreach ($ticketAssigns as $item) {
                $dayWorkings = explode(',', $item->days);
                if (in_array($dayOfWeek, $dayWorkings)) {
                    $retVal[$item->user_id] = $item->number_ticket_open;
                }
            }
        }
        // Giao lại việc cho đối tượng làm việc giờ hành chính khi không xác định được đối tượng giao
        if (empty($retVal)) {
            $ticketAssigns = TicketAssign::where('locate', '=', $locate)
                                        ->where('status', '=', 'active')
                                        ->where('from_hours', '>=', 8)
                                        ->where('to_hours', '<=', 17)
                                        ->orderBy('number_ticket_open', 'ASC')
                                        ->get(['number_ticket_open', 'user_id', 'days']);
            if (!empty($ticketAssigns)) {
                foreach ($ticketAssigns as $item) {
                    $dayWorkings = explode(',', $item->days);
                    if (!in_array(7, $dayWorkings) && !in_array(8, $dayWorkings)) {
                        $retVal[$item->user_id] = $item->number_ticket_open;
                    }
                }
            }
        }
        return $retVal;
    }

    private function reCalculateNumberTicketAssignment() {
        $assigns = TicketAssign::get();
        foreach ($assigns as $item) {
            $numberTicket = Ticket::where('user_id', '=', $item->user_id)
                                        ->where('locate', '=', $item->locate)
                                        ->whereIn('status', ['open', 'waiting_third_party', 'waiting_customer'])
                                        ->count();
            TicketAssign::where('user_id', '=', $item->user_id)
                                ->where('locate', '=', $item->locate)
                                ->update(['number_ticket_open' => $numberTicket]);
        }
    }
}