<?php

namespace Modules\Ticket\Controllers\Services;

use Modules\Ticket\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Modules\Ticket\Models\Order;
use Modules\Ticket\Models\Ticket;
use Modules\Ticket\Models\TicketItem;
use Modules\Ticket\Models\TicketNOrder;
use Modules\Ticket\Models\TicketAssign;
use Modules\Ticket\Models\TicketFilterSpam;
use Modules\Ticket\Models\User;
use App\Helpers\ApiClient;
use Modules\Ticket\Helpers\DBConnect;
use Illuminate\Support\Str;

class TicketService extends Controller
{

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $columns = $filter['columns'];
        $query = Ticket::buildTicketQuery($filter);
        //Get count
        $queryCount = clone $query;
        $queryCount->select($columns);

        if (isset($filter['statusses'])) {
            $count = $queryCount->whereIn('status', $filter['statusses'])->count();
        }else {
            $count = $queryCount->count();
        }
        //Get items
        $filter["page_id"] = $request->input("page_id", 0);
        $filter["page_size"] = $request->input("page_size", 20);
        $pageCount = $this->recordsCountToPagesCount($count, $filter['page_size']);
        
        $items = $this->getItems($query, $filter);
        $items = $items->toArray();
        if ($request->has("sort")) {
            $sort = $request->input('sort');
            $this->rebuildSortItems($items, $sort);
        }
        

        $meta = [
            "has_next" => $filter["page_size"] + 1 < $pageCount,
            "off_set" => (int) $filter["page_size"] * $filter["page_id"],
            "page_count" => (int) $pageCount,
            "page_id" => (int) $filter["page_id"],
            "total_count" => (int) $count,
            "page_size" => (int) $filter["page_size"],
        ];
        $response = array(
            "status" => 'successful',
            "result" => $items,
            "meta" => $meta,
        );
        return response()->json($response);
    }

    private function buildFilter($request){
        $retVal = [];
        $columns = ['status', 'id', 'ticket_items', 'user_id', 'is_reply', 'type', 'is_refund', 
                    'token_customer', 'keyword', 'locate', 'is_spam', 'is_all_open', 'from'];
        foreach($columns as $column) {
            if ($request->has($column)){
                $retVal[$column] = $request->input($column);
            }
        }
        if (!isset($retVal['is_spam'])) {
            $retVal['is_spam'] = 0;
        }
        if (!isset($retVal['status']) && isset($retVal['is_all_open']) && $retVal['is_all_open'] == 1) {
            unset($retVal['status']);
            $retVal['statusses'] = ['open', 'waiting_customer', 'waiting_third_party'];
        }
        if ($request->has('order') && $request->input('order') != '') {
            $orders = explode('-', $request->input('order'));
            if (isset($orders[1])) {
                $retVal['order'] = ['column' => $orders[0], 'direction' => $orders[1]];
            }
        }
        $retVal['columns'] = [
            "*"
        ];
        return $retVal;
    }

    private function getItems($query, $filter)
    {
        $pageId = $filter["page_id"] + 1;
        $pageSize = $filter["page_size"];
        $query->with(['ticketOrder']);
        if (array_key_exists('ticket_items', $filter)) {
            $query->with(['ticketItems' => function ($q) {
                $q->orderBy('created_at', 'DESC')
                    ->orderBy('type', 'DESC');
            }]);
        }
        if (array_key_exists('statusses', $filter) && !empty($filter['statusses'])) {
            $query->whereIn('status', $filter['statusses']);
        }

        if (array_key_exists('order', $filter)) {
            $order = $filter['order'];
            if ($order['column'] == 'deadline') {
                $query->orderBy(\DB::raw('ISNULL(deadline), deadline'), $order['direction']);
                $query->orderBy('created_at', 'DESC');
            } elseif($order['column'] == 'reply_at') {
                $query->orderBy(\DB::raw('ISNULL(reply_at), reply_at'), $order['direction']);
                $query->orderBy('updated_at', 'DESC');
            }  
            else {
                $query->orderBy($order['column'], $order['direction']);
            }
        } else {
            $query->orderBy('created_at', 'DESC');
        }

        if ($pageSize > 0) {
            $query->forPage(($filter["page_id"] + 1), $filter["page_size"]);
        }
        return $query->get($filter['columns']);
    }

    protected function recordsCountToPagesCount($recordsCount, $pageSize) {
        $retVal = (int) ($recordsCount / $pageSize);
        if ($recordsCount % $pageSize > 0) {
            $retVal++;
        }
        //return
        return $retVal;
    }

    public function saveCustomer(Request $request) {
        $response = [
            "status" => 'fail'
        ];
        $customerRules = [
            'email' => 'bail|required|email',
            'customer_name' => 'bail|required|max:200',
            'title' => 'bail|required|max:200',
            'token_customer' => 'bail|required|max:200',
            'token_ticket' => 'bail|required|max:200',
            'content' => 'bail|required|max:2000',
            'status' => 'bail|required',
        ];
        $this->validate($request, $customerRules);
        $isValid = true;
        $type = $request->input("type");
        $orderCode = $request->input("order_code", null);
        if (in_array($type, ["order", "cancel_order", "change_order", "tracking_code"])) {
            if (!$request->has("order_code") || !isset($orderCode)) {
                $isValid = false;
                $response = [
                    "status" => 'fail',
                    "message" => __('Please enter order code!'),
                ];
            }
        }
        if ($isValid) {
            $data = $this->buildDataTicket($request);
            if (!$request->has("id")) {
                $timeNow = time();
                $date = date('Y-m-d H:i:s', strtotime('+1 day', $timeNow));
                $deadline = \DateTime::createFromFormat('Y-m-d H:i:s', $date);
                $data['deadline'] = $deadline;
                $data['from'] = 'customer';
                $ticket = new Ticket;
                $ticket->fill($data);
                $ticket->save();
            }
            $ticketItem = null;
            if ($ticket && $ticket->id) {
                $dataItem = $this->buildDataItem($request, $ticket);
                $ticketItem = new TicketItem;
                $ticketItem->fill($dataItem);
                $ticketItem->save();
                if ($ticketItem) {
                    $this->callSendEmailRequest($ticketItem->id);
                }
                $listOrderCode = explode(",", $ticket->order_code);
                $codes = [];
                foreach ($listOrderCode as $code) {
                    $code = str_replace('#', '', $code);
                    $code = trim($code);
                    if ($code != '') {
                        $codes[] = $code;
                    }
                }
                if (!empty($codes)) {
                    $dataUpdate = [];
                    $locates = getModuleLocale();
                    $locateTicket = '';
                    foreach ($locates as $locate) {
                        if ($locate['enable'] && $locate['locale'] != 'central') {
                            try {
                                $connection = DBConnect::connect($locate['locale']);
                                $orders = $connection->table('order')->whereIn("code", $codes)->get(['id'])->toArray();
                                if (!empty($orders)) {
                                    foreach ($orders as $order) {
                                        $ticketDuplicates = $this->checkDuplicateOrder($ticket, $order->id);
                                        $update = [
                                            "ticket_id" => $ticket->id,
                                            "order_id" => $order->id,
                                            "type" => "order_code"
                                        ];
                                        if (!empty($ticketDuplicates)) {
                                            $update['ticket_related'] = implode(",", $ticketDuplicates);
                                        }
                                        $dataUpdate[] = $update;
                                    }
                                    $locateTicket = $locate['locale'];
                                    break;
                                } else {
                                    $customers = $connection->table('customer')->where("email", $ticket->email)->pluck("id")->toArray();
                                    if (!empty($customers)) {
                                        $orders = $connection->table('order')
                                                ->whereIn("customer_id", $customers)
                                                ->where("payment_status", "PAID")
                                                ->orderBy("created_at", 'DESC')
                                                ->get(["id", "customer_id", "created_at", "payment_status"]);
                                        if (!empty($orders)) {
                                            foreach ($orders as $item) {
                                                $ticketDuplicates = $this->checkDuplicateOrder($ticket, $item->id);
                                                $update = [
                                                        "ticket_id" => $ticket->id,
                                                        "order_id" => $item->id,
                                                        "type" => "email",
                                                ];
                                                if (!empty($ticketDuplicates)) {
                                                    $update['ticket_related'] = implode(",", $ticketDuplicates);
                                                }
                                            }
                                            $locateTicket = $locate['locale'];
                                            break;
                                        }
                                    }
                                }
                            } catch (\Exception $ex) {

                            }
                        }
                    }

                    if ($locateTicket != '') {
                        Ticket::where('id', '=', $ticket->id)->update(['locate' => $locateTicket]);
                    }
                    if (!empty($dataUpdate)) {
                        foreach ($dataUpdate as $item) {
                            TicketNOrder::create($item);
                        }
                    }
                }
                $this->triggerAsyncRequest(\URL::to('/') . "/ticket/assign?ticket_id=" . $ticket->id, "GET");
                $response = [
                    "status" => "successful",
                    "result" => $ticket,
                ];
            }
        }
        return response()->json($response);
    }

    private function buildDataTicket($request) {
        $result = [];
        $columns = ['email', 'customer_name', 'title', 'type', 'order_code', 'token_customer', 
                    'token_ticket', 'status', 'is_reply', 'is_refund', 'reply_email', 'user_id'];
        foreach ($columns as $column) {
            if ($request->has($column)) {
                $result[$column] = $request->input($column);
            }
        }
        $locate = env('APP_LOCALE');
        if ($locate && $locate != '' && $locate != null) {
            $result["locate"] = $locate;
        } else {
            $result["locate"] = "us";
        }
        return $result;
    }

    private function buildDataItem($request, $ticket) {
        $result = [];
        $result["ticket_id"] = $ticket->id;
        if ($request->has("content")) {
            $result["content"] = $request->input("content");
        }
        if ($request->has("files")) {
            $result["files"] = $request->input("files");
        }
        $result["type"] = "customer";
        return $result;

    }

    public function updateStatus(Request $request) {
        $response = [
            "status" => "fail"
        ];
        $id = $request->input("id");
        if ($id) {
            $ticket = Ticket::where("id", $id)->first();
            if ($ticket) {
                $data = [];
                if ($request->has('status')) {
                    $data['status'] = $request->input('status');
                    $this->updateNumberOpen($id, $data['status']);
                    if ($data['status'] == 'close') {
                        $this->requestTicketFulfillment([$id], 'close');
                    } else {
                        $this->requestTicketFulfillment([$id], 'open');
                    }
                }
                if ($request->has('deadline')) {
                    $deadline = \DateTime::createFromFormat('d/m/Y', $request->input('deadline'));
                    $deadline->setTime(23,59,59);
                    $data['deadline'] = $deadline;
                }
                if ($request->has('locate')) {
                    $data['locate'] = $request->input('locate');
                }
                if ($request->has('user_id')) {
                    $data['user_id'] = $request->input('user_id');
                }
                if ($request->has('note')) {
                    $data['note'] = $request->input('note');
                }
                if ($request->has('type')) {
                    $data['type'] = $request->input('type');
                }
                if(!empty($data)) {
                    $ticket->fill($data);
                    $ticket->save();
                }
                $response = [
                    "status" => "successful",
                    "result" => $ticket
                ];
            }
        }
        return response()->json($response);
    }

    private function requestTicketFulfillment($ids, $status) {
        $items = Ticket::with('ticketOrder')
                        ->whereIn('id', $ids)
                        ->get()
                        ->toArray();
        if (!empty($items)) {
            foreach ($items as $item) {
                if ($item['from'] == 'fulfillment' && !empty($item['ticket_order'])) {
                    $connection = DBConnect::connect($item['locate']);
                    $exists = $connection->table('order_meta')
                                ->where('order_id', $item['ticket_order']['order_id'])
                                ->where('key', '=', 'ticket_completed')
                                ->exists();
                    
                    $value = 0;
                    if ($status == 'close') {
                        $value = 1;
                    }
                    if (!$exists) {
                        $connection->table('order_meta')
                                    ->insert(['order_id' => $item['ticket_order']['order_id'], 'key' => 'ticket_completed', 'value' => $value]);
                    } else {
                        $connection->table('order_meta')
                                    ->where('order_id', '=', $item['ticket_order']['order_id'])
                                    ->where('key', '=', 'ticket_completed')
                                    ->update(['value' => $value]);
                    }
                }
            }
        }
    }

    private function updateNumberOpen($id, $status) {
        $ticket = Ticket::find($id);
        if ($ticket && $ticket->user_id != null) {
            $query = TicketAssign::where('user_id', '=', $ticket->user_id)
                    ->where('locate', '=', $ticket->locate);
            if ($status == 'close') {
                $query->decrement('number_ticket_open', 1);
            } else {
                $query->increment('number_ticket_open', 1);
            }
        }
    }

    public function createTicketItem(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('ticket_id')) {
            $user = \Auth::user();
            $userId = null;
            if ($user) {
                $userEmail = $user->email;
                $userCentral = User::where('email', '=', $userEmail)
                                        ->where('status', '=', 'active')
                                        ->first();
                if ($userCentral) {
                    $userId = $userCentral->id;
                }
            }
            $data = [
                'ticket_id' => $request->input('ticket_id'),
                'content' => $request->has('content') ? $request->input('content') : '',
                'user_id' => $userId,
                'files' => json_encode($request->input('files')),
                'type' => $request->has('type') ? $request->input('type') : 'staff'
            ];
            $ticketItem = new TicketItem;
            $ticketItem->fill($data);
            $ticketItem->save();
            Ticket::where('id', '=', $ticketItem->ticket_id)->update(['is_reply' => 1, 'deadline' => null, 'reply_at' => new \DateTime()]);
            $response['status'] = 'successful';
            $response['result'] = $ticketItem;
            $this->callSendEmailRequest($ticketItem->id);
        }
        return response()->json($response);
    }
    
    public function sendQuestion(Request $request) {
        $response = [
            "status" => "fail"
        ];
        if ($request->has("ticket_id") && $request->has("content")) {
            $isExists = Ticket::where("id", $request->input("ticket_id"))->exists();
            if ($isExists) {
                $data = $this->buildQuestionData($request);
                $ticketItem = new TicketItem;
                $ticketItem->fill($data);
                $ticketItem->save();
                $timeNow = time();
                $date = date('Y-m-d H:i:s', strtotime('+1 day', $timeNow));
                $deadline = \DateTime::createFromFormat('Y-m-d H:i:s', $date);
                $dataUpdate = ['deadline' => $deadline, 'is_reply' => 0];
                $checkTicketClose = Ticket::where("id", $request->input("ticket_id"))
                                            ->where('status', '=', 'close')
                                            ->exists();
                if ($checkTicketClose) {
                    $dataUpdate['status'] = 'open';
                }
                Ticket::where("id", $request->input("ticket_id"))
                        ->update($dataUpdate);
                $this->callSendEmailRequest($ticketItem->id);
                $response = [
                    "status" => "successful",
                    "result" => $ticketItem,
                ];
            }
        }
        
        return response()->json($response);
    }

    private function buildQuestionData($request) {
        $result = [];
        $columns = ['ticket_id', 'content', 'files'];
        foreach ($columns as $column) {
            if ($request->has($column)) {
                $result[$column] = $request->input($column);
            }
        }
        $result["type"] = "customer";
        return $result;
    }

    public function getOrder(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        $id = $request->input('id');
        if ($id) {
            $items = TicketNOrder::with('ticket')
                                    ->where('ticket_id', $id)
                                    ->get();
            if ($items && !empty($items[0]->ticket)) {
                $market = $items[0]->ticket->locate;
                $connection = DBConnect::connect($market);
                $orderIds = [];
                $isRefunds = [];
                foreach ($items as $item) {
                    $orderIds[] = $item->order_id;
                    $isRefunds[$item->order_id] = $item->is_refund;
                }
                $result = [];
                if (!empty($orderIds)) {
                    $result = $this->buildOrderRelateds($connection, $orderIds, $isRefunds, $market);
                }
                $response['status'] = 'successful';
                $response['result'] = $result;
                $response['id'] = $id;
            }
        }
        return response()->json($response);
    }

    private function buildOrderRelateds($connection, $orderIds, $isRefunds = [], $locate) {
        $locates = getModuleLocale();
        $locateByKeys = [];
        foreach ($locates as $loc) {
            $locateByKeys[$loc['locale']] = $loc;
        }
        $orders = $connection->table('order')
                            ->leftJoin('customer', 'customer.id', '=', 'order.customer_id')
                            ->whereIn('order.id', $orderIds)
                            ->get(['order.id', 'order.code' ,'order.created_at', 'order.status', 'order.payment_status', 'order.delivery_note', 'order.note',
                                'order.amount', 'order.verifier_id', 'order.shipping_fee', 'order.shipping_type','customer.full_name', 'customer.email', 'customer.phone', 
                                'order.delivery_address', 'order.province_id', 'order.country_id', 'order.state_name', 'order.city_name']);

        $orderItems = $connection->table('order_item')
                                ->leftJoin('product_info', 'product_info.product_id', '=', 'order_item.product_id')
                                ->whereIn('order_item.order_id', $orderIds)
                                ->where('product_info.is_default', '=', 1)
                                ->where(function($q) {
                                    $q->whereNull('order_item.product_sku_id');
                                    $q->orWhere('order_item.product_sku_id', '=', 0);
                                })
                                ->get(['order_item.order_id', 'order_item.quantity', 'product_info.name', 'product_info.slug']);

        $orderItemVariants = $connection->table('order_item')
                                ->leftJoin('product_info', 'product_info.product_sku_id', '=', 'order_item.product_sku_id')
                                ->whereIn('order_item.order_id', $orderIds)
                                ->where('order_item.product_sku_id', '>', 0)
                                ->get(['order_item.order_id', 'order_item.quantity', 'product_info.name', 'product_info.slug']);
        $orderMetas = $connection->table('order_meta')
                                ->whereIn('order_id', $orderIds)
                                ->where('key', '=', 'shipment_data')
                                ->pluck('value', 'order_id')->toArray();
        $countryIds = [];
        $provinceIds = [];
        $verifierIds = [];
        foreach ($orders as $item) {
            if (!empty($item->province_id) && !in_array($item->province_id, $provinceIds)) {
                $provinceIds[] = $item->province_id;
            }
            if (!empty($item->country_id) && !in_array($item->country_id, $countryIds)) {
                $countryIds[] = $item->country_id;
            }
            if (!empty($item->verifier_id)) {
                $verifierIds[] = $item->verifier_id;
            }
        }
        $provinces = [];
        if (!empty($provinceIds)) {
            $provinces = $connection->table('province')
                                    ->whereIn('id', $provinceIds)
                                    ->pluck('name', 'id')->toArray();
        }
        $countries = [];
        if (!empty($countryIds)) {
            $countries = $connection->table('countries')
                                    ->whereIn('id', $countryIds)
                                    ->pluck('name', 'id')->toArray();
        }
        $verifiers = [];
        if (!empty($verifierIds)) {
            $verifiers = $connection->table('users')
                                    ->whereIn('id', $verifierIds)
                                    ->pluck('name', 'id')->toArray();
        }
        $items = [];
        if (count($orderItems) > 0) {
            foreach ($orderItems as $item) {
                if (!array_key_exists($item->order_id, $items)) {
                    $items[$item->order_id] = [];
                }
                $items[$item->order_id][] = $item;
            }
        }
        if (count($orderItemVariants) > 0) {
            foreach ($orderItemVariants as $item) {
                if (!array_key_exists($item->order_id, $items)) {
                    $items[$item->order_id] = [];
                }
                $items[$item->order_id][] = $item;
            }
        }
        $retVal = [];
        foreach ($orders as $order) {
            $fullAddress = $order->delivery_address;
            if ($order->city_name != '') {
                $fullAddress .= ', ' . $order->city_name;
            }
            if (!empty($order->state_name)) {
                $fullAddress .= ', ' . $order->state_name;
            } else {
                if (array_key_exists($order->province_id, $provinces)) {
                    $fullAddress .= ', ' . $provinces[$order->province_id];
                }
            }
            if (array_key_exists($order->country_id, $countries)) {
                $fullAddress .= ', ' . $countries[$order->country_id];
            }
            $order->address = $fullAddress;
            if (array_key_exists($order->id, $items)) {
                $order->items = $items[$order->id];
            }
            $order->trackings = [];
            if (array_key_exists($order->id, $orderMetas)) {
                $order->trackings = json_decode($orderMetas[$order->id], true);
            }
            if (array_key_exists($order->id, $isRefunds)) {
                $order->is_refund = $isRefunds[$order->id];
            }
            $order->verifier = null;
            if (array_key_exists($order->verifier_id, $verifiers)) {
                $order->verifier = $verifiers[$order->verifier_id];
            }
            $order->amount_show = null;
            if (isset($locateByKeys[$locate])) {
                $order->amount_show = $this->formatPrice($order->amount, $locateByKeys[$locate]['currencyUnit']);
            }
            $order->shipping_fee = $this->formatPrice($order->shipping_fee, 'USD');
            
            $retVal[] = $order;
        }
        return $retVal;
    }

    private function formatPrice($price, $locate)
    {
        $price = doubleval($price);
        $template = $this->getCurrencyTemplate($locate);
        preg_match("/{money}{([^a-zA-z0-9]+)}{([0-9]+)}/", $template, $matches);
        $retval =  $price;
        if (count($matches) > 2) {
            $separate = $matches[1];
            $number = intval($matches[2]);
            $template = preg_replace("/{([^a-zA-z0-9]+)}/", "", $template);
            $template = preg_replace("/{([0-9]+)}/", "", $template);
            $decimal = $separate == ',' ? '.' : ',';
            $retval = str_replace('{money}', number_format($price, $number, $separate, $decimal), $template);
        }
    
        return $retval;
    }

    private function getCurrencyTemplate($locate) {
        $template = config('default.currency_template.' . $locate);
        return $template;
    }

    public function updateMultiple(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('ids') && $request->has('data')) {
            $data = [];
            $ids = $request->input('ids');
            foreach ($request->input('data') as $key => $value) {
                if ($value != '') {
                    $data[$key] = $value;
                }
                if ($key == 'deadline') {
                    $deadline = \DateTime::createFromFormat('d/m/Y', $value);
                    $deadline->setTime(23,59,59);
                    $data['deadline'] = $deadline;
                }
            }
            if (isset($data['status']) && $data['status'] == 'close') {
                $this->requestTicketFulfillment($ids, 'close');
            } else if (isset($data['status']) && $data['status'] != 'close') {
                $this->requestTicketFulfillment($ids, 'open');
            }
            // $tickets = Ticket::whereIn('id', $ids)->get(['user_id', 'locate']);
            if (!empty($data)) {
                Ticket::whereIn('id', $ids)
                        ->update($data);
            }
            // if (isset($data['status']) && !isset($data['user_id'])) {
            //     foreach ($tickets as $item) {
            //         if ($item->user_id != null) {
            //             $this->updateNumberTicketOpen($item->user_id, $item->locate, $data['status']);
            //         }
            //     }
            // }
            // if (isset($data['user_id']) && !isset($data['status'])) {
            //     foreach ($tickets as $item) {
            //         if ($item->user_id != $data['user_id']) {
            //             $this->updateNumberTicketOpen($data['user_id'], $item->locate, $item->status);
            //             if ($item->user_id != null) {
            //                 $this->updateNumberTicketOpen($item->user_id, $item->locate, 'close');
            //             }
            //         }
            //     }
            // }
            // if (isset($data['user_id']) && isset($data['status'])) {
            //     foreach ($tickets as $item) {
            //         if ($data['user_id'] != $item->user_id) {
            //             if ($item->user_id != null) {
            //                 $this->updateNumberTicketOpen($item->user_id, $item->locate, 'close');
            //             }
            //             $this->updateNumberTicketOpen($data['user_id'], $item->locate, $data['status']);
            //         }
            //     }
            // }
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    private function updateNumberTicketOpen($userId, $locate, $status) {
        $query = TicketAssign::where('user_id', '=', $userId)
                                ->where('locate', '=', $locate);
        if ($status == 'close') {
            $query->decrement('number_ticket_open', 1);
        } else {
            $query->increment('number_ticket_open', 1);
        }
    }

    public function mapOrder(Request $request) {
        $id = $request->input('id');
        $code = $request->input('order');
        $code = trim($code);
        $input = $request->all();
        $response = [
            'status' => 'fail',
            'ticket_n_order' => [],
            'locate' => ''
        ];
        $code = str_replace('#', '', $code);
        if (!empty($id) && !empty($code)) {
            $ticket = Ticket::find($id);
            if ($ticket && $ticket->locate != null && $ticket->locate != '') {
                $connection = DBConnect::connect($ticket->locate);
                $order = $connection->table('order')->where("code", '=', $code)->first(['id']);
                if ($order) {
                    $orderIds = [$order->id];
                    $data = [
                        'ticket_id' => $ticket->id,
                        'order_id' => $order->id,
                        'type' => 'order_code'
                    ];
                    $ticketNOrders = [$data];
                    if (!empty($orderIds)) {
                        $result = $this->buildOrderRelateds($connection, $orderIds, [], $ticket->locate);
                        $response['status'] = 'successful';
                        $response['ticket_n_order'] = $ticketNOrders;
                        $response['result'] = $result;
                        $response['locate'] = $ticket->locate;
                    }
                }
            }
        } else if (empty($id) && !empty($code)) {
            $locate = $request->input('locate');
            if (!empty($locate)) {
                $connection = DBConnect::connect($locate);
                $order = $connection->table('order as o')
                                            ->where('o.code', '=', $code)
                                            ->first(['o.id']);
                if ($order) {
                    $response['status'] = 'successful';
                    $orderIds = [$order->id];
                    $result = $this->buildOrderRelateds($connection, $orderIds, [], $locate);
                    $response['result'] = $result;
                    $response['locate'] = $locate;
                }
            } else {
                $locates = getModuleLocale();
                foreach ($locates as $locate) {
                    try {
                        if ($locate['enable'] && $locate['locale'] != 'central') {
                            $connection = DBConnect::connect($locate['locale']);
                            $order = $connection->table('order as o')
                                            ->where('o.code', '=', $code)
                                            ->first(['o.id']);
                            
                            if ($order) {
                                $response['status'] = 'successful';
                                $orderIds = [$order->id];
                                $result = $this->buildOrderRelateds($connection, $orderIds, [], $locate['locale']);
                                $response['result'] = $result;
                                $response['locate'] = $locate['locale'];
                                break; 
                            }
                        }
                    } catch (\Exception $e) {

                    }
                }
            }
            
        }
        return response()->json($response);
        
    }

    public function saveTicketOrder(Request $request) {
        $ticketNOrders = $request->get('ticket_n_order');
        $response = [
            'status' => 'fail'
        ];
        if (!empty($ticketNOrders)) {
            foreach ($ticketNOrders as $item) {
                $checkExists = TicketNOrder::where('ticket_id', '=', $item['ticket_id'])
                                            ->where('order_id', '=', $item['order_id'])
                                            ->exists();
                if (!$checkExists) {
                    TicketNOrder::create($item);
                }
            }
            if (!empty($request->get('id')) && !empty($request->get('order_code'))) {
                Ticket::where('id', '=', $request->get('id'))
                        ->update(['order_code' => $request->get('order_code')]);
            }
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function sendEmail(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        $type = $request->input('type');
        $email = null;
        $locate = $request->has('locate') ? $request->get('locate') : null;
        $userId = $request->has('user_id') ? $request->get('user_id') : null;
        $content = $request->input('content');
        $files = $request->input('data_files');
        
        $orderId = null;
        $code = null;
        if ($type == 'email') {
            $email = $request->input('value');
        } else if ($type == 'order') {
            $code = $request->input('value');
            if ($locate != null) {
                $connection = DBConnect::connect($locate);
                $order = $connection->table('order as o')
                                        ->leftJoin('customer as c', 'c.id', '=', 'o.customer_id')
                                        ->where('o.code', '=', $code)
                                        ->first(['o.id', 'c.email']);
                if ($order) {
                    $email = $order->email;
                    $orderId = $order->id;
                }
            } else {
                $locates = getModuleLocale();
                foreach ($locates as $locate) {
                    try {
                        if ($locate['enable'] && $locate['locale'] != 'central') {
                            $connection = DBConnect::connect($locate['locale']);
                            $locate = $locate['locale'];
                            $order = $connection->table('order as o')
                                            ->leftJoin('customer as c', 'c.id', '=', 'o.customer_id')
                                            ->where('o.code', '=', $code)
                                            ->first(['o.id', 'c.email']);
                            if ($order) {
                                $email = $order->email;
                                $orderId = $order->id;
                                break;
                            }
                        }
                    } catch(\Exception $ex) {

                    }
                }
            }
        }
        if ($locate == null) {
            $locate = 'us';
        }
        if ($email != null) {
            $token = md5($email);
            $tokenTicket = md5(time());
            $timeNow = time();
            $date = date('Y-m-d H:i:s', strtotime('+1 day', $timeNow));
            $deadline = \DateTime::createFromFormat('Y-m-d H:i:s', $date);
            $ticket = [
                'email' => $email,
                'title' => $request->input('title'),
                'order_code' => $code,
                'token_customer' => $token,
                'token_ticket' => $tokenTicket,
                'status' => 'open',
                'type' => 'complaint',
                'is_reply' => 1,
                'locate' => $locate,
                'user_id' => $userId,
                'deadline' => $deadline,
                'is_from_email' => 1
            ];
            $ticketFirst = Ticket::create($ticket);
            $ticketItem = [
                'ticket_id' => $ticketFirst->id,
                'content' => $content,
                'type' => 'staff',
                'files' => json_encode($files),
                'user_id' => $userId
            ];
            $item = TicketItem::create($ticketItem);
            $this->callSendEmailRequest($item->id);
            if ($orderId != null) {
                TicketNOrder::create([
                    'ticket_id' => $ticketFirst->id,
                    'order_id' => $orderId,
                    'type' => 'order_code'
                ]);
            }
            $response = [
                'status' => 'successful'
            ];
        }
        return response()->json($response);
    }


    public function createTicket(Request $request) {
        $input = $request->all();
        $timeNow = time();
        $date = date('Y-m-d H:i:s', strtotime('+1 day', $timeNow));
        $deadline = \DateTime::createFromFormat('Y-m-d H:i:s', $date);
        $response = [
            'status' => 'fail'
        ];
        $user = \Auth::user();
        $userId = null;
        if ($user) {
            $userEmail = $user->email;
            $userCentral = User::where('email', '=', $userEmail)
                                ->where('status', '=', 'active')
                                ->first();
           if ($userCentral) {
               $userId = $userCentral->id;
           }
        }
        
    
        if (array_key_exists('relastedOrders', $input) && !empty($input['relastedOrders'])) {
            $email = $input['relastedOrders'][0]['email'];
            $orderCode = $input['relastedOrders'][0]['code'];
            $token = md5($email);
            $tokenTicket = md5(time());
            $ticket = [
                'email' => $email,
                'title' => $input['title'],
                'order_code' => $orderCode,
                'token_customer' => $token . Str::random(4),
                'token_ticket' => $tokenTicket,
                'status' => $input['status'],
                'type' => 'order',
                'is_reply' => 1,
                'locate' => $input['locate'],
                'user_id' => $userId,
                'deadline' => null,
                'is_from_email' => 1,
                'from' => 'fulfillment',
                'note' => $request->has('note') ? $request->input('note') : '',
                'reply_at' => new \DateTime()
            ];
            $ticketFirst = Ticket::create($ticket);
            $ticketItem = [
                'ticket_id' => $ticketFirst->id,
                'content' => $input['answer_text'],
                'type' => 'staff',
                'user_id' => $userId,
                'files' => json_encode([])
            ];
            $item = TicketItem::create($ticketItem);
            $this->callSendEmailRequest($item->id);
            $ticketDuplicates = $this->checkDuplicateOrder($ticketFirst, $input['relastedOrders'][0]['id']);
            $ticketNOrder = [
                'ticket_id' => $ticketFirst->id,
                'order_id' => $input['relastedOrders'][0]['id'],
                'type' => 'order_code'
            ];
            if (!empty($ticketDuplicates)) {
                $ticketNOrder['ticket_related'] = implode(',', $ticketDuplicates);
            }
            TicketNOrder::create($ticketNOrder);
            $response['status'] = 'successful';
            $response['result'] = $ticketItem;
        } 
        return response()->json($response);
    }
    
    private function callSendEmailRequest($ticketItemId) {
        $baseUrl = config('sa.api_url');
        $url = $baseUrl . '/ticket/send-email/' . $ticketItemId;
        $this->triggerAsyncRequest($url, 'POST');
    }

    public function updateOrderNote(Request $request) {
        $input = $request->all();
        $response = [
            'status' => 'fail'
        ];
        if (array_key_exists('locate', $input) 
            && (array_key_exists('note', $input) || array_key_exists('delivery_note', $input))) {
            $connection = DBConnect::connect($input['locate']);
            $dataUpdate = [];
            if (array_key_exists('delivery_note', $input)) {
                $dataUpdate['delivery_note'] = $input['delivery_note'];
            }
            if (array_key_exists('note', $input)) {
                $dataUpdate['note'] = $input['note'];
            }
            if (!empty($dataUpdate)) {
                $query = $connection->table('order')
                        ->where('id', '=', $input['id']);
                foreach ($dataUpdate as $key => $value) {
                    $query->where($key, '=', $value);
                }
                $isExists= $query->exists();
                if (!$isExists) {
                    $query = $connection->table('order')
                        ->where('id', '=', $input['id'])
                        ->update($dataUpdate);
                        $user = \Auth::user();
                    $email = 'help_center@printerval.com';
                    if ($user) {
                        $email = $user->email;
                    }
                    $logData = [
                        'target_id' => $input['id'],
                        'target_type' => 'ORDER',
                        'actor_email' => $email,
                        'data' => json_encode($dataUpdate),
                        'event_type' => 'UPDATE_FROM_HELP_CENTER_NOTE',
                        'created_at' => new \DateTime()
                    ];
                    $connection->table('log')->insert($logData);
                }
            }
            
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function updateSpam(Request $request) {
        $ids = $request->input('ids');
        $response = ['status' => 'fail'];
        if (!empty($ids)) {
            $emails = Ticket::whereIn('id', $ids)
                            ->pluck('email')->toArray();
            Ticket::whereIn('id', $ids)->update(['is_spam'=> 1]);
            foreach ($emails as $email) {
                $checkExists = TicketFilterSpam::where('type', '=', 'email')
                                    ->where('value', '=', $email)
                                    ->exists();
                if (!$checkExists) {
                    TicketFilterSpam::create(['type' => 'email', 'value' => $email]);
                }
            }
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function updateNotSpam(Request $request) {
        $ids = $request->input('ids');
        $response = ['status' => 'fail'];
        if (!empty($ids)) {
            $emails = Ticket::whereIn('id', $ids)
                            ->pluck('email')->toArray();
            Ticket::whereIn('id', $ids)->update(['is_spam'=> 0]);
            foreach ($emails as $email) {
                TicketFilterSpam::where('type', '=', 'email')
                                    ->where('value', '=', $email)
                                    ->delete();
            }
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function createFilterSpam(Request $request) {
        $emails = $request->input('emails');
        $includes = $request->input('includes');
        $excludes = $request->input('excludes');
        $response = [
            'status' => 'successful'
        ];
        if (!empty($emails)) {
            $emails = explode(',', $emails);
            foreach ($emails as $email) {
                $email = trim($email);
                if (!empty($email)) {
                    $exists = TicketFilterSpam::where('type', '=', 'email')
                                    ->where('value', '=', $email)
                                    ->exists();
                    if (!$exists) {
                        TicketFilterSpam::create(['type' => 'email', 'value' => $email]);
                    }
                }
            }
        }
        if (!empty($includes)) {
            $includes = explode(',', $includes);
            foreach ($includes as $include) {
                $include = trim($include);
                if (!empty($include)) {
                    $exists = TicketFilterSpam::where('type', '=', 'include')
                                    ->where('value', '=', $include)
                                    ->exists();
                    if (!$exists) {
                        TicketFilterSpam::create(['type' => 'include', 'value' => $include]);
                    }
                }
            }
        }
        if (!empty($excludes)) {
            $excludes = explode(',', $excludes);
            foreach ($excludes as $exclude) {
                $exclude = trim($exclude);
                if (!empty($exclude)) {
                    $exists = TicketFilterSpam::where('type', '=', 'exclude')
                                    ->where('value', '=', $exclude)
                                    ->exists();
                    if (!$exists) {
                        TicketFilterSpam::create(['type' => 'exclude', 'value' => $exclude]);
                    }
                }
            }
        }

        $filters = $this->getFilterTicketSpam();
        $queryTime = date('Y-m-d H:i:s', strtotime('-14 days'));
        $query = Ticket::where('created_at', '>=', $queryTime)
                        ->where('is_spam', '=', 0);
        $queryTwo = clone $query;
        if (!empty($filters['emails'])) {
            $query->whereIn('email', $filters['emails']);
            $items = $query->get();
            $query->update(['is_spam' => 1]);
        }
        $items = [];
        if (!empty($filters['includes'])) {
            $queryTwo->where(function($q) use ($filters) {
                foreach($filters['includes'] as $string) {
                    $q->orWhere('title', 'LIKE', '%' . $string . '%');
                }
            });
            $items = $queryTwo->pluck('title', 'id')->toArray();
        }
        if (!empty($items) && !empty($filters['excludes'])) {
            foreach($items as $key => $vl) {
                foreach ($filters['excludes'] as $exclude) {
                    $pos = strpos($vl, $exclude);
                    if ($pos !== false) {
                        unset($items[$key]);
                    }
                }
            }
        }
        if (!empty($items)) {
            $spamIds = [];
            foreach ($items as $key => $value) {
                $spamIds[] = $key;
            }
            Ticket::whereIn('id', $spamIds)->update(['is_spam' => 1]);
        }

        return response()->json($response);
    }

    public function getFilterSpam() {
        $items = TicketFilterSpam::all();
        $emails = [];
        $includes = [];
        $excludes = [];
        foreach ($items as $item) {
            if ($item->type == 'email') {
                $emails[] = $item;
            }
            if ($item->type == 'include') {
                $includes[] = $item;
            }
            if ($item->type == 'exclude') {
                $excludes[] = $item;
            }
        }
        $response = [
            'status' => 'successful',
            'emails' => $emails,
            'includes' => $includes,
            'excludes' => $excludes
        ];
        return response()->json($response);
    }

    public function removeFilterItem(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('id') && $request->input('id') != '') {
            TicketFilterSpam::where('id', '=', $request->input('id'))->delete();
            $queryTime = date('Y-m-d H:i:s', strtotime('-14 days'));
            if ($request->input('type') == 'email') {
                $query = Ticket::where('created_at', '>=', $queryTime)
                        ->where('is_spam', '=', 1)
                        ->where('email', '=', $request->input('value'))
                        ->update(['is_spam' => 0]);
            }
            if ($request->input('type') == 'include') {
                $query = Ticket::where('created_at', '>=', $queryTime)
                        ->where('is_spam', '=', 1)
                        ->where('title', 'LIKE', '%' . $request->input('value') . '%')
                        ->update(['is_spam' => 0]);
            }
            $response = [
                'status' => 'successful'
            ];
        }
        return response()->json($response);
    }

    public function approveRefund(Request $request) {
        $input = $request->all();
        $response = ['status' => 'fail'];
        if (array_key_exists('ticket_id', $input) && array_key_exists('order_id', $input)) {
            $ticket = Ticket::find($input['ticket_id']);
            if ($ticket) {
                $connection = DBConnect::connect($ticket->locate);
                $updateTicketOrder = TicketNOrder::where('order_id', '=', $input['order_id'])
                                                    ->where('ticket_id', '=', $ticket->id);
                
                $isRefund = 0;
                if ($input['is_refund'] === 1) {
                    $isRefund = 1;
                    $updateTicketOrder->update(['is_refund' => 1]);
                    $ticket->is_refund = 1;
                } else {
                    $updateTicketOrder->update(['is_refund' => 0]);
                    $ticket->is_refund = 0;
                }
                $ticket->save();
                $orderMeta = [
                    'order_id' => $input['order_id'],
                    'key' => 'is_refund',
                    'value' => $isRefund
                ];
                $checkExists = $connection->table('order_meta')
                            ->where('order_id', '=', $orderMeta['order_id'])
                            ->where('key', '=', $orderMeta['key'])
                            ->exists();
                if (!$checkExists) {
                    $connection->table('order_meta')->insert($orderMeta);
                } else {
                    $checkExists = $connection->table('order_meta')
                            ->where('order_id', '=', $orderMeta['order_id'])
                            ->where('key', '=', $orderMeta['key'])
                            ->update($orderMeta);
                }
            }
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function rebuildReply(Request $request) {
        $response = ['status' => 'successful'];
        $queryTime = date('Y-m-d H:i:s', strtotime('-30 days'));
        $items = Ticket::with(['ticketItems' => function ($query) {
                                $query->orderBy('created_at', 'DESC');
                            }
                        ])
                        ->where('created_at', '>=', $queryTime)
                        ->get()->toArray();
        foreach ($items as $ticket) {
            if (count($ticket['ticket_items']) > 0) {
                $dataUpdate = [];
                if ($ticket['ticket_items'][0]['type'] == 'staff') {
                    $dataUpdate['is_reply'] = 1;
                }
                foreach ($ticket['ticket_items'] as $item) {
                    if ($item['type'] == 'staff') {
                        $dataUpdate['reply_at'] = $item['created_at'];
                        break;
                    }
                }
                if (!empty($dataUpdate)) {
                    Ticket::where('id', '=', $ticket['id'])
                            ->update($dataUpdate);
                }
            }
        }
        return response()->json($response);
    }

    public function createTicketFromOrder(Request $request) {
        $input = $request->all();
        $locates = getModuleLocale();
        $locateAvailables = [];
        foreach ($locates as $locate) {
            if ($locate['enable'] && $locate['locale'] != 'central') {
                $locateAvailables[] = $locate['locale'];
            }
        }
        $response = ['status' => 'fail'];
        if (array_key_exists('market', $input) && in_array($input['market'], $locateAvailables)) {
            $timeNow = time();
            $date = date('Y-m-d H:i:s', strtotime('+1 day', $timeNow));
            $deadline = \DateTime::createFromFormat('Y-m-d H:i:s', $date);
            $data = [
                'title' => __('Order confirmation - Order number ') . $input['order_code'] . ' Printerval',
                'email' => $input['email'],
                'customer_name' => $input['name'],
                'order_code' => $input['order_code'],
                'token_customer' => md5($input['email']),
                'token_ticket' => md5(time()) . Str::random(4),
                'status' => 'open',
                'type' => 'complaint',
                'from' => 'fulfillment',
                'locate' => $input['market'],
                'note' => $input['note'],
                'deadline' => $deadline
            ];
            $ticket = Ticket::create($data);
            $ticketDuplicates = $this->checkDuplicateOrder($ticket, $input['order_id']);
            $ticketNOrderData = ['ticket_id' => $ticket->id, 'order_id' => $input['order_id'], 'type' => 'order_code'];
            if (!empty($ticketDuplicates)) {
                $ticketNOrderData['ticket_related'] = implode(',', $ticketDuplicates);
            }
            TicketNOrder::create($ticketNOrderData);
            $this->triggerAsyncRequest(\URL::to('/') . "/ticket/assign?ticket_id=" . $ticket->id, "GET");
            $response['status'] = 'successful';
            $ticket->deadline = $date;
            $connection = DBConnect::connect($ticket->locate);
            $orderMeta = [
                'order_id' => $input['order_id'],
                'key' => 'is_ticket',
                'value' => 1
            ];
            $checkExists = $connection->table('order_meta')
                        ->where('order_id', '=', $orderMeta['order_id'])
                        ->where('key', '=', $orderMeta['key'])
                        ->exists();
            if (!$checkExists) {
                $connection->table('order_meta')->insert($orderMeta);
            }
            $response['result'] = $ticket;
            
        }
        return response()->json($response);
    }

    public function loadTicket(Request $request) {
        $items = $request->input('items');
        $orderIds = [];
        $keys = [];
        foreach ($items as $item) {
            $orderIds[] = $item['order_id'];
            $keys[] = $item['order_id'] . '-' . $item['locate'];
        }
        $tickets = Ticket::leftJoin('ticket_n_order', 'ticket_n_order.ticket_id', '=', 'ticket.id')
                        ->whereIn('ticket_n_order.order_id', $orderIds)
                        ->where('ticket.from', '=', 'fulfillment')
                        ->get(['ticket.id', 'ticket.status', 'ticket.locate', 'ticket.user_id', 'ticket.note', 'ticket.deadline', 'ticket.order_code', 'ticket_n_order.order_id']);
        $userIds = [];

        foreach ($tickets as $ticket) {
            $userIds[] = $ticket->user_id;
        }

        $users = User::whereIn('id', $userIds)
                        ->pluck('name', 'id')->toArray();
        $result = [];
        foreach($tickets as $ticket) {
            $key = $ticket->order_id . '-' . $ticket->locate;
            if (in_array($key, $keys)) {
                if (isset($users[$ticket->user_id])) {
                    $ticket->user = $users[$ticket->user_id];
                }
                $result[] = $ticket;
            }
        }
        $response = [
            'status' => 'successful',
            'result' => $result
        ];
        return response()->json($response);
    }

    public function rebuildTicket(Request $request) {
        $items = Ticket::with(['ticketItems' => function ($query) {
                    $query->orderBy('created_at', 'ASC');
                }])
                ->get();
        $customers = [];
        $emails = [];
        $fulfillments = [];
        foreach ($items as $item) {
            if ($item->is_from_email == 0) {
                if (isset($item->ticketItems[0]) && $item->ticketItems[0]->type == 'customer') {
                    $customers[] = $item->id;
                }
            } else {
                if (isset($item->ticketItems[0]) && $item->ticketItems[0]->type == 'staff') {
                    $fulfillments[] = $item->id;
                } else {
                    $emails[] = $item->id;
                }
            }
        }
        if (!empty($customers)) {
            Ticket::whereIn('id', $customers)->update(['from' => 'customer']);
        }
        if (!empty($emails)) {
            Ticket::whereIn('id', $emails)->update(['from' => 'email']);
        }
        if (!empty($fulfillments)) {
            Ticket::whereIn('id', $fulfillments)->update(['from' => 'fulfillment']);
        }
        return response()->json(['status' => 'successful']);
    }

}