<?php

namespace Modules\TrackingOrder\Controllers;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;
use Module;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Modules\TrackingOrder\Controllers\Controller;
use Modules\TrackingOrder\Models\Order;
use Modules\TrackingOrder\Models\OrderMeta;

class HomeController extends Controller
{
    const EXPIRE_TIME = 3 * 86400;
    const MAX_AFFILIATE_TIME = 45 * 86400;

    public function __construct()
    {        
        Module::onView("content", function() {
            return "This is content view from TrackingOrder Module HomeController";
        }, 5);
    }

    public function index(Request $request)
    {
        $message = config("tracking-order::app.message");
        return view('tracking-order::home.welcome', [
            'message' => $message,
        ]);
    }

    public function updateTrackingSource (Request $request) {
        $orderId = $request->input('order_id');
        $token = $request->input('token');
        if ($orderId && $token) {
            $dataLog = [
                'token' => $token
            ];
            DB::table('log')->insert([
                'target_type' => 'ORDER',
                'target_id' => $orderId,
                'event_type' => 'MODULE_UPDATE_TRACKING',
                'data' => json_encode($dataLog),
                'created_at' => new \DateTime()
            ]);
            $trackingAttributes = $this->buildTrackingAttributes($token);
            if (count($trackingAttributes)) {
                $this->storeOrderAttributes($trackingAttributes, $orderId);
            }
        }
        return response()->json([
            'status' => 'successful'
        ]);
    }

    private function storeOrderAttributes ($trackingAttributes, $orderId) {
        foreach ($trackingAttributes as $key => $value) {
            if ($value) {
                $this->storeAttribute($orderId, $key, $value);
            }
            try {
                if ($key == 'token') {
                    $orderToken = DB::table('order_token')
                        ->where('order_id', '=', $orderId)
                        ->first();
                    $date = date('Y-m-d H:i:s');
                    if (!$orderToken) {
                        DB::table('order_token')->insert([
                            'order_id' => $orderId,
                            'token' => $value,
                            'created_at' => $date,
                            'updated_at' => $date,
                        ]);
                    } else if ($value != $orderToken->token) {
                        DB::table('order_token')->where('id', '=', $orderToken->id)
                            ->update([
                                'token' => $value,
                                'updated_at' => $date,
                            ]);
                    }
                }
            } catch (\Exception $exception) {
                
            }
        }
    }

    private function storeAttribute ($orderId, $key, $value) {
        if (is_array($value)) {
            $value = json_encode($value);
        }
        $table = config('tracking-order::sa.tracking_table');
        $trackingObj = DB::table($table)
            ->where('order_id', '=', $orderId)
            ->where('key', '=', $key)
            ->first();
        if (!$trackingObj) {
            DB::table($table)->insert([
                'order_id' => $orderId,
                'key' => $key,
                'value' => $value
            ]);
        } else if ($value != $trackingObj->value) {
            if ($key == 'from'
                && $trackingObj->value
                && in_array($trackingObj->value, ['facebook_ads', 'instagram_ads'])) {
                //@todo update clicker
                return;
            }
            DB::table($table)->where('id', '=', $trackingObj->id)
                ->update([
                    'value' => $value
                ]);
        }
    }


    public function buildTrackingAttributes($tokenUserQuery) {
        $retVal = [];
        if ($tokenUserQuery) {
            $retVal['token'] = $tokenUserQuery;
            $clickerSource = Redis::get('source::'. $tokenUserQuery);
            if($clickerSource) {
                $trackingInfo = [];
                $this->buildTrackingByFrom($tokenUserQuery, $clickerSource, $trackingInfo);
                $retVal["from"] = $clickerSource;
                if (count($trackingInfo)) {
                    $retVal["tracking_info"] = $trackingInfo;
                }
                $sources = ['adwords', 'bing ads', 'facebook_ads', 'instagram_ads'];
                if (in_array($clickerSource, $sources)) {
                    $campaignId = $this->getDataByKey($tokenUserQuery, 'campaignid');
                    if (!$campaignId) {
                        $campaignId = $this->getDataByKey($tokenUserQuery, 'campaign_id');
                    }
                    $retVal['campaign_id'] = $campaignId;
                }
                $campaignName = $this->getDataByKey($tokenUserQuery, 'campaign_name');
                if ($campaignName) {
                    $retVal['campaign_name'] = $campaignName;
                }

                if ($clickerSource == 'adwords') {
                    $retVal['gclid'] = $this->getDataByKey($tokenUserQuery, 'gclid');
                }

                if ($clickerSource == 'affiliate') {
                    //chỉ tính affiliate trong vòng 45 ngày
                    $retVal['affiliate_code'] = $this->getDataByKey($tokenUserQuery, 'affiliate_id', [self::EXPIRE_TIME, self::MAX_AFFILIATE_TIME]);
                }
                $listAttrsTracking = config('tracking-order::clicker.attrs_tracking', []);
                if ($listAttrsTracking && !empty($listAttrsTracking[$clickerSource])) {
                    foreach ($listAttrsTracking[$clickerSource] as $attrTracking) {
                        $valueTracking = $this->getDataByKey($tokenUserQuery, $attrTracking);
                        if ($valueTracking) {
                            $retVal[$attrTracking] = $valueTracking;
                        }
                    }
                }
            }
            $clickerSourceFlow = Redis::get('source_flow::'. $tokenUserQuery);
            if($clickerSourceFlow) {
                $retVal["from_flow"] = $clickerSourceFlow;
            }
            $result = $this->getUserAgentFromClicker($tokenUserQuery);
            if (count($result)) {
                foreach ($result as $key => $value) {
                    $retVal[$key] = $value;
                }
            }
        }

        return $retVal;
    }

    private function buildTrackingByFrom($tokenUserQuery, $from, &$trackingInfo) {
        if (!$tokenUserQuery) {
            return;
        }

        $clickerConfig = config('tracking-order::clicker.sources');
        if($from && $clickerConfig && array_key_exists($from, $clickerConfig)) {
            $trackingSource = Redis::get('source::' . $from . $tokenUserQuery);
            $trackingColumn = isset($clickerConfig[$from]) && isset($clickerConfig[$from]['save_column'])
                ? $clickerConfig[$from]['save_column']
                : '';
            if($trackingSource && $trackingColumn) {
                $trackingInfo[$trackingColumn] = $trackingSource;
            }
        }
        $attrsJson = Redis::get('attrs::' . $tokenUserQuery);
        if($attrsJson) {
            try {
                $attrs = json_decode($attrsJson, true);
                if(is_array($attrs) && count($attrs) > 0) {
                    foreach ($attrs as $attrItem) {
                        $attr = isset($attrItem['attr']) ? $attrItem['attr'] : '';
                        $value = isset($attrItem['value']) ? $attrItem['value'] : '';
                        $trackingInfo[$attr] = $value;
                    }
                }
            } catch (\Exception $ex) {
                Log::error('buildTrackingByFrom: ' . $ex->getMessage() . " Line: {$ex->getLine()} File: {$ex->getFile()}");
            }
        }
    }

    //maxTime[0] : expire time in second
    //maxTime[1] : time in second

    public function getDataByKey($tokenUserQuery, $key, $maxTime = [-1, -1]) {
        $retVal = "";
        if ($tokenUserQuery) {
            $cacheKey = 'source::clickerDataTracking::'. $tokenUserQuery;
            $clickerDataTracking = Redis::get($cacheKey);
            if ($clickerDataTracking) {
                try {
                    $clickerDataTracking = json_decode($clickerDataTracking, true);
                    if (isset($clickerDataTracking[$key])) {
                        $retVal = $clickerDataTracking[$key];
                    }
                } catch (\Exception $ex) {
                    $retVal = "";
                }
            }
        }

        return $retVal;
    }

    public function getSource(Request $request) {
        $token = $request->get('token');
        if (!$token) {
            return [
                'status' => 'fail',
                'result' => ''
            ];
        }
        $sourceKey = 'source::'. $token;
        $clickerSource = '';
        if (Redis::get($sourceKey)) {
            $clickerSource = Redis::get($sourceKey);
        } else {
            $clickerObj = $this->getSourceFromClicker($token);
            if($clickerObj && isset($clickerObj->source)) {
                $clickerSource = $clickerObj->source;
                Redis::set($sourceKey, $clickerSource, 'EX', self::EXPIRE_TIME);
            }
        }

        return [
            'status' => 'successful',
            'result' => $clickerSource
        ];
    }

    public function buildTrackingSource(Request $request)
    {
        $result = [
            'status' => 'successful'
        ];
        $token = $request->get('token');
        if($token) {
            $clickerDataTrackingKey = 'source::clickerDataTracking::' . $token;
            $sourceKey = 'source::'. $token;
            $sourceFlowKey = 'source_flow::'. $token;
            $attrsKey = 'attrs::' . $token;
            $sourceUrlKey = 'source::url::' . $token;
            $clickerConfig = config('tracking-order::clicker.sources');
            $clickerObj = $this->getSourceFromClicker($token);
            $clickerSources = $this->getSourceFlowFromClicker($token);
            $clickerSource = '';
            if($clickerObj && isset($clickerObj->source)) {
                $clickerSource = $clickerObj->source;
            }
            $clickerDataTracking = null;
            try {
                if (isset($clickerObj->data)) {
                    Redis::set($clickerDataTrackingKey, $clickerObj->data, 'EX', self::EXPIRE_TIME);
                    $clickerDataTracking = json_decode($clickerObj->data, true);
                }
            } catch(\Exception $e) {
                $clickerDataTracking = null;
            }
            if ($clickerObj
                && $clickerSource
                && array_key_exists($clickerSource, $clickerConfig)
                && $clickerDataTracking) {
                $paramTracking = $clickerConfig[$clickerSource]['tracking_column'];
                if(isset($clickerDataTracking[$paramTracking])) {
                    $valueTracking = $clickerDataTracking[$paramTracking];
                    $keyTracking = 'source::' . $clickerSource . $token;
                    Redis::set($keyTracking, $valueTracking, 'EX', self::EXPIRE_TIME);
                }
            }

            Redis::set($sourceKey, $clickerSource, 'EX', self::EXPIRE_TIME);
            Redis::set($sourceFlowKey, $clickerSources, 'EX', self::EXPIRE_TIME);
            if ($clickerObj && isset($clickerObj->attrs) && is_array($clickerObj->attrs) && count($clickerObj->attrs) > 0) {
                Redis::set($attrsKey, json_encode($clickerObj->attrs));
            }
            if($clickerObj && isset($clickerObj->url) && $clickerObj->url) {
                $urlExplode = explode('?', $clickerObj->url);
                $url = $urlExplode && isset($urlExplode[0]) ? $urlExplode[0] : '';
                Redis::set($sourceUrlKey, $url, 'EX', self::EXPIRE_TIME);
            }

        }
        return response()->json($result);
    }

    private function getSourceFromClicker($tokenUserQuery){
        $url = config('tracking-order::clicker.domain') . "/service/tracking/get-source?_token=$tokenUserQuery";
        $clickerReponse = $this->triggerSyncRequest($url, 'GET', []);

        $result = '';
        if($clickerReponse
            && isset($clickerReponse->status)
            && isset($clickerReponse->data->item)
            && $clickerReponse->status == 'successful'){
            $result = $clickerReponse->data->item;
        }

        return $result;
    }

    private function getSourceFlowFromClicker($tokenUserQuery){
        $url = config('tracking-order::clicker.domain') . "/service/tracking/get-history-from-token?_token=$tokenUserQuery";
        $clickerReponse = $this->triggerSyncRequest($url, 'GET', []);
        $sources = '';
        if($clickerReponse
            && isset($clickerReponse->status)
            && isset($clickerReponse->data)
            && $clickerReponse->status == 'successful'){
            $sources = $clickerReponse->data->sources;
        }
        return $sources;
    }

    protected function triggerSyncRequest($url, $method = 'GET', $params = [], $headers = []) {
        $ch = curl_init();
        $timeout = 500;
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        if ($headers) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        if ($method != 'GET') {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
        }

        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        $data = curl_exec($ch);
        curl_close($ch);
        return json_decode($data);
    }

    public static function getBrowser($userAgent)
    {

        $platform = "N/A";
        $browser = "N/A";
            $osRegex = array(
                '/windows nt 10/i' => 'Windows 10',
                '/windows nt 6.3/i' => 'Windows 8.1',
                '/windows nt 6.2/i' => 'Windows 8',
                '/windows nt 6.1/i' => 'Windows 7',
                '/windows nt 6.0/i' => 'Windows Vista',
                '/windows nt 5.2/i' => 'Windows Server 2003/XP x64',
                '/windows nt 5.1/i' => 'Windows XP',
                '/windows xp/i' => 'Windows XP',
                '/windows nt 5.0/i' => 'Windows 2000',
                '/windows me/i' => 'Windows ME',
                '/win98/i' => 'Windows 98',
                '/win95/i' => 'Windows 95',
                '/win16/i' => 'Windows 3.11',
                '/macintosh|mac os x/i' => 'Mac OS X',
                '/mac_powerpc/i' => 'Mac OS 9',
                '/linux/i' => 'Linux',
                '/ubuntu/i' => 'Ubuntu',
                '/iphone/i' => 'iPhone',
                '/ipod/i' => 'iPod',
                '/ipad/i' => 'iPad',
                '/android/i' => 'Android',
                '/blackberry/i' => 'BlackBerry',
                '/webos/i' => 'Mobile',
                '/windows phone/i' => 'Windows Phone',
                '/PrintervalApp\/IOS/i' => 'App IOS',
                '/PrintervalApp\/Android/i' => 'App Android',
        );
        $browserRegex = array(
            '/MSIE/i' => 'IE',
            '/Firefox/i' => 'Firefox',
            '/Chrome/i' => 'Chrome',
            '/Safari/i' => 'Safari',
            '/Opera/i' => 'Opera',
            '/Netscape/i' => 'Netscape'
        );

        foreach ($osRegex as $regex => $value) {

            if (preg_match($regex, $userAgent)) {
                $platform = $value;
            }
        }
        foreach ($browserRegex as $regex => $value) {

            if (preg_match($regex, $userAgent)) {
                $browser = $value;
                break;
            }
        }

        return array(
            "platform" => $platform,
            "browser" => $browser
        );
    }

    private function getUserAgentFromClicker($tokenUserQuery) {
        $url = config('tracking-order::clicker.domain') . "/service/tracking/get-user-agent?_token=$tokenUserQuery";
        $clickerReponse = $this->triggerSyncRequest($url, 'GET', []);
        $retVal = [
            'ip' => '',
            'user_agent' => '',
            'device' => '',
            'browser' => '',
            'platform' => ''
        ];
        if(
            $clickerReponse &&
            isset($clickerReponse->status) &&
            isset($clickerReponse->result) && 
            $clickerReponse->status == 'successful'
        ){
            $userAgent = $clickerReponse->result->user_agent;
            $ip = $clickerReponse->result->ip;
            $device = self::getUserDevice($userAgent);
            $result = self::getBrowser($userAgent);

            $retVal['user_agent'] = $userAgent;
            $retVal['ip'] = $ip;
            $retVal['device'] = $device;
            $retVal['browser'] = $result['browser'];
            $retVal['platform'] = $result['platform'];
        }


        return $retVal;
    }

    public static function isMobile($userAgent)
    {
        return preg_match("/(android|avantgo|blackberry|bolt|boost|cricket|docomo|fone|hiptop|mini|palm|phone|pie|tablet|up\.browser|up\.link|webos|wos)/i", isset($userAgent) ? $userAgent : "");
    }

    public static function isTablet($userAgent)
    {
        return preg_match("/(ipad|tablet)/i", isset($userAgent) ? $userAgent : "");
    }

    public static function getUserDevice($userAgent)
    {
        $result = "Desktop";
        if (self::isTablet($userAgent)) {
            $result = 'Tablet';
        } else if (self::isMobile($userAgent)) {
            $result = 'Mobile';
        }
        return $result;
    }

    public function getUserDeviceV2($userAgent)
    {
        $result = "Desktop";

        try {
            $detect = new MobileDetect;
            if ($detect->isMobile()) {
                $result = 'Mobile';
            }
            if($detect->isTablet()) {
                $result = 'Tablet';
            }
        }
        catch (\Exception $ex) 
        {
            DB::table('log')->insert([
                'target_type' => 'user_agent',
                'target_id' => null,
                'event_type' => 'ERROR_DETECT_DEVICE',
                'data' => json_encode([
                    'user_agent' => $userAgent,
                    'message' => $ex->getMessage()
                ]),
                'created_at' => new \DateTime()
            ]);

            if (self::isTablet($userAgent)) {
                $result = 'Tablet';
            } else if (self::isMobile($userAgent)) {
                $result = 'Mobile';
            }
        }

        return $result;
    }

    public function checkDetector(Request $request)
    {
        $userAgent = $request->header('User-Agent');

        return [
            'device' => $this->getUserDeviceV2($userAgent)
        ];
    }

    public function rebuildTracking(Request $request) {
        set_time_limit(3600);
        $site = $request->input('site');
        if ($request->get('order_ids')) {
            $items = Order::with(['orderMeta' => function ($q) {
                $q->where('key', '=', 'user_agent');
            }])
            ->whereIn('id', explode(',', $request->get('order_ids')))
            ->get(['id', 'created_at']);
        } else {
            $from = $request->has('from') ? $request->get('from') : date('d/m/Y', strtotime('-1 day'));
            $to = $request->has('to') ? $request->get('to') : date('d/m/Y');
            $from = \DateTime::createFromFormat('d/m/Y', $from);
            $to = \DateTime::createFromFormat('d/m/Y', $to);
            $from->setTime(0,0,0);
            $to->setTime(23,59,59);
            $items = Order::with(['orderMeta' => function ($q) {
                                $q->where('key', '=', 'user_agent');
                            }])
                            ->where('created_at', '>=', $from)
                            ->where('created_at', '<=', $to)
                            ->get(['id', 'created_at']);
        }
        $orderIds = [];
        $username = config('tracking-order::clicker.authBasic.username');
        $password = config('tracking-order::clicker.authBasic.password');
        if ($username == null || $password == null) {
            throw new \Exception('The module Product Advertising require "clickerAuthBasic.username" and "clickerAuthBasic.password" in /config/app.php');
        }
        $headers = array(
            'Content-Type:application/json',
            'Authorization: Basic '. base64_encode($username . ":" . $password)
        );
        $locale = env('APP_LOCALE');
        foreach($items as $item) {
            $isExistsMetaSource = $this->isExistsMetaSource($item->id);
            if ($item->orderMeta->count() == 0 ||  $request->has('force_rebuild') || !$isExistsMetaSource) {
                $orderIds[] = $item->id;
                $tokenTracking = $this->getTokenTracking($item->id);
                $url = config('tracking-order::clicker.domain') . "/service/get-order-success-data?id=" . $item->id . "&from_time=" . urlencode(date('Y-m-d H:i:s', strtotime('-2 hours', strtotime($item->created_at)))) . "&to_time=" . urlencode(date('Y-m-d H:i:s', strtotime('+2 hours', strtotime($item->created_at))));
                if ($site) {
                    $url .= '&site=' . $site;
                }
                if ($locale) {
                    $url .= '&locale=' . $locale;
                }
                if ($tokenTracking) {
                    $url .= '&tokens=' . implode(',', $tokenTracking);
                }
                $output = $this->triggerSyncRequest($url, 'GET', [], $headers);
                if ($output && !empty($output->data)) {
                    $listAttribute = [];
                    if (!empty($output->data->dataSource)) {
                        $clickerSource = $output->data->dataSource;
                        $listAttribute['token'] = $clickerSource->item->token;
                        $listAttribute["from"] = $clickerSource->source;
                        $sources = ['adwords', 'bing ads', 'facebook_ads', 'instagram_ads'];
                        $dataSource = json_decode($clickerSource->item->data, true);
                        if (in_array($clickerSource->source, $sources)) {
                            $campaignId = isset($dataSource['campaignid']) ? $dataSource['campaignid']: '';
                            if (!$campaignId) {
                                $campaignId = isset($dataSource['campaign_id']) ? $dataSource['campaign_id']: '';
                            }
                            if ($campaignId) {
                                $listAttribute['campaign_id'] = $campaignId;
                            }
                        }
                        $campaignName = isset($dataSource['campaign_name']) ? $dataSource['campaign_name']: '';
                        if ($campaignName) {
                            $listAttribute['campaign_name'] = $campaignName;
                        }
                        if ($clickerSource->source == 'adwords') {
                            $listAttribute['gclid'] = isset($dataSource['gclid']) ? $dataSource['gclid']: '';
                        }
                    }
                    if (!empty($output->data->dataSourceFlow)) {
                        $listAttribute["from_flow"] = $output->data->dataSourceFlow;
                    }
                    if (!empty($output->data->dataUserAgent)) {
                        $userAgent = $output->data->dataUserAgent->user_agent;
                        $ip = $output->data->dataUserAgent->ip;
                        $device = self::getUserDevice($userAgent);
                        $resultBrowser = self::getBrowser($userAgent);

                        $listAttribute['user_agent'] = $userAgent;
                        $listAttribute['ip'] = $ip;
                        $listAttribute['device'] = $device;
                        $listAttribute['browser'] = $resultBrowser['browser'];
                        $listAttribute['platform'] = $resultBrowser['platform'];
                    }
                    if ($listAttribute) {
                        $this->storeOrderAttributes($listAttribute, $item->id);
                    }
                }
            }
        }
        return response()->json([
            'status' => 'successful',
            'data' => $orderIds
        ]);
    }

    public function getTokenTracking($orderId) {
        $result = [];
        $cartMeta = DB::table('order_meta')
            ->where('order_id', $orderId)
            ->where('key', 'cart_id')
            ->first(['value']);
        $cartId = isset($cartMeta->value) ? $cartMeta->value : null;
        if ($cartId && \Illuminate\Support\Facades\Schema::hasTable('cart_tracking')) {
            $tokens = DB::table('cart_tracking')
                ->where('cart_id', $cartId)
                ->groupBy('token')
                ->pluck('token')
                ->toArray();
            if ($tokens && count($tokens) > 1) {
                $result = $tokens;
            }
        }
        return $result;
    }

    public function isExistsMetaSource($orderId) {
        return DB::table('order_meta')
            ->where('order_id', '=', $orderId)
            ->where('key', '=', 'from')
            ->exists();
    }
    public function rebuildTrackinOrderLossSource(Request $request) {
        set_time_limit(3600);
        $site = $request->input('site');
        $retVal = [];
        $from = $request->has('from') ? $request->get('from') : date('d/m/Y', strtotime('-1 day'));
        $to = $request->has('to') ? $request->get('to') : date('d/m/Y');
        $orders = Order::leftJoin('order_meta', function($join) {
                $join->on('order.id', '=', 'order_meta.order_id')
                    ->where('order_meta.key', '=', 'from');
            })
            ->whereNull('order_meta.id')
            ->where('order.created_at', '>=', $from)
            ->where('order.created_at', '<=', $to)
            ->get(['order.id', 'order.created_at']);
        //from orders build finish order url. from finish order url get 
        $paths = ["cloudflare-http-request-20240411_00.csv","cloudflare-http-request-20240412_06.csv","cloudflare-http-request-20240413_12.csv","cloudflare-http-request-20240414_18.csv","cloudflare-http-request-20240411_01.csv","cloudflare-http-request-20240412_07.csv","cloudflare-http-request-20240413_13.csv","cloudflare-http-request-20240414_19.csv","cloudflare-http-request-20240411_02.csv","cloudflare-http-request-20240412_08.csv","cloudflare-http-request-20240413_14.csv","cloudflare-http-request-20240414_20.csv","cloudflare-http-request-20240411_03.csv","cloudflare-http-request-20240412_09.csv","cloudflare-http-request-20240413_15.csv","cloudflare-http-request-20240414_21.csv","cloudflare-http-request-20240411_04.csv","cloudflare-http-request-20240412_10.csv","cloudflare-http-request-20240413_16.csv","cloudflare-http-request-20240414_22.csv","cloudflare-http-request-20240411_05.csv","cloudflare-http-request-20240412_11.csv","cloudflare-http-request-20240413_17.csv","cloudflare-http-request-20240414_23.csv","cloudflare-http-request-20240411_06.csv","cloudflare-http-request-20240412_12.csv","cloudflare-http-request-20240413_18.csv","cloudflare-http-request-20240415_00.csv","cloudflare-http-request-20240411_07.csv","cloudflare-http-request-20240412_13.csv","cloudflare-http-request-20240413_19.csv","cloudflare-http-request-20240415_01.csv","cloudflare-http-request-20240411_08.csv","cloudflare-http-request-20240412_14.csv","cloudflare-http-request-20240413_20.csv","cloudflare-http-request-20240415_02.csv","cloudflare-http-request-20240411_09.csv","cloudflare-http-request-20240412_15.csv","cloudflare-http-request-20240413_21.csv","cloudflare-http-request-20240415_03.csv","cloudflare-http-request-20240411_10.csv","cloudflare-http-request-20240412_16.csv","cloudflare-http-request-20240413_22.csv","cloudflare-http-request-20240415_04.csv","cloudflare-http-request-20240411_11.csv","cloudflare-http-request-20240412_17.csv","cloudflare-http-request-20240413_23.csv","cloudflare-http-request-20240415_05.csv","cloudflare-http-request-20240411_12.csv","cloudflare-http-request-20240412_18.csv","cloudflare-http-request-20240414_00.csv","cloudflare-http-request-20240415_06.csv","cloudflare-http-request-20240411_13.csv","cloudflare-http-request-20240412_19.csv","cloudflare-http-request-20240414_01.csv","cloudflare-http-request-20240415_07.csv","cloudflare-http-request-20240411_14.csv","cloudflare-http-request-20240412_20.csv","cloudflare-http-request-20240414_02.csv","cloudflare-http-request-20240415_08.csv","cloudflare-http-request-20240411_15.csv","cloudflare-http-request-20240412_21.csv","cloudflare-http-request-20240414_03.csv","cloudflare-http-request-20240415_09.csv","cloudflare-http-request-20240411_16.csv","cloudflare-http-request-20240412_22.csv","cloudflare-http-request-20240414_04.csv","cloudflare-http-request-20240415_10.csv","cloudflare-http-request-20240411_17.csv","cloudflare-http-request-20240412_23.csv","cloudflare-http-request-20240414_05.csv","cloudflare-http-request-20240415_11.csv","cloudflare-http-request-20240411_18.csv","cloudflare-http-request-20240413_00.csv","cloudflare-http-request-20240414_06.csv","cloudflare-http-request-20240415_12.csv","cloudflare-http-request-20240411_19.csv","cloudflare-http-request-20240413_01.csv","cloudflare-http-request-20240414_07.csv","cloudflare-http-request-20240415_13.csv","cloudflare-http-request-20240411_20.csv","cloudflare-http-request-20240413_02.csv","cloudflare-http-request-20240414_08.csv","cloudflare-http-request-20240415_14.csv","cloudflare-http-request-20240411_21.csv","cloudflare-http-request-20240413_03.csv","cloudflare-http-request-20240414_09.csv","cloudflare-http-request-20240415_15.csv","cloudflare-http-request-20240411_22.csv","cloudflare-http-request-20240413_04.csv","cloudflare-http-request-20240414_10.csv","cloudflare-http-request-20240415_16.csv","cloudflare-http-request-20240411_23.csv","cloudflare-http-request-20240413_05.csv","cloudflare-http-request-20240414_11.csv","cloudflare-http-request-20240415_17.csv","cloudflare-http-request-20240412_00.csv","cloudflare-http-request-20240413_06.csv","cloudflare-http-request-20240414_12.csv","cloudflare-http-request-20240415_18.csv","cloudflare-http-request-20240412_01.csv","cloudflare-http-request-20240413_07.csv","cloudflare-http-request-20240414_13.csv","cloudflare-http-request-20240415_19.csv","cloudflare-http-request-20240412_02.csv","cloudflare-http-request-20240413_08.csv","cloudflare-http-request-20240414_14.csv","cloudflare-http-request-20240415_20.csv","cloudflare-http-request-20240412_03.csv","cloudflare-http-request-20240413_09.csv","cloudflare-http-request-20240414_15.csv","cloudflare-http-request-20240415_21.csv","cloudflare-http-request-20240412_04.csv","cloudflare-http-request-20240413_10.csv","cloudflare-http-request-20240414_16.csv","cloudflare-http-request-20240415_22.csv","cloudflare-http-request-20240412_05.csv","cloudflare-http-request-20240413_11.csv","cloudflare-http-request-20240414_17.csv","cloudflare-http-request-20240415_23.csv"];
        $dataCSV = [];
        $cacheKey = 'dataCSV::tracking:rebuild';
        
        
        if (Redis::get($cacheKey)) {
            $dataCSV =  json_decode(Redis::get($cacheKey), true);
        } else {
            foreach ($paths as $path) {
                $path = 'csv/cloudflare-http-events/' . $path;
                $csvFile = public_path($path);
                if (file_exists($csvFile)) {
                    // Open the file for reading
                    if (($handle = fopen($csvFile, 'r')) !== false) {
                        // Loop through each line
                        while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                            // Process the data as needed, for example, you can echo it
                            if ($data['0'] != '#' && isset($data['1']) && isset($data['2']) && isset($data['3']) && isset($data['4']))
                            {
                                if (!isset($dataCSV[$data['1']])) {
                                    $dataCSV[$data['1']] = [];
                                }
                                $dataCSV[$data['1']][] = [
                                    'ip' => $data['1'],
                                    'path' => $data['2'],
                                    'query' => $data['3'],
                                    'create_time' => $data['4']
                                ];
                            }
                        }
                        // Close the file handle
                        fclose($handle);
                    } else {
                        echo "Error opening file";
                    }
                } else {
                    echo "File not found";
                }
            }
            Redis::set($cacheKey, json_encode($dataCSV), 'EX',  60 * 24);
        }
        
        if ($request->has('is_debug')){
            return $orders;
        }
        foreach ($orders as $order) {
            $token = '';
            $orderIP = '';
            $orderAtributes = OrderMeta::where('order_id', $order->id)
                ->get();
            foreach ($orderAtributes as $item) {
                if (isset($item->key) && $item->key == 'token') {
                    $token = $item->value;
                }
                if (isset($item->key) && $item->key == 'ip') {
                    $orderIP = $item->value;
                }
            }
            if ($token && $orderIP && isset($dataCSV[$orderIP])) {
                $values = $dataCSV[$orderIP];
                foreach ($values as $value) {
                    $fullUrl = $value['path'] . $value['query'];
                    $this->buildClicker($token, $fullUrl, $value['create_time']);
                }
                $retVal[] = $order;
            }
        }
        $this->rebuildClickerFromIds(array_column($retVal, 'id'));

        return $retVal;
    }
    private function buildClicker($token, $fullUrl, $createTime) {
        $curl = curl_init();
        $createDateTime = new \DateTime($createTime, new \DateTimeZone('UTC'));
        $createDateTime->setTimezone(new \DateTimeZone('Asia/Bangkok'));
        $createTime = $createDateTime->format('Y-m-d\TH:i:s\Z');

        curl_setopt_array($curl, array(
        CURLOPT_URL => 'https://clicker.agoz.me/t/ts',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => 'POST',
        CURLOPT_POSTFIELDS => json_encode([
            'url' => urlencode(url($fullUrl)),
            'host' => 'printerval.com',
            '_token' => $token,
            'create_time' => $createTime
        ]),
        CURLOPT_HTTPHEADER => array(
                'accept: application/json',
                'accept-language: en-US,en;q=0.9,vi;q=0.8',
                'content-type: application/json',
                'origin: https://printerval.com',
                'referer: https://printerval.com/',
                'sec-ch-ua: "Google Chrome";v="123", "Not:A-Brand";v="8", "Chromium";v="123"',
                'sec-ch-ua-mobile: ?0',
                'sec-ch-ua-platform: "macOS"',
                'sec-fetch-dest: empty',
                'sec-fetch-mode: cors',
                'sec-fetch-site: cross-site',
                'user-agent: Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/123.0.0.0 Safari/537.36',
                'Cookie: XSRF-TOKEN=eyJpdiI6IjIrY3hWNHA4eGtQcHY5aWF0RGg2bnc9PSIsInZhbHVlIjoiWldrWThHS1FEV3VkMkhpdjFLWkg1a2VBUVl4V3huVlBLaDU2ZnJOM3FjVjRaTjh6TFpjVHpMaWUxWUZZTUh4NjQzZ29zMnJGK1hQK1l0ZFl1QTZBd2c9PSIsIm1hYyI6IjZhNTExYTk5ZDNkNTdhZDQ4NjA5ODhhNTA4OTFiMWUwYTczMmY0NWFlYmI0OWRkOTkzNjdiYzI4YzNlZDdkOGIifQ%3D%3D; laravel_session=eyJpdiI6IlBCS3U0VEpzV3ZPMHRkQ0dtZGw0Y2c9PSIsInZhbHVlIjoicCs3K3RlUXlScHBlMVhrZEZjWUdhWFhkUERhQVwvQUkrQldja1F1ZFZtRE5idm1LbFNreDlpaTh4d2V1eHFZMkJJUlY2R3kxRWM2dTc0Sm52QjcrZUJ3PT0iLCJtYWMiOiJkOTk4N2Y2YzRmOGY0ZTE5NThjZDgzYWQzNjAwM2Y0OGJiZjA5ODZiMTIzNGQ5YzkzZTQxMzQ0NmQ5OWMwMWE0In0%3D'
                ),
        ));
            $response = curl_exec($curl);
            curl_close($curl);
    }

    private function rebuildClickerFromIds ($orderIds)
    {
        $this->triggerSyncRequest(
            'https://printerval.com/tracking-order/rebuild-tracking?order_ids='. implode(',', $orderIds) .'&site=printerval.com', 'GET'
        );
    }
}
