<?php

namespace Modules\TrackingOrder\Controllers;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Modules\TrackingOrder\Controllers\Controller;
use Modules\TrackingOrder\Models\Order;
use App\Http\Controllers\Services\ExportService;

class TrackingTimelineController extends Controller
{
    public function index(Request $request) {
        return view('tracking-order::system.tracking.index');
    }

    public function find(Request $request)
    {
        set_time_limit(1800);

        $pageSize = intval($request->get('page_size', 50));
        $pageId = intval($request->get('page_id', 0));

        $query = Order::with(['orderItems', 'timelines' => function ($query) {
            $query->orderBy('time', 'desc');
        }])->join('order_timeline', 'order.id', 'order_timeline.order_id');

        if ($request->get('printer')) {
            $query->printer(['value' => $request->get('printer')]);
        }

        if ($request->get('search')) {
            $query->search(['keyword' => $request->get('search')]);
        }
        
        $orderStatus = $request->get('status', 'DELIVERING');
        $query->where('order.status', $orderStatus);

        if ($request->get('delay_days', 0) > 0) {
            $query->having(DB::raw('max(sb_order_timeline.time)'), '<=', date('Y-m-d 23:59:59', time() - ($request->get('delay_days', 0) + 1) * 86400));
        }
        if ($request->get('min_date')) {
            $query->where('order.created_at', '>=', $request->get('min_date'));
        }
        if ($request->get('max_date')) {
            $query->where('order.created_at', '<=', $request->get('max_date'));
        }
        $query->groupBy('order.id');
        $query->orderBy('order_timeline.time', 'desc');

        $total = $this->getTotal($query);

        if ($pageSize > 0) {
            $query->limit($pageSize)->offset($pageId * $pageSize);
        }
            
        $result = $query->get(['order.*', DB::raw('max(sb_order_timeline.time) as timeline_time')]);

        return [
            'status' => 'successful',
            'meta' => $this->getMetaData([
                'page_size' => $pageSize,
                'page_id' => $pageId,
            ], $total),
            'result' => $result,
        ];
    }

    public function getTotal($query) {
        $cloneQuery = clone $query;
        return $cloneQuery->get(['order.*', DB::raw('max(sb_order_timeline.time) as timeline_time')])->count();
    }

    public function getMetaData($data, $total)
    {
        $pageId = 0;
        $pageSize = 20;
        if (!empty($data['page_size'])) {
            $pageSize = $data['page_size'];
        }
        if (!empty($data['page_id'])) {
            $pageId = $data['page_id'];
        }
        $meta = [
            'page_id' => $pageId,
            'page_size' => $pageSize,
            'page_count' => 0,
            'has_next' => false,
            'total_count' => $total,
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function export(Request $request)
    {
        $request->merge([
            'page_size' => -1,
        ]);

        $data = $this->find($request);
        if (isset($data['result']) && count($data['result'])) {
            $data = $data['result'];
        } else {
            $data = [];
        }

        $exportData = [];
        for ($i=0; $i < count($data); $i++) { // $data[$i]

            $timelines = $data[$i]->timelines;
            $printingCodes = array_map(function($orderItem) {
                return $orderItem['printing_code'];
            }, $data[$i]->orderItems->toArray());

            $row = [
                'code' => $data[$i]->code,
                'status' => $data[$i]->status,
                'printer' => implode(', ', array_unique($printingCodes)),
                'delivering_status' => isset($timelines[0]) ? $timelines[0]->content : '',
                'tracking_code' => isset($timelines[0]) ? $timelines[0]->code : '',
                'timelines' => implode(",\n", array_map(function($timeline) {
                    return $timeline['content'];
                }, $timelines->toArray())),
            ];

            $exportData[] = $row;
        }

        $exportService = new ExportService();

        $user = Auth::user();
        $userEmail = null;
        if ($user && $user->email) {
            $userEmail = $user->email;
        }
        
        return $exportService->exportJsonToSheet($exportData, null, $userEmail);
    }
}