<?php

namespace Modules\Trademarks\Controllers;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\Localization\Helpers\DBConnect;
use Modules\Trademarks\Controllers\Controller;
use Module;
use Modules\Trademarks\Models\OptionModel;
use Modules\Trademarks\Models\ProductModel;
use Modules\Trademarks\Repositories\LogRepository;
use App\Utils\EventHelper;
class HomeController extends Controller
{

    protected $visionHost = 'https://megavision.megaads.vn';

    protected $listTrademarksKeywords = [];
    protected $listTradeMarksIds = [];
    protected $minMaxTrademarkId = [];
    protected $listViolationKeywords = [];
    protected $listViolationIds = [];
    protected $minMaxViolationId = [];
    protected $listWarningKeywords = [];
    protected $listWarningIds = [];
    protected $minMaxWarningId = [];

    protected $newKeywordHour = 0;
    protected $isDebug = 0;
    protected $logRepository;

    public function __construct()
    {
        $this->logRepository = new LogRepository();
        $this->initKeywords();
        Module::onView("trademarks::btn", function () {
            return "This is content view from Cart Module HomeController";
        }, 5);
    }
    public function index(Request $request)
    {
        $message = config("trademarks::app.message");
        return view('trademarks::home.welcome', [
            'message' => $message,
        ]);
    }

    public function check(Request $request)
    {
        $retVal = [
            'status' => 'fail',
            'message' => 'Invalid product id'
        ];
        $productId = $request->input('id');
        if ($productId) {
            $product = ProductModel::where('id', '=', $productId)->first(['id', 'name', 'description', 'content', 'trademarks', 'is_trademark']);
            if ($product) {
                $dataCheck = $this->checkTrademarks($product);
                $retVal = [
                    'status' => 'successful',
                    'message' => 'Successful',
                    'data' => $dataCheck
                ];
            }

        }
        return response()->json($retVal);
    }

    public function initKeywords() {
        $this->listTrademarksKeywords = $this->getTrademarksKeywords();
        $this->listViolationKeywords = $this->getViolationKeywords();
        $this->listWarningKeywords = $this->getListWarningKeywords();
    }

    public function checkAll(Request $request)
    {
        set_time_limit(10 * 3600);
        ini_set('memory_limit', '2048M');


        $query = ProductModel::query();
        if ($request->input('trademarksIds')) {
            $this->listTradeMarksIds = explode(',', $request->input('trademarksIds'));
        }
        if ($request->input('minMaxTrademarkId')) {
            $this->minMaxTrademarkId = explode(',', $request->input('minMaxTrademarkId'));
        }

        if ($request->input('listWarningIds')) {
            $this->listWarningIds = explode(',', $request->input('listWarningIds'));
        }
        if ($request->input('minMaxWarningId')) {
            $this->minMaxWarningId = explode(',', $request->input('minMaxWarningId'));
        }

        if ($request->input('violationIds')) {
            $this->listViolationIds = explode(',', $request->input('violationIds'));
        }
        if ($request->input('minMaxViolationId')) {
            $this->minMaxViolationId = explode(',', $request->input('minMaxViolationId'));
        }

        if ($request->input('new_keyword')) {
            $this->newKeywordHour = $request->get('hour', 3);
        }

        if ($request->input('hour')) {
            $hour = $request->input('hour');
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $hour . ' hours')));
        }

        if ($request->has('mt-check')) {
            $query->whereIn('id', explode(',', $request->input('mt-check')));
            $this->isDebug = 1;
        } else {
            $step = $request->input('step', 10000);
            $fromId = $this->getFromId($request, $query);
            $toId = $this->getToId($request, $query);
        }

        $query->whereNull('deleted_at');

        if (!$request->has('rebuild')) {
            $query->whereIn('status', ['ACTIVE', 'INACTIVE']);
        }
        
        if (!count($this->listTrademarksKeywords) && !count($this->listViolationKeywords) && !count($this->listWarningKeywords)) {
            return [
                'status' => 'successful',
                'message' => 'empty keyword',
                'result' => []
            ];
        }
        $productIsTrademarks = [];
        $productIsViolations = [];
        $productIsWarings = [];
        if ($this->isDebug) {
            $cloneQuery = clone $query;
            $allProduct = $cloneQuery->orderBy('id', 'asc')
                ->get(['id', 'name', 'description', 'content', 'trademarks', 'is_trademark', 'is_violation', 'status']);
            $res = $this->checkProducts($allProduct);
            $productIsTrademarks = array_merge($productIsTrademarks, $res['productIsTrademarks']);
            $productIsViolations = array_merge($productIsViolations, $res['productIsViolations']);
            $productIsWarings = array_merge($productIsWarings, $res['productIsWarings']);
            
            return response()->json([
                'status' => 'successful',
                'message' => 'Successful',
                'trademarks' => $productIsTrademarks,
                'violations' => $productIsViolations,
                'warning' => $productIsWarings,
                'trade_keywords' => count($this->listTrademarksKeywords),
                'violation_keywords' => count($this->listViolationKeywords),
                'warning_keywords' => count($this->listWarningKeywords),
            ]);
        }

        $i = 0;
        while ($fromId <= $toId) {
            $stepToId = $fromId + $step;
            if ($toId < $stepToId) {
                $stepToId = $toId;
            }
            $i++;
            $cloneQuery = clone $query;
            $allProduct = $cloneQuery->orderBy('id', 'asc')
                ->where('id', '>=', $fromId)
                ->where('id', '<=', $stepToId)
                ->get(['id', 'name', 'description', 'content', 'trademarks', 'is_trademark', 'is_violation', 'status']);
            $res = $this->checkProducts($allProduct);
            $productIsTrademarks = array_merge($productIsTrademarks, $res['productIsTrademarks']);
            $productIsViolations = array_merge($productIsViolations, $res['productIsViolations']);
            $productIsWarings = array_merge($productIsWarings, $res['productIsWarings']);
            $fromId += $step + 1;
        }
  
        return response()->json([
            'status' => 'successful',
            'message' => 'Successful',
            'trademarks' => $productIsTrademarks,
            'violations' => $productIsViolations,
            'warnings' => $productIsWarings,
            'trade_keywords' => count($this->listTrademarksKeywords),
            'violation_keywords' => count($this->listViolationKeywords),
            'warning_keywords' => count($this->listWarningKeywords),
            'page' => $i,
        ]);
    }

    public function checkProducts($products) {
        $retVal = [
            'productIsTrademarks' => [],
            'productIsViolations' => [],
            'productIsWarings' => []
        ];
        foreach ($products as $product) {
            if (count($this->listTrademarksKeywords)) {
                $check = $this->checkTrademarks($product);
                foreach ($check as $productTrademark) {
                    if (!empty($productTrademark)) {
                        $retVal['productIsTrademarks'][] = $product->id;
                    }
                }
            }
            if (count($this->listViolationKeywords)) {
                $check = $this->checkViolation($product);
                foreach ($check as $productViolation) {
                    if (!empty($productViolation)) {
                        $retVal['productIsViolations'][] = $product->id;
                    }
                }
            }
            if (count($this->listWarningKeywords)) {
                $check = $this->checkWarning($product);
                foreach ($check as $productWarning) {
                    if (!empty($productWarning)) {
                        $retVal['productIsWarings'][] = $product->id;
                    }
                }
            }
        }

        return $retVal;
    }

    public function getFromId($request, $query, $orderField = 'created_at', $getField = 'id') {
        if ($request->input('minId')) {
            return $request->input('minId');
        }
        $cloneQuery = clone $query;

        $from = $cloneQuery->orderBy($orderField, 'asc')->first();
        if ($from) {
            return $from->{$getField};
        }

        return 0;
    }

    public function getToId($request, $query, $orderField = 'created_at', $getField = 'id') {
        if ($request->input('maxId')) {
            return $request->input('maxId');
        }
        $cloneQuery = clone $query;

        $to = $cloneQuery->orderBy($orderField, 'desc')->first();
        if ($to) {
            return $to->{$getField};
        }

        return 0;
    }

    public function getPendingTrademarkIds($request, $type) {
        $query = DB::table('trademarks')
            ->where('status', 'PENDING')
            ->where('type', $type == 'trademark' ? ['trademark', 'warning'] : ['violation']);
        if ($request->get('hour')) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $request->get('hour') . ' hours')));
        }
        $pendingIds = $query
            ->get(['id'])
            ->pluck('id')
            ->toArray();
            
        return $pendingIds;
    }

    public function recheckProductIsViolate(Request $request)
    {
        set_time_limit(10 * 3600);

        $retVal = [
            'status' => 'successful',
            'result' => []
        ];
        $type = $request->get('type');
        if (!$type) {
            return [
                'status' => 'fail',
                'message' => 'missing type = violation || trademark',
                'result' => []
            ];
        }
        
        $pendingIds = $this->getPendingTrademarkIds($request, $type);
        if (count($pendingIds)) {
            $query = DB::table('product_n_trademark')
                ->whereIn('trademark_id', $pendingIds)
                ->whereIn('type', $type == 'trademark' ? ['trademark', 'warning'] : ['violation']);
            if ($request->get('trademark_id')) {
                $query->where('trademark_id', $request->get('trademark_id'));
            }
            if ($request->get('product_id')) {
                $query->where('product_id', $request->get('product_id'));
            }
            $fromId = $this->getFromId($request, $query, 'product_id', 'product_id');
            $toId = $this->getToId($request, $query, 'product_id', 'product_id');
            $step = $request->get('step', 10000);
            while($fromId <= $toId) {
                $stepToId = $fromId + $step;
                if ($toId < $stepToId) {
                    $stepToId = $toId;
                }
                $cloneQuery = clone $query;
                $productIds = $cloneQuery->where('product_n_trademark.product_id', '>=', $fromId)
                    ->where('product_n_trademark.product_id', '<=', $stepToId)
                    ->get(['product_id'])
                    ->pluck('product_id')
                    ->toArray();
                if(count($productIds)) {
                    foreach ($productIds as $productId) {
                        $check = DB::table('product_n_trademark')
                            ->whereNotIn('trademark_id', $pendingIds)
                            ->where('type', $type)
                            ->where('product_id', $productId)
                            ->exists();
                        if (!$check && !$request->has('hour')) {
                            if ($type == 'trademark') {
                                $check = DB::table('product')
                                    ->where('id', $productId)
                                    ->where('is_trademark', 0)
                                    ->whereNull('deleted_at')
                                    ->exists();
                            } else if ($type == 'violation') {
                                $check = DB::table('product')
                                    ->where('id', $productId)
                                    ->where('is_violation', 0)
                                    ->whereNull('deleted_at')
                                    ->exists();
                            }
                        }

                        if (!$check) {
                            if ($type == 'trademark') {
                                $updateData = [
                                    'is_trademark' => 0,
                                    'status' => 'ACTIVE',
                                    'updated_at' => date('Y-m-d H:i:s')
                                ];
                                DB::table('product')
                                    ->where('id', $productId)
                                    ->update($updateData);
                                $this->logRepository->logTrademarkProduct($productId, $type, json_encode($updateData));
                            } else if ($type == 'violation') {
                                $updateData = [
                                    'is_violation' => 0,
                                    'updated_at' => date('Y-m-d H:i:s')
                                ];
                                DB::table('product')
                                    ->where('id', $productId)
                                    ->update($updateData);
                                $this->logRepository->logTrademarkProduct($productId, $type, json_encode($updateData));
                            }
                            $retVal['result'][] = $productId;
                        }
                    }
                }
    
                $fromId += $step + 1;
            }
        }

        return $retVal;
    }

    public function checkingAll(Request $request)
    {
        $isChecking = Cache::get('checking-all');
        if ($request->input('reset')) {
            Cache::forget('checking-all');
        }
        return response()->json([
            'status' => 'successful',
            'data' => $isChecking
        ]);
    }

    public function testTrademark(Request $request) {
        $value = $request->get('value');
        if (!is_array($value)) {
            return [
                'status' => 'successful',
                'result' => [
                    'trademark' => $this->checkTextTrademarks($value),
                    'violation' => $this->checkTextViolation($value)
                ]
            ];
        } else {
            $retVal = [
                'status' => 'successful',
                'result' => []
            ];
            foreach ($value as $key => $val) {
                $retVal['result'][$key] = [
                    'trademark' => $this->checkTextTrademarks($val),
                    'violation' => $this->checkTextViolation($val)
                ];
            }

            return $retVal;
        }
    }

    public function checkTrademarks($product)
    {
        $productTrademarks = [];
        try {
            if (!empty($product->id)) {

//                $vectorTextVisionDetect = $this->checkVectorVision($product);
//                $vectorTrademarks = $this->checkTextTrademarks($vectorTextVisionDetect);
//                if (!count($vectorTrademarks)) {
//                    $vectorTextVisionImageSearch = $this->checkVectorVision($product, $this->visionHost . '/google-image/search');
//                    $vectorTrademarks = $this->checkTextTrademarks($vectorTextVisionImageSearch);
//                }
                $productTrademarks = [
                    'title' => $this->checkTextTrademarks($product->name),
                    'description' => $this->checkTextTrademarks($product->description),
                    'content' => $this->checkTextTrademarks($product->content),
//                    'vector' => $vectorTrademarks
                ];
                $product->trademarks = json_encode($productTrademarks);
                $notDeleteIds = [];
                $isTrademark = false;
                $hideIds = [];
                foreach ($productTrademarks as $trademarks) {
                    if (count($trademarks)) {
                        $isTrademark = true;
                        foreach ($trademarks as $item) {
                            $notDeleteIds[] = $item->id;
                            DB::table('product_n_trademark')
                                ->updateOrInsert(['product_id' => $product->id, 'trademark_id' => $item->id], [
                                    'type' => 'trademark',
                                    'updated_at' => date('Y-m-d H:i:s', time())
                                ]);
                            $productsDesignAlsoAvailable = $this->getProductsDesignAlsoAvailable($product->id);
                            foreach ($productsDesignAlsoAvailable as $productAvailable) {
                                DB::table('product_n_trademark')
                                    ->updateOrInsert(['product_id' => $productAvailable->id, 'trademark_id' => $item->id], [
                                        'type' => 'trademark',
                                        'updated_at' => date('Y-m-d H:i:s', time())
                                    ]);
                                $productAvailable->trademarks = json_encode($productTrademarks);
                                if (!$productAvailable->is_trademark || $productAvailable->status == 'ACTIVE') {
                                    $productAvailable->is_trademark = 1;
                                    $productAvailable->status = 'PENDING';
                                    $productAvailable->save();
                                    $hideIds[] = $productAvailable->id;
                                }
                            }
                        }
                    }
                }

                if ($isTrademark) {
                    if (!$product->is_trademark || $product->status == 'ACTIVE') {
                        $this->createTicket($product->id, $product->trademarks);
                        DB::table('product')->where('id', $product->id)->update([
                            'is_trademark' => 1,
                            'status' => 'PENDING',
                            'updated_at' => date('Y-m-d H:i:s', time())
                        ]);
                        $hideIds[] = $product->id;
                        \Log::info('CHECK TRADEMARK', [$product->id, $isTrademark]);
                    }
                }
                if (count($hideIds)) {
                    EventHelper::pushEvent('bulkHideProduct', ['ids' => $hideIds]);
                }
            }
        } catch (\Exception $exception) {
            \Log::info('CHECK TRADEMARK', [$exception->getMessage() . ' line: ' . $exception->getLine()]);
        }
        return $productTrademarks;
    }

    public function checkViolation($product)
    {
        $productViolations = [
            'title' => $this->checkTextViolation($product->name),
            'description' => $this->checkTextViolation($product->description),
            'content' => $this->checkTextViolation($product->content),
        ];
        $notDeleteIds = [];
        $isViolation = false;
        foreach ($productViolations as $trademarks) {
            if (count($trademarks)) {
                $isViolation = 1;
                foreach ($trademarks as $item) {
                    $notDeleteIds[] = $item->id;
                    DB::table('product_n_trademark')
                        ->updateOrInsert(['product_id' => $product->id, 'trademark_id' => $item->id], [
                            'type' => 'violation',
                            'updated_at' => date('Y-m-d H:i:s', time())
                        ]);
                    $productsDesignAlsoAvailable = $this->getProductsDesignAlsoAvailable($product->id);
                    foreach ($productsDesignAlsoAvailable as $productAvailable) {
                        DB::table('product_n_trademark')
                            ->updateOrInsert(['product_id' => $productAvailable->id, 'trademark_id' => $item->id], [
                                'type' => 'violation',
                                'updated_at' => date('Y-m-d H:i:s', time())
                            ]);
                        if (!$productAvailable->is_violation) {
                            $productAvailable->is_violation = 1;
                            $productAvailable->save();

                            EventHelper::pushEvent('productIsViolation', ['id' => $productAvailable->id]);
                        }
                    }
                }
            }
        }
        if ($isViolation) {
            if (!$product->is_violation) {
                DB::table('product')->where('id', $product->id)->update([
                    'is_violation' => 1,
                    'updated_at' => date('Y-m-d H:i:s', time())
                ]);

                EventHelper::pushEvent('productIsViolation', ['id' => $product->id]);
            }
        }

        return $productViolations;
    }

    public function checkWarning($product)
    {
        $productViolations = [
            'title' => $this->checkTextWarning($product->name),
            'description' => $this->checkTextWarning($product->description),
            'content' => $this->checkTextWarning($product->content),
        ];
        foreach ($productViolations as $trademarks) {
            if (count($trademarks)) {
                foreach ($trademarks as $item) {
                    if (!DB::table('product_n_trademark')->where(['product_id' => $product->id, 'trademark_id' => $item->id])->whereIn('type', ['warning', 'not-warning'])->exists()) {
                        DB::table('product_n_trademark')
                            ->insert([
                                'product_id' => $product->id, 
                                'trademark_id' => $item->id,
                                'type' => 'warning',
                                'created_at' => date('Y-m-d H:i:s', time()),
                                'updated_at' => date('Y-m-d H:i:s', time())
                            ]);
                    }
                    $productsDesignAlsoAvailable = $this->getProductsDesignAlsoAvailable($product->id);
                    foreach ($productsDesignAlsoAvailable as $productAvailable) {
                        if (!DB::table('product_n_trademark')->where(['product_id' => $productAvailable->id, 'trademark_id' => $item->id])->whereIn('type', ['warning', 'not-warning'])->exists()) {
                            DB::table('product_n_trademark')
                                ->insert([
                                    'product_id' => $productAvailable->id, 
                                    'trademark_id' => $item->id,
                                    'type' => 'warning',
                                    'created_at' => date('Y-m-d H:i:s', time()),
                                    'updated_at' => date('Y-m-d H:i:s', time())
                                ]);
                        }
                    }
                }
            }
        }

        return $productViolations;
    }

    protected function checkTextTrademarks($text)
    {
        $text = strtolower(trim($text));

        if (!$this->listTrademarksKeywords) {
            $this->listTrademarksKeywords = $this->getTrademarksKeywords();
        }
        $result = $this->getTrademarks($text);

        $text = str_replace(' and ', ' ', $text);
        $text = preg_replace("/[^A-Za-z ]/", ' ', $text);
        $text = trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $text)));
        $result = array_merge($result, $this->getTrademarks($text));

        $text = preg_replace("/[^A-Za-z ]/", '', $text);
        $result = array_merge($result, $this->getTrademarks($text));
        $retVal = [];
        foreach ($result as $item) {
            if (!isset($retVal[$item->id])) {
                $retVal[$item->id] = $item;
            }
        }

        return array_values($retVal);
    }

    protected function getTrademarks ($text) {
        $text = strtolower(trim($text));
        $retVal = [];
        if (!$this->listTrademarksKeywords) {
            $this->listTrademarksKeywords = $this->getTrademarksKeywords();
        }
        foreach ($this->listTrademarksKeywords as $item) {
            if (
                preg_match("/\b" . $item->keyword . "\b/i", $text) || 
                preg_match("/\b" . str_replace(' ', '', $item->keyword) . "\b/i", $text)
            ) {
                $retVal[] = $item;
            }
        }
        return $retVal;
    }

    protected function checkTextViolation($text) {
        $text = strtolower(trim($text));

        if (!$this->listViolationKeywords) {
            $this->listViolationKeywords = $this->getViolationKeywords();
        }

        $result = $this->getViolation($text);

        $text = str_replace(' and ', ' ', $text);
        $text = preg_replace("/[^A-Za-z ]/", ' ', $text);
        $text = trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $text)));
        $result = array_merge($result, $this->getViolation($text));

        $text = preg_replace("/[^A-Za-z ]/", '', $text);
        $result = array_merge($result, $this->getViolation($text));

        $retVal = [];
        foreach ($result as $item) {
            if (!isset($retVal[$item->id])) {
                $retVal[$item->id] = $item;
            }
        }

        return array_values($retVal);
    }

    protected function getViolation ($text) {
        $text = strtolower(trim($text));
        $retVal = [];
        if (!$this->listViolationKeywords) {
            $this->listViolationKeywords = $this->getViolationKeywords();
        }
        foreach ($this->listViolationKeywords as $item) {
            if (
                preg_match("/\b" . $item->keyword . "\b/i", $text) || 
                preg_match("/\b" . str_replace(' ', '', $item->keyword) . "\b/i", $text)
            ) {
                $retVal[] = $item;
            }
        }
        return $retVal;
    }

    protected function checkTextWarning($text) {
        $text = strtolower(trim($text));

        if (!$this->listWarningKeywords) {
            $this->listWarningKeywords = $this->getListWarningKeywords();
        }

        $result = $this->getWarning($text);

        $text = str_replace(' and ', ' ', $text);
        $text = preg_replace("/[^A-Za-z ]/", ' ', $text);
        $text = trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $text)));
        $result = array_merge($result, $this->getWarning($text));

        $text = preg_replace("/[^A-Za-z ]/", '', $text);
        $result = array_merge($result, $this->getWarning($text));

        $retVal = [];
        foreach ($result as $item) {
            if (!isset($retVal[$item->id])) {
                $retVal[$item->id] = $item;
            }
        }

        return array_values($retVal);
    }

    protected function getWarning ($text) {
        $text = strtolower(trim($text));
        $retVal = [];
        if (!$this->listWarningKeywords) {
            $this->listWarningKeywords = $this->getListWarningKeywords();
        }
        foreach ($this->listWarningKeywords as $item) {
            if (
                preg_match("/\b" . $item->keyword . "\b/i", $text) || 
                preg_match("/\b" . str_replace(' ', '', $item->keyword) . "\b/i", $text)
            ) {
                $retVal[] = $item;
            }
        }
        return $retVal;
    }

    protected function checkVectorVision($product, $apiEndpoint = null)
    {
        if (!$product->image_url) {
            return '';
        }
        if (!$apiEndpoint) $apiEndpoint = $this->visionHost . '/vision/detect/text';

        $images = [$this->getImageCdn($product->image_url, 540, 540, false, false)];
        $textDetect = '';

        /* foreach ($product->productSkus as $sku) {
            if ($sku->image_url)
                $images[] = $sku->image_url;
        } */

        $visionDetect = $this->postCurlAPI($apiEndpoint, $images);
        foreach ($visionDetect as $value) {
            if ($value->texts && count($value->texts)) {
                $textDetect = implode(' ', $value->texts);
            }
        }

        return $textDetect;
    }

    function postCurlAPI($url, $data, $json = true)
    {
        $curl = curl_init();

        $data = [
            'image_url' => $data
        ];

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/json'
            ),
        ));

        $response = curl_exec($curl);

        if (curl_errno($curl)) {
            $error_msg = curl_error($curl);
        }

        if (isset($error_msg)) {
            dd($error_msg);
        }

        if ($json) $response = json_decode($response);
        return $response;
    }

    protected function getImageCdn($url, $width = 0, $height = 0, $fitIn = true, $webp = true)
    {
        if (!$url) {
            $url = config('app.url') . "/images/logo.png";
            $fitIn = true;
        }
        if (substr($url, -3) == 'svg') {
            return $url;
        }
        $originUrl = $url;
        if (substr($url, 0, 4) == 'http') {
            $url = str_replace('https://', '', $url);
            $url = str_replace('http://', '', $url);
        } else {
            $url = config('app.url') . $url;
        }
        if ($webp && isset($_SERVER['HTTP_ACCEPT'])) {
            if (strpos($_SERVER['HTTP_ACCEPT'], 'image/webp') !== false) {
                $webp = true;
            } else {
                $webp = false;
            }
        }

        $baseCdnUrl = config('sa.cdn_url') . '/';
        $fitIn = ($fitIn && $width && $height);
        $fitIn = false;
        if ($fitIn) {
            $baseCdnUrl .= "fit-in/";
        }
        if ($width || $height) {
            $baseCdnUrl .= $width . "x" . $height . "/";
        }
        if ($fitIn || $webp) {
            $baseCdnUrl .= "filters";
        }
        if ($fitIn) {
            $baseCdnUrl .= ":fill(fff)";
        }
        if ($webp) {
            $baseCdnUrl .= ":format(webp)";
        }
        if ($fitIn || $webp) {
            $baseCdnUrl .= "/";
        }
        $baseCdnUrl .= $url;
        return $baseCdnUrl;
    }

    protected function getClient($authconfig = "credentials",$credentials = 'sheet', $authCode = '4/Ajm2xQHHJJpPgKGl6cq-A7cfarCCU8meTr6xnsXcm1U' ) {
        $client = new \Google_Client();
        $client->setApplicationName("Sheet API");
        $client->setScopes(\Google_Service_Sheets::SPREADSHEETS);

        $client->setAuthConfig(base_path() . "/app/Modules/Trademarks/Config/" . $authconfig . ".json");
        $client->setAccessType('offline');

        // Load previously authorized credentials from a file.
        $credentialsPath = base_path() . "/app/Modules/Trademarks/Config/" . $credentials . ".json";
        if (file_exists($credentialsPath)) {
            $accessToken = json_decode(file_get_contents($credentialsPath), true);
        } else {
            // Request authorization from the user.
            $authUrl = $client->createAuthUrl();
            printf("Open the following link in your browser:\n%s\n", $authUrl);
            print 'Enter verification code: ';
            $authCode = trim($authCode);

            // Exchange authorization code for an access token.
            $accessToken = $client->fetchAccessTokenWithAuthCode($authCode);

            // Store the credentials to disk.
            if (!file_exists(dirname($credentialsPath))) {
                mkdir(dirname($credentialsPath), 0700, true);
            }
            file_put_contents($credentialsPath, json_encode($accessToken));
            printf("Credentials saved to %s\n", $credentialsPath);
        }
        $client->setAccessToken($accessToken);

        // Refresh the token if it's expired.
        if ($client->isAccessTokenExpired()) {
            $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            file_put_contents($credentialsPath, json_encode($client->getAccessToken()));
        }
        return $client;
    }

    protected function getTrademarksKeywords () {
        $query = DB::table('trademarks');
        if ($this->listTradeMarksIds) {
            $query->whereIn('id', $this->listTradeMarksIds);
        }
        if ($this->minMaxTrademarkId) {
            $query->where('id', '>=', $this->minMaxTrademarkId[0])
                ->where('id', '<=', $this->minMaxTrademarkId[1]);
        }
        if ($this->newKeywordHour) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $this->newKeywordHour . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->where('type', 'trademark');
        $keywords = $query->get(['keyword', 'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = $this->replaceKeyword($item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    protected function replaceKeyword($keyword) {
        $keyword = strtolower(trim($keyword));
        $keyword = str_replace(' and ', ' ', $keyword);
        $keyword = str_replace('\\', '\\\\', $keyword);
        $keyword = str_replace('/', '\/', $keyword);
        $keyword = str_replace('.', '\.', $keyword);

        return $keyword;
    }

    protected function getViolationKeywords () {
        $query = DB::table('trademarks');
        if ($this->listViolationIds) {
            $query->whereIn('id', $this->listViolationIds);
        }
        if ($this->minMaxViolationId) {
            $query->where('id', '>=', $this->minMaxViolationId[0])
                ->where('id', '<=', $this->minMaxViolationId[1]);
        }
        if ($this->newKeywordHour) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $this->newKeywordHour . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->where('type', 'violation');
        $keywords = $query->get(['keyword', 'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = strtolower(trim($item->keyword));
            $item->keyword = str_replace(' and ', ' ', $item->keyword);
            $item->keyword = str_replace('\\', ' ', $item->keyword);
            $item->keyword = str_replace('/', ' ', $item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    protected function getListWarningKeywords() {
        $query = DB::table('trademarks');
        if ($this->listWarningIds) {
            $query->whereIn('id', $this->listWarningIds);
        }
        if ($this->minMaxWarningId) {
            $query->where('id', '>=', $this->minMaxWarningId[0])
                ->where('id', '<=', $this->minMaxWarningId[1]);
        }
        if ($this->newKeywordHour) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $this->newKeywordHour . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->where('type', 'warning');
        $keywords = $query->get(['keyword',  'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = strtolower(trim($item->keyword));
            $item->keyword = str_replace(' and ', ' ', $item->keyword);
            $item->keyword = str_replace('\\', ' ', $item->keyword);
            $item->keyword = str_replace('/', ' ', $item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    public function management () {
        return view('trademarks::trademark.index');
    }

    public function checkTag(Request $request){
        set_time_limit( 3600);
        ini_set('memory_limit', '2048M');
        $response = [
            'status' => 'successful'
        ];
        $data = [];
        $ignore = $request->get('ignore','');
        $filter['ignores'] = $ignore ? explode(',',$ignore) : [];
        $filter['hour'] = $request->get('hour',25);
        $filter['is_all'] = $request->get('is_all',false);
        try{
            $locales = getModuleLocale();
            foreach ($locales as $locale){
                if (!$locale['enable']) {
                    continue;
                }
                $data[$locale['locale']] = $this->checkTrademarkTagFromLocale($locale, $filter);
            }
        }catch (Exception $ex){
            print_r($ex->getTraceAsString());
        }
        $response['result'] = $data;
        return response()->json($response);

    }

    protected function checkTrademarkTagFromLocale($locale, $filter)
    {
        $connection = DBConnect::connect($locale['locale']);
        $trademarks = $this->getTrademarksData($connection, $filter);
        $pageSize = 1000;
        $result = [];
        if (!empty($trademarks)) {
            $maxId = $this->getMaxTagId($connection);
            foreach (array_chunk($trademarks, 50) as $chunkTrademark) {
                $fromId = 0;
                while ($fromId < $maxId) {
                    $toId = $fromId + $pageSize;
                    $pattern = '(' . implode('|', $chunkTrademark) . ')( |$)';
                    $tagIds = $this->getTagTrademark($connection, $fromId, $toId, $pattern);
                    if ($tagIds) {
                        $this->deleteTagTrademark($connection, $tagIds);
                        $result[] = [$tagIds, $pattern];
                    }
                    $fromId = $toId;
                }
            }
        }
        return $result;
    }

    protected function deleteTagTrademark($connection, $tagIds) {
        foreach ($tagIds as $tagId) {
            $this->onDeleteTag($connection, $tagId);
        }
        $connection->table('tag_refer')->whereIn('tag_id', $tagIds)->delete();
        $connection->table('tag')->whereIn('id', $tagIds)->delete();
    }

    protected function onDeleteTag($connection, $tagId) {
        $connection->table('slug_manager')
            ->where('target_id', $tagId)
            ->where('type', 'tag')
            ->update([
                'type' => 'delete',
                'priority' => 0,
            ]);
        $pIds = $connection->table('tag_refer')
            ->where('tag_id', $tagId)
            ->where('refer_type', 'PRODUCT')
            ->pluck('refer_id')
            ->toArray();
        if (count($pIds) > 0) {
            foreach (array_chunk($pIds, 200) as $chunkPIds) {
                $connection->table('product')->whereIn('id', $chunkPIds)->update([
                    'updated_at' => new \DateTime()
                ]);
            }
        }
    }

    protected function getTrademarksData($connection, $filter)
    {
        $query = $connection->table('trademarks');
        if (array_key_exists('is_all', $filter) && $filter['is_all']) {
            //@todo on all
        } else {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', time() - $filter['hour'] * 3600));
        }
        if (array_key_exists('ignores', $filter) && $filter['ignores']) {
            $query->whereNotIn('keyword', $filter['ignores']);
        }
        return $query->pluck('keyword')->toArray();
    }

    protected function getMaxTagId($connection) {
        $result = $connection->table('tag')->orderBy('id', 'desc')->first(['id']);
        return $result->id;
    }

    protected function getTagTrademark($connection, $fromId, $toId, $pattern) {
        return $connection->table('tag')
            ->where('id', '>', $fromId)
            ->where('id', '<=', $toId)
            ->where('title', 'REGEXP', $pattern)
            ->pluck('id')
            ->toArray();
    }

    public function deleteTrademark(Request $request, $id) {
        set_time_limit(30 * 60);

        DB::table('trademarks')
            ->where('id', $id)
            ->delete();
        DB::table('product_n_trademark')
            ->where('trademark_id', $id)
            ->delete();

        return [
            'status' => 'successful'
        ];
    }

    public function syncKeywordTrademarkToOtherLocation(Request $request)
    {
        set_time_limit(60 * 10);
        $token = $request->get('token');
        $locales = ['uk', 'ca', 'au', 'de', 'fr', 'it', 'es', 'pt', 'jp'];
        $datetime = $request->get('date', '-1');
        $fromDate = date('Y-m-d 00:00:00', strtotime($datetime . ' days'));
        $USKeywords = DB::table('trademarks')
            ->where('updated_at', '>=', $fromDate)
            ->get(['keyword', 'image_url', 'link', 'type', 'status', 'claim']);
        foreach ($locales as $locale) {
            $connection = \Modules\Pod\Helpers\DBConnect::connect($locale);
            foreach ($USKeywords as $US) {
                $keyword = $connection->table('trademarks')->where('keyword', $US->keyword)->first();
                if (!$keyword) {
                    $connection->table('trademarks')->insert([
                        'keyword' => $US->keyword,
                        'image_url' => $US->image_url,
                        'link' => $US->link,
                        'created_at' => new \DateTime(),
                        'updated_at' => new \DateTime(),
                        'type' => $US->type,
                        'status' => $US->status,
                        'claim' => $US->claim,
                    ]);
                } else {
                    $connection->table('trademarks')
                        ->where('id', $keyword->id)
                        ->update([
                        'keyword' => $US->keyword,
                        'image_url' => $US->image_url,
                        'link' => $US->link,
                        'updated_at' => new \DateTime(),
                        'type' => $US->type,
                        'status' => $US->status,
                        'claim' => $US->claim,
                    ]);
                }
            }
        }
        $listApi = [
            'https://api.printblur.com/',
            "https://api.printblur.de/",
            "https://api.printblur.fr/",
            "https://api.printblur.es/",
            "https://api.printblur.co.uk/"
        ];
        foreach ($USKeywords as $US) {
            $data = array(
                'keyword' => $US->keyword,
                'image_url' => $US->image_url,
                'link' => $US->link,
                'type' => $US->type,
                'status' => $US->status,
                'claim' => $US->claim,
                'service_token' => 'megaads@123'
            );
            foreach ($listApi as $api) {
                $this->syncTrademark($data, $api);
            }
        }
        return $USKeywords;
    }

    protected function getProductsDesignAlsoAvailable($productId)
    {
        $retVal = [];
        $designIds = DB::table('product_n_design')
                    ->where('product_id', $productId)
                    ->pluck('design_id')->toArray();
        $productIds = DB::table('product_n_design')
                        ->whereIn('design_id', $designIds)
                        ->pluck('product_id')
                        ->toArray();
        if (!empty($productIds) && !empty($designIds)) {
            $retVal = ProductModel::whereIn('id', $productIds)
                ->get(['product.id', 'product.is_violation', 'product.is_trademark', 'product.status']);
        }
        return $retVal;
    }

    protected function syncTrademark ($data, $apiUrl)
    {
        $query = http_build_query($data);
        $isAvailable =  $this->triggerSyncRequest($apiUrl . 'trademarks?filters=keyword=' . $data['keyword'],"GET",
            [], []
        );
        if ($isAvailable && $isAvailable['status'] == 'successful' && count($isAvailable['result']) > 0) {
            $this->triggerSyncRequest($apiUrl . 'trademarks/' . $isAvailable['result'][0]['id'],"PATCH",
                $data, []
            );
        } else {
            $this->triggerSyncRequest($apiUrl . 'trademarks?' . $query,"POST",
                [], []
            );
        }
    }

    public function hasVioloateProduct(Request $request) {
        $ids = explode(',', $request->get('ids'));
        $retVal = [
            'status' => 'successful',
            'result' => []
        ];

        foreach ($ids as $id) {
            $retVal['result'][$id] = DB::table('product_n_trademark')
                ->where('trademark_id', $id)
                ->exists();
        }

        return $retVal;
    }

    protected function createTicket ($productId, $keyword) {
        $client = new \GuzzleHttp\Client();
        $product = DB::table('product')
            ->where('sold', '>', 0)
            ->where('id', $productId)
            ->first(['name', 'slug', 'id']);
        $result = null;
        $locale = env('APP_LOCALE');
        if ($locale == '' || !$locale) {
            $locale = 'us';
        }
        $locale = strtolower($locale);
        if ($product) {
            $data = [
                "status"  => "PENDING",
                "title"  => "Kiểm tra trademark sản phẩm " . $product->name,
                "description"  => 'Sản phẩm được đánh trademark với từ các từ: \n' . $keyword,
                "team"  => 'CONTENT',
                "locale"  => $locale,
                "product_id"  => $productId,
                "product_code"  => null,
                "product_url"  => url('/') . clroute($product->slug . '-p' . $product->id),
                "email"  =>  'check-trademark',
                "contents"  => [
                    ["type"  => "content", "data"  => NULL,"team"  => "CS"],
                    ["type"  => "content","data"  => NULL,"team"  => "FULFILL"],
                    ["type"  => "content","data"  => NULL,"team"  => "DESIGN"],
                    ["type"  => "content","data"  => NULL,"team"  => "CONTENT"]
                ],
                "assigns"  => [
                    ["type"  => "assign","data"  => NULL,"team"  => "CS"],
                    ["type"  => "assign","data"  => NULL,"team"  => "FULFILL"],
                    ["type"  => "assign","data"  => NULL,"team"  => "DESIGN"],
                    ["type"  => "assign","data"  => NULL,"team"  => "CONTENT"]
                ],
                'service_token' => 'megaads@123'
            ];
            $request = new Request($data);
//            $result  = $client->request("POST", url('/') . '/central/staff-job/save' , [ 'body' => json_encode($data)]);
//            $result = $this->triggerSyncRequest(url('/') . '/central/staff-job/save',"POST",
//                $data, []
//            );
            if (
                class_exists('\Modules\Ticket\Controllers\Services\StaffJobService')
                && method_exists('\Modules\Ticket\Controllers\Services\StaffJobService','save')
            ) {
                $staffJobService = new \Modules\Ticket\Controllers\Services\StaffJobService;
                $result = $staffJobService->save($request);
            }
        }
        return $result;
    }

    public function violationKeywords() {
        return view('trademarks::violation-keywords.index');
    }
}


