<?php

namespace Modules\Trademarks\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Trademarks\Controllers\Controller;
use Modules\Trademarks\Models\Trademark;

class SyncOriginalProductController extends Controller
{
    public function sync(Request $request) {
        $step = $request->input('step', 1000);
        $fromId = $this->getFromId($request);
        $toId = $this->getToId($request);

        while($fromId < $toId) {
            $stepToId = $fromId + $step;
            if ($toId < $stepToId) {
                $stepToId = $toId;
            }

            $originalProducts = DB::table('original_product')
                ->where('id', '>=', $fromId)
                ->where('id', '<=', $stepToId)
                ->get();

            $productBySource = $this->getItemBySource($originalProducts);

            foreach ($productBySource as $source => $items) {
                $arr = explode('-', $source);
                if (count($arr) !== 2) {
                    continue;
                }
                $productWithKeywords = $this->getProductWithKeywords($source, $items->pluck('product_id')->toArray());
            }

            $fromId += $step + 1;
        }
    }

    public function save($productWithKeywords, $items) {
        $productByOriginal = [];
        foreach ($items as $item) {
            $productByOriginal[$item->original_product_id] = $item->product_id;
        }
        $logData = [];
        $productNTrademarkData = [];
        foreach ($productWithKeywords as $orignalId => $item) {
            if (!isset($productByOriginal[$orignalId])) {
                continue;
            }
            $productId = $productByOriginal[$orignalId];
            $updateData = [];
            if ($item['is_hide']) {
                $updateData['status'] = 'PENDING';
            }
            if ($item['is_trademark']) {
                $updateData['is_trademark'] = 1;
            }
            if ($item['is_violation']) {
                $updateData['is_violation'] = 1;
            }
            if (count($updateData)) {
                $updateData['updated_at'] = date('Y-m-d H:i:s', time());
                DB::table('product')->where('id', $productId)->update($updateData);
                $logData[] = $this->getLogData($productId, $updateData);
            }
            if (count($item['trademarks'])) {
                foreach ($item['trademarks'] as $trademarkData) {
                    $trademark = $this->syncTrademark($trademarkData['trademark']);
                    $productNTrademarkData[] = [
                        'product_id' => $productId,
                        'trademark_id' => $trademark->id,
                        'type' => $trademarkData['type'],
                        'hide_by' => 'sync'
                    ];
                }
            }
        }

        if (count($logData)) {
            DB::table('log')->insert($logData);
        }
        if (count($productNTrademarkData)) {
            DB::table('product_n_trademark')->insert($productNTrademarkData);
        }
    }

    private function syncTrademark($trademark) {
        $result = Trademark::where('keyword', $trademark['keyword'])->first();
        if (!$result) {
            $result = Trademark::create($trademark);
        }

        return $result;
    }

    private function getProductWithKeywords($source, $productIds) {
        $sourceName = explode('-', $source)[0];
        $sourceLocale = explode('-', $source)[1];
        $baseUrl = $this->buildBaseUrl($sourceName, $sourceLocale);
        if (!$baseUrl) {
            return [];
        }
        $response =  $this->triggerSyncRequest(
            $baseUrl . '/trademarks/product-with-keyword',
            'POST',
            [
                'product_ids' => implode(',', $productIds)
            ]
        );

        if (empty($response['result'])) {
            return [];
        }

        return $response['result'];
    }

    public function getLogData($productId, $updateData) {
        return [
            'actor_email' => '',
            'target_type' => 'PRODUCT',
            'target_id' => $productId,
            'event_type' => 'UPDATE',
            'data' => json_encode($updateData),
            'created_at' => date('Y-m-d H:i:s', time())
        ];
    }

    public function buildBaseUrl($sourceName, $sourceLocale) {
        $urlBySources = [
            'printerval' => 'https://printerval.com',
            'printblur' => 'https://printblur.com',
        ];

        if (!isset($urlBySources[$sourceName])) {
            return '';
        }

        return $urlBySources[$sourceName] . ($sourceLocale !== 'us' ? ('/' . $sourceLocale) : '');
    }

    public function getItemBySource($items) {
        $itemBySource = [];

        foreach ($items as $item) {
            if (!isset($itemBySource[$item->source])) {
                $itemBySource[$item->source] = collect();
            }

            $itemBySource[$item->source]->push($item);
        }

        return $itemBySource;
    }

    public function productWithKeywords(Request $request)
    {
        set_time_limit(10 * 60);
        $productIds = explode(',', $request->get('product_ids'));
        $pnt = DB::table('product_n_trademark')->whereIn('product_id', $productIds)->get();
        $trademarks = DB::table('trademarks')->whereIn('id', $pnt->pluck('trademark_id'))->get()->keyBy('id');
        $products = DB::table('product')->whereIn('id', $productIds)->get(['id', 'status', 'is_trademark', 'is_violation', 'deleted_at'])->keyBy('id');
        $result = [];

        foreach ($products as $product) {
            $data = [
                'id' => $product->id,
                'is_hide' => $product->deleted_at || $product->status != "ACTIVE",
                'is_trademark' => $product->is_trademark,
                'is_violation' => $product->is_violation,
                'trademarks' => []
            ];

            $result[$product->id] = $data;
        }

        foreach ($pnt as $item) {
            if (isset($result[$item->product_id]) && isset($trademarks[$item->trademark_id])) {
                $trademark = $trademarks[$item->trademark_id];
                $result[$item->product_id]['trademarks'][] = [
                    'type' => $item->type,
                    'trademark' => $trademark
                ];
            }
        }

        return [
            'status' => 'successful',
            'result' => $result
        ];
    }

    private function getFromId($request) {
        $fromId = 0;
        if ($request->has('from_id')) {
            $fromId = $request->get('from_id');
        } else {
            $minId = DB::table('original_product')->orderBy('id', 'asc')
                ->first(['id']);
            if ($minId) {
                $fromId = $minId->id;
            }
        }

        return $fromId;
    }

    private function getToId($request) {
        $maxId = 0;
        if ($request->has('to_id')) {
            $maxId = $request->get('to_id');
        } else {
            $maxId = DB::table('original_product')->orderBy('id', 'desc')->first();
            if ($maxId) {
                $maxId = $maxId->id;
            }
        }

        return $maxId;
    }
}
