<?php

namespace Modules\Trademarks\Controllers;

use Exception;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\Trademarks\Controllers\Controller;
use Modules\Trademarks\Models\ProductNTrademarkModel;

class TrademarkProductController extends HideProductController
{
    public function index() {
        $staffs = DB::table('users')
            ->whereIn('role', ['ADMIN', 'STAFF'])
            ->get(['id', 'name']);
        
        return view('trademarks::trademark-product.index', ['staffs' => $staffs]);
    }

    public function getTotal($request, $query) {
        $locale = env('APP_LOCALE');
        if (!$locale) {
            $locale = 'us';
        }
        $key = 'PRODUCT_TRADEMARK_TOTAL::' . $locale;
        $filters = $request->except(['page_id', 'page_size']);
        if (!count($filters) && Cache::get($key)) {
            return Cache::get($key);
        }
        $cloneQuery = clone $query;
        $total = $cloneQuery->count();
        if (!count($filters)) {
            Cache::put($key, $total, 3600);
        }
        return $total;
    }

    public function find(Request $request) {
        $query = ProductNTrademarkModel::with(['trademark', 'actor' => function ($query) {
            $query->select(['id', 'name', 'email']);
        }])
            ->join('product', 'product.id', 'product_n_trademark.product_id')
            ->whereNull('product.deleted_at')
            ->whereIn('product_n_trademark.type', ['violation', 'trademark', 'hided', 'brand'])
            ->where('product.status', 'PENDING');

        if ($request->get('search')) {
            $keyword = strtolower($request->get('search'));
            if (preg_match("/^\d+$/", $keyword)) {
                $query->where('product_id', $keyword);
            } else if (preg_match("/^p\d+$/", $keyword)) {
                $query->where('product_id', str_replace('p', '', $keyword));
            } else if (preg_match("/^t\d+$/", $keyword)) {
                $query->where('trademark_id', str_replace('t', '', $keyword));
            }
        }

        if ($request->get('trademark_id')) {
            $query->where('trademark_id', $request->get('trademark_id'));
        }
        if ($request->get('date_from')) {
            $query->where('product_n_trademark.created_at', '>=', $request->get('date_from'));
        }
        if ($request->get('date_to')) {
            $query->where('product_n_trademark.created_at', '<=', $request->get('date_to'));
        }
        if ($request->get('update_from')) {
            $query->where('product_n_trademark.updated_at', '>=', $request->get('update_from'));
        }
        if ($request->get('update_to')) {
            $query->where('product_n_trademark.updated_at', '<=', $request->get('update_to'));
        }

        if ($request->has('actor_id')) {
            $query->where('product_n_trademark.actor_id', $request->get('actor_id'));
        }

        if ($request->has('product_actor_id')) {
            $query->where('product.actor_id', $request->get('product_actor_id'));
        }

        if ($request->get('hide_by')) {
            $query->where('hide_by', $request->get('hide_by'));
        }
        if ($request->get('reason')) {
            $query->where('reason', $request->get('reason'));
        } else {
            $query->where('reason', '!=', 'auto');
        }
        // if ($request->get('product_status')) {
        //     $query->where('product.status', $request->get('product_status'));
        // } else if ($request->get('type') && !$request->has('actor_id')) {
        //     $query->where('product.status', 'ACTIVE');
        // }
        $query->orderBy('product_n_trademark.created_at', 'desc');

        $simplePaginate = $request->get('simple_paginate', 1);
        $total = 0;
        if (!$simplePaginate) {
            $total = $this->getTotal($request, $query);
        } else {
            $pageSize = $request->get('page_size', 50);
            $pageId = $request->get('page_id', 0);
            $query->limit($pageSize)->offset($pageSize * $pageId);
        }

        $meta = $this->getMetaData($request->all(), $total, $simplePaginate);

        $items = $query->get([
            'product.id', 'product.name', 'product.slug', 'product.image_url', 'product.sold', 'product.view_count', 
            'product_n_trademark.product_id', 'product_n_trademark.trademark_id', 'product_n_trademark.actor_id', 'product_n_trademark.reason', 'product_n_trademark.note',
            'product_n_trademark.created_at', 'product_n_trademark.updated_at'
        ]);

        return [
            'status' => 'successful',
            'meta' => $meta,
            'result' => $items
        ];
    }

    public function getMetaData($filter, $total, $simplePaginate = false)
    {
        $pageId = 0;
        $pageSize = 50;
        if (!empty($filter['page_size'])) {
            $pageSize = intval($filter['page_size']);
        }
        if (!empty($filter['page_id'])) {
            $pageId = intval($filter['page_id']);
        }
        $meta = [
            'page_id' => $pageId,
            'page_size' => $pageSize,
            'page_count' => 0,
            'has_next' => false,
            'total_count' => $total,
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1 || $simplePaginate) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function hideWarning(Request $request) {
        $token = $request->header('token');
        $user = DB::table('users')
            ->where('token', $token)
            ->whereIn('role', ['STAFF', 'ADMIN'])
            ->where('status', 'ACTIVE')
            ->first(['id', 'email']);

        if (!$user) {
            return [
                'status' => 'fail',
                'message' => 'Permission denied'
            ];
        }

        $productIds = explode(',', $request->get('product_ids', ''));

        $this->hideWarningProductNTrademark($user, $productIds, $request->get('trademark_ids'));
        if($request->get('hide_same_design')) {
            $designIds = DB::table('product_n_design')->whereIn('product_id', $productIds)->get(['design_id'])->pluck('design_id');
            $sameDesignProductIds = DB::table('product_n_design')
                ->whereNotIn('product_id', $productIds)
                ->whereIn('design_id', $designIds)
                ->get(['product_id'])
                ->pluck('product_id')
                ->toArray();

            if (!empty($request->get('trademark_ids'))) {
                $this->hideWarningProductNTrademark($user, $sameDesignProductIds, $request->get('trademark_ids'));
            }
        }

        return [
            'status' => 'successful'
        ];
    }

    public function hideWarningProductNTrademark($user, $productIds, $tradeMarkIds) {
        $updateData = [
            'status' => 'PENDING',
            'actor_id' => $user->id,
            'updated_at' => date('Y-m-d H:i:s', time()),
        ];

        DB::table('product_n_trademark')
            ->whereIn('product_id', $productIds)
            ->whereIn('trademark_id', $tradeMarkIds)
            ->update($updateData);

        $violationIds = DB::table('product_n_trademark')
            ->join('trademarks', 'trademarks.id', 'product_n_trademark.trademark_id')
            ->where('trademarks.type')
            ->whereIn('product_id', $productIds)
            ->whereIn('product_n_trademark.type', ['violation', 'trademark', 'hided', 'brand'])
            ->where('trademarks.status', 'ACTIVE')
            ->get(['product_id'])
            ->pluck('product_id')
            ->toArray();
        $activeIds = array_diff($productIds, $violationIds);
        if (count($activeIds)) {
            $updateProductData = [
                'status' => 'ACTIVE',
                'is_trademark' => 0,
                'is_violation' => 0,
                'updated_at' => date('Y-m-d H:i:s', time()),
            ];
            $this->activeProduct($user, $activeIds, $updateProductData);
        }
    }

    public function activeProduct($user, $productIds, $updateData) {

        DB::table('product')
            ->whereIn('id', $productIds)
            ->update($updateData);
        $logData = [];
        $updateData['reason'] = 'Bật từ tool bỏ qua trademark';

        foreach ($productIds as $productId) {
            $logData[] = [
                'actor_email' => $user ? $user->email : '',
                'target_type' => 'PRODUCT',
                'target_id' => $productId,
                'event_type' => 'UPDATE',
                'data' => json_encode($updateData),
                'created_at' => date('Y-m-d H:i:s', time())
            ];
        }

        if (count($logData)) {
            DB::table('log')->insert($logData);
        }
    }

    public function export(Request $request) {
        set_time_limit(3600);
        ini_set('memory_limit', '2048M');
        $user = Auth::user();
        if (!$user) {
            return [
                'status' => 'fail',
                'message' => 'Permission denied'
            ];
        }
        $filterUrl = trim($request->get('filter_url', ''), '?');
        if (empty($filterUrl)) {
            return [
                'status' => 'fail',
                'message' => 'filter_url is required'
            ];
        }
        $filterUrl = $filterUrl . (strpos($filterUrl, '?') !== false ? '&' : '?') . 'page_id=';
        $pageId = 0;
        $items = [];
        while (true) {
            $res = $this->triggerSyncRequest($filterUrl . $pageId, 'GET');
            if ($res && !empty($res['status']) && $res['status'] == 'successful') {
                $items = array_merge($items, $res['result']);
                if (count($res['result'])) {
                    $pageId++;
                } else {
                    break;
                }
            } else {
                break;
            }
        }
        $exportData = [];
        foreach ($items as $item) {
            $exportData[] = [
                'id' => $item['id'],
                'name' => $item['name'],
                'slug' => $item['slug'],
                'actor' => $item['actor']['email'],
                'keyword' => !empty($item['trademark']) ? $item['trademark']['keyword'] : '',
                'keyword_type' => !empty($item['trademark']) ? $item['trademark']['type'] : '',
                'reason' => $item['reason'],
                'view' => $item['view_count'],
                'sold' => $item['sold'],
                'created_at' => $item['created_at'],
                'updated_at' => $item['updated_at'],
            ];
        }

        return $this->triggerSyncRequest(config('app.url') . $this->getLocalePrefix() . '/admin/export-service/export', 'POST', [
            'json_data' => $exportData,
            'mode' => 'download',
            'email' => $user->email,
        ], ['Content-Type: application/json']);

        return $exportData;
    }

    protected function getLocalePrefix() {
        $locale = env('APP_LOCALE', config('localization::module.default_locale', 'us'));
        if (!$locale || $locale == config('localization::module.default_locale', 'us')) {
            $locale = '';
        } else {
            $locale = '/' . $locale;
        }

        return $locale;
    }
}