<?php

namespace Modules\Trademarks\Controllers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Modules\Trademarks\Models\SellerProductNTrademarkModel;

class TrademarkProductSellerController extends Controller
{
    public function index()
    {
        $staffs = DB::table('users')
            ->whereIn('role', ['ADMIN', 'STAFF'])
            ->get(['id', 'name']);

        $seller = DB::table('users')
            ->where('role', 'SELLER')
            ->get(['id', 'name']);

        return view('trademarks::trademark-product-seller.index', ['staffs' => $staffs, 'sellers' => $seller]);
    }

    public function find(Request $request)
    {
        $query = SellerProductNTrademarkModel::with(['trademark', 'actor' => function ($query) {
            $query->select(['id', 'name', 'email']);
        }])
            ->join('product', 'product.id', 'seller_product_n_trademark.product_id')
            ->whereNull('product.deleted_at')
            ->whereIn('seller_product_n_trademark.type', ['violation', 'trademark', 'hided', 'brand'])
            ->where('product.status', 'PENDING');

        if ($request->get('search')) {
            $keyword = strtolower($request->get('search'));
            if (preg_match("/^\d+$/", $keyword)) {
                $query->where('product_id', $keyword);
            } else if (preg_match("/^p\d+$/", $keyword)) {
                $query->where('product_id', str_replace('p', '', $keyword));
            } else if (preg_match("/^t\d+$/", $keyword)) {
                $query->where('trademark_id', str_replace('t', '', $keyword));
            }
        }

        if ($request->get('trademark_id')) {
            $query->where('trademark_id', $request->get('trademark_id'));
        }
        if ($request->get('date_from')) {
            $query->where('seller_product_n_trademark.created_at', '>=', $request->get('date_from'));
        }
        if ($request->get('date_to')) {
            $query->where('seller_product_n_trademark.created_at', '<=', $request->get('date_to'));
        }
        if ($request->get('update_from')) {
            $query->where('seller_product_n_trademark.updated_at', '>=', $request->get('update_from'));
        }
        if ($request->get('update_to')) {
            $query->where('seller_product_n_trademark.updated_at', '<=', $request->get('update_to'));
        }

        if ($request->has('actor_id')) {
            $query->where('seller_product_n_trademark.actor_id', $request->get('actor_id'));
        }

        if ($request->has('product_actor_id')) {
            $query->where('product.actor_id', $request->get('product_actor_id'));
        }

        if ($request->get('hide_by')) {
            $query->where('seller_product_n_trademark.hide_by', $request->get('hide_by'));
        }
        if ($request->get('reason')) {
            $query->where('seller_product_n_trademark.reason', $request->get('reason'));
        } else {
            $query->where(function ($query) {
                $query->where('seller_product_n_trademark.reason', '!=', 'auto')
                    ->orWhereNull('seller_product_n_trademark.reason');
            });
        }

        $query->orderBy('seller_product_n_trademark.created_at', 'desc');

        $simplePaginate = $request->get('simple_paginate', 1);
        $total = 0;
        if (!$simplePaginate) {
            $total = $this->getTotal($request, $query);
        } else {
            $pageSize = $request->get('page_size', 50);
            $pageId = $request->get('page_id', 0);
            $query->limit($pageSize)->offset($pageSize * $pageId);
        }

        $meta = $this->getMetaData($request->all(), $total, $simplePaginate);

        $items = $query->get([
            'product.id', 'product.name', 'product.slug', 'product.image_url', 'product.sold', 'product.view_count',
            'seller_product_n_trademark.product_id', 'seller_product_n_trademark.trademark_id', 'seller_product_n_trademark.actor_id', 'seller_product_n_trademark.reason',
            'seller_product_n_trademark.note', 'seller_product_n_trademark.created_at', 'seller_product_n_trademark.updated_at'
        ]);

        return [
            'status' => 'successful',
            'meta' => $meta,
            'result' => $items
        ];
    }

    public function getTotal($request, $query)
    {
        $locale = env('APP_LOCALE');
        if (!$locale) {
            $locale = 'us';
        }
        $key = 'SELLER_PRODUCT_TRADEMARK_TOTAL::' . $locale;
        $filters = $request->except(['page_id', 'page_size']);
        if (!count($filters) && Cache::get($key)) {
            return Cache::get($key);
        }
        $cloneQuery = clone $query;
        $total = $cloneQuery->count();
        if (!count($filters)) {
            Cache::put($key, $total, 3600);
        }
        return $total;
    }

    public function getMetaData($filter, $total, $simplePaginate = false)
    {
        $pageId = 0;
        $pageSize = 50;
        if (!empty($filter['page_size'])) {
            $pageSize = intval($filter['page_size']);
        }
        if (!empty($filter['page_id'])) {
            $pageId = intval($filter['page_id']);
        }
        $meta = [
            'page_id' => $pageId,
            'page_size' => $pageSize,
            'page_count' => 0,
            'has_next' => false,
            'total_count' => $total,
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1 || $simplePaginate) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function export(Request $request)
    {
        set_time_limit(3600);
        ini_set('memory_limit', '2048M');
        $user = Auth::user();
        if (!$user) {
            return [
                'status' => 'fail',
                'message' => 'Permission denied'
            ];
        }
        $filterUrl = trim($request->get('filter_url', ''), '?');
        if (empty($filterUrl)) {
            return [
                'status' => 'fail',
                'message' => 'filter_url is required'
            ];
        }
        $filterUrl = $filterUrl . (strpos($filterUrl, '?') !== false ? '&' : '?') . 'page_id=';
        $pageId = 0;
        $items = [];
        while (true) {
            $res = $this->triggerSyncRequest($filterUrl . $pageId, 'GET');
            if ($res && !empty($res['status']) && $res['status'] == 'successful') {
                $items = array_merge($items, $res['result']);
                if (count($res['result'])) {
                    $pageId++;
                } else {
                    break;
                }
            } else {
                break;
            }
        }
        $exportData = [];
        foreach ($items as $item) {
            $exportData[] = [
                'id' => $item['id'],
                'name' => $item['name'],
                'slug' => $item['slug'],
                'actor' => $item['actor']['email'],
                'keyword' => !empty($item['trademark']) ? $item['trademark']['keyword'] : '',
                'keyword_type' => !empty($item['trademark']) ? $item['trademark']['type'] : '',
                'reason' => $item['reason'],
                'view' => $item['view_count'],
                'sold' => $item['sold'],
                'created_at' => $item['created_at'],
                'updated_at' => $item['updated_at'],
            ];
        }

        return $this->triggerSyncRequest(config('app.url') . $this->getLocalePrefix() . '/admin/export-service/export', 'POST', [
            'json_data' => $exportData,
            'mode' => 'download',
            'email' => $user->email,
        ], ['Content-Type: application/json']);

        return $exportData;
    }

    protected function getLocalePrefix()
    {
        $locale = env('APP_LOCALE', config('localization::module.default_locale', 'us'));
        if (!$locale || $locale == config('localization::module.default_locale', 'us')) {
            $locale = '';
        } else {
            $locale = '/' . $locale;
        }

        return $locale;
    }
}