<?php

namespace Modules\Trademarks\Controllers;

use App\Utils\EventHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\Trademarks\Models\ProductModel;
use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\DB;
use Modules\Trademarks\Models\ProductNUser;

class TrademarkSellerController extends Controller
{
    protected $visionHost = 'https://megavision.megaads.vn';

    protected $listTrademarksKeywords = [];
    protected $listTradeMarksIds = [];
    protected $minMaxTrademarkId = [];
    protected $listViolationKeywords = [];
    protected $listViolationIds = [];
    protected $minMaxViolationId = [];
    protected $listWarningKeywords = [];
    protected $listWarningIds = [];
    protected $minMaxWarningId = [];

    protected $newKeywordHour = 0;
    protected $isDebug = 0;

    public function management()
    {
        return view('trademarks::trademark-seller.index');
    }

    public function testTrademark(Request $request)
    {
        $value = $request->get('value');
        if (!is_array($value)) {
            return [
                'status' => 'successful',
                'result' => [
                    'trademark' => $this->checkTextTrademarks($value),
                    'violation' => $this->checkTextViolation($value)
                ]
            ];
        } else {
            $retVal = [
                'status' => 'successful',
                'result' => []
            ];
            foreach ($value as $key => $val) {
                $retVal['result'][$key] = [
                    'trademark' => $this->checkTextTrademarks($val),
                    'violation' => $this->checkTextViolation($val)
                ];
            }
            return $retVal;
        }
    }

    protected function getViolationKeywords()
    {
        $query = DB::table('seller_trademarks_keyword');
        if ($this->listViolationIds) {
            $query->whereIn('id', $this->listViolationIds);
        }
        if ($this->minMaxViolationId) {
            $query->where('id', '>=', $this->minMaxViolationId[0])
                ->where('id', '<=', $this->minMaxViolationId[1]);
        }
        if ($this->newKeywordHour) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $this->newKeywordHour . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->where('type', 'violation');
        $keywords = $query->get(['keyword', 'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = strtolower(trim($item->keyword));
            $item->keyword = str_replace(' and ', ' ', $item->keyword);
            $item->keyword = str_replace('\\', ' ', $item->keyword);
            $item->keyword = str_replace('/', ' ', $item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    protected function checkTextViolation($text)
    {
        $text = strtolower(trim($text));

        if (!$this->listViolationKeywords) {
            $this->listViolationKeywords = $this->getViolationKeywords();
        }

        $result = $this->getViolation($text);

        $text = str_replace(' and ', ' ', $text);
        $text = preg_replace("/[^A-Za-z ]/", ' ', $text);
        $text = trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $text)));
        $result = array_merge($result, $this->getViolation($text));

        $text = preg_replace("/[^A-Za-z ]/", '', $text);
        $result = array_merge($result, $this->getViolation($text));

        $retVal = [];
        foreach ($result as $item) {
            if (!isset($retVal[$item->id])) {
                $retVal[$item->id] = $item;
            }
        }
        return array_values($retVal);
    }

    protected function checkTextTrademarks($text)
    {
        $text = strtolower(trim($text));

        if (!$this->listTrademarksKeywords) {
            $this->listTrademarksKeywords = $this->getTrademarksKeywords();
        }
        $result = $this->getTrademarks($text);

        $text = str_replace(' and ', ' ', $text);
        $text = preg_replace("/[^A-Za-z ]/", ' ', $text);
        $text = trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $text)));
        $result = array_merge($result, $this->getTrademarks($text));

        $text = preg_replace("/[^A-Za-z ]/", '', $text);
        $result = array_merge($result, $this->getTrademarks($text));
        $retVal = [];
        foreach ($result as $item) {
            if (!isset($retVal[$item->id])) {
                $retVal[$item->id] = $item;
            }
        }

        return array_values($retVal);
    }

    protected function getTrademarksKeywords()
    {
        $query = DB::table('seller_trademarks_keyword');
        if ($this->listTradeMarksIds) {
            $query->whereIn('id', $this->listTradeMarksIds);
        }
        if ($this->minMaxTrademarkId) {
            $query->where('id', '>=', $this->minMaxTrademarkId[0])
                ->where('id', '<=', $this->minMaxTrademarkId[1]);
        }
        if ($this->newKeywordHour) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $this->newKeywordHour . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->where('type', 'trademark');
        $keywords = $query->get(['keyword', 'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = $this->replaceKeyword($item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    protected function getTrademarks($text)
    {
        $text = strtolower(trim($text));
        $retVal = [];
        if (!$this->listTrademarksKeywords) {
            $this->listTrademarksKeywords = $this->getTrademarksKeywords();
        }
        foreach ($this->listTrademarksKeywords as $item) {
            if (
                preg_match("/\b" . $item->keyword . "\b/i", $text) ||
                preg_match("/\b" . str_replace(' ', '', $item->keyword) . "\b/i", $text)
            ) {
                $retVal[] = $item;
            }
        }
        return $retVal;
    }

    protected function replaceKeyword($keyword)
    {
        $keyword = strtolower(trim($keyword));
        $keyword = str_replace(' and ', ' ', $keyword);
        $keyword = str_replace('\\', '\\\\', $keyword);
        $keyword = str_replace('/', '\/', $keyword);
        $keyword = str_replace('.', '\.', $keyword);

        return $keyword;
    }

    protected function getViolation($text)
    {
        $text = strtolower(trim($text));
        $retVal = [];
        if (!$this->listViolationKeywords) {
            $this->listViolationKeywords = $this->getViolationKeywords();
        }
        foreach ($this->listViolationKeywords as $item) {
            if (
                preg_match("/\b" . $item->keyword . "\b/i", $text) ||
                preg_match("/\b" . str_replace(' ', '', $item->keyword) . "\b/i", $text)
            ) {
                $retVal[] = $item;
            }
        }
        return $retVal;
    }

    public function getFromId($request, $query, $orderField = 'created_at', $getField = 'id') {
        if ($request->input('minId')) {
            return $request->input('minId');
        }
        $cloneQuery = clone $query;

        $from = $cloneQuery->orderBy($orderField, 'asc')->first();
        if ($from) {
            return $from->{$getField};
        }

        return 0;
    }

    public function getToId($request, $query, $orderField = 'created_at', $getField = 'id') {
        if ($request->input('maxId')) {
            return $request->input('maxId');
        }
        $cloneQuery = clone $query;

        $to = $cloneQuery->orderBy($orderField, 'desc')->first();
        if ($to) {
            return $to->{$getField};
        }

        return 0;
    }

    public function checkAll(Request $request)
    {
        set_time_limit(10 * 3600);
        ini_set('memory_limit', '2048M');

        $query = ProductModel::query();
        if ($request->input('trademarksIds')) {
            $this->listTradeMarksIds = explode(',', $request->input('trademarksIds'));
        }
        if ($request->input('minMaxTrademarkId')) {
            $this->minMaxTrademarkId = explode(',', $request->input('minMaxTrademarkId'));
        }

        if ($request->input('listWarningIds')) {
            $this->listWarningIds = explode(',', $request->input('listWarningIds'));
        }
        if ($request->input('minMaxWarningId')) {
            $this->minMaxWarningId = explode(',', $request->input('minMaxWarningId'));
        }

        if ($request->input('violationIds')) {
            $this->listViolationIds = explode(',', $request->input('violationIds'));
        }
        if ($request->input('minMaxViolationId')) {
            $this->minMaxViolationId = explode(',', $request->input('minMaxViolationId'));
        }

        if ($request->input('new_keyword')) {
            $this->newKeywordHour = $request->get('hour', 3);
        }

        if ($request->input('hour')) {
            $hour = $request->input('hour');
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $hour . ' hours')));
        }

        if ($request->has('mt-check')) {
            $query->whereIn('id', explode(',', $request->input('mt-check')));
            $this->isDebug = 1;
        } else {
            $step = $request->input('step', 10000);
            $fromId = $this->getFromId($request, $query);
            $toId = $this->getToId($request, $query);
        }

        $query->whereNull('deleted_at');

        if (!$request->has('rebuild')) {
            $query->whereIn('status', ['ACTIVE', 'INACTIVE']);
        }

        if (!count($this->listTrademarksKeywords) && !count($this->listViolationKeywords) && !count($this->listWarningKeywords)) {
            return [
                'status' => 'successful',
                'message' => 'empty keyword',
                'result' => []
            ];
        }
        $productIsTrademarks = [];
        $productIsViolations = [];
        $productIsWarings = [];
        if ($this->isDebug) {
            $cloneQuery = clone $query;
            $allProduct = $cloneQuery
                ->join('product_n_user', 'product_n_user.product_id', '=', 'product.id')
                ->join('users', 'users.id', '=', 'product_n_user.user_id')
                ->orderBy('id')
                ->get(['id', 'name', 'description', 'content', 'trademarks', 'is_trademark', 'is_violation', 'status']);
            $res = $this->checkProducts($allProduct);
            $productIsTrademarks = array_merge($productIsTrademarks, $res['productIsTrademarks']);
            $productIsViolations = array_merge($productIsViolations, $res['productIsViolations']);
            $productIsWarings = array_merge($productIsWarings, $res['productIsWarings']);

            return response()->json([
                'status' => 'successful',
                'message' => 'Successful',
                'trademarks' => $productIsTrademarks,
                'violations' => $productIsViolations,
                'warning' => $productIsWarings,
                'trade_keywords' => count($this->listTrademarksKeywords),
                'violation_keywords' => count($this->listViolationKeywords),
                'warning_keywords' => count($this->listWarningKeywords),
            ]);
        }

        $i = 0;
        while ($fromId <= $toId) {
            $stepToId = $fromId + $step;
            if ($toId < $stepToId) {
                $stepToId = $toId;
            }
            $i++;
            $cloneQuery = clone $query;
            $allProduct = $cloneQuery->orderBy('id')
                ->where('id', '>=', $fromId)
                ->where('id', '<=', $stepToId)
                ->get(['id', 'name', 'description', 'content', 'trademarks', 'is_trademark', 'is_violation', 'status']);
            $res = $this->checkProducts($allProduct);
            $productIsTrademarks = array_merge($productIsTrademarks, $res['productIsTrademarks']);
            $productIsViolations = array_merge($productIsViolations, $res['productIsViolations']);
            $productIsWarings = array_merge($productIsWarings, $res['productIsWarings']);
            $fromId += $step + 1;
        }

        return response()->json([
            'status' => 'successful',
            'message' => 'Successful',
            'trademarks' => $productIsTrademarks,
            'violations' => $productIsViolations,
            'warnings' => $productIsWarings,
            'trade_keywords' => count($this->listTrademarksKeywords),
            'violation_keywords' => count($this->listViolationKeywords),
            'warning_keywords' => count($this->listWarningKeywords),
            'page' => $i,
        ]);
    }

    public function checkProducts($products)
    {
        $retVal = [
            'productIsTrademarks' => [],
            'productIsViolations' => [],
            'productIsWarings' => []
        ];
        foreach ($products as $product) {
            if (count($this->listTrademarksKeywords)) {
                $check = $this->checkTrademarks($product);
                foreach ($check as $productTrademark) {
                    if (!empty($productTrademark)) {
                        $retVal['productIsTrademarks'][] = $product->id;
                    }
                }
            }
            if (count($this->listViolationKeywords)) {
                $check = $this->checkViolation($product);
                foreach ($check as $productViolation) {
                    if (!empty($productViolation)) {
                        $retVal['productIsViolations'][] = $product->id;
                    }
                }
            }
            if (count($this->listWarningKeywords)) {
                $check = $this->checkWarning($product);
                foreach ($check as $productWarning) {
                    if (!empty($productWarning)) {
                        $retVal['productIsWarings'][] = $product->id;
                    }
                }
            }
        }

        return $retVal;
    }

    public function checkWarning($product)
    {
        $productViolations = [
            'title' => $this->checkTextWarning($product->name),
            'description' => $this->checkTextWarning($product->description),
            'content' => $this->checkTextWarning($product->content),
        ];
        foreach ($productViolations as $trademarks) {
            if (count($trademarks)) {
                foreach ($trademarks as $item) {
                    if (!DB::table('seller_product_n_trademark')->where(['product_id' => $product->id, 'trademark_id' => $item->id])->whereIn('type', ['warning', 'not-warning'])->exists()) {
                        DB::table('seller_product_n_trademark')
                            ->insert([
                                'product_id' => $product->id,
                                'trademark_id' => $item->id,
                                'type' => 'warning',
                                'created_at' => date('Y-m-d H:i:s', time()),
                                'updated_at' => date('Y-m-d H:i:s', time())
                            ]);
                    }
                    $productsDesignAlsoAvailable = $this->getProductsDesignAlsoAvailable($product->id);
                    foreach ($productsDesignAlsoAvailable as $productAvailable) {
                        if (!DB::table('seller_product_n_trademark')->where(['product_id' => $productAvailable->id, 'trademark_id' => $item->id])->whereIn('type', ['warning', 'not-warning'])->exists()) {
                            DB::table('seller_product_n_trademark')
                                ->insert([
                                    'product_id' => $productAvailable->id,
                                    'trademark_id' => $item->id,
                                    'type' => 'warning',
                                    'created_at' => date('Y-m-d H:i:s', time()),
                                    'updated_at' => date('Y-m-d H:i:s', time())
                                ]);
                        }
                    }
                }
            }
        }

        return $productViolations;
    }

    protected function checkTextWarning($text)
    {
        $text = strtolower(trim($text));

        if (!$this->listWarningKeywords) {
            $this->listWarningKeywords = $this->getListWarningKeywords();
        }

        $result = $this->getWarning($text);

        $text = str_replace(' and ', ' ', $text);
        $text = preg_replace("/[^A-Za-z ]/", ' ', $text);
        $text = trim(preg_replace('/\s\s+/', ' ', str_replace("\n", " ", $text)));
        $result = array_merge($result, $this->getWarning($text));

        $text = preg_replace("/[^A-Za-z ]/", '', $text);
        $result = array_merge($result, $this->getWarning($text));

        $retVal = [];
        foreach ($result as $item) {
            if (!isset($retVal[$item->id])) {
                $retVal[$item->id] = $item;
            }
        }

        return array_values($retVal);
    }

    protected function getWarning($text)
    {
        $text = strtolower(trim($text));
        $retVal = [];
        if (!$this->listWarningKeywords) {
            $this->listWarningKeywords = $this->getListWarningKeywords();
        }
        foreach ($this->listWarningKeywords as $item) {
            if (
                preg_match("/\b" . $item->keyword . "\b/i", $text) ||
                preg_match("/\b" . str_replace(' ', '', $item->keyword) . "\b/i", $text)
            ) {
                $retVal[] = $item;
            }
        }
        return $retVal;
    }

    protected function getListWarningKeywords()
    {
        $query = DB::table('trademarks');
        if ($this->listWarningIds) {
            $query->whereIn('id', $this->listWarningIds);
        }
        if ($this->minMaxWarningId) {
            $query->where('id', '>=', $this->minMaxWarningId[0])
                ->where('id', '<=', $this->minMaxWarningId[1]);
        }
        if ($this->newKeywordHour) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $this->newKeywordHour . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->where('type', 'warning');
        $keywords = $query->get(['keyword', 'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = strtolower(trim($item->keyword));
            $item->keyword = str_replace(' and ', ' ', $item->keyword);
            $item->keyword = str_replace('\\', ' ', $item->keyword);
            $item->keyword = str_replace('/', ' ', $item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    public function checkViolation($product)
    {
        $productViolations = [
            'title' => $this->checkTextViolation($product->name),
            'description' => $this->checkTextViolation($product->description),
            'content' => $this->checkTextViolation($product->content),
        ];
        $isViolation = false;
        foreach ($productViolations as $trademarks) {
            if (count($trademarks)) {
                $isViolation = 1;
                foreach ($trademarks as $item) {
                    DB::table('seller_product_n_trademark')
                        ->updateOrInsert(['product_id' => $product->id, 'trademark_id' => $item->id], [
                            'type' => 'violation',
                            'updated_at' => date('Y-m-d H:i:s', time())
                        ]);
                    $productsDesignAlsoAvailable = $this->getProductsDesignAlsoAvailable($product->id);
                    foreach ($productsDesignAlsoAvailable as $productAvailable) {
                        DB::table('seller_product_n_trademark')
                            ->updateOrInsert(['product_id' => $productAvailable->id, 'trademark_id' => $item->id], [
                                'type' => 'violation',
                                'updated_at' => date('Y-m-d H:i:s', time())
                            ]);
                        if (!$productAvailable->is_violation) {
                            $productAvailable->is_violation = 1;
                            $productAvailable->save();

                            EventHelper::pushEvent('productIsViolation', ['id' => $productAvailable->id]);
                        }
                    }
                }
            }
        }
        if ($isViolation) {
            if (!$product->is_violation) {
                DB::table('product')->where('id', $product->id)->update([
                    'is_violation' => 1,
                    'updated_at' => date('Y-m-d H:i:s', time())
                ]);

                EventHelper::pushEvent('productIsViolation', ['id' => $product->id]);
            }
        }

        return $productViolations;
    }

    public function checkTrademarks($product)
    {
        $productTrademarks = [];
        try {
            if (!empty($product->id)) {
                $productTrademarks = [
                    'title' => $this->checkTextTrademarks($product->name),
                    'description' => $this->checkTextTrademarks($product->description),
                    'content' => $this->checkTextTrademarks($product->content),
                ];
                $product->trademarks = json_encode($productTrademarks);
                $isTrademark = false;
                $hideIds = [];
                foreach ($productTrademarks as $trademarks) {
                    if (count($trademarks)) {
                        $isTrademark = true;
                        foreach ($trademarks as $item) {
                            DB::table('seller_product_n_trademark')
                                ->updateOrInsert(['product_id' => $product->id, 'trademark_id' => $item->id], [
                                    'type' => 'trademark',
                                    'updated_at' => date('Y-m-d H:i:s', time())
                                ]);
                            $productsDesignAlsoAvailable = $this->getProductsDesignAlsoAvailable($product->id);
                            foreach ($productsDesignAlsoAvailable as $productAvailable) {
                                DB::table('seller_product_n_trademark')
                                    ->updateOrInsert(['product_id' => $productAvailable->id, 'trademark_id' => $item->id], [
                                        'type' => 'trademark',
                                        'updated_at' => date('Y-m-d H:i:s', time())
                                    ]);
                                $productAvailable->trademarks = json_encode($productTrademarks);
                                if (!$productAvailable->is_trademark || $productAvailable->status == 'ACTIVE') {
                                    $productAvailable->is_trademark = 1;
                                    $productAvailable->status = 'PENDING';
                                    $productAvailable->save();
                                    $hideIds[] = $productAvailable->id;
                                }
                            }
                        }
                    }
                }

                if ($isTrademark) {
                    if (!$product->is_trademark || $product->status == 'ACTIVE') {
                        $this->createTicket($product->id, $product->trademarks);
                        DB::table('product')->where('id', $product->id)->update([
                            'is_trademark' => 1,
                            'status' => 'PENDING',
                            'updated_at' => date('Y-m-d H:i:s', time())
                        ]);
                        $hideIds[] = $product->id;
                        Log::info('CHECK TRADEMARK', [$product->id, $isTrademark]);
                    }
                }
                if (count($hideIds)) {
                    EventHelper::pushEvent('bulkHideProduct', ['ids' => $hideIds]);
                }
            }
        } catch (\Exception $exception) {
            Log::info('CHECK TRADEMARK', [$exception->getMessage() . ' line: ' . $exception->getLine()]);
        }
        return $productTrademarks;
    }

    protected function getProductsDesignAlsoAvailable($productId)
    {
        $retVal = [];
        $designIds = DB::table('product_n_design')
            ->where('product_id', $productId)
            ->pluck('design_id')->toArray();
        $productIds = DB::table('product_n_design')
            ->whereIn('design_id', $designIds)
            ->pluck('product_id')
            ->toArray();
        if (!empty($productIds) && !empty($designIds)) {
            $retVal = ProductModel::whereIn('id', $productIds)
                ->get(['product.id', 'product.is_violation', 'product.is_trademark', 'product.status']);
        }
        return $retVal;
    }

    protected function createTicket($productId, $keyword)
    {
        $product = DB::table('product')
            ->where('sold', '>', 0)
            ->where('id', $productId)
            ->first(['name', 'slug', 'id']);
        $result = null;
        $locale = env('APP_LOCALE');
        if ($locale == '' || !$locale) {
            $locale = 'us';
        }
        $locale = strtolower($locale);
        if ($product) {
            $data = [
                "status" => "PENDING",
                "title" => "Kiểm tra trademark sản phẩm " . $product->name,
                "description" => 'Sản phẩm được đánh trademark với từ các từ: \n' . $keyword,
                "team" => 'CONTENT',
                "locale" => $locale,
                "product_id" => $productId,
                "product_code" => null,
                "product_url" => url('/') . clroute($product->slug . '-p' . $product->id),
                "email" => 'check-trademark',
                "contents" => [
                    ["type" => "content", "data" => NULL, "team" => "CS"],
                    ["type" => "content", "data" => NULL, "team" => "FULFILL"],
                    ["type" => "content", "data" => NULL, "team" => "DESIGN"],
                    ["type" => "content", "data" => NULL, "team" => "CONTENT"]
                ],
                "assigns" => [
                    ["type" => "assign", "data" => NULL, "team" => "CS"],
                    ["type" => "assign", "data" => NULL, "team" => "FULFILL"],
                    ["type" => "assign", "data" => NULL, "team" => "DESIGN"],
                    ["type" => "assign", "data" => NULL, "team" => "CONTENT"]
                ],
                'service_token' => 'megaads@123'
            ];
            $request = new Request($data);
            if (
                class_exists('\Modules\Ticket\Controllers\Services\StaffJobService')
                && method_exists('\Modules\Ticket\Controllers\Services\StaffJobService', 'save')
            ) {
                $staffJobService = new \Modules\Ticket\Controllers\Services\StaffJobService;
                $result = $staffJobService->save($request);
            }
        }
        return $result;
    }

    public function checkBySearchService(Request $request)
    {
        $cacheKey = decorCacheKey('check_trademark_by_search');
        if ($request->has('product_hours')) {
            $cacheKey = $cacheKey . '_propduct';
        } else {
            $cacheKey = $cacheKey . '_keyword';
        }
        if (Redis::exists($cacheKey) && !$request->has('clear_cache')) {
            $retVal['result'] = Redis::get($cacheKey);
            return $retVal;
        } else {
            Redis::set($cacheKey, 1, 'EX', 60 * 60 * 6);
        }
        set_time_limit(3 * 3600);
        $keywords = $this->getTrademarksKeywordsService($request);
        $result = [];
        $minId = 0;
        if ($request->get('product_hours')) {
            $minProd = DB::table('product')->where('created_at', '>=', date('Y-m-d H:00:00', time() - $request->get('product_hours') * 3600))
                ->orderBy('created_at', 'asc')
                ->first(['id']);
            if ($minProd) {
                $minId = $minProd->id;
            }
        }
        foreach ($keywords as $keyword) {
            $this->checkKeyword($keyword, $result, $minId, 0);
        }
        Redis::del($cacheKey);
        return [
            'status' => 'successful',
            'result' => $result
        ];
    }

    protected function getTrademarksKeywordsService($request)
    {
        $query = DB::table('seller_trademarks_keyword');
        if ($request->get('trademark_ids')) {
            $query->whereIn('id', explode(',', $request->get('trademark_ids')));
        }
        if (!$request->get('all_keywords')) {
            $query->where('updated_at', '>=', date('Y-m-d H:i:s', strtotime('-' . $request->get('hour', 2) . ' hours')));
        }
        $query->where('status', 'ACTIVE');
        $query->whereIn('type', explode(',', $request->get('types', 'trademark,violation,warning')));
        $keywords = $query->get(['keyword', 'type', 'id'])->toArray();
        $retVal = [];
        foreach ($keywords as $item) {
            $item->keyword = $this->replaceKeyword($item->keyword);
            $retVal[] = $item;
        }
        return $retVal;
    }

    protected function getLocalePrefix()
    {
        $locale = env('APP_LOCALE', config('localization::module.default_locale', 'us'));
        if (!$locale || $locale == config('localization::module.default_locale', 'us')) {
            $locale = '';
        } else {
            $locale = '/' . $locale;
        }

        return $locale;
    }

    private function checkKeyword($trademark, &$retVal, $minId, $pageId = 0)
    {
        $url = config('app.url') . $this->getLocalePrefix() . "/z-search/trademark/search?page_size=1000&q=" . urlencode($trademark->keyword) . '&page_id=' . $pageId;
        if ($minId) {
            $url .= '&from_id=' . $minId;
        }

        $result = $this->triggerSyncRequest($url);
        if (!empty($result['result'])) {
            $productIds = [];
            foreach ($result['result'] as $item) {
                $productIds[] = $item['id'];
            }

            //loc ra các sp của seller
            $productBySellerIds = ProductNUser::query()
                ->join('users', 'users.id', '=', 'product_n_user.user_id')
                ->whereIn('product_n_user.product_id', $productIds)
                ->where('users.role', '=', 'SELLER')
                ->pluck('product_n_user.product_id')->toArray();

            $highRiskProductIds = DB::table('product_meta')
                ->whereIn('product_id', $productBySellerIds)
                ->where('key', 'is_high_risk')
                ->where('value', 1)
                ->pluck('product_id')
                ->toArray();
            $productBySellerIds = array_diff($productBySellerIds, $highRiskProductIds);
            if (count($productBySellerIds)) {
                $this->updateProduct($trademark, $productBySellerIds);
                $retVal = array_merge($retVal, $productBySellerIds);
            }

            if (!empty($result['meta']['has_next'])) {
                $this->checkKeyword($trademark, $retVal, $minId, ++$pageId);
            }
        }
    }

    public function updateProduct($trademark, $productIds)
    {
        $updateData = [];
        $productNTrademark = [
            'trademark_id' => $trademark->id,
            'type' => $trademark->type
        ];
        switch ($trademark->type) {
            case 'trademark':
                $updateData = [
                    'updated_at' => date('Y-m-d H:i:s', time())
                ];
                $updateData['is_trademark'] = 1;
                $updateData['status'] = 'PENDING';
                break;
            case 'violation':
                $updateData = [
                    'updated_at' => date('Y-m-d H:i:s', time())
                ];
                $updateData['is_violation'] = 1;
                break;
            default:
                break;
        }
        if (count($updateData)) {
            foreach (array_chunk($productIds, 100) as $partIds) {
                if ($trademark->type == 'trademark') {
                    DB::table('product')->whereIn('id', $partIds)->where(function ($query) {
                        $query->where('is_trademark', 0)
                            ->orWhere('status', 'ACTIVE');
                    })->update($updateData);
                } else if ($trademark->type == 'violation') {
                    DB::table('product')->whereIn('id', $partIds)->where('is_violation', 0)->update($updateData);
                }
            }
        }

        $insertData = [];
        $existIds = DB::table('seller_product_n_trademark')->where('trademark_id', $trademark->id)->whereIn('product_id', $productIds)->get(['product_id'])->pluck('product_id')->toArray();
        $productIds = array_diff($productIds, $existIds);
        foreach ($productIds as $productId) {
            $tmpData = $productNTrademark;
            $tmpData['product_id'] = $productId;
            $tmpData['created_at'] = date('Y-m-d H:i:s', time());
            $tmpData['updated_at'] = date('Y-m-d H:i:s', time());
            $insertData[] = $tmpData;
        }

        foreach (array_chunk($insertData, 100) as $tmpData) {
            DB::table('seller_product_n_trademark')->insert($tmpData);
        }
    }
}