<?php

namespace Modules\Trademarks\Controllers;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\Trademarks\Controllers\Controller;
use Modules\Trademarks\Models\ProductNTrademarkModel;

class WarningProductController extends HideProductController
{
    public function index() {
        $staffs = DB::table('users')
            ->whereIn('role', ['ADMIN', 'STAFF'])
            ->get(['id', 'name']);
        
        return view('trademarks::warning-product.index', ['staffs' => $staffs]);
    }

    public function getTotal($request, $query) {
        $locale = env('APP_LOCALE');
        if (!$locale) {
            $locale = 'us';
        }
        $key = 'WARNING_PRODUCT_TOTAL::' .  $request->get('type', 'warning') . $locale;
        $filters = $request->except(['page_id', 'page_size']);
        if (count($filters) == 1 && $request->get('type') && Cache::get($key)) {
            return Cache::get($key);
        }
        $cloneQuery = clone $query;
        $total = $cloneQuery->count();
        if (count($filters) == 1 && $request->get('type')) {
            Cache::put($key, $total, 3600);
        }
        return $total;
    }

    public function find(Request $request) {
        $query = ProductNTrademarkModel::with(['trademark', 'actor' => function ($query) {
            $query->select(['id', 'name', 'email']);
        }])
            ->join('product', 'product.id', 'product_n_trademark.product_id')
            ->whereNull('product.deleted_at');

        if ($request->get('search')) {
            $keyword = strtolower($request->get('search'));
            if (preg_match("/^\d+$/", $keyword)) {
                $query->where('product_id', $keyword);
            } else if (preg_match("/^p\d+$/", $keyword)) {
                $query->where('product_id', str_replace('p', '', $keyword));
            } else if (preg_match("/^t\d+$/", $keyword)) {
                $query->where('trademark_id', str_replace('t', '', $keyword));
            }
        }

        if ($request->get('trademark_id')) {
            $query->where('trademark_id', $request->get('trademark_id'));
        }
        if ($request->get('date_from')) {
            $query->where('product_n_trademark.created_at', '>=', $request->get('date_from'));
        }
        if ($request->get('date_to')) {
            $query->where('product_n_trademark.created_at', '<=', $request->get('date_to'));
        }
        if ($request->get('update_from')) {
            $query->where('product_n_trademark.updated_at', '>=', $request->get('update_from'));
        }
        if ($request->get('update_to')) {
            $query->where('product_n_trademark.updated_at', '<=', $request->get('update_to'));
        }

        if ($request->get('type')) {
            $query->where('type', $request->get('type'));
        }

        if ($request->has('actor_id')) {
            $query->where('product_n_trademark.actor_id', $request->get('actor_id'));
        }

        if ($request->has('product_actor_id')) {
            $query->where('product.actor_id', $request->get('product_actor_id'));
        }

        if ($request->get('hide_by')) {
            $query->where('hide_by', $request->get('hide_by'));
        }
        if ($request->get('product_status')) {
            $query->where('product.status', $request->get('product_status'));
        } else if ($request->get('type') && !$request->has('actor_id')) {
            $query->where('product.status', 'ACTIVE');
        }
        $query->orderBy('product.sold', 'desc')
            ->orderBy('product.view_count', 'desc')
            ->orderBy('product_n_trademark.created_at', 'desc');

        $simplePaginate = $request->get('simple_paginate', 0);
        $total = 0;
        if (!$simplePaginate) {
            $total = $this->getTotal($request, $query);
            $pageSize = $request->get('page_size', 50);
            $pageId = $request->get('page_id', 0);
            
            $query->limit($pageSize)->offset($pageSize * $pageId);
        }

        $meta = $this->getMetaData($request->all(), $total, $simplePaginate);

        $items = $query->get([
            'product.id', 'product.name', 'product.slug', 'product.image_url', 'product.sold', 'product.view_count', 'product.approve_advertising',
            'product_n_trademark.product_id', 'product_n_trademark.trademark_id', 'product_n_trademark.actor_id', 'product_n_trademark.created_at', 'product_n_trademark.updated_at'
        ]);

        return [
            'status' => 'successful',
            'meta' => $meta,
            'result' => $items
        ];
    }

    public function getMetaData($filter, $total, $simplePaginate = false)
    {
        $pageId = 0;
        $pageSize = 50;
        if (!empty($filter['page_size'])) {
            $pageSize = intval($filter['page_size']);
        }
        if (!empty($filter['page_id'])) {
            $pageId = intval($filter['page_id']);
        }
        $meta = [
            'page_id' => intval($pageId),
            'page_size' => intval($pageSize),
            'page_count' => 0,
            'has_next' => false,
            'total_count' => intval($total),
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1 || $simplePaginate) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function hideWarning(Request $request) {
        $token = $request->header('token');
        $user = DB::table('users')
            ->where('token', $token)
            ->whereIn('role', ['STAFF', 'ADMIN'])
            ->where('status', 'ACTIVE')
            ->first(['id', 'email']);

        if (!$user) {
            return [
                'status' => 'fail',
                'message' => 'Permission denied'
            ];
        }

        $productIds = explode(',', $request->get('product_ids', ''));

        $this->hideWarningProductNTrademark($user, $productIds, $request->get('trademark_ids'));
        if($request->get('hide_same_design')) {
            $designIds = DB::table('product_n_design')->whereIn('product_id', $productIds)->get(['design_id'])->pluck('design_id');
            $sameDesignProductIds = DB::table('product_n_design')
                ->whereNotIn('product_id', $productIds)
                ->whereIn('design_id', $designIds)
                ->get(['product_id'])
                ->pluck('product_id')
                ->toArray();

            if (!empty($request->get('trademark_ids'))) {
                $this->hideWarningProductNTrademark($user, $sameDesignProductIds, $request->get('trademark_ids'));
            }
        }

        return [
            'status' => 'successful'
        ];
    }

    public function hideWarningProductNTrademark($user, $productIds, $tradeMarkIds) {
        $updateData = [
            'type' => 'not-warning',
            'actor_id' => $user->id,
            'updated_at' => date('Y-m-d H:i:s', time()),
        ];

        DB::table('product_n_trademark')
            ->whereIn('product_id', $productIds)
            ->whereIn('trademark_id', $tradeMarkIds)
            ->where('type', 'warning')
            ->update($updateData);
    }
}