<?php

namespace Modules\Trademarks\Controllers;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\Trademarks\Controllers\Controller;
use Modules\Trademarks\Models\ProductNTrademarkModel;

class WrongLanguageController extends HideProductController
{
    public function index() {
        $staffs = DB::table('users')
            ->whereIn('role', ['ADMIN', 'STAFF'])
            ->get(['id', 'name']);
        
        return view('trademarks::wrong-language.index', ['staffs' => $staffs]);
    }

    public function find(Request $request) {
        $query = DB::table('product_language_warning')
        ->join('product', 'product.id', 'product_language_warning.product_id')
        ->whereNull('product.deleted_at');

        if ($request->get('search')) {
            $keyword = strtolower($request->get('search'));
            if (preg_match("/^\d+$/", $keyword)) {
                $query->where('product_id', $keyword);
            } else if (preg_match("/^p\d+$/", $keyword)) {
                $query->where('product_id', str_replace('p', '', $keyword));
            }
        }

        if ($request->get('date_from')) {
            $query->where('product.created_at', '>=', $request->get('date_from'));
        }
        if ($request->get('date_to')) {
            $query->where('product.created_at', '<=', $request->get('date_to'));
        }
        if ($request->has('actor_id')) {
            $query->where('product_language_warning.actor_id', $request->get('actor_id'));
        }
        if ($request->has('product_actor_id')) {
            $query->where('product.actor_id', $request->get('product_actor_id'));
        }
        if ($request->get('status', 'ACTIVE')) {
            $query->where('product_language_warning.status', $request->get('status', 'ACTIVE'));
        }

        $query->orderBy('product.sold', 'desc')
            ->orderBy('product.view_count', 'desc')
            ->orderBy('product.created_at', 'desc');

        $simplePaginate = $request->get('simple_paginate', 1);
        $total = 0;
        $cloneQuery = clone $query;
        $total = $cloneQuery->count();
        if (!$simplePaginate) {
            $cloneQuery = clone $query;
            $total = $cloneQuery->count();
        } else {
            $pageSize = $request->get('page_size', 50);
            $pageId = $request->get('page_id', 0);
            
            $query->limit($pageSize)->offset($pageSize * $pageId);
        }

        $meta = $this->getMetaData($request->all(), $total, $simplePaginate);

        $items = $query->get([
            'product.id as product_id', 'product.image_url', 'product.name', 'product_language_warning.other_language', 
            'product_language_warning.status as warning_status', 'product.status', 'product.actor_id', 'product.created_at',
            'product.view_count', 'product.sold',
        ]);

        return [
            'status' => 'successful',
            'meta' => $meta,
            'result' => $items
        ];
    }

    public function getMetaData($filter, $total, $simplePaginate = false)
    {
        $pageId = 0;
        $pageSize = 50;
        if (!empty($filter['page_size'])) {
            $pageSize = intval($filter['page_size']);
        }
        if (!empty($filter['page_id'])) {
            $pageId = intval($filter['page_id']);
        }
        $meta = [
            'page_id' => intval($pageId),
            'page_size' => intval($pageSize),
            'page_count' => 0,
            'has_next' => false,
            'total_count' => intval($total),
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1 || $simplePaginate) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    public function hideWarning(Request $request) {
        $token = $request->header('token');
        $user = DB::table('users')
            ->where('token', $token)
            ->whereIn('role', ['STAFF', 'ADMIN'])
            ->where('status', 'ACTIVE')
            ->first(['id', 'email']);

        if (!$user) {
            return [
                'status' => 'fail',
                'message' => 'Permission denied'
            ];
        }

        $productIds = explode(',', $request->get('product_ids', ''));

        if($request->get('hide_same_design')) {
            $designIds = DB::table('product_n_design')->whereIn('product_id', $productIds)->get(['design_id'])->pluck('design_id');
            $sameDesignProductIds = DB::table('product_n_design')
                ->whereNotIn('product_id', $productIds)
                ->whereIn('design_id', $designIds)
                ->get(['product_id'])
                ->pluck('product_id')
                ->toArray();
            $productIds = array_merge($productIds, $sameDesignProductIds);
        }

        DB::table('product_language_warning')->whereIn('product_id', $productIds)->update([
            'status' => 'PENDING',
            'actor_id' => $user->id,
            'updated_at' => date('Y-m-d H:i:s', time())
        ]);

        return [
            'status' => 'successful'
        ];
    }
}