<?php

namespace Modules\Trademarks\Service;
use App\Utils\Email;
use App\Utils\EventHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Trademarks\Models\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Trademarks\Models\ProductModel;

class HideProductService {
    public function hideProductByIds($user, $input) {
        $result = [];
        $productIds = explode(',', $input['product_ids']);
        $updateData = [
            'status' => 'PENDING',
            'updater_id' => $user->id,
            'updated_at' => date('Y-m-d H:i:s'),
        ];
        if (!empty($input['is_trademark'])) {
            $updateData['is_trademark'] = 1;
        }
        if (!empty($input['is_violation'])) {
            $updateData['is_violation'] = 1;
        }
        if (!empty($input['trademark_ids'])) {
            $updateData['is_trademark'] = DB::table('trademarks')->whereIn('id', $input['trademark_ids'])->whereIn('type', ['trademark', 'brand', 'warning'])->exists();
            $updateData['is_violation'] = DB::table('trademarks')->whereIn('id', $input['trademark_ids'])->where('type', 'violation')->exists();
        }

        $result = array_merge($result, $productIds);

        $this->hideProduct($productIds, $updateData);
        $this->saveLog($productIds, $user, $updateData);
        $options = [
            'hide_by' => !empty($input['hide_by']) ? $input['hide_by'] : 'id',
        ];
        if (!empty($input['note'])) {
            $options['note'] = $input['note'];
        }
        if (!empty($input['reason'])) {
            $options['reason'] = $input['reason'];
        }
        if (!empty($input['trademark_ids'])) {
            $this->saveProductNTrademark($user, $productIds, $input['trademark_ids'], $options);
        }

        if(!empty($input['hide_same_design'])) {
            $designIds = DB::table('product_n_design')->whereIn('product_id', $productIds)->get(['design_id'])->pluck('design_id');
            $sameDesignProductIds = DB::table('product_n_design')
                ->whereNotIn('product_id', $productIds)
                ->whereIn('design_id', $designIds)
                ->get(['product_id'])
                ->pluck('product_id')
                ->toArray();

            $this->hideProduct($sameDesignProductIds, $updateData);
            $this->saveLog($sameDesignProductIds, $user, $updateData);
            if (!empty($input['trademark_ids'])) {
                $this->saveProductNTrademark($user, $sameDesignProductIds, $input['trademark_ids'], $options);
            }
            
            $result = array_merge($result, $sameDesignProductIds);

            DB::table('small_design')->whereIn('design_id', $designIds)
                ->update([
                    'status' => 'PENDING',
                    'updated_at' => new \DateTime(),
                ]);
        }

        $result = $this->productIdsGroupBySeller($result);

        if (!empty($updateData['is_trademark'])) {
            foreach ($result as $item) {
                $this->sendEmailHideProduct($item['user'], $item['products']);
            }
        }

        return $result;
    }

    public function hideProduct($productIds, $updateData) {
        $products = DB::table('product')->whereIn('id', $productIds)->get(['id', 'is_trademark', 'is_violation']);
        foreach ($products as $product) {
            $tmpData = $updateData;
            if (isset($tmpData['is_trademark'])) {
                $tmpData['is_trademark'] = $product->is_trademark ? $product->is_trademark : $tmpData['is_trademark'];
            }
            if (isset($tmpData['is_violation'])) {
                $tmpData['is_violation'] = $product->is_violation ? $product->is_violation : $tmpData['is_violation'];
            }
            DB::table('product')->where('id', $product->id)
                ->update($tmpData);
        }
        if (count($productIds)) {
            EventHelper::pushEvent('bulkHideProduct', ['ids' => $productIds]);
        }
    }

    public function saveLog($productIds, $user, $updateData) {
        $logData = [];
        foreach ($productIds as $productId) {
            $logData[] = [
                'actor_email' => $user ? $user->email : '',
                'target_type' => 'PRODUCT',
                'target_id' => $productId,
                'event_type' => 'UPDATE',
                'data' => json_encode($updateData),
                'created_at' => date('Y-m-d H:i:s', time())
            ];
        }
        DB::table('log')->insert($logData);
    }

    public function saveProductNTrademark($user, $productIds, $trademarkIds, $options = []) {
        $trademarks = DB::table('trademarks')->whereIn('id', $trademarkIds)->get();
        $baseSaveData = [];
        foreach ($trademarks as $item) {
            $saveItem = [
                'trademark_id' => $item->id,
                'type' => $item->type == 'warning' ? 'hided' : $item->type,
                'actor_id' => $user->id,
                'created_at' => date('Y-m-d H:i:s', time()),
                'updated_at' => date('Y-m-d H:i:s', time())
            ];

            if (count($options)) {
                $saveItem = array_merge($saveItem, $options);
            }

            $baseSaveData[] = $saveItem;
        }

        foreach ($productIds as $productId) {
            $saveData = [];
            foreach ($baseSaveData as $item) {
                if (DB::table('product_n_trademark')->where('product_id', $productId)->where('trademark_id', $item['trademark_id'])->exists()) {
                    continue;
                }
                $item['product_id'] = $productId;
                $saveData[] = $item;
            }
            if (count($saveData)) {
                DB::table('product_n_trademark')->insert($saveData);
            }
        }
    }

    public function productIdsGroupBySeller($productIds) {
        $retVal = [];
        $pnu = DB::table('product_n_user')
            ->whereIn('product_id', $productIds)
            ->get();
        $productIdGroupByUserId = [];
        foreach ($pnu as $item) {
            if (!isset($productIdGroupByUserId[$item->user_id])) {
                $productIdGroupByUserId[$item->user_id] = [];
            }
            $productIdGroupByUserId[$item->user_id][] = $item->product_id;
        }

        $users = DB::table('users')->whereIn('id', array_keys($productIdGroupByUserId))
            ->whereNotNull('seller_token')
            ->get(['id', 'name', 'email']);
        foreach ($users as $user) {
            if (isset($productIdGroupByUserId[$user->id])) {
                $products = DB::table('product')
                    ->whereIn('id', $productIdGroupByUserId[$user->id])
                    // ->where('status', 'ACTIVE')
                    ->get(['id', 'sku', 'slug', 'name'])
                    ->toArray();
                if (count($products)) {
                    $retVal[] = [
                        'user' => (array) $user,
                        'products' => $products
                    ];
                }
            }
        }

        return $retVal;
    }

    public function sendEmailHideProduct($user, $products) {
        $emailDefault = [
            'driver' => 'ticket',
            'from' => '"' . config('app.name') . ' Support" <' . config('ticket::sa.send_email.email', 'support@printerval.com') . '>'
        ];
        $emailConfig = config('ticket::sa.send_email', $emailDefault);
        $param = [
            'driver' => $emailConfig['driver'],
            'subject' => 'DMCA takedown notice - ' . config('app.name') ,
            'from' => $emailConfig['from'],
            'to' => $user['email'],
            'html' => utf8_encode(view('trademarks::emails.hide-product', ['user' => $user, 'products' => $products])),
        ];
        $this->logSendEmail($user, $products);
        try {
            Email::sendToGearman($param);  
        } catch (\Exception $ex) {
            \Log::error('Send Email Hide Product Error:' . $ex->getMessage());
        }                  
    }

    private function logSendEmail($user, $products) {
        $data = [
            'user_id' => $user['id'],
            'to' => $user['email'],
            'products' => $products
        ];
        $logData = [
            'actor_type' => 'DMCA',
            'actor_email' => (Auth::user() ? Auth::user()->email : ''),
            'target_type' => 'DMCA_SELLER',
            'target_id' => $user['id'],
            'event_type' => 'SEND_EMAIL',
            'data' => json_encode($data),
            'created_at' => new \DateTime()
        ];
        DB::table('log')->insert([$logData]);
    }
}