<?php

namespace Modules\WarehousePod\Controllers\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\WarehousePod\Controllers\Controller;
use Modules\WarehousePod\Models\Warehouse;
use Modules\WarehousePod\Models\WarehouseConfig;
use Modules\WarehousePod\Models\WarehouseConfigItem;
use Modules\WarehousePod\Models\WarehouseConfigItemValue;
use Modules\WarehousePod\Models\WarehouseConfigItemVariant;
use Modules\WarehousePod\Models\WarehouseNProduct;
use Modules\WarehousePod\Models\Product;
use Modules\WarehousePod\Models\ProductInfo;
use Modules\WarehousePod\Models\WarehouseCategory;
use Modules\WarehousePod\Models\WarehouseCategoryValue;
use Modules\WarehousePod\Models\WarehouseCategoryVariant;
use Modules\WarehousePod\Models\ProductVariantOption;
use Modules\WarehousePod\Models\ProductNCategory;
use Modules\WarehousePod\Models\ProductSkuValue;
use Modules\WarehousePod\Models\WarehouseProductUpdate;
use Modules\WarehousePod\Models\Log;
use Modules\WarehousePod\Models\Category;
use Modules\WarehousePod\Models\ProductVariant;
use Modules\WarehousePod\Models\ProductNVariantOption;
use Modules\WarehousePod\Models\OrderWarningProfit;
use Modules\WarehousePod\Models\Country;
use Modules\WarehousePod\Models\ProviderNWarehouse;
use Modules\Ticket\Helpers\DBconnect;
use Illuminate\Support\Facades\Redis;
use Maatwebsite\Excel\Facades\Excel;

use Module;

class WarehouseConfigService extends Controller
{
    protected $connection = null;
    protected $pageSize = 20;
    static $variant = [];
    public function __construct()
    {        
        parent::__construct();
        if ($this->connection == null) {
            $this->connection = DBconnect::connect($this->market);
        }
    }

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $result = $this->getResult($filter);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function getResult($filter) {
        $query = Warehouse::on($this->configName);
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            $query->where(function ($q) use ($filter) {
                $q->orWhere('name', 'LIKE', '%' . $filter['keyword'] . '%');
                $q->orWhere('location', 'LIKE', '%' . $filter['keyword'] . '%');
            });
        }
        if (isset($filter['type']) && $filter['type'] != '') {
            $query->where('type', '=', $filter['type']);
        }
        if (isset($filter['!name']) && $filter['!name'] != '') {
            $query->where('name', '!=', $filter['!name']);
        }
        if (isset($filter['name']) && $filter['name'] != '') {
            $query->where('name', '=', $filter['name']);
        }
        if (isset($filter['id']) && $filter['id'] != '') {
            $query->where('id', '=', $filter['id']);
        }
        if (isset($filter['is_active']) && $filter['is_active'] != '') {
            $query->where('is_active', '=', $filter['is_active']);
        }
        if (isset($filter['country_id']) && $filter['country_id'] != '') {
            $query->where('country_id', '=', $filter['country_id']);
        }
        if (isset($filter['seller_email']) && $filter['seller_email'] != '') {
            $seller = \DB::table("users")->where("email", $filter["seller_email"])->where("status", "ACTIVE")->whereNotNull("seller_token")->first();
            if ($seller) {
                $query->where('seller_id', '=', $seller->id);
            } else {
                $query->where('seller_id', '=', -1);
            }
        } else {
            $query->whereNull('seller_id');
        }
        if (isset($filter['printing_code']) && $filter['printing_code'] != '') {
            $query->where('printing_code', '=', $filter['printing_code']);
        }
        $query->orderBy('created_at', 'DESC');
        $query->with(['warehouseConfigs.warehouseConfigItems']);
        if (isset($filter['category_id']) && $filter['category_id'] != '') {
            $query->whereHas('categoryConfigs',function ($q) use ($filter) {
                $q->where('category_id', '=', $filter['category_id']);
            });
        }
        if (isset($filter['ship_type']) && $filter['ship_type'] != '') {
            $query->whereHas('warehouseConfigs',function ($q) use ($filter) {
                $q->where('type', '=', $filter['ship_type']);
            });
        }
        if (isset($filter['has_ratio']) && $filter['has_ratio'] == 'has_config_ratio') {
            $query->whereHas('categoryConfigs',function ($q) use ($filter) {
                $q->whereNotNull('ratio_id');
            });
        }
        $query->with(['providers']);
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function buildFilter($request) {
        $retVal = [];
        $columns = ['id', 'keyword', 'country_id', 'is_active', 'printing_code', 'category_id', 'ship_type', 'type', 'seller_email', 'has_ratio'];
        foreach($columns as $column) {
            if ($request->has($column)){
                $retVal[$column] = $request->input($column);
            }
        }
        $retVal['page_size'] = $this->pageSize;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function getWarehouseCategory(Request $request) {
        $response = ['status' => 'fail'];
        $warehouseCategoryId = $request->has('warehouse_category_id') ? $request->input('warehouse_category_id') : -1;
        if ($request->has('warehouse_id')) {
            if ($warehouseCategoryId == -1) {
                $query = WarehouseCategory::on($this->configName)
                                        ->where('warehouse_id', $request->input('warehouse_id'));
    
                if ($request->has('page_id') && $request->has('page_size')) {
                    $pageId = (int) $request->input('page_id', 1);
                    $pageSize = (int) $request->input('page_size', 3);
                    $query->forPage($pageId, $pageSize);
                }
            } else {
                $query = WarehouseCategory::on($this->configName)
                    ->with(['options', 'products'])
                    ->where('id', $warehouseCategoryId);

                if ($request->has('page_id') && $request->has('page_size')) {
                    $pageId = (int) $request->input('page_id', 1);
                    $pageSize = (int) $request->input('page_size', 3);
                    $query->forPage($pageId, $pageSize);
                }
            }
            
            $items = $query->get()->toArray();

            $this->decorWarehouseCategory($items);
            $response['status'] = 'successful';
            $response['result'] = $items;
        }
        return response()->json($response);
    }

    private function decorWarehouseCategory(&$items) {
        foreach ($items as &$item) {
            if (isset($item['options'])) {
                foreach ($item['options'] as $key => $option) {
                    $varians = ProductVariantOption::on($this->configName)
                                                                    ->where('variant_id', '=', $option['option_id'])
                                                                    ->pluck('name', 'id')->toArray();
                    $notApplyVariants = $varians;
                    $item['options'][$key]['variants'] = $varians;
                }
            } else {
                $item['options'] = [];
            }
            if (isset($item['products'])) {
                foreach ($item['products'] as $key => $product) {
                    $item['products'][$key]['variants'] = json_decode($product['variants']);
                }
            } else {
                $item['products'] = [];
            }
        }
    }

    public function storeWarehouseCategory(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('warehouse_id')) {
            $warehouseData = Warehouse::on($this->configName)->find($request->input("warehouse_id"));
            if ($warehouseData->is_editable == 0) {
                $retval = [
                    'status' => 'fail', 
                    'message' => 'Bạn không có quyền lưu kho. Vui lòng liên hệ bộ phận kỹ thuật!', 
                ];
                return $retval;
            }
            $warehouse = Warehouse::on($this->configName)
                    ->where('id', '=', $request->input('warehouse_id'))
                    ->first();
            if ($warehouse) {
                if ($request->has('categoryConfig')) {
                    $categoryConfig = $request->input("categoryConfig");
                    if (!empty($categoryConfig['category_id'])) {
                        $data = [
                            'warehouse_id' => $warehouse->id, 
                            'category_id' => $categoryConfig['category_id'], 
                            'cost' => isset($categoryConfig['cost']) ? $categoryConfig['cost'] : null,
                            'origin_cost' => isset($categoryConfig['origin_cost']) ? $categoryConfig['origin_cost'] : null,
                            'cost_bulk' => isset($categoryConfig['cost_bulk']) ? $categoryConfig['cost_bulk'] : null,
                            'origin_cost_bulk' => isset($categoryConfig['origin_cost_bulk']) ? $categoryConfig['origin_cost_bulk'] : null,
                            'ratio_id' => isset($categoryConfig['ratio_id']) ? $categoryConfig['ratio_id'] : null,
                        ];    
                        if (isset($categoryConfig['id'])) {
                            $warehouseCategory = WarehouseCategory::on($this->configName)->where('id', '=', $categoryConfig['id'])->first();
                            if (!$warehouseCategory) {
                                $warehouseCategory = WarehouseCategory::on($this->configName)->create($data);
                                $warehouseCategory->save();
                            } else {
                                WarehouseCategory::on($this->configName)->where('id', '=', $categoryConfig['id'])->update($data);
                            }
                        } else {
                            $warehouseCategory = WarehouseCategory::on($this->configName)->create($data);
                            $warehouseCategory->save();
                        }
                        if (isset($categoryConfig['options'])) {
                            $this->saveWarehouseCategoryValue($categoryConfig['options'], $warehouseCategory);
                        }
                        if (isset($categoryConfig['products'])) {
                            $this->saveWarehouseCategoryVariant($categoryConfig['products'], $warehouseCategory);
                        }
                    }
                }
            }
            $response = [
                'status' => 'successful'
            ];
            return response()->json($response);
        }
    }

    public function deleteWarehouseCategoryById(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('id')) {
            $warehouseCategoryId = $request->input('id');
            WarehouseCategoryVariant::on($this->configName)
                                ->where('warehouse_category_id', $warehouseCategoryId)
                                ->delete();
            WarehouseCategoryValue::on($this->configName)
                                ->where('warehouse_category_id', $warehouseCategoryId)
                                ->delete();
            WarehouseCategory::on($this->configName)
                                ->where('id', $warehouseCategoryId)
                                ->delete();
            $response = [
                'status' => 'successful'
            ];
            return response()->json($response);
        }
    }

    public function enableEdit(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if (hasPermission('enable_edit_warehouse')) {
            if ($request->has("id") && $request->has("is_editable")) {
                \DB::table("warehouse")->where("id", $request->input("id"))->update(["is_editable" => $request->input("is_editable")]);
                $email = \Auth::user()->email;
                $log = [
                    'actor_type' => 'WAREHOUSE',
                    'actor_email' => $email,
                    'target_type' => 'WAREHOUSE',
                    'event_type' => 'ENABLE_EDIT',
                    'created_at' => new \DateTime()
                ];
                $log['target_id'] = $request->input("id");
                $log['data'] = json_encode($request->all());
                Log::on($this->configName)->create($log);
                $response = [
                    'status' => 'successful'
                ];
            }
        } else {
            $response["message"] = "Permission deny!";
        }
        return response()->json($response);
    }

    public function store(Request $request)
    {
        if ($request->has("updated_at")) {
            $updatedAt = $request->input("updated_at");
        }
        if (isset($updatedAt) && $request->has("id")) {
            $warehouseData = Warehouse::on($this->configName)->find($request->input("id"));
            $date = new \Datetime($updatedAt);
            $warehouseUpdatedAt = new \Datetime($warehouseData->updated_at);
            if ($warehouseData->is_editable == 0) {
                $retval = [
                    'status' => 'fail', 
                    'message' => 'Bạn không có quyền lưu kho. Vui lòng liên hệ bộ phận kỹ thuật!', 
                ];
                return $retval;
            }
            if ($warehouseUpdatedAt > $date) {
                $retval = [
                    'status' => 'fail', 
                    'message' => 'Kho được cập nhật trước đó. Vui lòng kiểm tra lại!', 
                ];
                return $retval;
            }
        }
        $email = $request->has('email') ? $request->get('email') : null;
        $data = $this->buildDataWarehouse($request);
        $log = [
            'actor_type' => 'WAREHOUSE',
            'actor_email' => $email,
            'target_type' => 'WAREHOUSE_CONFIG',
            'created_at' => new \DateTime()
        ];
        if (isset($data['id']) && !empty($data['id'])) {
            if (isset($data["seller_id"])) {
                $data["is_editable"] = 1;
            }
            Warehouse::on($this->configName)
                    ->where('id', '=', $data['id'])
                    ->update($data);
            $warehouse = Warehouse::on($this->configName)->find($data['id']);
            $log['event_type'] = 'UPDATE';
        } else {
            $data["is_editable"] = 1;
            $warehouse = Warehouse::on($this->configName)->create($data);
            $log['event_type'] = 'CREATE';
        }
        $log['target_id'] = $warehouse->id;
        $log['data'] = json_encode($request->all());
        Log::on($this->configName)->create($log);
        $this->saveConfigs($request, 'shipping_configs', $warehouse);
        $this->saveConfigs($request, 'cost_configs', $warehouse);
        $this->saveWarehouseCategory($request, $warehouse);
        $this->saveWarehouseProviders($request, $warehouse);
        if ($request->has("complete_save") && $request->input("complete_save")) {
            \DB::table("warehouse")->where("id", $request->input("id"))->update(["is_editable" => 0]);
        }
        $response = [
            'status' => 'successful',
            'result' => $warehouse
        ];
        return response()->json($response);
    }

    private function saveWarehouseCategory($request, $warehouse) {
        if ($request->has('category_configs')) {
            $this->deleteWarehouseCategory($warehouse, $request->get('category_configs'));
            foreach ($request->get('category_configs') as $categoryConfig) {
                if (isset($categoryConfig['category_id']) && $categoryConfig['category_id'] != null) {
                    if (isset($categoryConfig['id'])) {
                        $warehouseCategory = WarehouseCategory::on($this->configName)
                                                    ->where('id', '=', $categoryConfig['id'])
                                                    ->first();
                        if (!$warehouseCategory) {
                            $warehouseCategory = WarehouseCategory::on($this->configName)
                                                                            ->create([
                                                                                'warehouse_id' => $warehouse->id, 
                                                                                'category_id' => $categoryConfig['category_id'], 
                                                                                'cost' => isset($categoryConfig['cost']) ? $categoryConfig['cost'] : null,
                                                                                'origin_cost' => isset($categoryConfig['origin_cost']) ? $categoryConfig['origin_cost'] : 0,
                                                                                'ratio_id' => isset($categoryConfig['ratio_id']) ? $categoryConfig['ratio_id'] : null
                                                                            ]);
                            $warehouseCategory->save();
                        } else {
                            WarehouseCategory::on($this->configName)->where('id', '=', $categoryConfig['id'])
                                                                    ->update([
                                                                        'category_id' => $categoryConfig['category_id'], 
                                                                        'cost' => isset($categoryConfig['cost']) ? $categoryConfig['cost'] : null,
                                                                        'origin_cost' => isset($categoryConfig['origin_cost']) ? $categoryConfig['origin_cost'] : 0,
                                                                        'ratio_id' => isset($categoryConfig['ratio_id']) ? $categoryConfig['ratio_id'] : null
                                                                    ]);
                        }
                    } else {
                        $warehouseCategory = WarehouseCategory::on($this->configName)
                                            ->create([
                                                'warehouse_id' => $warehouse->id, 
                                                'category_id' => $categoryConfig['category_id'], 
                                                'cost' => isset($categoryConfig['cost']) ? $categoryConfig['cost'] : null,
                                                'origin_cost' => isset($categoryConfig['origin_cost']) ? $categoryConfig['origin_cost'] : 0,
                                                'ratio_id' => isset($categoryConfig['ratio_id']) ? $categoryConfig['ratio_id'] : null
                                            ]);
                        $warehouseCategory->save();
                    }
                    $this->saveWarehouseCategoryValue($categoryConfig['options'], $warehouseCategory);
                    $this->saveWarehouseCategoryVariant($categoryConfig['products'], $warehouseCategory);
                }
            }
        }
    }

    private function deleteWarehouseCategory($warehouse, $categoryConfig) {
        $requestIds = [];
        foreach ($categoryConfig as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
        }
        $idExists = WarehouseCategory::on($this->configName)
                                        ->where('warehouse_id', '=', $warehouse->id)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            $deletedIds = array_values($deletedIds);
            if (!empty($deletedIds)) {
                WarehouseCategory::on($this->configName)
                                ->whereIn('id', $deletedIds)
                                ->delete();
            }
        }
    }

    private function deleteWarehouseCategoryVariant($warehouseCategory, $products) {
        $requestIds = [];
        foreach ($products as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
        }
        $idExists = WarehouseCategoryVariant::on($this->configName)
                                        ->where('warehouse_category_id', '=', $warehouseCategory->id)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            $deletedIds = array_values($deletedIds);
            if (!empty($deletedIds)) {
                WarehouseCategoryVariant::on($this->configName)
                                ->whereIn('id', $deletedIds)
                                ->delete();
            }
        }

    }

    private function saveWarehouseCategoryVariant($products, $warehouseCategory) {
        $this->deleteWarehouseCategoryVariant($warehouseCategory, $products);
        foreach ($products as $product) {
            $data = $this->buildWarehouseCategoryVariant($warehouseCategory, $product);
            if (isset($data['id'])) {
                $first = WarehouseCategoryVariant::on($this->configName)
                                                    ->where('id', '=', $data['id'])
                                                    ->first();
                if (!$first) {
                    unset($data['id']);
                    $item = WarehouseCategoryVariant::on($this->configName)->create($data);
                    $item->save();
                } else {
                    WarehouseCategoryVariant::on($this->configName)
                                                ->where('id', '=', $first->id)
                                                ->update($data);
                }
            } else {
                $item = WarehouseCategoryVariant::on($this->configName)->create($data);
                $item->save();
            }
        }
    }

    private function buildWarehouseCategoryVariant($warehouseCategory, $product) {
        $retVal = [
            'warehouse_category_id' => $warehouseCategory->id,
            'str_id' => $product['str_id'],
            'deleted' => $product['deleted'],
            'instock' => $product['instock'],
            'variants' => json_encode($product['variants']),
            'cost' => isset($product['cost']) ? $product['cost'] : null,
            'origin_cost' => isset($product['origin_cost']) ? $product['origin_cost'] : 0,
            'cost_bulk' => isset($product['cost_bulk']) ? $product['cost_bulk'] : null,
            'origin_cost_bulk' => isset($product['origin_cost_bulk']) ? $product['origin_cost_bulk'] : null,
            'ratio_id' => isset($product['ratio_id']) ? $product['ratio_id'] : null
        ];
        if (isset($product['id'])) {
            $retVal['id'] = $product['id'];
        }
        return $retVal;
    }

    private function deleteWarehouseCategoryValue($warehouseCategory, $options) {
        $requestIds = [];
        foreach ($options as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
        }
        $idExists = WarehouseCategoryValue::on($this->configName)
                                        ->where('warehouse_category_id', '=', $warehouseCategory->id)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            $deletedIds = array_values($deletedIds);
            if (!empty($deletedIds)) {
                WarehouseCategoryValue::on($this->configName)
                                ->whereIn('id', $deletedIds)
                                ->delete();
            }
        }

    }

    private function saveWarehouseCategoryValue($options, $warehouseCategory) {
        $this->deleteWarehouseCategoryValue($warehouseCategory, $options);
        foreach ($options as $option) {
            if (isset($option['option_id'])) {
                $data = $this->buildWarehouseCategoryValue($warehouseCategory, $option);
                if (isset($data['id'])) {
                    $first = WarehouseCategoryValue::on($this->configName)
                                                    ->where('id', '=', $data['id'])
                                                    ->first();
                    if (!$first) {
                        unset($data['id']);
                        $item = WarehouseCategoryValue::on($this->configName)
                                                ->create($data);
                        $item->save();
                    } else {
                        WarehouseCategoryValue::on($this->configName)
                                                ->where('id', '=', $first->id)
                                                ->update($data);
                    }
                } else {
                    $item = WarehouseCategoryValue::on($this->configName)
                                                ->create($data);
                    $item->save();
                }
            }
        }
    }

    private function buildWarehouseCategoryValue($warehouseCategory, $option) {
        if (empty($option['variant_ids'])) {
            $option['variant_ids'] = [0];
        }
        $data = [
            'warehouse_category_id' => $warehouseCategory->id, 
            'option_id' => $option['option_id'], 
            'option_value_ids' => implode(',', $option['variant_ids'])
        ];
        $data['not_apply_ids'] = null;
        if (isset($option['not_apply_ids']) && !empty($option['not_apply_ids'])) {
            $data['not_apply_ids'] = implode(',', $option['not_apply_ids']);
        }
        if (isset($option['id']) && !empty($option['id'])) {
            $data['id'] = $option['id'];
        }
        return $data;
    }

    private function buildDataWarehouse($request) {
        $columns = ['id', 'is_active', 'printing_code', 'location', 'name', 'country_id', 'type', 'rate'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = ($request->has($column)) ? $request->get($column) : '';
        }
        if (!isset($retVal['country_id']) || $retVal['country_id'] == '') {
            $retVal['country_id'] = -1;
        }
        if ($request->has("seller_email")) {
            $seller = \DB::table("users")->where("email", $request->input("seller_email"))->where("status", "ACTIVE")->whereNotNull("seller_token")->first();
            if ($seller) {
                $retVal['seller_id'] = $seller->id;
            }
        }
        return $retVal;
    }

    private function saveWarehouseProviders($request, $warehouse) {
        if ($request->has('providerIds')) {
            ProviderNWarehouse::on($this->configName)
                ->where('warehouse_id', '=', $warehouse->id)
                ->whereNotIn('provider_id', $request->input('providerIds'))
                ->delete();
            foreach ($request->input('providerIds') as $provider) {
                $warehouseProvider = ProviderNWarehouse::on($this->configName)
                                            ->where('warehouse_id', '=', $warehouse->id)
                                            ->where('provider_id', '=', $provider)
                                            ->first();
                if (!$warehouseProvider) {
                    ProviderNWarehouse::on($this->configName)->create([
                        'warehouse_id' => $warehouse->id, 
                        'provider_id' => $provider, 
                        'created_at' => new \DateTime(),
                        'updated_at' => new \DateTime(),
                    ]);
                }
            }
        }
    }

    private function saveConfigs($request, $type, $warehouse) {
        if ($request->has($type) && !empty($request->input($type))) {
            $this->deleteConfig($warehouse, $request->input($type));
            foreach($request->input($type) as $item) {
                $data = $this->buildConfig($item, $warehouse);
                if (!isset($data['id'])) {
                    $config = WarehouseConfig::on($this->configName)
                                                ->create($data);
                    $config->save();
                } else {
                    WarehouseConfig::on($this->configName)
                                    ->where('id', $data['id'])
                                    ->update($data);
                    $config = WarehouseConfig::on($this->configName)
                                                ->find($data['id']);
                }

                if (array_key_exists('config_items', $item) && !empty($item['config_items'])) {
                    $this->deleteConfigItem($config, $item['config_items']);
                    $retVal['config_items'] = [];
                    foreach ($item['config_items'] as $configItem) {
                         $dataItem = $this->buildDataConfigItem($configItem, $config);
                         if (!isset($dataItem['id'])) {
                            $configItem = WarehouseConfigItem::on($this->configName)->create($dataItem);
                            $configItem->save();
                         } else {
                             WarehouseConfigItem::on($this->configName)
                                                    ->where('id', '=', $dataItem['id'])
                                                    ->update($dataItem);
                         }
                    }
                }
            }
        } else {
            if ($type == 'shipping_configs') {
                $this->clearConfig($warehouse->id, 'shipping');
            } else if ($type == 'cost_configs') {
                $this->clearConfig($warehouse->id, 'cost');
            }
        }
    }

    private function clearConfig($warehouseId, $type) {
        $configIds = WarehouseConfig::on($this->configName)
                                        ->where('warehouse_id', '=', $warehouseId)
                                        ->where('config_type', '=', $type)
                                        ->pluck('id')
                                        ->toArray();
        if (!empty($configIds)) {
            WarehouseConfigItem::on($this->configName)
                                 ->whereIn('warehouse_config_id', $configIds)
                                 ->delete();
            WarehouseConfig::on($this->configName)
                            ->whereIn('id', $configIds)
                            ->delete();
        }
    }


    private function deleteConfig($warehouse, $items) {
        $type = '';
        $requestIds = [];
        foreach ($items as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
            $type = $item['config_type'];
        }
        $idExists = WarehouseConfig::on($this->configName)
                                        ->where('warehouse_id', '=', $warehouse->id)
                                        ->where('config_type', '=', $type)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            $deletedIds = array_values($deletedIds);
            if (!empty($deletedIds)) {
                WarehouseConfig::on($this->configName)->whereIn('id', $deletedIds)
                                                ->delete();
            }
        }        
    }

    private function deleteConfigItem($config, $items) {
        $requestIds = [];
        foreach ($items as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
        }
        $idExists = WarehouseConfigItem::on($this->configName)
                                        ->where('warehouse_config_id', '=', $config->id)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            if (!empty($deletedIds)) {
                WarehouseConfigItem::on($this->configName)
                                ->whereIn('id', $deletedIds)
                                ->delete();
            }
        }
    }

    private function buildConfig($config, $warehouse) {
        $retVal = [
            'warehouse_id' => $warehouse->id
        ];
        $columns = ['id','config_type', 'type', 'name', 'is_active'];
        foreach ($columns as $column) {
            if (array_key_exists($column, $config)) {
                $retVal[$column] = $config[$column];
            }
        }
        $retVal['invalid_cate'] = null;
        if (array_key_exists('invalid_cate', $config) && !empty($config['invalid_cate'])) {
            $retVal['invalid_cate'] = implode(',', $config['invalid_cate']);
        }
        return $retVal;
    }

    private function buildDataConfigItem($item, $config) {
        $retVal = [
            'warehouse_config_id' => $config->id
        ];
        $columns = array_merge(WarehouseConfigItem::COLUMNS, ['id']);
        $columnDefaults = ['default_shipping_fee', 'origin_shipping_fee', 'fee_limit', 'fee_if_limit', 'default_adding_item', 'origin_adding_fee', 'shipping_min_time', 'shipping_max_time', 'handling_min_time', 'handling_max_time', 'tax'];
        foreach ($columnDefaults as $column) {
            $retVal[$column] = 0;
        }
        foreach ($columns as $column) {
            if (array_key_exists($column, $item)) {
                $retVal[$column] = $item[$column];
            }
        }
        $retVal['type'] = 'custom';
        if (array_key_exists('is_default', $item) && $item['is_default']) {
            $retVal['type'] = 'default';
        }
        $retVal['apply_cate'] = null;
        if (array_key_exists('apply_cate', $item) && !empty($item['apply_cate'])) {
            if (is_array($item['apply_cate'])) {
                $item['apply_cate'] = implode(',', $item['apply_cate']);
            }
            $retVal['apply_cate'] = $item['apply_cate'];
        }

        return $retVal;
    }

    public function getOptionValue(Request $request) {
        $optionId = $request->get('option_id');
        $response = [
            'status' => 'fail'
        ];
        $result = [];
        if ($optionId) {
            $key = 'VARIANT:OPTION:' . $this->market . ':' . $optionId;
            $variantIds = [-1];
            if (Redis::exists($key)) {
                $values = Redis::get($key);
                $variantIds = json_decode($values, true);
            }
            $result = $this->connection->table('product_variant_option')
                                    ->whereIn('id', $variantIds)
                                    ->pluck('name', 'id')->toArray();
            if (count($result) == 0) {
                $result = $this->connection->table('product_variant_option')
                                    ->where('variant_id','=', $optionId)
                                    ->pluck('name', 'id')->toArray();
                
            }
            if (count($result) > 0) {
                $options = $result;
                $response['status'] = 'successful';
                $response['result'] = [
                    'apply_options' => $result,
                    'not_apply_options' => $options
                ];
            }
        }
        return response()->json($response);
    }

    public function buildStatus(Request $request) {
        set_time_limit(-1);
        $statusByCategory = $this->buildWarehouseStatus();
        $result = $this->removeStatusDuplicate($statusByCategory);
        $this->saveUpdateData($result['instocks'], 'instock');
        $this->saveUpdateData($result['outstocks'], 'outstock');
        $response = [
            'status' => 'successful'
        ];
        return response()->json($response);
    }

    private function removeStatusDuplicate($statusByCategory) {
        $instocks = $statusByCategory['instock'];
        $outstocks = [];
        foreach ($statusByCategory['outstock'] as $categoryId => $values) {
            if (!isset($outstocks[$categoryId])) {
                $outstocks[$categoryId] = [];
            }
            foreach ($values as $value) {
                $keys = [];
                foreach ($value as $item) {
                    $keys[] = $item['variant_option_id'];
                }
                sort($keys);
                $key = implode('-', $keys);
                if (!in_array($key, $statusByCategory['keyInstocks'][$categoryId])) {
                    $outstocks[$categoryId][] = $value;
                }
            }
        }
        return [
            'instocks' => $instocks,
            'outstocks' => $outstocks,
        ];
    }

    private function saveUpdateData($items, $type) {
        foreach ($items as $key => $values) {
            foreach ($values as $value) {
                $data = [
                    'category_id' => $key,
                    'type' => $type,
                    'variant' => json_encode($value)
                ];
                $uniqId = $key;
                $variantOptions = [];
                foreach ($value as $variant) {
                    $variantOptions[] = $variant['variant_option_id'];
                }
                sort($variantOptions);
                $data['uniq_id'] = $uniqId . '-' . implode('-', $variantOptions);
                $lastUpdate = WarehouseProductUpdate::on($this->configName)
                                        ->where('uniq_id', '=', $data['uniq_id'])
                                        ->orderBy('created_at', 'DESC')
                                        ->first();

                if (!$lastUpdate) {
                    WarehouseProductUpdate::on($this->configName)
                                            ->create($data);
                } else {
                    if ($lastUpdate->type != $data['type']) {
                        WarehouseProductUpdate::on($this->configName)
                                                ->where('id', '=', $lastUpdate->id)
                                                ->where('status', '=', 'waiting')
                                                ->update(['status' => 'done']);
                        WarehouseProductUpdate::on($this->configName)
                                                ->create($data);
                    }
                }
            }
        }
    }

    private function buildWarehouseStatus($warehouseId = null) {
        $query = Warehouse::on($this->configName)
                                ->where('is_active', '=', 1);
        if ($warehouseId != null) {
            $query->where('id', '=', $warehouseId);
        }
        $warehouses = $query->get();
        $retVal = [];
        $instocks = [];
        $outstocks = [];
        $instockByKeys = [];
        $outstockByKeys = [];
        foreach ($warehouses as $warehouse) {
            $items = WarehouseCategory::on($this->configName)
                                                ->where('warehouse_id', '=', $warehouse->id)
                                                ->with(['options', 'products' => function ($q) { 
                                                    $q->where('deleted', '=', 0);
                                                }])
                                                ->get();
            foreach ($items as $item) {
                if (!isset($instocks[$item->category_id])) {
                    $instocks[$item->category_id] = [];
                    $instockByKeys[$item->category_id] = [];
                }
                if (!isset($outstocks[$item->category_id])) {
                    $outstocks[$item->category_id] = [];
                    $outstockByKeys[$item->category_id] = [];
                }
                $optionByValueIds = [];
                foreach ($item->options as $option) {
                    $optionValueIds = explode(',', $option->option_value_ids);
                    foreach ($optionValueIds as $id) {
                        $optionByValueIds[$id] = $option->option_id;
                    }
                }
                foreach ($item->products as $product) {
                    $variantIds = explode('-', $product->str_id);
                    $variants = [];
                    $keys = [];
                    foreach ($variantIds as $id) {
                        $variant = [
                            'variant_option_id' => $id,
                            'variant_id' => $optionByValueIds[$id]
                        ];
                        $keys[] = $id;
                        $variants[] = $variant;
                    }
                    sort($keys);
                    $key = implode('-', $keys);
                    if ($product->instock) {
                        if (!in_array($key, $instockByKeys[$item->category_id])) {
                            $instocks[$item->category_id][] = $variants;
                            $instockByKeys[$item->category_id][] = $key;
                        }
                        
                    } else {
                        if (!in_array($key, $outstockByKeys[$item->category_id])) {
                            $outstocks[$item->category_id][] = $variants;
                            $outstockByKeys[$item->category_id][] = $key;
                        }
                    }
                }
        
            }
        }
        $retVal['instock'] = $instocks;
        $retVal['outstock'] = $outstocks;
        $retVal['keyInstocks'] = $instockByKeys;
        $retVal['keyOutstocks'] = $outstockByKeys;
        return $retVal;
    }

    public function updateStatus(Request $request) {
        set_time_limit(-1);
        $items = WarehouseProductUpdate::on($this->configName)
                                        ->where('status', '=', 'waiting')
                                        ->get();
        if (count($items) > 0) {
            foreach ($items as $item) {
                WarehouseProductUpdate::on($this->configName)
                                        ->where('id', $item->id)
                                        ->update(['status' => 'inprocess']);
                $this->updateData($item);
                WarehouseProductUpdate::on($this->configName)
                                            ->where('id', $item->id)
                                            ->update(['status' => 'done']);
            }
        }
        $response = [
            'status' => 'successful'
        ];
        return response()->json($response);
    }

    private function updateData($item) {
        $categoryId = $item->category_id;
        $varians = json_decode($item->variant, true);
        $retVal = [];
        $products = ProductSkuValue::on($this->configName)
                                    ->leftJoin('product_n_category', 'product_n_category.product_id', '=', 'product_sku_value.product_id')
                                    ->select(\DB::raw("COUNT(sb_product_sku_value.sku_id) as count, sb_product_sku_value.sku_id, sb_product_sku_value.product_id"))
                                    ->where('product_n_category.category_id', '=', $categoryId)
                                    ->where(function($q) use ($varians) {
                                        foreach ($varians as $variant) {
                                            $q->orWhere(function($q1) use ($variant) {
                                                $q1->where('product_sku_value.variant_id', '=', $variant['variant_id'])
                                                ->where('product_sku_value.variant_option_id', '=', $variant['variant_option_id']);
                                            });
                                        } 
                                    })
                                    ->groupBy('product_sku_value.sku_id')
                                    ->havingRaw('COUNT(sb_product_sku_value.sku_id) = ' . count($varians))
                                    ->get();
        $skuIds = [];
        $productIds = [];
        foreach ($products as $product) {
            $skuIds[] = $product->sku_id;
            if (!in_array($product->product_id, $productIds)) {
                $productIds[] = $product->product_id;
            }
        }
        $updatedAt = new \DateTime();
        $updatedAtString = $updatedAt->format('Y-m-d H:i:s');
        $dataUpdate = [
            'updated_at' => $updatedAtString
        ];

        if ($item->type == 'instock') {
            $dataUpdate['status'] = 'ACTIVE';
        }
        if ($item->type == 'outstock') {
            $dataUpdate['status'] = 'PENDING';
        }
        $this->updateProduct($skuIds, 'sb_product_sku', $dataUpdate);
        $dataUpdate = [
            'updated_at' => $updatedAtString
        ];
        $this->updateProduct($productIds, 'sb_product', $dataUpdate);
    }

    private function updateProduct($items, $table, $data) {
        $tmp = [];
        $subItem = [];
        $count = count($items);
        $i = 0;
        foreach($items as $item) {
            $subItem[] = $item;
            if (count($subItem) == 500 || $i == $count - 1) {
                $tmp[] = $subItem;
                $subItem = [];
            }
            $i++;
        }
        $sql = 'UPDATE ' . $table . ' SET ';
        $params = [];
        $valueStrs = [];
        foreach ($data as $key => $value) {
            $valueStrs[] = $key . ' = ? ' ;
            $params[] = $value;
        }
        $sql .= implode(',', $valueStrs);
        foreach ($tmp as $ids) {
            $myQuery = $sql . 'WHERE id IN (' . implode(',', $ids) . ')';
            \DB::update($myQuery, $params);
        }
    }

    public function syncProductStatus(Request $request) {
        $id = $request->has('id') ? $request->input('id') : null;
        $response = [
            'status' => 'fail'
        ];
        if ($id == null) {
            return response()->json($response);
        }
        $statusByCategory = $this->buildWarehouseStatus($id);
        $result = $this->removeStatusDuplicate($statusByCategory);
        $this->saveStatusByWh($result['instocks'], 'instock');
        $this->saveStatusByWh($result['outstocks'], 'outstock');
        $response = [
            'status' => 'successful'
        ];
        return response()->json($response);
    }

    private function saveStatusByWh($items, $type) {
        foreach ($items as $key => $values) {
            foreach ($values as $value) {
                $data = [
                    'category_id' => $key,
                    'type' => $type,
                    'variant' => json_encode($value)
                ];
                $uniqId = $key;
                $variantOptions = [];
                foreach ($value as $variant) {
                    $variantOptions[] = $variant['variant_option_id'];
                }
                sort($variantOptions);
                $data['uniq_id'] = $uniqId . '-' . implode('-', $variantOptions);
                $lastUpdate = WarehouseProductUpdate::on($this->configName)
                                        ->where('uniq_id', '=', $data['uniq_id'])
                                        ->orderBy('updated_at', 'DESC')
                                        ->first();
                if ($lastUpdate) {
                    $query = WarehouseProductUpdate::on($this->configName)
                            ->where('uniq_id', '=', $data['uniq_id'])
                            ->where('status', '=', 'done');
                    if (($lastUpdate->type == 'instock' && $type == 'instock') 
                        || ($lastUpdate->type == 'outstock' && $type == 'instock')
                        || ($lastUpdate->type == 'instock' && $type == 'outstock')) {
                        $query->where('type', '=', 'instock');     
                    } elseif ($lastUpdate->type == 'outstock' && $type == 'outstock') {
                        $query->where('type', '=', 'outstock');
                    }
                    $query->update(['status' => 'waiting']);
                }
            }
        }
    }

    public function categoryVariant(Request $request) {
        // Redis::del(Redis::keys('VARIANT:OPTION:*'));
        //@todo slow query optimize
        exit;
        set_time_limit(3600);
        $categoryId = $request->has('category_id') ? $request->get('category_id') : '';
        $variantId = $request->has('variant_id') ? $request->get('variant_id') : '';
        if ($categoryId != '' && $variantId != '') {
            $keys = [
                ['category_id' => $categoryId,
                'variant_id' => $variantId]
            ];
        } else {
            $keys = $this->buildKeyCache();
        }
        foreach ($keys as $key) {
            $optionsIds = $this->getOptionFromVariant($key['variant_id']);
            if (!$optionsIds) {
                continue;
            }
            $items = $this->connection->table(DB::raw('sb_product_n_category as sb_pnc USE INDEX (cId_Sorder)') )
                ->leftJoin('product_n_variant_option as pnvo', 'pnc.product_id', '=', 'pnvo.product_id')
                ->where('pnc.category_id', '=', $key['category_id'])
                ->whereIn('pnvo.variant_option_id', $optionsIds)
                ->groupBy('pnvo.variant_option_id')
                ->pluck('pnvo.variant_option_id')
                ->toArray();
            if (!empty($items)) {
                $key = 'VARIANT:OPTION:' . $this->market . ':' . $key['category_id'] . ':' . $key['variant_id'];
                Redis::set($key, json_encode($items));
            }
        }
        $response['status'] = 'successful';
        return response()->json($response);
    }

    private function getOptionFromVariant($variantId) {
        if (array_key_exists($variantId, self::$variant)) {
            return self::$variant[$variantId];
        }
        $result = $this->connection->table('product_variant_option')
            ->where('variant_id', $variantId)
            ->pluck('id')
            ->toArray();
        self::$variant[$variantId] = $result;
        return $result;

    }
    
    private function buildKeyCache() {
        $categories = Category::where('type', '=', 'PRODUCT')
                                ->where('is_hidden', '=', 0)
                                ->pluck('id')
                                ->toArray();
        $varians = ProductVariant::pluck('id')
                                    ->toArray();
        $tmp = [];
        foreach ($categories as $category) {
            foreach ($varians as $variant) {
                $new = [
                    'category_id' => $category,
                    'variant_id' => $variant
                ];
                $tmp[] = $new;
            }
        }
        return $tmp;
    }


    public function warningOrder() {
        ini_set("memory_limit", "2048M");
        set_time_limit(3600);
        $warningFrom = config('warehouse-pod::sa.margin_warning_profit_from');
        $warningTo = config('warehouse-pod::sa.margin_warning_profit_to');
        if ($this->market == 'us') {
            $urlApi = 'https://' . config('warehouse-pod::sa.api_domain') . '/api/get-data-alert-profit?margin_profit_from=' . $warningFrom . '&margin_profit_to=' . $warningTo;
        } else {
            $urlApi = 'https://' . $this->market . '.' . config('warehouse-pod::sa.api_domain') . '/api/get-data-alert-profit?margin_profit_from=' . $warningFrom . '&margin_profit_to=' . $warningTo;
        }
        $output = $this->sendRequest($urlApi);
        if ($output && !empty($output['result'])) {
            foreach ($output['result'] as $item) {
                try {
                    $item['order_time'] = \DateTime::createFromFormat('Y-m-d H:i:s', $item['order_time']);
                    if ($item['ratio_profit'] > $warningFrom && $item['ratio_profit'] < $warningTo) {
                        $item['is_hidden'] = 1;
                    }               
                    $orderWarning = OrderWarningProfit::on($this->configName)
                                                    ->where('code', '=', $item['code'])
                                                    ->exists();
                    if ($orderWarning) {
                        OrderWarningProfit::on($this->configName)
                                                    ->where('code', '=', $item['code'])
                                                    ->update($item);
                    } else {
                        $orderWarning = OrderWarningProfit::on($this->configName)->create();
                        foreach ($item as $key => $val) {
                            $orderWarning->{$key} = $val;
                        }
                        $orderWarning->save();
                    }
                } catch (\Exception $e) {

                }   
            }
        }
        $response = ['status' => 'successful'];
        return response()->json($response);
    }

    public function warningOrderFind(Request $request) {
        $filter = $this->buildFilterWarningOrder($request);
        $result = $this->getResultWarningOrder($filter);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResultWarningOrder($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    public function warningOrderExport(Request $request) {
        $filter = $this->buildFilterWarningOrder($request);
        unset($filter['page_id']);
        unset($filter['page_size']);
        $result = $this->getResultWarningOrder($filter);
    
        $columns = ["code", "profit", "amount", "ratio_profit", "tax", "payment_gate_fee", "shipping_fee", "shipping_fee_cost", "total_fee", "order_time"];
        foreach ($result as $item) {
            $data = [];
            foreach ($columns as $column) {
                if ($item[$column]) {
                    $data[$column] = $item[$column];
                } else {
                    $data[$column] = 0;
                }
            }
            
            $exportData[] = $data;
        }
        ob_end_clean();
        $retVal = Excel::create('warehouse-warning-export-' . time(), function($excel) use ($exportData) {
            $excel->setTitle('export_warehouse')->setCreator('Megaads')->setCompany('Megaads');
            $excel->sheet('Order profit warning', function($sheet) use ($exportData) {
                $sheet->row(1, [
                    'Mã đơn', 'Profit', 'Amount', "Profit rate", "Tax", 'Payment gate fee', 'Shipping fee', 'Shipping fee cost', 'Total fee', 'Order time'
                ]);

                foreach ($exportData as $value) {
                    $sheet->appendRow($value);
                }
            });
        
        })->store('xlsx', false, true);
        $urlDownload = route('system::download', ['fileName' => $retVal['file'], 'type' => 'exports', 'time' => time()]);
        $response = [
            "status" => "successful",
            "result" => $urlDownload
        ];
        return response()->json($response);
    }

    private function getResultWarningOrder($filter) {
        $query = OrderWarningProfit::on($this->configName)
                                    ->where('is_hidden', '=', 0);
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            $query->where('code', 'LIKE', '%' . $filter['keyword'] . '%');
        }
        if (isset($filter['from']) && $filter['from'] != '') {
            $query->where('order_time', '>=', $filter['from']);
        }
        if (isset($filter['to']) && $filter['to'] != '') {
            $query->where('order_time', '<=', $filter['to']);
        }
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['orders']) && count($filter['orders']) > 0) {
                foreach ($filter['orders'] as $key => $value) {
                    $query->orderBy($key, $value);
                }
            } else {
                $query->orderBy('order_time', 'DESC');
            }
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function buildFilterWarningOrder($request) {
        $retVal = [];
        $columns = ['keyword'];
        foreach($columns as $column) {
            if ($request->has($column)){
                $retVal[$column] = $request->input($column);
            }
        }
        $from = $request->has('from') ? $request->input('from') : '';
        if ($from != '') {
            $from = \DateTime::createFromFormat('d/m/Y', $from);
            $from->setTime(0,0,0);
            $retVal['from'] = $from;
        }
        $to = $request->has('to') ? $request->input('to') : '';
        if ($to != '') {
            $to = \DateTime::createFromFormat('d/m/Y', $to);
            $to->setTime(23,59,59);
            $retVal['to'] = $to;
        }
        $sort = $request->has('sort') ? $request->input('sort') : '';
        if ($sort != '') {
            $retVal['sort'] = $sort;
            $arrOrders = explode('-', $retVal['sort']);
            $retVal['orders'] = [$arrOrders[0] => $arrOrders[1]];
        }
        $retVal['page_size'] = 50;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function warningOrderDelete(Request $request) {
        $ids = $request->has('ids') ? $request->input('ids') : [];
        $response = ['status' => 'fail'];
        if (!empty($ids)) {
            OrderWarningProfit::on($this->configName)
                                ->whereIn('id', $ids)
                                ->update(['is_hidden' => 1]);
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }
    
    public function getFreeShippingCate() {
        $countryByIp = \App\Utils::countryFromIp();
        $country = Country::where("iso", $countryByIp)->first();
        $result = [];
        if ($country) {
            $countryId = $country->id;
            $warehouseConfigs = WarehouseConfig::join("warehouse", function ($q) use ($countryId) {
                $q->on("warehouse_config.warehouse_id", "=", "warehouse.id");
                $q->where("country_id", $countryId);
                $q->where("warehouse.is_active", 1);
            })->with("warehouseConfigItems")
            ->where("warehouse_config.is_active", 1)
            ->where("config_type", "shipping")
            ->get(["warehouse_config.id"]);
            
            foreach ($warehouseConfigs as $config) {
                foreach ($config->warehouseConfigItems as $item) {
                    if ($item->default_shipping_fee == 0) {
                        $result = array_unique(array_merge($result, explode(",", $item->apply_cate)));
                    }
                }
            }
        }
        return $result;
    }

    public function buildHandlingTime(Request $request) {
        $response = [
            "status" => 'fail',
            "result" => []
        ];
        $warehouseConfigItems = WarehouseConfigItem::join("warehouse_config", "warehouse_config_item.warehouse_config_id", "=", "warehouse_config.id")
            ->get(["warehouse_config_item.id", "shipping_min_time", "shipping_max_time", "warehouse_config.type"]);
        if (count($warehouseConfigItems) > 0) {
            foreach ($warehouseConfigItems as $key => $warehouseConfigItem) {
                if ($warehouseConfigItem->shipping_min_time && $warehouseConfigItem->shipping_max_time && $warehouseConfigItem->shipping_min_time != 0 && $warehouseConfigItem->shipping_max_time != 0) {
                    $handlingMaxTime = 5;
                    if ($warehouseConfigItem->type == 'premium') {
                        $handlingMaxTime = 4;
                    } else if ($warehouseConfigItem->type == 'express') {
                        $handlingMaxTime = 2;
                    }
                    $data = [
                        "handling_min_time" => 1,
                        "handling_max_time" => $handlingMaxTime,
                        "shipping_min_time" => 1,
                        "shipping_max_time" => $warehouseConfigItem->shipping_max_time - $handlingMaxTime,
                    ];
                    WarehouseConfigItem::where("id", $warehouseConfigItem->id)->update($data);
                    $response["result"][] = $warehouseConfigItem->id;
                }
            }
        }
        return json_encode($response);
    }

    public function rebuildHandlingTime() {
        $response = [
            "status" => 'fail',
            "result" => []
        ];
        $warehouseConfigItems = WarehouseConfigItem::join("warehouse_config", "warehouse_config_item.warehouse_config_id", "=", "warehouse_config.id")
            ->get(["warehouse_config_item.id", "shipping_min_time", "shipping_max_time", "warehouse_config.type"]);
        if (count($warehouseConfigItems) > 0) {
            foreach ($warehouseConfigItems as $key => $warehouseConfigItem) {
                if ($warehouseConfigItem->shipping_min_time && $warehouseConfigItem->shipping_max_time && $warehouseConfigItem->shipping_min_time != 0 && $warehouseConfigItem->shipping_max_time != 0) {
                    $shippingMinTime = 5;
                    if ($warehouseConfigItem->type == 'premium') {
                        $shippingMinTime = 4;
                    } else if ($warehouseConfigItem->type == 'express') {
                        $shippingMinTime = 2;
                    }
                    $data = [
                        "shipping_min_time" => $shippingMinTime,
                    ];
                    WarehouseConfigItem::where("id", $warehouseConfigItem->id)->update($data);
                    $response["result"][] = $warehouseConfigItem->id;
                }
            }
        }
        return json_encode($response);
    }

    public function exportExcel(Request $request) {
        $categories = Category::on($this->configName)
                                ->where('type', '=', 'PRODUCT')
                                ->where('is_hidden', '=', 0)
                                ->orderBy('name', 'ASC')
                                ->pluck('slug', 'id')
                                ->toArray();
        $exportData = [];
        $warehouse = Warehouse::where("id", $request->input("id"))->first();
        $warehouseConfigs = WarehouseConfigItem::join("warehouse_config", function($q) {
            $q->on("warehouse_config.id", "=", "warehouse_config_item.warehouse_config_id");
        })
            ->where("warehouse_id", $request->input("id"))
            ->get(["config_type", "warehouse_config.type", "warehouse_config_item.name", "warehouse_config_item.type as item_type", "fee_limit", "fee_if_limit", "handling_min_time", "handling_max_time", "shipping_min_time", "shipping_max_time", "default_shipping_fee", "default_adding_item", "tax", "adding_item", "warehouse_config_item.created_at", "warehouse_config_item.updated_at", "apply_cate"]);
        $columns = ["config_type", "type"];
        $endColumns = ["item_type", "fee_limit", "fee_if_limit", "handling_min_time", "handling_max_time", "shipping_min_time", "shipping_max_time", "default_shipping_fee", "default_adding_item", "tax", "adding_item", "created_at", "updated_at"];
        foreach ($warehouseConfigs as $item) {
            $data = [];
            foreach ($columns as $column) {
                if ($item[$column]) {
                    $data[$column] = $item[$column];
                } else {
                    $data[$column] = 0;
                }
            }
            $data["name"] = "";
            if ($item["name"] != "" && $item["name"] != null) {
                $data["name"] = $item["name"];
            }
            $data['apply_cate'] = '';
            if (isset($item['apply_cate'])) {
                $cateIds = explode(",", $item['apply_cate']);
                $index = 0;
                foreach ($cateIds as $key => $value) {
                    $data['apply_cate'] .= isset($categories[$value]) ? $categories[$value] : $value;
                    if ($index < count($cateIds) - 1) {
                        $data['apply_cate'] .= ",";
                    }
                    $index++;
                }
            }
            foreach ($endColumns as $column) {
                if ($item[$column]) {
                    $data[$column] = $item[$column];
                } else {
                    $data[$column] = NULL;
                }
            }
            
            $exportData[] = $data;
        }
        ob_end_clean();
        $retVal = Excel::create('warehouse-export-' . date("Y-m-d_H-i-s", time()), function($excel) use ($exportData, $warehouse) {
            $excel->setTitle('export_warehouse')->setCreator('Megaads')->setCompany('Megaads');
            $excel->sheet('Cấu hình kho', function($sheet) use ($exportData) {
                $sheet->row(1, [
                    'Loại cấu hình ship', 'Loại ship', 'Tên cấu hình', "Danh mục áp dụng", "Loại cấu hình", 'Fee limit', 'Fee if limit', 'Handling min time', 'Handling max time', 'Shipping min time', 'Shipping max time', 'Phí ship mặc định', 'Phí cộng thêm', 'Thuế', 'Phí cộng thêm theo thứ tự', 'Thời gian tạo', 'Thời gian cập nhật'
                ]);

                foreach ($exportData as $value) {
                    $sheet->appendRow($value);
                }
            });
        
        })->store('xlsx', false, true);
        $urlDownload = route('system::download', ['fileName' => $retVal['file'], 'type' => 'exports', 'time' => date("Y-m-d_H-i-s", time())]);
        $response = [
            "status" => "successful",
            "result" => $urlDownload
        ];
        return response()->json($response);
    }

    public function importExcel(Request $request) {
        $categories = Category::on($this->configName)
                                ->where('type', '=', 'PRODUCT')
                                ->where('is_hidden', '=', 0)
                                ->orderBy('name', 'ASC')
                                ->pluck('id', "slug")
                                ->toArray();
        $id = $request->input("id");
        $file = $request->file('file');
        if ($file) {
            $baseName = $file->getClientOriginalName();
            $filename = date("Y-m-d_H-i-s", time()) . '_' . $baseName;
            $uploadFolder = public_path('files/import');

            if (!is_dir($uploadFolder)) {
                mkdir($uploadFolder, 0777, true);
            }

            $path = $file->move(public_path('files/import'), $filename);
        }
        $response = ["status" => "fail"];
        Excel::load($path, function ($reader) use ($categories, $id, &$response) {
            $results = $reader->get();
            $insertDataWarehouseConfig = [];
            $insertDataWarehouseConfigItem = [];
            foreach ($results as $key => $row) {
                $stringTime = date("Y-m-d H:i:s");

                $warehouseConfig = WarehouseConfig::where("warehouse_id", $id)
                    ->where("config_type", $row->loai_cau_hinh_ship)
                    ->where("type", $row->loai_ship)
                    ->first();
                if (!$warehouseConfig) {
                    $warehouseConfig = [
                        'config_type' => $row->loai_cau_hinh_ship,
                        'type' => $row->loai_ship,
                        'name' => ucfirst($row->loai_ship),
                        'is_active' => 1,
                        'warehouse_id' => $id,
                        'created_at' => $stringTime,
                        'updated_at' => $stringTime
                    ];
                    WarehouseConfig::inssert($warehouseConfig);
                    $warehouseConfig = WarehouseConfig::where("warehouse_id", $id)
                        ->where("config_type", $row->loai_cau_hinh_ship)
                        ->where("type", $row->loai_ship)
                        ->first();
                }
                $isDefault = 0;
                if ($row->loai_cau_hinh == 'default') {
                    $isDefault = 1;
                }
                $applyCate = NULL;
                if ($row->danh_muc_ap_dung && $row->danh_muc_ap_dung != "") {
                    $cates = explode(',', $row->danh_muc_ap_dung);
                    $index = 0;
                    foreach ($cates as $cate) {
                        if (isset($categories[$cate])) {
                            $applyCate .= $categories[$cate];
                            if ($index < count($cates) - 1) {
                                $applyCate .= ",";
                            }
                        }
                        $index ++;
                    }
                }
                if ($isDefault == 0) {
                    $warehouseConfigItem = [
                        'warehouse_config_id' => $warehouseConfig->id,
                        'name' => $row->ten_cau_hinh != "" ? $row->ten_cau_hinh : NULL,
                        'type' => $isDefault ? 'default' : 'custom',
                        'fee_limit' => $row->fee_limit,
                        'fee_if_limit' => $row->fee_if_limit,
                        'handling_min_time' => $row->handling_min_time,
                        'handling_max_time' => $row->handling_max_time,
                        'shipping_min_time' => $row->shipping_min_time,
                        'shipping_max_time' => $row->shipping_max_time,
                        'default_shipping_fee' => $row->phi_ship_mac_dinh,
                        'default_adding_item' => $row->phi_cong_them,
                        'tax' => $row->thue,
                        'adding_item' => $row->phi_cong_them_theo_thu_tu != 0 ? $row->phi_cong_them_theo_thu_tu : NULL,
                        'apply_cate' => $applyCate,
                        'created_at' => $stringTime,
                        'updated_at' => $stringTime
                    ];
                    $insertDataWarehouseConfigItem[] = $warehouseConfigItem;
                    WarehouseConfigItem::insert($warehouseConfigItem);
                } else {
                    $defaultWarehouseConfig = WarehouseConfigItem::where("warehouse_config_id", $warehouseConfig->id)->where("type", "default")->first();
                    if ($defaultWarehouseConfig) {
                        WarehouseConfigItem::where("id", $defaultWarehouseConfig->id)->update([
                            'name' => $row->ten_cau_hinh != "" ? $row->ten_cau_hinh : NULL,
                            'fee_limit' => $row->fee_limit,
                            'fee_if_limit' => $row->fee_if_limit,
                            'handling_min_time' => $row->handling_min_time,
                            'handling_max_time' => $row->handling_max_time,
                            'shipping_min_time' => $row->shipping_min_time,
                            'shipping_max_time' => $row->shipping_max_time,
                            'default_shipping_fee' => $row->phi_ship_mac_dinh,
                            'default_adding_item' => $row->phi_cong_them,
                            'tax' => $row->thue,
                            'adding_item' => $row->phi_cong_them_theo_thu_tu != 0 ? $row->phi_cong_them_theo_thu_tu : NULL,
                            'apply_cate' => $applyCate,
                            'updated_at' => $stringTime
                        ]);
                    }
                    $insertDataWarehouseConfigItem[] = WarehouseConfigItem::where("id", $defaultWarehouseConfig->id)->first()->toArray();
                }
                if (!array_key_exists($warehouseConfig->id, $insertDataWarehouseConfig)) {
                    $insertDataWarehouseConfig[$warehouseConfig->id] = $warehouseConfig;
                }
            }

            $response = [
                'status' => 'successful',
                'result' => [
                    "warehouseConfigItem" => $insertDataWarehouseConfigItem,
                    "warehouseConfigItem" => $insertDataWarehouseConfigItem
                ]
            ];
        });
        return response()->json($response);
    }

    public function duplicateCost(Request $request) {
        $response = ["status" => "fail"];
        $warehouseName = $request->input("warehouse_name");
        $warehouseOrigin = $request->input("warehouse_id");
        $category = $request->input("category_id");
        $warehouseCategory = WarehouseCategory::where("category_id", $category)->where("warehouse_id", $warehouseOrigin)->first();
        $warehouse = Warehouse::where("name", $warehouseName)->first();
        if ($warehouseCategory && $warehouse) {
            $warehouseCategoryNew = WarehouseCategory::where("category_id", $category)->where("warehouse_id", $warehouse->id)->first();
            if (!$warehouseCategoryNew) {
                $warehouseCategoryNew = new WarehouseCategory;
                $warehouseCategoryNew->warehouse_id = $warehouse->id;
                $warehouseCategoryNew->category_id = $category;
                $warehouseCategoryNew->created_at = new \Datetime();
                $warehouseCategoryNew->updated_at = new \Datetime();
            }
            $warehouseCategoryNew->cost = $warehouseCategory->cost;
            $warehouseCategoryNew->save();
            $categoryValue = WarehouseCategoryValue::where("warehouse_category_id", $warehouseCategory->id)->get();
            $categoryVariant = WarehouseCategoryVariant::where("warehouse_category_id", $warehouseCategory->id)->get();
            foreach ($categoryValue as $key => $value) {
                $optionIds = explode(",", $value->option_value_ids);
                $oldWarehouseCategoryValue = WarehouseCategoryValue::where("warehouse_category_id", $warehouseCategoryNew->id)
                    ->where("option_id", $value->option_id)
                    ->first();
                if ($oldWarehouseCategoryValue) {
                    $oldOptionIds = explode(",", $oldWarehouseCategoryValue->option_value_ids);
                    $optionIds = array_unique(array_merge($optionIds, $oldOptionIds));
                    $oldWarehouseCategoryValue->option_value_ids = implode(",", $optionIds);
                    $oldWarehouseCategoryValue->save();
                } else {
                    $newWarehouseCategoryValue = new WarehouseCategoryValue();
                    $newWarehouseCategoryValue->warehouse_category_id = $warehouseCategoryNew->id;
                    $newWarehouseCategoryValue->option_id = $value->option_id;
                    $newWarehouseCategoryValue->option_value_ids = implode(",", $optionIds);
                    $newWarehouseCategoryValue->not_apply_ids = $value->not_apply_ids;
                    $newWarehouseCategoryValue->created_at = new \Datetime();
                    $newWarehouseCategoryValue->updated_at = new \Datetime();
                    $newWarehouseCategoryValue->save();
                }
            }
            foreach ($categoryVariant as $key => $value) {
                $oldWarehouseCategoryVariant = WarehouseCategoryVariant::where("warehouse_category_id", $warehouseCategoryNew->id)
                    ->where("str_id", $value->str_id)
                    ->first();
                if ($oldWarehouseCategoryVariant) {
                    $oldWarehouseCategoryVariant->cost = $value->cost;
                    $oldWarehouseCategoryVariant->save();
                } else {
                    $newWarehouseCategoryVariant = new WarehouseCategoryVariant();
                    $newWarehouseCategoryVariant->warehouse_category_id = $warehouseCategoryNew->id;
                    $newWarehouseCategoryVariant->deleted = $value->deleted;
                    $newWarehouseCategoryVariant->instock = $value->instock;
                    $newWarehouseCategoryVariant->str_id = $value->str_id;
                    $newWarehouseCategoryVariant->variants = $value->variants;
                    $newWarehouseCategoryVariant->cost = $value->cost;
                    $newWarehouseCategoryVariant->created_at = new \Datetime();
                    $newWarehouseCategoryVariant->updated_at = new \Datetime();
                    $newWarehouseCategoryVariant->save();
                }
            }
            $response = [
                'status' => 'successful',
                'result' => [
                ]
            ];
        }
        return response()->json($response);
    }

    public function cloneConfig(Request $request) {
        $response = ["status" => "fail"];
        if ($request->has("id")) {
            $listIds = explode(",", $request->input("id"));
            $result = [];
            foreach ($listIds as $listId) {
                $configLocale = $request->input("to_locale", $this->configName);
                $email = $request->has('email') ? $request->get('email') : null;
                $warehouse = Warehouse::on($this->configName)->find($listId);
                $warehouseCategories = WarehouseCategory::on($this->configName)->with(['options', 'products'])
                                            ->where('warehouse_id', $listId)->get()->toArray();
                $warehouseConfigs = WarehouseConfig::on($this->configName)->with("warehouseConfigItems")
                                            ->where('warehouse_id', '=', $listId)->get()->toArray();
                $name = $warehouse->name . " (Copy) " .time();
                $data = $warehouse->toArray();
                unset($data['id']);
                $data["name"] = $name;
                $data["is_active"] = "0";
                $data["created_at"] = new \Datetime();
                $data["updated_at"] = new \Datetime();
                $log = [
                    'actor_type' => 'WAREHOUSE',
                    'actor_email' => $email,
                    'target_type' => 'WAREHOUSE_CONFIG',
                    'created_at' => new \DateTime(),
                    'event_type' => 'CLONE',
                ];
                $splitLocale = explode('_', $configLocale);
                $connection = DBConnect::connect($splitLocale[1]);
                $newWarehouse = $connection->table('warehouse')->insertGetId($data);
                foreach ($warehouseConfigs as $key => $warehouseConfig) {
                    $dataWarehouseConfigItem = [];
                    $newWarehouseConfig = $warehouseConfig;
                    unset($newWarehouseConfig["id"]);
                    unset($newWarehouseConfig["warehouse_config_items"]);
                    $newWarehouseConfig["warehouse_id"] = $newWarehouse;
                    $newWarehouseConfig["created_at"] = new \Datetime();
                    $newWarehouseConfig["updated_at"] = new \Datetime();
                    $newCreateWarehouseConfig = $connection->table('warehouse_config')->insertGetId($newWarehouseConfig);
                    foreach ($warehouseConfig['warehouse_config_items'] as $item) {
                        $newWarehouseConfigItem = $item;
                        unset($newWarehouseConfigItem["id"]);
                        unset($newWarehouseConfigItem["map_id"]);
                        $newWarehouseConfigItem["warehouse_config_id"] = $newCreateWarehouseConfig;
                        $newWarehouseConfigItem["created_at"] = new \Datetime();
                        $newWarehouseConfigItem["updated_at"] = new \Datetime();
                        $dataWarehouseConfigItem[] = $newWarehouseConfigItem;
                    }
                    $connection->table('warehouse_config_item')->insert($dataWarehouseConfigItem);
                }
                foreach ($warehouseCategories as $key => $warehouseCategory) {
                    $dataWarehouseValue = [];
                    $dataWarehouseVariant = [];
                    $newWarehouseCategory = $warehouseCategory;
                    $category = Category::where("id", $warehouseCategory['category_id'])->first();
                    $categoryId = $warehouseCategory['category_id'];
                    if ($category) {
                        $categoryInMarket = $connection->table('category')->where("name", $category->name)->first();
                        if ($categoryInMarket) {
                            $categoryId = $categoryInMarket->id;
                        }
                    }
                    unset($newWarehouseCategory["id"]);
                    unset($newWarehouseCategory["warehouse_id"]);
                    unset($newWarehouseCategory["options"]);
                    unset($newWarehouseCategory["products"]);
                    $newWarehouseCategory["warehouse_id"] = $newWarehouse;
                    $newWarehouseCategory["category_id"] = $categoryId;
                    $newWarehouseCategory["created_at"] = new \Datetime();
                    $newWarehouseCategory["updated_at"] = new \Datetime();
                    $newCreateWarehouseCategory = $connection->table('warehouse_category')->insertGetId($newWarehouseCategory);
                    $mapOptionValue = [];
                    foreach ($warehouseCategory['options'] as $item) {
                        $newWarehouseConfigValue = $item;
                        unset($newWarehouseConfigValue["id"]);
                        unset($newWarehouseConfigValue["warehouse_category_id"]);
                        $option = ProductVariant::where("id", $item['option_id'])->first();
                        $optionId = $item['option_id'];
                        if ($option) {
                            $optionInMarket = $connection->table('product_variant')->where("slug", $option->slug)->first();
                            if ($optionInMarket) {
                                $optionId = $optionInMarket->id;
                            }
                        }
                        $optionValueIds = $item['option_value_ids'];
                        $listOptionValueIds = explode(",", $optionValueIds);
                        $marketOptionValueIds = [];
                        foreach ($listOptionValueIds as $optionValueId) {
                            $mapOptionValue[$optionValueId] = $optionValueId;
                            $optionValue = ProductVariantOption::where("id", $optionValueId)->first();
                            if ($optionValue) {
                                $optionValueInMarket = $connection->table('product_variant_option')
                                    ->where("variant_id", $optionInMarket ? $optionInMarket->id : null)
                                    ->where("name", $optionValue->name)
                                    ->first();
                                if ($optionValueInMarket) {
                                    $marketOptionValueIds[] = $optionValueInMarket->id;
                                    $mapOptionValue[$optionValueId] = $optionValueInMarket->id;
                                }
                            }
                        }
                        if (count($marketOptionValueIds) > 0) {
                            $optionValueIds = implode(",", $marketOptionValueIds);
                        }
                        $newWarehouseConfigValue["option_id"] = $optionId;
                        $newWarehouseConfigValue["option_value_ids"] = $optionValueIds;
                        $newWarehouseConfigValue["warehouse_category_id"] = $newCreateWarehouseCategory;
                        $newWarehouseConfigValue["created_at"] = new \Datetime();
                        $newWarehouseConfigValue["updated_at"] = new \Datetime();
                        $dataWarehouseValue[] = $newWarehouseConfigValue;
                    }
                    foreach ($warehouseCategory['products'] as $item) {
                        $newWarehouseVariant = $item;
                        unset($newWarehouseVariant["id"]);
                        unset($newWarehouseVariant["warehouse_category_id"]);
                        $strIds = $item['str_id'];
                        $listStrIds = explode("-", $strIds);
                        $marketStrIds = [];
                        $variants = $item['variants'];
                        foreach ($listStrIds as $strId) {
                            if (isset($mapOptionValue[$strId])) {
                                $marketStrIds[] = $mapOptionValue[$strId];
                                $variants = str_replace('"id":"' . $strId . '"', '"id":"' . $mapOptionValue[$strId] . '"', $variants);
                            }
                        }
                        $newWarehouseVariant["variants"] = $variants;
                        $newWarehouseVariant["warehouse_category_id"] = $newCreateWarehouseCategory;
                        $newWarehouseVariant["created_at"] = new \Datetime();
                        $newWarehouseVariant["updated_at"] = new \Datetime();
                        $dataWarehouseVariant[] = $newWarehouseVariant;
                    }
                    $connection->table('warehouse_category_value')->insert($dataWarehouseValue);
                    $connection->table('warehouse_category_variant')->insert($dataWarehouseVariant);
                }
                $newWarehouseData = $connection->table('warehouse')->where("id", $newWarehouse)->get();
                $log['target_id'] = $newWarehouse;
                $log['data'] = json_encode($newWarehouseData);
                $connection->table('log')->insert($log);
                $result[] = $newWarehouse;
            }
            $response = [
                'status' => 'successful',
                'result' => $result
            ];
        }
        return response()->json($response);
    }

    public function cloneShippingCost(Request $request) {
        $response = ["status" => "fail"];
        if ($request->has("from_id") && $request->has("to_id")) {
            $fromId = $request->input('from_id');
            $listIds = explode(",", $request->input("to_id"));
            $warehouseConfigs = WarehouseConfig::on($this->configName)->with("warehouseConfigItems")
                                        ->where("config_type", "cost")
                                        ->where('warehouse_id', '=', $fromId)->get()->toArray();
            foreach ($listIds as $listId) {
                foreach ($warehouseConfigs as $key => $warehouseConfig) {
                    $cloneWarehouseConfig = WarehouseConfig::on($this->configName)
                        ->where("config_type", "cost")
                        ->where("type", $warehouseConfig["type"])
                        ->where('warehouse_id', '=', $listId)->first();

                    if ($cloneWarehouseConfig) {
                        WarehouseConfigItem::on($this->configName)
                                     ->where('warehouse_config_id', $cloneWarehouseConfig->id)
                                     ->delete();
                        $dataWarehouseConfigItem = [];

                        foreach ($warehouseConfig['warehouse_config_items'] as $item) {
                            $newWarehouseConfigItem = $item;
                            unset($newWarehouseConfigItem["id"]);
                            unset($newWarehouseConfigItem["map_id"]);
                            $newWarehouseConfigItem["warehouse_config_id"] = $cloneWarehouseConfig->id;
                            $newWarehouseConfigItem["created_at"] = new \Datetime();
                            $newWarehouseConfigItem["updated_at"] = new \Datetime();
                            $dataWarehouseConfigItem[] = $newWarehouseConfigItem;
                        }
                        WarehouseConfigItem::on($this->configName)->insert($dataWarehouseConfigItem);
                    }
                }
            }
            
            $result = [];
            $response = [
                'status' => 'successful',
                'result' => $result
            ];
        }
        return response()->json($response);
    }

    public function checkBulk(Request $request) {
        $response = ["status" => "fail"];
        if ($request->has("ids")) {
            $ids = explode('-', $request->input("ids"));
            $result = WarehouseConfigItemVariant::query()
                                    ->whereIn('warehouse_config_item_id', $ids)
                                    ->where('is_bulk', '=', 1)
                                    ->select('warehouse_config_item_id')
                                    ->pluck('warehouse_config_item_id')
                                    ->toArray();
            $response = [
                'status' => 'successful',
                'result' => $result
            ];
        }
        return response()->json($response);
    }

    public function getConfigItemVariant(Request $request) {
        $response = ['status' => 'fail'];
        $warehouseConfigItemId = $request->input('warehouse_config_item_id');

        if ($warehouseConfigItemId) {
            $items = WarehouseConfigItem::on($this->configName)->with(["options", "products"])
                                                        ->where('id', '=', $warehouseConfigItemId)
                                                        ->get();

            $this->decorConfigItemVariant($items);
            $response['status'] = 'successful';
            $response['result'] = $items;
        } else {
            $type = $request->input('type', 'shipping');
            $warehouseConfigIds = WarehouseConfig::on($this->configName)
                                                ->where('warehouse_id', '=', $request->input('warehouse_id'))
                                                ->where('config_type', '=', $type)
                                                ->pluck('id')->toArray();
            $items = WarehouseConfigItem::on($this->configName)->with(["options", "products"])
                                                        ->whereIn('warehouse_config_id', $warehouseConfigIds)
                                                        ->join("warehouse_config", "warehouse_config_item.warehouse_config_id", "=", "warehouse_config.id")
                                                        ->get(["warehouse_config_item.*", "warehouse_config.type"]);

            $this->decorConfigItemVariant($items);
            $response['status'] = 'successful';
            $response['result'] = $items;
        }
        return response()->json($response);
    }

    private function decorConfigItemVariant(&$items) {
        foreach ($items as &$item) {
            if (isset($item['options'])) {
                foreach ($item['options'] as $key => $option) {
                    $varians = ProductVariantOption::on($this->configName)
                                                                    ->where('variant_id', '=', $option['option_id'])
                                                                    ->pluck('name', 'id')->toArray();
                    $item['options'][$key]['variants'] = $varians;
                }
            } else {
                $item['options'] = [];
            }
            if (isset($item['products'])) {
                foreach ($item['products'] as $key => $product) {
                    $item['products'][$key]['variants'] = json_decode($product['variants']);
                }
            } else {
                $item['products'] = [];
            }
        }
    }

    public function storeConfigItemVariant(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        if ($request->has('warehouse_id') && $request->has('warehouse_config_item_id')) {
            $warehouseData = Warehouse::on($this->configName)->find($request->input("warehouse_id"));
            if ($warehouseData->is_editable == 0) {
                $retval = [
                    'status' => 'fail', 
                    'message' => 'Bạn không có quyền lưu kho. Vui lòng liên hệ bộ phận kỹ thuật!', 
                ];
                return $retval;
            }
            $warehouse = Warehouse::on($this->configName)
                    ->where('id', '=', $request->input('warehouse_id'))
                    ->first();
            if ($warehouse) {
                if ($request->has('data')) {
                    $warehouseConfigItem = WarehouseConfigItem::on($this->configName)
                        ->where('id', '=', $request->input('warehouse_config_item_id'))
                        ->first();
                    $data = $request->input("data");
                    if (isset($data['options'])) {
                        $this->saveWarehouseConfigItemValue($data['options'], $warehouseConfigItem);
                    }
                    if (isset($data['products'])) {
                        $this->saveWarehouseConfigItemVariant($data['products'], $warehouseConfigItem);
                    }
                }
            }
            $response = [
                'status' => 'successful'
            ];
            return response()->json($response);
        }
    }

    private function deleteWarehouseConfigItemVariant($warehouseConfigItem, $products) {
        $requestIds = [];
        foreach ($products as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
        }
        $idExists = WarehouseConfigItemVariant::on($this->configName)
                                        ->where('warehouse_config_item_id', '=', $warehouseConfigItem->id)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            $deletedIds = array_values($deletedIds);
            if (!empty($deletedIds)) {
                WarehouseConfigItemVariant::on($this->configName)
                                ->whereIn('id', $deletedIds)
                                ->delete();
            }
        }

    }

    private function saveWarehouseConfigItemVariant($products, $warehouseConfigItem) {
        $this->deleteWarehouseConfigItemVariant($warehouseConfigItem, $products);
        foreach ($products as $product) {
            $data = $this->buildWarehouseConfigItemVariant($warehouseConfigItem, $product);
            if (isset($data['id'])) {
                $first = WarehouseConfigItemVariant::on($this->configName)
                                                    ->where('id', '=', $data['id'])
                                                    ->first();
                if (!$first) {
                    unset($data['id']);
                    $item = WarehouseConfigItemVariant::on($this->configName)
                                                ->create($data);
                    $item->save();
                } else {
                    WarehouseConfigItemVariant::on($this->configName)
                                                ->where('id', '=', $first->id)
                                                ->update($data);
                }
            } else {
                $item = WarehouseConfigItemVariant::on($this->configName)
                                                ->create($data);
                $item->save();
            }
        }
    }

    private function buildWarehouseConfigItemVariant($warehouseConfigItem, $product) {
        $retVal = [
            'warehouse_config_item_id' => $warehouseConfigItem->id,
            'str_id' => $product['str_id'],
            'variants' => json_encode($product['variants']),
            'default_shipping_fee' => isset($product['default_shipping_fee']) ? $product['default_shipping_fee'] : null,
            'default_adding_item' => isset($product['default_adding_item']) ? $product['default_adding_item'] : 0,
            'origin_shipping_fee' => isset($product['origin_shipping_fee']) ? $product['origin_shipping_fee'] : null,
            'tax' => isset($product['tax']) ? $product['tax'] : 0,
            'origin_adding_fee' => isset($product['origin_adding_fee']) ? $product['origin_adding_fee'] : 0,
            'ratio_id' => isset($product['ratio_id']) ? $product['ratio_id'] : null,

            'is_bulk' => isset($product['is_bulk']) ? $product['is_bulk'] : 0,
            'origin_shipping_fee_bulk' => isset($product['origin_shipping_fee_bulk']) ? $product['origin_shipping_fee_bulk'] : null,
            'origin_adding_fee_bulk' => isset($product['origin_adding_fee_bulk']) ? $product['origin_adding_fee_bulk'] : null,
            'default_shipping_fee_bulk' => isset($product['default_shipping_fee_bulk']) ? $product['default_shipping_fee_bulk'] : null,
            'tax_bulk' => isset($product['tax_bulk']) ? $product['tax_bulk'] : null,
            'default_adding_item_bulk' => isset($product['default_adding_item_bulk']) ? $product['default_adding_item_bulk'] : null,
        ];
        if (isset($product['id'])) {
            $retVal['id'] = $product['id'];
        }
        return $retVal;
    }

    private function deleteWarehouseConfigItemValue($warehouseConfigItem, $options) {
        $requestIds = [];
        foreach ($options as $item) {
            if (isset($item['id'])) {
                $requestIds[] = $item['id'];
            }
        }
        $idExists = WarehouseConfigItemValue::on($this->configName)
                                        ->where('warehouse_config_item_id', '=', $warehouseConfigItem->id)
                                        ->pluck('id')->toArray();
        if (!empty($idExists)) {
            $deletedIds = array_diff($idExists, $requestIds);
            $deletedIds = array_values($deletedIds);
            if (!empty($deletedIds)) {
                WarehouseConfigItemValue::on($this->configName)
                                ->whereIn('id', $deletedIds)
                                ->delete();
            }
        }

    }

    private function saveWarehouseConfigItemValue($options, $warehouseConfigItem) {
        $this->deleteWarehouseConfigItemValue($warehouseConfigItem, $options);
        foreach ($options as $option) {
            if (isset($option['option_id'])) {
                $data = $this->buildWarehouseConfigItemValue($warehouseConfigItem, $option);
                if (isset($data['id'])) {
                    $first = WarehouseConfigItemValue::on($this->configName)
                                                    ->where('id', '=', $data['id'])
                                                    ->first();
                    if (!$first) {
                        unset($data['id']);
                        $item = WarehouseConfigItemValue::on($this->configName)
                                                ->create($data);
                        $item->save();
                    } else {
                        WarehouseConfigItemValue::on($this->configName)
                                                ->where('id', '=', $first->id)
                                                ->update($data);
                    }
                } else {
                    $item = WarehouseConfigItemValue::on($this->configName)
                                                ->create($data);
                    $item->save();
                }
            }
        }
    }

    private function buildWarehouseConfigItemValue($warehouseConfigItem, $option) {
        if (empty($option['variant_ids'])) {
            $option['variant_ids'] = [0];
        }
        $data = [
            'warehouse_config_item_id' => $warehouseConfigItem->id,
            'option_id' => $option['option_id'], 
            'option_value_ids' => implode(',', $option['variant_ids'])
        ];
        if (isset($option['id']) && !empty($option['id'])) {
            $data['id'] = $option['id'];
        }
        return $data;
    }
}
