<?php

namespace Modules\WarehousePod\Controllers\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Models\Warehouse;
use Modules\WarehousePod\Controllers\Controller;
use Modules\WarehousePod\Models\Log;
use Modules\WarehousePod\Models\WarehouseCurrencyRatio;
use Modules\WarehousePod\Models\Country;
use Modules\Ticket\Helpers\DBconnect;
use Modules\WarehousePod\Models\WarehouseCategory;
use Modules\WarehousePod\Models\WarehouseCategoryVariant;
use Modules\WarehousePod\Models\WarehouseConfigItem;
use Modules\WarehousePod\Models\WarehouseConfigItemVariant;

class WarehouseCurrencyRatioService extends Controller
{
    protected $connection = null;
    protected $warehouseCurrencyRatios = [];
    public function __construct()
    {
        parent::__construct();
        if ($this->connection == null) {
            $this->connection = DBconnect::connect($this->market);
        }
    }

    public function find(Request $request)
    {
        $filter = $this->buildFilter($request);
        $result = $this->getResult($filter);
        $this->decorResult($result);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function decorResult(&$items)
    {
    }

    private function getResult($filter)
    {
        $query = WarehouseCurrencyRatio::on($this->configName);
        if (isset($filter['id']) && $filter['id'] != '') {
            $query->where('id', '=', $filter['id']);
        }
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            $query->where('from_currency', 'LIKE', '%' . $filter['keyword'] . '%');
            $query->orWhere('to_currency', 'LIKE', '%' . $filter['keyword'] . '%');
        }

        $query->orderBy('created_at', 'DESC');
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function buildFilter($request)
    {
        $retVal = [];
        $columns = ['id', 'keyword'];
        foreach ($columns as $column) {
            if ($request->has($column)) {
                $retVal[$column] = $request->input($column);
            }
        }
        $retVal['page_size'] = 50;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function store(Request $request)
    {
        $data = $this->buildDataWarehouseCurrencyRatio($request);
        $email = $request->has('email') ? $request->get('email') : null;
        $log = [
            'actor_type' => 'staff',
            'actor_email' => $email,
            'target_type' => 'CURRENCY_RATIO',
            'created_at' => new \DateTime()
        ];
        $oldData = null;
        if ($request->has("id")) {
            $data["id"] = $request->get("id");
        }
        if (isset($data['id']) && !empty($data['id'])) {
            $oldData = WarehouseCurrencyRatio::on($this->configName)->find($data['id']);
            WarehouseCurrencyRatio::on($this->configName)
                ->where('id', '=', $data['id'])
                ->update($data);
            $warehouseCurrencyRatio = WarehouseCurrencyRatio::on($this->configName)->find($data['id']);
            $this->updateRateWarehouse($warehouseCurrencyRatio);
            $log['event_type'] = 'UPDATE';
        } else {
            $isExists = WarehouseCurrencyRatio::on($this->configName)
                ->where('from_currency', '=', $data['from_currency'])
                ->where('to_currency', '=', $data['to_currency'])
                ->exists();
            if ($isExists) {
                $response = [
                    'status' => 'fail',
                    'message' => 'Currency is already exists'
                ];
                return response()->json($response);
            }
            $warehouseCurrencyRatio = WarehouseCurrencyRatio::on($this->configName)->create($data);
            $log['event_type'] = 'CREATE';
        }
        $log['target_id'] = $warehouseCurrencyRatio->id;
        $log['data'] = json_encode($request->all());
        Log::on($this->configName)->create($log);
        $response = [
            'status' => 'successful',
            'result' => $warehouseCurrencyRatio
        ];
        return response()->json($response);
    }

    private function buildDataWarehouseCurrencyRatio($request)
    {
        $columns = ['from_currency', 'to_currency', 'ratio'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = ($request->has($column)) ? $request->get($column) : NULL;
        }
        return $retVal;
    }

    public function delete(Request $request)
    {
        $id = $request->has('id') ? $request->input('id') : [];
        $email = $request->has('email') ? $request->get('email') : null;
        $response = ['status' => 'fail'];
        if (!empty($id)) {
            $log = [
                'actor_type' => 'staff',
                'actor_email' => $email,
                'target_type' => 'warehouseCurrencyRatio',
                'event_type' => 'DELETE',
                'created_at' => new \DateTime()
            ];
            WarehouseCurrencyRatio::on($this->configName)
                ->where('id', $id)
                ->delete();
            $log['target_id'] = $id;
            $log['data'] = json_encode($request->all());
            Log::on($this->configName)->create($log);
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    private function updateRateWarehouse($warehouseCurrencyRatio) {
        //Warehouse Category
        $warehouseCategories = WarehouseCategory::on($this->configName)
            ->with(["products"])
            ->where("ratio_id", $warehouseCurrencyRatio->id)
            ->get(["id", "origin_cost"]);
        foreach ($warehouseCategories as $key => $warehouseCategory) {
            WarehouseCategory::on($this->configName)
                ->where("id", $warehouseCategory->id)
                ->update([
                    "cost" => $warehouseCategory->origin_cost * $warehouseCurrencyRatio->ratio
                ]);
            foreach ($warehouseCategory->products as $key => $product) {
                if ($product->ratio_id == null || $product->ratio_id == $warehouseCurrencyRatio->id) {
                    WarehouseCategoryVariant::on($this->configName)
                        ->where("id", $product->id)
                        ->update([
                            "cost" => $product->origin_cost * $warehouseCurrencyRatio->ratio
                        ]);
                }
            }
        }
        $warehouseCategoryVariants = WarehouseCategoryVariant::on($this->configName)
            ->where("ratio_id", $warehouseCurrencyRatio->id)
            ->get(["id", "origin_cost"]);
        foreach ($warehouseCategoryVariants as $key => $warehouseCategoryVariant) {
            WarehouseCategoryVariant::on($this->configName)
                ->where("id", $warehouseCategoryVariant->id)
                ->update([
                    "cost" => $warehouseCategoryVariant->origin_cost * $warehouseCurrencyRatio->ratio
                ]);
        }

        //Config Item
        $warehouseConfigItems = WarehouseConfigItem::on($this->configName)
            ->with(["products"])
            ->where("ratio_id", $warehouseCurrencyRatio->id)
            ->get(["id", "origin_shipping_fee", "origin_adding_fee"]);
        foreach ($warehouseConfigItems as $key => $warehouseConfigItem) {
            WarehouseConfigItem::on($this->configName)
                ->where("id", $warehouseConfigItem->id)
                ->update([
                    "default_shipping_fee" => $warehouseConfigItem->origin_shipping_fee * $warehouseCurrencyRatio->ratio,
                    "default_adding_item" => $warehouseConfigItem->origin_adding_fee * $warehouseCurrencyRatio->ratio,
                ]);
            foreach ($warehouseConfigItem->products as $key => $product) {
                if ($product->ratio_id == null || $product->ratio_id == $warehouseCurrencyRatio->id) {
                    WarehouseConfigItemVariant::on($this->configName)
                        ->where("id", $product->id)
                        ->update([
                            "default_shipping_fee" => $product->origin_shipping_fee * $warehouseCurrencyRatio->ratio,
                            "default_adding_item" => $product->origin_adding_fee * $warehouseCurrencyRatio->ratio,
                        ]);
                }
            }
        }

        $warehouseConfigItemVariants = WarehouseConfigItemVariant::on($this->configName)
            ->where("ratio_id", $warehouseCurrencyRatio->id)
            ->get(["id", "origin_shipping_fee", "origin_adding_fee"]);
        foreach ($warehouseConfigItemVariants as $key => $warehouseConfigItemVariant) {
            WarehouseConfigItemVariant::on($this->configName)
                ->where("id", $warehouseConfigItemVariant->id)
                ->update([
                    "default_shipping_fee" => $warehouseConfigItemVariant->origin_shipping_fee * $warehouseCurrencyRatio->ratio,
                    "default_adding_item" => $warehouseConfigItemVariant->origin_adding_fee * $warehouseCurrencyRatio->ratio,
                ]);
        }
    }
}
