<?php

namespace Modules\WarehousePod\Controllers\Services;

use App\Utils\SkuHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\WarehousePod\Controllers\Controller;
use Modules\WarehousePod\Models\Warehouse;
use Modules\WarehousePod\Models\WarehouseConfig;
use Modules\WarehousePod\Models\WarehouseConfigItem;
use Modules\WarehousePod\Models\WarehouseNProduct;
use Modules\WarehousePod\Models\Product;
use Modules\WarehousePod\Models\ProductInfo;
use Modules\Ticket\Helpers\DBconnect;
use Modules\WarehousePod\Models\Country;

use Module;

class WarehouseProductService extends Controller
{
    protected $connection = null;
    protected $printings = [];
    public function __construct()
    {        
        parent::__construct();
        if ($this->connection == null) {
            $this->connection = DBconnect::connect($this->market);
        }
        if (!empty($printings)) {
            $this->printings = Printing::on($this->configName)
                                        ->pluck('name', 'code')
                                        ->toArray();
        }
    }

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $result = $this->getResult($filter);
        $this->decorResult($result);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function decorResult(&$items) {
        $warehouseIds = [];
        $productIds = [];
        $productSkuIds = [];
        foreach($items as $item) {
            if (!in_array($item->warehouse_id, $warehouseIds)) {
                $warehouseIds[] = $item->warehouse_id;
            }
            if (!in_array($item->product_id, $productIds)) {
                $productIds[] = $item->product_id;
            }
            if (!in_array($item->product_sku_id, $productSkuIds)) {
                $productSkuIds[] = $item->product_sku_id;
            }
        }
        $warehouses = Warehouse::on($this->configName)
                                ->whereIn('id', $warehouseIds)
                                ->get(['name', 'id', 'printing_code']);
        $warehouseByIds = [];
        foreach ($warehouses as $item) {
            $warehouseByIds[$item->id] = $item;
        }
        //@todo product-info done
//        $products = ProductInfo::on($this->configName)
//                                ->whereIn('product_id', $productIds)
//                                ->whereIn('product_sku_id', $productSkuIds)
//                                ->get(['product_id', 'product_sku_id', 'sku', 'image_url', 'name', 'slug']);
        $products = DB::table('product_sku')
            ->whereIn('id',  $productSkuIds)
            ->whereIn('product_id', $productIds)
            ->get(['product_id', 'id as product_sku_id', 'sku', 'image_url']);
        SkuHelper::decorProducts($products, 'product_sku_id', 'name', [
            'sku_slug_decor_col' => 'slug',
        ]);
        $productByKeys = [];
        foreach ($products as $product) {
            $key = $product->product_id . '-' . $product->product_sku_id;
            $productByKeys[$key] = $product;
        }
        foreach ($items as $item) {
            if (isset($warehouseByIds[$item->warehouse_id])) {
                $item->warehouse_name = $warehouseByIds[$item->warehouse_id]->name;
                $item->printing_code = $warehouseByIds[$item->warehouse_id]->printing_code;
            }
            $key = $item->product_id . '-' . $item->product_sku_id;
            if (isset($productByKeys[$key])) {
                $item->name = $productByKeys[$key]->name;
                $item->image_url = $productByKeys[$key]->image_url;
                $item->slug = $productByKeys[$key]->slug;
                $item->sku = $productByKeys[$key]->sku;
            }
        }
    }

    private function getResult($filter) {
        $query = WarehouseNProduct::on($this->configName);
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            //@todo delete
            $query->leftJoin('product_info', 'product_info.product_sku_id', '=', 'warehouse_n_product.product_sku_id');
            $query->where('product_info.search', 'LIKE', '%' . $filter['keyword'] . '%');
        }
        if (isset($filter['warehouse_id']) && $filter['warehouse_id'] != '') {
            $query->where('warehouse_n_product.warehouse_id', '=', $filter['warehouse_id']);
        }

        if (isset($filter['printing_code']) && $filter['printing_code'] != '') {
            $warehouseIds = Warehouse::on($this->configName)
                                        ->where('printing_code', '=', $filter['printing_code'])
                                        ->pluck('id')
                                        ->toArray();
            if (empty($warehouseIds)) {
                $warehouseIds = [-1];
            }
            $query->whereIn('warehouse_n_product.warehouse_id', $warehouseIds);
        }
        $query->orderBy('warehouse_n_product.created_at', 'DESC');
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function buildFilter($request) {
        $retVal = [];
        $columns = ['warehouse_id', 'printing_code', 'keyword'];
        foreach($columns as $column) {
            if ($request->has($column)){
                $retVal[$column] = $request->input($column);
            }
        }
        $retVal['page_size'] = 50;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function store(Request $request)
    {
       $input = $request->all();
       if (array_key_exists('warehouse_id', $input) && array_key_exists('apply_product', $input)) {
            foreach ($input['apply_product'] as $item) {
                $dataSave = [
                    'warehouse_id' => $input['warehouse_id'], 
                    'product_id' => $item['product_id'],
                    'product_sku_id' => 0
                ];
                if (isset($item['product_sku_id']) && $item['product_sku_id'] != '') {
                    $dataSave['product_sku_id'] = $item['product_sku_id'];
                }
                $checkExists = WarehouseNProduct::on($this->configName)
                                                ->where('warehouse_id', '=', $dataSave['warehouse_id'])
                                                ->where('product_id', '=', $dataSave['product_id'])
                                                ->where('product_sku_id', '=', $dataSave['product_sku_id'])
                                                ->exists();
                if (!$checkExists) {
                    WarehouseNProduct::on($this->configName)
                                    ->create($dataSave);
                }
            }
       }
        $response = [
            'status' => 'successful'
        ];
        return response()->json($response);
    }


    public function product(Request $request){
        $keyword = $request->has('keyword') ? $request->input('keyword') : null;
        $response = [
            'status' => 'fail'
        ];
        if ($keyword != null) {
            //@todo product-info
            $items = ProductInfo::where('name', 'LIKE', '%' . $keyword . '%')
                        ->limit(10)
                        ->get(['id', 'product_id', 'product_sku_id', 'name', 'image_url']);
            $response['status'] = 'successful';
            $response['result'] = $items;
        }
        return response()->json($response);
    }

    public function delete(Request $request) {
        $ids = $request->has('ids') ? $request->input('ids') : [];
        $response = ['status' => 'fail'];
        if (!empty($ids)) {
            WarehouseNProduct::on($this->configName)
                                ->whereIn('id', $ids)
                                ->delete();
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function rebuild(Request $request) {
        ini_set("memory_limit", "-1");
        set_time_limit(0);
        $maps = [
            'tools.usps.com' => 'United States',
            'track.easypost.com' => [
                '16' => 'Canada',
                '22' => 'United States',
                '10' => 'United States'
            ],
            'auspost.com.au' => 'Australia',
            'wwwapps.ups.com' => 'United States',
            'a1.asendiausa.com' => 'United States',
            'webtrack.dhlglobalmail.com' => 'United States',
            'yuntrack.com' => 'China',
            'webtrack.dhlglobalmail.com' => 'United States',
            'www.ups.com' => 'United States',
            't.17track.net' => 'China',
            'www.dhl.com/global-en' => 'United States',
            'www.dhl.com/en' => 'Germany',
            'www.dhl.com/vn-en' => 'Germany',
            'dhl.de' => 'Germany',
            'www.fedex.com' => 'United States',
            'purolator.com' => 'Canada',
            'webtrack.dhlglobalmail.com' => 'United States',
            'mydhl.express.dhl/nz' => 'New Zealand',
            'canadapost-postescanada.ca' => 'Canada',
            'www.canadapost.ca' => 'Canada',
            'www.royalmail.com' => 'United Kingdom',
            'mailingtechnology.com' => 'Czech Republic',
            'laposte.fr' => 'France',
            'colisprive.com' => 'France'
        ];
        foreach ($maps as $key => $value) {
            $connection = DBconnect::connect($this->market);
            $count = $this->getOrderMeta($key, $connection, 0, true);
            if ($count > 0) {
                $limitSize = ceil($count / 20);
                for ($pageId = 0; $pageId < $limitSize; $pageId++) {
                    $result = $this->getOrderMeta($key, $connection, $pageId, false);
                    $this->buildWarehouseProduct($connection, $result, $key, $maps);
                }
            }
        }

        $response = [
            'status' => 'successful'
        ];
        return response()->json($response);
    }

    private function buildWarehouseProduct($connection, $items, $key, $maps) {
        foreach ($items as $item) {
            $shipments = json_decode($item->value);
            if (count($shipments) == 1) {
                $orders = $connection->table('order as o')
                                    ->leftJoin('order_item as o_i', 'o_i.order_id', '=', 'o.id')
                                    ->where('o.id', '=', $item->order_id)
                                    ->get(['o_i.product_id', 'o_i.product_sku_id', 'o_i.printing_code', 'o.country_id']);
                if (count($orders) > 0) {
                    $location = $maps[$key];
                    if (is_array($maps[$key])) {
                        if (strlen($shipments[0]->code) == 16) {
                            $location = 'Canada';
                        } else {
                            $location = 'United States';
                        }
                    }
                    foreach ($orders as $order) {
                        if ($order->printing_code != '') {
                            $warehouse = $connection->table('warehouse')
                                                    ->where('printing_code', '=', $order->printing_code)
                                                    ->where('country_id', '=', $order->country_id)
                                                    ->where('location', '=', $location)
                                                    ->first();
                            if (!$warehouse) {
                                $name = ucfirst($order->printing_code);
                                if (isset($this->printings[$order->printing_code])) {
                                    $name = $this->printings[$order->printing_code];
                                }
                                $name .= ' - ' . $location;
                                $saveData = [
                                    'printing_code' => $order->printing_code,
                                    'country_id' => $order->country_id,
                                    'location' => $location,
                                    'is_default' => 0,
                                    'is_active' => 1,
                                    'name' => $name
                                ];
                                $warehouse = Warehouse::on($this->configName)
                                                        ->create($saveData);
                            }
                            $productSkuId = 0;
                            if ($order->product_sku_id != null && $order->product_sku_id != '') {
                                $productSkuId = $order->product_sku_id;
                            }
                            $warehouseNProduct = [
                                'warehouse_id' => $warehouse->id,
                                'product_id' => $order->product_id,
                                'product_sku_id' => $productSkuId
                            ];
                            $checkExists = WarehouseNProduct::on($this->configName)
                                                                ->where('warehouse_id', '=', $warehouse->id)
                                                                ->where('product_id', '=', $order->product_id)
                                                                ->where('product_sku_id', '=', $productSkuId)
                                                                ->exists();
                            if (!$checkExists) {
                                WarehouseNProduct::on($this->configName)
                                                    ->create($warehouseNProduct);
                            }
                        }
                        
                    }
                }
            }
        }
    }


    private function getOrderMeta($keyword, $connection, $pageId = 0, $count = false) {
        $query = $connection->table('order_meta');
        $query->where('key', '=', 'shipment_data');
        $query->where('value', 'LIKE', '%' . $keyword . '%');
        if ($count) {
            return $query->count();
        } else {
            $query->forPage($pageId, 20);
            return $query->get();
        }
    }

}
