<?php

namespace Modules\WarehousePod\Controllers\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\WarehousePod\Controllers\Controller;
use Modules\WarehousePod\Models\Log;
use Modules\WarehousePod\Models\ProductSkuValue;
use Modules\WarehousePod\Models\ProductVariant;
use Modules\WarehousePod\Models\Country;
use Modules\WarehousePod\Models\WarehouseProvider;
use Modules\WarehousePod\Models\WarehouseProviderInfo;
use Modules\WarehousePod\Models\WarehouseProviderVariant;
use Modules\WarehousePod\Models\WarehouseProviderLocation;
use Modules\WarehousePod\Models\WarehouseProviderLocationVariant;
use Modules\WarehousePod\Models\WarehouseProviderVariantMapping;
use Modules\WarehousePod\Models\WarehouseProviderVariantBuild;
use Modules\WarehousePod\Models\WarehouseProviderVariantConfig;
use Modules\WarehousePod\Models\WarehouseProviderVariantOption;
use Modules\WarehousePod\Models\ProductVariantOption;
use Modules\Ticket\Helpers\DBconnect;

class WarehouseProviderService extends Controller
{
    protected $connection = null;
    protected $printings = [];
    public function __construct()
    {
        parent::__construct();
        if ($this->connection == null) {
            $this->connection = DBconnect::connect($this->market);
        }
    }

    public function find(Request $request)
    {
        $filter = $this->buildFilter($request);
        $result = $this->getResult($filter);
        $this->decorResult($result);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function decorResult(&$items)
    {
        foreach ($items as &$item) {
            foreach ($item->warehouseProviderVariants as $key => $option) {
                $varians = ProductVariantOption::on($this->configName)
                    ->where('variant_id', '=', $option['option_id'])
                    ->pluck('name', 'id')->toArray();
                $item->warehouseProviderVariants[$key]['variants'] = $varians;
            }
        }
    }

    private function getResult($filter)
    {
        $query = WarehouseProvider::on($this->configName)->with(["warehouseProviderVariantConfigs", "warehouseProviderVariants"]);
        if (isset($filter['id']) && $filter['id'] != '') {
            $query->where('id', '=', $filter['id']);
        }
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            $query->where('name', 'LIKE', '%' . $filter['keyword'] . '%');
        }
        if (isset($filter['status']) && $filter['status'] != '') {
            $query->where('status', '=', $filter['status']);
        }
        if (isset($filter['supplier']) && $filter['supplier'] != '') {
            $query->where('supplier', 'like', '%' . $filter['supplier'] . '%');
        }
        if (isset($filter['supplier_id']) && $filter['supplier_id'] != '') {
            $query->where('supplier_id', 'like', '%' . $filter['supplier_id'] . '%');
        }
        if (isset($filter['category_id']) && $filter['category_id'] != '') {
            $query->where('category_id', $filter['category_id']);
        }
        if (isset($filter['country_id']) && $filter['country_id'] != '') {
            $query->whereHas('warehouseProviderLocations', function ($q) use ($filter) {
                $q->where("country_id", $filter['country_id']);
            });
        }

        $query->orderBy('created_at', 'DESC');
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function buildFilter($request)
    {
        $retVal = [];
        $columns = ['id', 'status', 'keyword', 'supplier', 'country_id'];
        foreach ($columns as $column) {
            if ($request->has($column)) {
                $retVal[$column] = $request->input($column);
            }
        }
        $retVal['page_size'] = 50;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function store(Request $request)
    {
        $data = $this->buildDataWarehouseProvider($request);
        $email = $request->has('email') ? $request->get('email') : null;
        $log = [
            'actor_type' => 'staff',
            'actor_email' => $email,
            'target_type' => 'WAREHOUSE_PROVIDER',
            'created_at' => new \DateTime()
        ];
        $oldData = null;
        if (isset($data['id']) && !empty($data['id'])) {
            $oldData = WarehouseProvider::on($this->configName)->find($data['id']);
            WarehouseProvider::on($this->configName)
                ->where('id', '=', $data['id'])
                ->update($data);
            $warehouseProvider = WarehouseProvider::on($this->configName)->find($data['id']);
            $log['event_type'] = 'UPDATE';
        } else {
            $warehouseProvider = WarehouseProvider::on($this->configName)->create($data);
            $log['event_type'] = 'CREATE';
        }
        $log['target_id'] = $warehouseProvider->id;
        $log['data'] = json_encode($request->all());
        Log::on($this->configName)->create($log);
        $response = [
            'status' => 'successful'
        ];
        $this->storeVariantConfig($request, $oldData, $warehouseProvider);
        return response()->json($response);
    }

    private function buildDataWarehouseProvider($request)
    {
        $columns = ['id', 'name', 'supplier', 'supplier_id', 'status', 'category_id', 'provider_info_id'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = ($request->has($column)) ? $request->get($column) : NULL;
        }
        return $retVal;
    }

    public function delete(Request $request)
    {
        $id = $request->has('id') ? $request->input('id') : [];
        $email = $request->has('email') ? $request->get('email') : null;
        $response = ['status' => 'fail'];
        if (!empty($id)) {
            $log = [
                'actor_type' => 'staff',
                'actor_email' => $email,
                'target_type' => 'WAREHOUSE_PROVIDER',
                'event_type' => 'DELETE',
                'created_at' => new \DateTime()
            ];
            WarehouseProvider::on($this->configName)
                ->where('id', $id)
                ->delete();
            WarehouseProviderLocation::on($this->configName)
                ->where('warehouse_provider_id', $id)
                ->delete();
            WarehouseProviderVariantConfig::on($this->configName)
                ->where('warehouse_provider_id', $id)
                ->delete();
            $log['target_id'] = $id;
            $log['data'] = json_encode($request->all());
            Log::on($this->configName)->create($log);
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function storeVariantConfig(Request $request, $oldData, $newData)
    {
        if ($request->has("products")) {
            $listIds = [];
            foreach ($request->input("products") as $product) {
                $dataArr = (array) $product;
                if (isset($dataArr["id"])) {
                    $listIds[] = $dataArr['id'];
                }
            }
            WarehouseProviderVariantConfig::on($this->configName)
                ->whereNotIn('id', $listIds)
                ->delete();
            foreach ($request->input("products") as $product) {
                $dataArr = (array) $product;
                if (isset($dataArr["id"])) {
                    WarehouseProviderVariantConfig::on($this->configName)
                        ->where('id', $dataArr["id"])
                        ->update([
                            "cost" => $dataArr["cost"],
                        ]);
                } else {
                    WarehouseProviderVariantConfig::on($this->configName)
                        ->create([
                            "warehouse_provider_id" => $newData->id,
                            "instock" => $dataArr["instock"],
                            "str_id" => $dataArr["str_id"],
                            "cost" => $dataArr["cost"],
                        ]);
                }
            }
        } else {
            WarehouseProviderVariantConfig::on($this->configName)
                ->where('warehouse_provider_id', $newData->id)
                ->delete();
        }
    }

    public function createProviderVariant(Request $request)
    {
        $response = ['status' => 'fail'];
        if ($request->has("name") && $request->has("type")) {
            $isExists = WarehouseProviderVariantOption::on($this->configName)
                ->where("name", $request->input("name"))
                ->where("type", $request->input("type"))
                ->exists();
            if (!$isExists) {
                WarehouseProviderVariantOption::on($this->configName)
                    ->create([
                        "name" => $request->input("name"),
                        "type" => $request->input("type"),
                    ]);
                $newVaraint = WarehouseProviderVariantOption::on($this->configName)
                    ->where("name", $request->input("name"))
                    ->where("type", $request->input("type"))
                    ->first();
                $response['status'] = 'successful';
                $response['result'] = $newVaraint;
            }
        }
        return response()->json($response);
    }

    public function findProviderLocation(Request $request)
    {
        $response = ['status' => 'fail'];
        $filter = $this->buildFilterProviderLocation($request);
        $result = $this->getResultProviderLocation($filter);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResultProviderLocation($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function getResultProviderLocation($filter)
    {
        $query = WarehouseProviderLocation::on($this->configName)->with(["products"]);
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            $query->where('name', 'LIKE', '%' . $filter['keyword'] . '%');
        }
        if (isset($filter['status']) && $filter['status'] != '') {
            $query->where('status', '=', $filter['status']);
        }
        if (isset($filter['country_id']) && $filter['country_id'] != '') {
            $query->where("country_id", $filter['country_id']);
        }
        if (isset($filter['warehouse_provider_id']) && $filter['warehouse_provider_id'] != '') {
            $query->where("warehouse_provider_id", $filter['warehouse_provider_id']);
        }

        $query->orderBy('created_at', 'DESC');
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function buildFilterProviderLocation($request)
    {
        $retVal = [];
        $columns = ['status', 'keyword', 'warehouse_provider_id', 'country_id'];
        foreach ($columns as $column) {
            if ($request->has($column)) {
                $retVal[$column] = $request->input($column);
            }
        }
        $retVal['page_size'] = 50;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function storeProviderLocation(Request $request)
    {
        $response = ['status' => 'fail'];
        $data = $this->buildDataProviderLocation($request);
        $email = $request->has('email') ? $request->get('email') : null;
        $log = [
            'actor_type' => 'staff',
            'actor_email' => $email,
            'target_type' => 'WH_PROVIDER_LOCATION',
            'created_at' => new \DateTime()
        ];
        if (isset($data['id']) && !empty($data['id'])) {
            WarehouseProviderLocation::on($this->configName)
                ->where('id', '=', $data['id'])
                ->update($data);
            $warehouseProviderLocation = WarehouseProviderLocation::on($this->configName)->find($data['id']);
            $log['event_type'] = 'UPDATE';
            $log['target_id'] = $warehouseProviderLocation->id;
            $log['data'] = json_encode($request->all());
            Log::on($this->configName)->create($log);
            $response['status'] = 'successful';
        } else {
            $isExists = WarehouseProviderLocation::on($this->configName)
                ->where("warehouse_provider_id", $data["warehouse_provider_id"])
                ->where("country_id", $data["country_id"])
                ->where("shipping_type", $data["shipping_type"])
                ->exists();
            if (!$isExists) {
                $warehouseProviderLocation = WarehouseProviderLocation::on($this->configName)->create($data);
                $log['event_type'] = 'CREATE';
                $log['target_id'] = $warehouseProviderLocation->id;
                $log['data'] = json_encode($request->all());
                Log::on($this->configName)->create($log);
                $response['status'] = 'successful';
            } else {
                $response['message'] = 'Dupplicate data!';
            }
        }
        return response()->json($response);
    }

    private function buildDataProviderLocation($request)
    {
        $columns = ['id', 'name', 'status', 'warehouse_provider_id', 'country_id', 'production_min_time', 'production_max_time', 'shipping_min_time', 'shipping_max_time', 'shipping_fee', 'adding_fee', 'shipping_type'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = ($request->has($column)) ? $request->get($column) : '';
        }
        return $retVal;
    }

    public function deleteProviderLocation(Request $request)
    {
        $id = $request->has('id') ? $request->input('id') : [];
        $response = ['status' => 'fail'];
        if (!empty($id)) {
            WarehouseProviderLocation::on($this->configName)
                ->where('id', $id)
                ->delete();
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function saveProviderLocationProducts(Request $request)
    {
        $response = ['status' => 'fail'];
        $products = $request->input("products");
        $locationId = $request->input("location_id");
        if ($products) {
            $listIds = [];
            foreach ($products as $item) {
                if (isset($item['id'])) {
                    $listIds[] = $item['id'];
                }
            }
            WarehouseProviderLocationVariant::on($this->configName)
                ->where("warehouse_provider_location_id", $locationId)
                ->whereNotIn("id", $listIds)
                ->delete();
            foreach ($products as $item) {
                $data = $this->buildDataProviderLocationProducts($item);
                if (isset($data["id"])) {
                    WarehouseProviderLocationVariant::on($this->configName)
                        ->where("id", $item["id"])->update($data);
                } else {
                    WarehouseProviderLocationVariant::on($this->configName)->create($data);
                }
            }
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    private function buildDataProviderLocationProducts($item)
    {
        $columns = ['id', 'warehouse_provider_location_id', 'str_id', 'production_min_time', 'production_max_time', 'shipping_min_time', 'shipping_max_time', 'shipping_fee', 'adding_fee'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = isset($item[$column]) ? $item[$column] : null;
        }
        return $retVal;
    }

    public function mappingProduct(Request $request)
    {
        $response = [
            "status" => "fail",
        ];
        $productId = $request->has('product_id') ? $request->input('product_id') : null;
        $productSkuId = $request->has('product_sku_id') ? $request->input('product_sku_id') : null;
        $countryId = $request->has('country_id') ? $request->input('country_id') : null;
        $shippingType = $request->has('shipping_type') ? $request->input('shipping_type') : 'standard';
        $mapWarehouseProvider = [];
        if ($productId && $countryId) {
            $productCategory = \DB::table("product_n_category")
                ->where("product_id", $productId)
                ->where("is_parent", 0)
                ->orderBy("id", "DESC")
                ->first();
            $result = $this->getCombinations($productId, $productSkuId, $productCategory);
            $mapWarehouseProvider = $this->mapByVariantOption($productId, $countryId, $result, $productCategory, $shippingType);
            $response = [
                "combinations" => $result["combinations"],
                "listVariantsDefault" => $result["listVariantsDefault"],
                "mapWarehouseProvider" => $mapWarehouseProvider,
            ];
        }
        return $response;
    }

    private function getCombinations($productId, $productSkuId, $productCategory)
    {
        $listChangeById = [];
        $productSkus = ProductSkuValue::on($this->configName)
            ->where('sku_id', '=', $productSkuId)
            ->orderBy('variant_option_id', 'ASC')->get();
        $listVariants = [];
        $listVariantsDefault = [];
        $warehouseProviderIds = [];
        foreach ($productSkus as $key => $item) {
            $listVariantsDefault[] = $item["variant_option_id"];
            $mappings = WarehouseProviderVariantMapping::on($this->configName)
                ->join("warehouse_provider", "warehouse_provider_id", "warehouse_provider.id")
                ->where("category_id", $productCategory->category_id)
                ->where("default_option_id", $item->variant_id)
                ->where("default_option_value", $item->variant_option_id)
                ->get();

            $mappingIds = [];
            if (count($mappings) > 0) {
                foreach ($mappings as $map) {
                    $mappingIds[] = $map["provider_option_value"];
                    if (!isset($listChangeById[$map["warehouse_provider_id"]])) {
                        $listChangeById[$map["warehouse_provider_id"]] = [];
                    }
                    $listChangeById[$map["warehouse_provider_id"]][$map["default_option_value"]] = $map["provider_option_value"];
                    $warehouseProviderIds[] = $map["warehouse_provider_id"];
                }
            }
            $listVariants[] = $item["variant_option_id"];
        }
        $newChangeVariant = [];
        foreach ($listChangeById as $item) {
            $newItem = [];
            foreach ($listVariants as $variant) {
                if (isset($item[$variant])) {
                    $newItem[] = $item[$variant];
                } else {
                    $newItem[] = $variant;
                }
            }
            $newChangeVariant[$key] = $newItem;
        }
        foreach ($newChangeVariant as $key => &$combination) {
            if (gettype($combination) == 'array') {
                sort($combination);
            } else {
                $combination = [$combination];
                sort($combination);
            }
        }
        return [
            "combinations" => $newChangeVariant,
            "listVariantsDefault" => $listVariantsDefault,
            "listChangeById" => $listChangeById,
        ];
    }

    private function combinations($arrays, $i = 0)
    {
        if (!isset($arrays[$i])) {
            return array();
        }
        if ($i == count($arrays) - 1) {
            return $arrays[$i];
        }

        // get combinations from subsequent arrays
        $tmp = $this->combinations($arrays, $i + 1);

        $result = array();

        // concat each array from tmp with each element from $arrays[$i]
        foreach ($arrays[$i] as $v) {
            foreach ($tmp as $t) {
                $result[] = is_array($t) ?
                    array_merge(array($v), $t) :
                    array($v, $t);
            }
        }
        return $result;
    }

    public function mapByVariantOption($productId, $countryId, $resultMapping, $productCategory, $shippingType)
    {
        $countStrValueList = 0;
        if (count($resultMapping["combinations"]) > 0) {
            foreach ($resultMapping["combinations"] as $item) {
                $countStrValueList = count($item);
                break;
            }
        }
        $listProviderIds = array_keys($resultMapping["listChangeById"]);
        $warehouseProviders = WarehouseProvider::on($this->configName)
            ->with(["warehouseProviderVariants"])
            ->where("warehouse_provider.status", "ACTIVE")
            ->whereIn("warehouse_provider.id", $listProviderIds)
            ->join("warehouse_provider_location", function ($q) use ($countryId, $shippingType) {
                $q->on("warehouse_provider_id", "=", "warehouse_provider.id");
                $q->where("country_id", $countryId);
                $q->where("shipping_type", ucfirst($shippingType));
                $q->where("warehouse_provider_location.status", "ACTIVE");
            })
            ->where("category_id", $productCategory->category_id)
            ->get(["warehouse_provider.*", 'country_id', 'production_min_time', 'production_max_time', 'shipping_min_time', 'shipping_max_time', 'shipping_fee', 'adding_fee', 'shipping_type']);
        if (count($warehouseProviders) == 0) {
            $warehouseProviders = WarehouseProvider::on($this->configName)
                ->with(["warehouseProviderVariants"])
                ->where("warehouse_provider.status", "ACTIVE")
                ->whereIn("warehouse_provider.id", $listProviderIds)
                ->join("warehouse_provider_location", function ($q) use ($countryId, $shippingType) {
                    $q->on("warehouse_provider_id", "=", "warehouse_provider.id");
                    $q->where("country_id", -1);
                    $q->where("shipping_type", ucfirst($shippingType));
                    $q->where("warehouse_provider_location.status", "ACTIVE");
                })
                ->where("category_id", $productCategory->category_id)
                ->get(["warehouse_provider.*", 'country_id', 'production_min_time', 'production_max_time', 'shipping_min_time', 'shipping_max_time', 'shipping_fee', 'adding_fee', 'shipping_type']);
        }

        // Map by skus
        $countItems = [];
        $countStrItems = [];
        $retValItem = [];
        $warehouseProviderVariantBuildIds = [];
        $costItems = [];
        $countItemLefts = [];
        $mapWarehouseProvider = [];

        foreach ($warehouseProviders as $warehouseProvider) {
            $warehouseProviderVariantBuilds = WarehouseProviderVariantBuild::on($this->configName)
                ->where("warehouse_provider_id", $warehouseProvider->id)
                ->get();
            foreach ($resultMapping["combinations"] as $key => $combination) {
                if ($key == $warehouseProvider->id) {
                    foreach ($warehouseProviderVariantBuilds as $warehouseProviderVariantBuild) {
                        $strIds = explode('-', $warehouseProviderVariantBuild->str_id);
                        $resultVariant = array_diff($combination, $strIds);
                        $countResultVariant = count($resultVariant);
                        // Nếu khớp chính xác thì lấy luôn chuyển kho khác
                        if ($countResultVariant == 0) {
                            $mapWarehouseProvider[] = [
                                "warehouseProviderVariantBuild" => $warehouseProviderVariantBuild,
                                "provider" => $warehouseProvider,
                                "is_full_correct" => 1,
                            ];
                            break;
                        } else {
                            // Nếu không lấy chính xác thì đếm số khớp nhiều nhất
                            if (!in_array($warehouseProviderVariantBuild["id"], $warehouseProviderVariantBuildIds)) {
                                $countItem = count($resultVariant);
                                $countItemLeft = $countStrValueList - $countItem;
                                array_push($countItems, $countItem);
                                array_push($countItemLefts, $countItemLeft);
                                array_push($countStrItems, count($strIds));
                                array_push($costItems, $warehouseProvider->cost);
                                $warehouseProviderVariantBuildIds[] = $warehouseProviderVariantBuild["id"];
                                $data = [
                                    'count' => $countItem,
                                    'count_left_str' => $countItemLeft,
                                    'strIds' => $strIds,
                                    'cost' => $warehouseProviderVariantBuild->cost,
                                    'warehouse_provider_build' => $warehouseProviderVariantBuild,
                                    'warehouseProvider' => $warehouseProvider,
                                ];
                                array_push($retValItem, $data);
                            }
                        }
                    }
                }
            }
        }
        // Nếu không có kho nào khớp chính xác thì lấy theo kho khớp nhiều nhất
        if (!$mapWarehouseProvider && count($retValItem) > 0) {
            array_multisort($countItems, SORT_ASC, $countStrItems, SORT_ASC, $countItemLefts, SORT_DESC, $costItems, SORT_ASC, $retValItem);
            $topCount = $retValItem[0]["count"];
            foreach ($retValItem as $item) {
                if ($topCount == $item["count"]) {
                    $mapWarehouseProvider[] = [
                        "warehouseProviderVariantBuild" => $item["warehouse_provider_build"],
                        "mapStrIds" => $countStrValueList - $item["count"],
                        "provider" => $item["warehouseProvider"],
                    ];
                }
            }
        } else {
            $mapWarehouseProvider = $warehouseProviders;
        }
        return $mapWarehouseProvider;
    }

    public function findVariant(Request $request)
    {
        $response = [
            "status" => "fail"
        ];
        if ($request->input("type") == 'size') {
            $sizeOptions = WarehouseProviderVariantOption::on($this->configName)
                ->where("type", "size")
                ->get();
            $response = [
                "status" => "successful",
                "result" => $sizeOptions
            ];
        } else if ($request->input("type") == 'color') {
            $colorOptions = WarehouseProviderVariantOption::on($this->configName)
                ->where("type", "color")
                ->get();
            $response = [
                "status" => "successful",
                "result" => $colorOptions
            ];
        }
        return response()->json($response);
    }

    public function storeProviderProducts(Request $request)
    {
        $response = ['status' => 'fail'];
        $email = $request->has('email') ? $request->get('email') : null;
        $log = [
            'actor_type' => 'staff',
            'actor_email' => $email,
            'target_type' => 'WH_PROVIDER_PRODUCTS',
            'event_type' => 'CREATE_OR_UPDATE',
            'created_at' => new \DateTime()
        ];
        $providerId = $request->input("id");
        if ($providerId) {
            $this->saveProviderVariant($request, $providerId);
            $this->saveProviderVariantConfig($request, $providerId);
            $this->saveProviderSizeAndColor($request, $providerId);
            $this->buildProviderAllVariant($providerId);
            $log['target_id'] = $providerId;
            $log['data'] = json_encode($request->all());
            Log::on($this->configName)->create($log);

            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    private function saveProviderVariant($request, $providerId)
    {
        $mappingVariants = $request->input("mappingVariant", []);
        if ($mappingVariants || count($mappingVariants) == 0) {
            $listIds = [];
            foreach ($mappingVariants as $item) {
                if (isset($item['id'])) {
                    $listIds[] = $item['id'];
                }
            }
            WarehouseProviderVariant::on($this->configName)
                ->where("warehouse_provider_id", $providerId)
                ->whereNotIn("id", $listIds)
                ->delete();
            foreach ($mappingVariants as $item) {
                if (isset($item["id"])) {
                    WarehouseProviderVariant::on($this->configName)
                        ->where("id", $item["id"])->update([
                            "option_id" => $item["option_id"],
                            "option_value_ids" => implode(',', $item["variant_ids"])
                        ]);
                } else {
                    WarehouseProviderVariant::on($this->configName)->create([
                        "warehouse_provider_id" => $providerId,
                        "option_type" => "default",
                        "option_id" => $item["option_id"],
                        "option_value_ids" => implode(',', $item["variant_ids"])
                    ]);
                }
            }
        }
    }

    private function saveProviderSizeAndColor($request, $providerId)
    {
        $retVal["has_size"] = 0;
        if ($request->has("sizes")) {
            $sizes = $request->input("sizes");
            $retVal["size_value_ids"] = implode(",", $sizes);
            if (count($sizes) > 0) {
                $retVal["has_size"] = 1;
            }
        }
        $retVal["has_color"] = 0;
        if ($request->has("colors")) {
            $colors = $request->input("colors");
            $retVal["color_value_ids"] = implode(",", $colors);
            if (count($colors) > 0) {
                $retVal["has_color"] = 1;
            }
        }
        WarehouseProvider::on($this->configName)->where("id", $providerId)->update($retVal);
    }

    private function saveProviderVariantConfig($request, $providerId)
    {
        $products = $request->input('products');
        $listIds = [];
        foreach ($products as $product) {
            $arrProduct = (array) $product;
            if (isset($arrProduct['id'])) {
                $listIds[] = $arrProduct['id'];
            }
        }
        WarehouseProviderVariantConfig::on($this->configName)
            ->where("warehouse_provider_id", $providerId)
            ->whereNotIn("id", $listIds)
            ->delete();
        foreach ($products as $product) {
            $arrProduct = (array) $product;
            if (isset($arrProduct['id'])) {
                WarehouseProviderVariantConfig::on($this->configName)
                    ->where("id", $arrProduct["id"])
                    ->update([
                        "cost" => $arrProduct["cost"],
                        "instock" => $arrProduct["instock"],
                        "str_id" => $arrProduct["str_id"],
                    ]);
            } else {
                WarehouseProviderVariantConfig::on($this->configName)
                    ->create([
                        "warehouse_provider_id" => $providerId,
                        "cost" => $arrProduct["cost"],
                        "instock" => $arrProduct["instock"],
                        "str_id" => $arrProduct["str_id"],
                    ]);
            }
        }
    }

    private function buildProviderAllVariant($providerId)
    {
        WarehouseProviderVariantBuild::on($this->configName)->where("warehouse_provider_id", $providerId)->delete();
        $configs = WarehouseProviderVariantConfig::on($this->configName)
            ->where("warehouse_provider_id", $providerId)
            ->get();
        $variants = WarehouseProviderVariant::on($this->configName)
            ->where("warehouse_provider_id", $providerId)
            ->where("option_type", "default")
            ->get();
        if (count($configs) > 0) {
            $combineByVariants =  [];
            if (count($variants) > 1) {
                $list1 = explode(",", $variants[0]["option_value_ids"]);
                $list2 = explode(",", $variants[1]["option_value_ids"]);
                $combineByVariants = $this->makeCombinations($list1, $list2);
            } else if (count($variants) == 1) {
                $combineByVariants = explode(",", $variants[0]["option_value_ids"]);
            }
            $listBuildVariant = [];
            foreach ($configs as $key => $config) {
                $newData = [
                    "warehouse_provider_id" => $providerId,
                    "provider_str_id" => $config["str_id"],
                    "str_id" => $config["str_id"],
                    "instock" => $config["instock"],
                    "cost" => $config["cost"],
                    "created_at" => new \Datetime(),
                    "updated_at" => new \Datetime(),
                ];
                if (count($combineByVariants) > 0) {
                    foreach ($combineByVariants as $variant) {
                        $newData["str_id"] = $config["str_id"] . "-" . $variant;
                        $listBuildVariant[] = $newData;
                    }
                } else {
                    $listBuildVariant[] = $newData;
                }
            }
            foreach ($listBuildVariant as $key => $item) {
                $str = explode("-", $item["str_id"]);
                sort($str);
                $listBuildVariant[$key]["str_id"] = implode("-", $str);
            }
            WarehouseProviderVariantBuild::on($this->configName)->where("warehouse_provider_id", $providerId)->insert($listBuildVariant);
        }
    }

    private function makeCombinations($list1, $list2)
    {
        $combinations = [];
        foreach ($list1 as $item) {
            foreach ($list2 as $item2) {
                $combinations[] = $item . '-' . $item2;
            }
        }
        return $combinations;
    }

    public function findProviderMappingVariant(Request $request)
    {
        $response = ['status' => 'fail'];
        $filter = $this->buildFilterProviderMappingVariant($request);
        $result = $this->getResultProviderMappingVariant($filter);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResultProviderMappingVariant($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function getResultProviderMappingVariant($filter)
    {
        $query = WarehouseProviderVariantMapping::on($this->configName);
        if (isset($filter['default_option_id']) && $filter['default_option_id'] != '') {
            $query->where('default_option_id', '=', $filter['default_option_id']);
        }
        if (isset($filter['default_option_value']) && $filter['default_option_value'] != '') {
            $query->where('default_option_value', '=', $filter['default_option_value']);
        }
        if (isset($filter['provider_option_value']) && $filter['provider_option_value'] != '') {
            $query->where('provider_option_value', '=', $filter['provider_option_value']);
        }
        if (isset($filter['warehouse_provider_id']) && $filter['warehouse_provider_id'] != '') {
            $query->where('warehouse_provider_id', '=', $filter['warehouse_provider_id']);
        }

        $query->orderBy('created_at', 'DESC');
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            $query->join("product_variant", "product_variant.id", "warehouse_provider_variant_mapping.default_option_id");
            $query->join("warehouse_provider", "warehouse_provider.id", "warehouse_provider_variant_mapping.warehouse_provider_id");
            $query->join("product_variant_option", "product_variant_option.id", "warehouse_provider_variant_mapping.default_option_value");
            $query->join("warehouse_provider_variant_option", "warehouse_provider_variant_option.id", "warehouse_provider_variant_mapping.provider_option_value");
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get(["warehouse_provider_variant_mapping.*", "product_variant_option.name as variant_name", "product_variant.name as option_name", "warehouse_provider_variant_option.name as provider_option_name", "warehouse_provider.name as provider_name"]);
        }
    }

    private function buildFilterProviderMappingVariant($request)
    {
        $retVal = [];
        $columns = ['default_option_id', 'default_option_value', 'provider_option_value'];
        foreach ($columns as $column) {
            if ($request->has($column)) {
                $retVal[$column] = $request->input($column);
            }
        }
        $retVal['page_size'] = 50;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    public function storeProviderMappingVariant(Request $request)
    {
        $response = ['status' => 'fail'];
        $data = $this->buildDataProviderMappingVariant($request);
        $email = $request->has('email') ? $request->get('email') : null;
        $log = [
            'actor_type' => 'staff',
            'actor_email' => $email,
            'target_type' => 'WH_PROVIDER_MAPPING',
            'created_at' => new \DateTime()
        ];
        $isExists = WarehouseProviderVariantMapping::on($this->configName)
            ->where("default_option_value", $data["default_option_value"])
            ->where("provider_option_value", $data["provider_option_value"])
            ->where("warehouse_provider_id", $data["warehouse_provider_id"])
            ->exists();
        if (!$isExists) {
            $warehouseProviderMappingVariant = WarehouseProviderVariantMapping::on($this->configName)->create($data);
            $log['event_type'] = 'CREATE';
            $log['target_id'] = $warehouseProviderMappingVariant->id;
            $log['data'] = json_encode($request->all());
            Log::on($this->configName)->create($log);
            $response['status'] = 'successful';
        } else {
            $response['message'] = 'Dupplicate data!';
        }
        return response()->json($response);
    }

    private function buildDataProviderMappingVariant($request)
    {
        $columns = ['default_option_id', 'default_option_value', 'provider_option_value', 'warehouse_provider_id'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = ($request->has($column)) ? $request->get($column) : '';
        }
        return $retVal;
    }

    public function deleteProviderMappingVariant(Request $request)
    {
        $email = $request->has('email') ? $request->get('email') : null;
        $log = [
            'actor_type' => 'staff',
            'actor_email' => $email,
            'target_type' => 'WH_PROVIDER_MAPPING',
            'created_at' => new \DateTime()
        ];
        $id = $request->has('id') ? $request->input('id') : [];
        $response = ['status' => 'fail'];
        if (!empty($id)) {
            $warehouseMapping = WarehouseProviderVariantMapping::on($this->configName)
            ->where('id', $id)->first();
            WarehouseProviderVariantMapping::on($this->configName)
                ->where('id', $id)
                ->delete();
            $log['event_type'] = 'DELETE';
            $log['target_id'] = $id;
            $data = $request->all();
            $data["data"] = $warehouseMapping;
            $log['data'] = json_encode($data);
            Log::on($this->configName)->create($log);
            $response['status'] = 'successful';
        }
        return response()->json($response);
    }

    public function findProviderInfo(Request $request)
    {
        $response = ['status' => 'successful'];
        $providerInfos = WarehouseProviderInfo::on($this->configName)
            ->get();
        $response['result'] = $providerInfos;
        return response()->json($response);
    }

    public function storeProviderInfo(Request $request)
    {
        $response = ['status' => 'fail'];
        $data = $request->all();
        if (isset($data['id']) && !empty($data['id'])) {
            WarehouseProviderInfo::on($this->configName)
                ->where('id', '=', $data['id'])
                ->update($data);
            WarehouseProviderInfo::on($this->configName)->find($data['id']);
            $response['status'] = 'successful';
        } else {
            $isExists = WarehouseProviderInfo::on($this->configName)
                ->where("name", $data["name"])
                ->exists();
            if (!$isExists) {
                WarehouseProviderInfo::on($this->configName)->create($data);
                $response['status'] = 'successful';
            } else {
                $response['message'] = 'Dupplicate data!';
            }
        }
        return response()->json($response);
    }

    public function findProviderVariant(Request $request) {
        $response = ['status' => 'fail'];
        $providerId = $request->input("provider_id");
        if ($providerId) {
            $provider = WarehouseProvider::on($this->configName)
                ->where("id", $providerId)->first();
            $ids = array_merge(explode(',', $provider->size_value_ids), explode(',', $provider->color_value_ids));
            $result = WarehouseProviderVariantOption::on($this->configName)
                ->whereIn("id", $ids)
                ->get(['name', 'id']);
            $response['status'] = 'successful';
            $response['result'] = $result;
        }
        return response()->json($response);

    }

    public function storePrintifyVariant(Request $request) {
        $response = ['status' => 'fail'];
        $data = $request->all();
        if (isset($data['type']) && isset($data['items'])) {
            foreach ($data['items'] as $item) {
                $variantOption = WarehouseProviderVariantOption::on($this->configName)
                ->where("name", $item['name'])
                ->where("type", ucfirst($data['type']))
                ->first();
                if (!$variantOption) {
                    WarehouseProviderVariantOption::on($this->configName)
                        ->create([
                            "name" => $item["name"],
                            "type" => ucfirst($data['type']),
                        ]);
                }
            }
            $response['status'] = 'successful';
            $response['result'] = [];
        }
        return response()->json($response);
    }

    public function storePrintifyProduct(Request $request) {
        return;
        set_time_limit(-1);
        $response = ['status' => 'fail'];
        $allData = $request->all();
        foreach ($allData as $key => $data) {
            $providerInfo = WarehouseProviderInfo::on($this->configName)
                ->where("name", $data["provider_name"])
                ->first();
            if (!$providerInfo) {
                WarehouseProviderInfo::on($this->configName)->create([
                    "name" => $data["provider_name"]
                ]);
                $providerInfo = WarehouseProviderInfo::on($this->configName)
                    ->where("name", $data["provider_name"])
                    ->first();
            }
            $warehouseProvider = WarehouseProvider::on($this->configName)
                ->where("name", $data["name"] . "_" . $data["provider_name"])
                ->first();
            if (!$warehouseProvider) {
                WarehouseProvider::on($this->configName)->create([
                    "name" => $data["name"] . "_" . $data["provider_name"],
                    "provider_info_id" => $providerInfo->id,
                    "supplier_id" => 8,
                ]);
                $warehouseProvider = WarehouseProvider::on($this->configName)
                    ->where("name", $data["name"] . "_" . $data["provider_name"])
                    ->first();
            }
            if (isset($data["skus"])) {
                WarehouseProviderVariantConfig::on($this->configName)
                    ->where("warehouse_provider_id", $warehouseProvider->id)
                    ->delete();
                $skus = $data["skus"];
                $sizes = [];
                $colors = [];
                foreach ($skus as $key => $sku) {
                    $variantOption = $this->buildVariantOption($sku);
                    $str = [];
                    if ($variantOption["size"] && !in_array($variantOption["size"]->id, $sizes)) {
                        $sizes[] = $variantOption["size"]->id;
                    }
                    if ($variantOption["color"] && !in_array($variantOption["color"]->id, $colors)) {
                        $colors[] = $variantOption["color"]->id;
                    }
                    if ($variantOption["size"]) {
                        $str[] = $variantOption["size"]->id;
                    }
                    if ($variantOption["color"]) {
                        $str[] = $variantOption["color"]->id;
                    }
                    sort($str);
                    WarehouseProviderVariantConfig::on($this->configName)
                        ->create([
                            "warehouse_provider_id" => $warehouseProvider->id,
                            "instock" => isset($sku["status"]) && $sku["status"] == 'ACTIVE' ? 1 : 0,
                            "str_id" => implode("-", $str),
                            "cost" => $sku["price"],
                        ]);
                }
                sort($sizes);
                sort($colors);
                WarehouseProvider::on($this->configName)
                    ->where("id", $warehouseProvider->id)
                    ->update([
                        "has_size" => count($sizes) > 0 ? 1 : 0,
                        "has_color" => count($colors) > 0 ? 1 : 0,
                        "size_value_ids" => implode(",", $sizes),
                        "color_value_ids" => implode(",", $colors),
                    ]);
            }
            if (isset($data["locations"])) {
                $locations = $data["locations"];
                WarehouseProviderLocation::on($this->configName)
                            ->where("warehouse_provider_id", $warehouseProvider->id)
                            ->delete();
                foreach ($locations as $key => $location) {
                    $countryId = null;
                    if ($location["country"] == "Rest of World") {
                        $countryId = -1;
                    } else {
                        $country = Country::where("nicename", $location["country"])->first();
                        if (!$country) {
                            $country = Country::where("iso", $location["country_code"])->first();
                        }
                        if ($country) {
                            $countryId = $country->id;
                        }
                    }
                    if ($countryId) {
                        $isExists = WarehouseProviderLocation::on($this->configName)
                            ->where("warehouse_provider_id", $warehouseProvider->id)
                            ->where('country_id', '=', $countryId)
                            ->where("shipping_type", ucfirst($location["method"]))
                            ->exists();
                        if (!$isExists) {
                            WarehouseProviderLocation::on($this->configName)
                            ->create([
                                "name" => $location["country"] . '_' . $location["method"],
                                "warehouse_provider_id" => $warehouseProvider->id,
                                "country_id" => $countryId,
                                "shipping_fee" => $location["first_item"],
                                "adding_fee" => $location["next_item"],
                                "shipping_type" => ucfirst($location["method"]),
                            ]);
                        }
                        $warehouseProviderLocation = WarehouseProviderLocation::on($this->configName)
                            ->where("warehouse_provider_id", $warehouseProvider->id)
                            ->where("country_id", $countryId)
                            ->where("shipping_type", ucfirst($location["method"]))
                            ->first();
                        if ($location["type"] != "All") {
                            $variants = explode("/", $location["type"]);
                            $str = [];
                            foreach ($variants as $variant) {
                                $variantOption = WarehouseProviderVariantOption::on($this->configName)
                                    ->where("name", trim($variant))
                                    ->first();
                                if ($variantOption) {
                                    $str[] = $variantOption->id;
                                }
                            }
                            sort($str);
                            $isExistsLocationVariant = WarehouseProviderLocationVariant::on($this->configName)
                                ->where("warehouse_provider_location_id", $warehouseProviderLocation->id)
                                ->where("str_id", implode("-", $str))
                                ->exists();
                            if (!$isExistsLocationVariant) {
                                WarehouseProviderLocationVariant::on($this->configName)
                                ->create([
                                    "warehouse_provider_location_id" => $warehouseProviderLocation->id,
                                    "str_id" => implode("-", $str),
                                    "shipping_fee" => $location["first_item"],
                                    "adding_fee" => $location["next_item"],
                                ]);
                            }
                        }
                    }
                }
            }
        }
        $response['status'] = 'successful';
        $response['result'] = [];
        return response()->json($response);
    }

    private function buildVariantOption($sku) {
        $size = null;
        $color = null;
        if (isset($sku["color"])) {
            $color = WarehouseProviderVariantOption::on($this->configName)
                ->where("name", $sku["color"])
                ->where("type", "Color")
                ->first();
            if (!$color) {
                WarehouseProviderVariantOption::on($this->configName)->create([
                    "name" => $sku["color"],
                    "type" => "Color",
                ]);
                $color = WarehouseProviderVariantOption::on($this->configName)
                    ->where("name", $sku["color"])
                    ->where("type", "Color")
                    ->first();
            }
        }
        if (isset($sku["size"])) {
            $size = WarehouseProviderVariantOption::on($this->configName)
                ->where("name", $sku["size"])
                ->where("type", "Size")
                ->first();
            if (!$size) {
                WarehouseProviderVariantOption::on($this->configName)->create([
                    "name" => $sku["size"],
                    "type" => "Size",
                ]);
                $size = WarehouseProviderVariantOption::on($this->configName)
                    ->where("name", $sku["size"])
                    ->where("type", "Size")
                    ->first();
            }
        }
        return [
            "size" => $size,
            "color" => $color
        ];
    }

    private function buildDataPrintifyProduct($request)
    {
        $columns = ['name', 'provider_name'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = ($request->has($column)) ? $request->get($column) : NULL;
        }
        return $retVal;
    }

    public function cloneLocation(Request $request) {
        $response = ['status' => 'fail'];
        $providerId = $request->input("provider_id");
        $targetProviderId = $request->input("target_provider_id");
        if ($providerId) {
            $locations = WarehouseProviderLocation::on($this->configName)
                ->with(["products"])
                ->where("warehouse_provider_id", $providerId)
                ->get();
            foreach ($locations as $key => $location) {
                $data = $this->buildDataProviderLocationClone($location);
                $data["warehouse_provider_id"] = $targetProviderId;
                WarehouseProviderLocation::on($this->configName)
                    ->create($data);
                $newLocation = WarehouseProviderLocation::on($this->configName)
                    ->where("warehouse_provider_id", $targetProviderId)
                    ->where("country_id", $data["country_id"])
                    ->where("shipping_type", $data["shipping_type"])
                    ->first();
                foreach ($location["products"] as $index => $product) {
                    $dataLocationProduct = $this->buildDataProviderLocationProductsClone($product);
                    $dataLocationProduct["warehouse_provider_location_id"] = $newLocation->id;
                    WarehouseProviderLocationVariant::on($this->configName)->create($dataLocationProduct);
                }
            }
            $response['status'] = 'successful';
            $response['result'] = [];
        }
        return response()->json($response);
    }

    private function buildDataProviderLocationClone($data)
    {
        $columns = ['name', 'status', 'country_id', 'production_min_time', 'production_max_time', 'shipping_min_time', 'shipping_max_time', 'shipping_fee', 'adding_fee', 'shipping_type'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = isset($data[$column]) ? $data[$column] : '';
        }
        return $retVal;
    }

    private function buildDataProviderLocationProductsClone($item)
    {
        $columns = ['str_id', 'production_min_time', 'production_max_time', 'shipping_min_time', 'shipping_max_time', 'shipping_fee', 'adding_fee'];
        $retVal = [];
        foreach ($columns as $column) {
            $retVal[$column] = isset($item[$column]) ? $item[$column] : null;
        }
        return $retVal;
    }
}
