<?php

namespace Modules\WarehousePod\Controllers;

use Illuminate\Http\Request;
use Modules\WarehousePod\Controllers\Controller;
use Modules\Ticket\Helpers\DBconnect;
use Module;
use Modules\WarehousePod\Models\Printing;
use Modules\WarehousePod\Models\Country;
use Modules\WarehousePod\Models\Category;
use Modules\WarehousePod\Models\ProductNCategory;
use Modules\WarehousePod\Models\ProductSkuValue;
use Modules\WarehousePod\Models\Product;
use Modules\WarehousePod\Models\Warehouse;
use Modules\WarehousePod\Models\WarehouseCategory;
use Modules\WarehousePod\Models\WarehouseCategoryValue;
use Modules\WarehousePod\Models\WarehouseConfig;
use Modules\WarehousePod\Models\WarehouseConfigItem;
use Modules\WarehousePod\Models\ErrorLog;
use Illuminate\Support\Facades\Cache;
use Modules\WarehousePod\Models\WarehouseCurrencyRatio;

class WarehouseConfigController extends Controller
{
    protected $connection = null;

    public function __construct()
    {   
        parent::__construct();
        if ($this->connection == null) {
            $this->connection = DBconnect::connect($this->market);
        }
    }

    public function config(Request $request)
    {
        $printing = Printing::on($this->configName)
                                ->pluck('name', 'code')
                                ->toArray();
        $printing['default'] = 'Default';
        $countries = Country::on($this->configName)
                                ->orderBy('name', 'ASC')
                                ->pluck('name', 'id')
                                ->toArray();
        $countries[-1] = "Tất cả các quốc gia";
        $categories = Category::on($this->configName)
                                ->where('type', '=', 'PRODUCT')
                                ->orderBy('name', 'ASC')
                                ->pluck('name', 'id')
                                ->toArray();
        $allActiveWareHouseCate = WarehouseConfigItem::on($this->configName)
                                    ->join("warehouse_config", "warehouse_config.id", "=", "warehouse_config_id")
                                    ->where('warehouse_config.is_active', 1)
                                    ->join("warehouse", "warehouse.id", "=", "warehouse_id")
                                    ->where('warehouse.is_active', 1)
                                    ->get(["apply_cate"]);
        $currencyRatios = WarehouseCurrencyRatio::on($this->configName)
                                    ->get()->toArray();
        $ratioByIds = [];
        foreach ($currencyRatios as $key => $item) {
            $currencyRatios[$key]["name"] = $item["from_currency"] . " -> " . $item["to_currency"];
            $ratioByIds[$item['id']] = $item['ratio'];
        }
        $listCateApply = [];
        foreach ($allActiveWareHouseCate as $item) {
            $applyCate = explode(",", $item->apply_cate);
            $listCateApply = array_unique (array_merge ($listCateApply, $applyCate));
        }
        $listCateApply = array_diff($listCateApply, array(""));
        $listByKey = [];
        foreach ($listCateApply as $item) {
            $listByKey[$item] = $item;
        }
        $listNotApply = [];
        foreach ($categories as $key => $cate) {
            if (!array_key_exists($key, $listByKey)) {
                $listNotApply[] = $key;
            }
        }
        $options = $this->connection->table('product_variant')
                                    ->get(['name', 'id']);
        return view('warehouse-pod::system.config.index', [
            'printings' => $printing,
            "listNotApply" => $listNotApply,
            'countries' => $countries,
            'categories' => $categories,
            'options' => $options,
            "currencyRatios" => $currencyRatios,
            "ratioByIds" => $ratioByIds,
        ]);
    }

    public function detect($products = [], $countryId = -1) {
        $retVal = [];
        if (!empty($products)) {
            $productIds = [];
            $productSkuIds = [];
            foreach($products as $product) {
                $arrProduct = (array) $product;
                $productIds[] = $arrProduct['product_id'];
                $productSkuIds[] = $arrProduct['product_sku_id'];
            }
            $productCategory = $this->getCategory($productIds);
            $variantBySkuIds = [];
            if (!empty($productSkuIds)) {
                $variantBySkuIds = $this->getVariants($productSkuIds);
            }
            $warehouses = $this->getWarehouses($productCategory['categoryIds'], $countryId);
            $retVal = $this->matchingProductWarehouse($products, $productCategory['categoryByProducts'], $variantBySkuIds, $warehouses);
        }
        return $retVal;
    }

    private function matchingProductWarehouse($products, $productCategory, $variantBySkuIds, $warehouses) {
        $retVal = [];
        foreach ($products as $product) {
            $arrProduct = (array) $product;
            if (isset($productCategory[$arrProduct['product_id']])) {
                if (!isset($variantBySkuIds[$arrProduct['product_sku_id']])) {
                    $results = $this->matchingByCategory($productCategory[$arrProduct['product_id']], $warehouses);
                    $arrProduct['warehouse_ids'] =  $this->buildWarehouse($results);
                } else {
                    $results = $this->matchingByVariant($productCategory[$arrProduct['product_id']], $variantBySkuIds[$arrProduct['product_sku_id']], $warehouses);
                    $arrProduct['warehouse_ids'] = $this->buildWarehouse($results);
                }
            } else {
                $arrProduct['warehouse_ids'] = [];
            }
            $retVal[] = $arrProduct;
        }
        return $retVal;
    }

    private function buildWarehouse($items){
        $retVal = [];
        if (!empty($items)) {
            foreach($items as $item) {
                $retVal[] = $item['warehouse_id'];
            }
        }
        return $retVal;
    }

    private function matchingByCategory($categories, $warehouses) {
        $retVal = [];
        foreach ($categories as $category) {
            if (isset($warehouses[$category])) {
                usort($warehouses[$category], function($a, $b) {
                    return $a["country_id"] < $b["country_id"];
                });
                $tmp = [];
                foreach ($warehouses[$category] as $ware) {
                    $tmp[] = $ware;
                }
                $retVal = array_merge($retVal, $tmp);
            }
        }
        return $retVal;
    }

    private function matchingByVariant($categories, $variants, $warehouses) {
        $retVal = [];
        $product = [];
        foreach($variants as $variant) {
            $product[$variant['variant_id']] = $variant['variant_option_id'];
        }
        $tmp = $this->matchingByCategory($categories, $warehouses);
        foreach ($tmp as $item) {
            $count = 0;
            foreach ($item['options'] as $option) {
                if (isset($product[$option['option_id']])) {
                    $variantTmps = explode(',', $option['option_value_ids']);
                    if (in_array($product[$option['option_id']], $variantTmps)) {
                        $count++;
                    }
                }
            }
            if ($count == count($item['options'])) {
                $retVal[] = $item;
            }
        }
        $retVal = $this->sortByInstock($variants, $retVal);
        if (count($retVal) > 1) {
            $retVal = $this->sortByShippingFee($retVal, $categories);
        }
        return $retVal;
    }

    private function sortByShippingFee($items, $categories) {
        $retVal = $items;
        $warehouseIds = [];
        $warehouseByIds = [];
        foreach ($items as $item) {
            $warehouseIds[] = $item['warehouse_id'];
            $warehouseByIds[$item['warehouse_id']] = $item;
        }
        $itemShips = WarehouseConfig::join('warehouse', 'warehouse.id', '=', 'warehouse_config.warehouse_id')
                                    ->join('warehouse_config_item', 'warehouse_config_item.warehouse_config_id', '=', 'warehouse_config.id')
                                    ->whereIn('warehouse.id', $warehouseIds)
                                    ->where('warehouse_config.config_type', '=', 'shipping')
                                    ->where('warehouse_config.type', '=', 'standard')
                                    ->where(function ($q) use ($categories) {
                                        $q->orWhere('warehouse_config_item.type', '=', 'default');
                                        $q->orWhere(function ($q1) use ($categories) {
                                            $q1->where('warehouse_config_item.type', '=', 'custom');
                                            $q1->whereIn('warehouse_config_item.apply_cate', $categories);
                                        });
                                    })
                                    ->get(['warehouse.id', 'warehouse_config_item.default_shipping_fee', 'warehouse_config_item.apply_cate']);
        foreach ($itemShips as $item) {
            if (empty($item->apply_cate)) {
                $item->apply_cate = 0;
            }
        }
        $arrItems = $itemShips->toArray();
        $columns1 = array_column($arrItems, 'default_shipping_fee');
        $columns2 = array_column($arrItems, 'apply_cate');
        array_multisort($columns1, SORT_ASC, $columns2, SORT_DESC, $arrItems);
        $tmp = [];
        $newRetVal = [];
        foreach ($arrItems as $item) {
            if (!in_array($item['id'], $tmp)) {
                $tmp[] = $item['id'];
                $newRetVal[] = $warehouseByIds[$item['id']];
            }
        }
        if (!empty($newRetVal)) {
            $retVal = $newRetVal;
        }
        return $retVal;
    }

    private function sortByInstock($variants, $items) {
        $retVal = [];
        $source = [];
        foreach($variants as $variant) {
            $source[] = $variant['variant_option_id'];
        }
        foreach($items as $item) {
            $item['instock'] = 0;
            foreach ($item['products'] as $product) {
                $search = explode('-', $product['str_id']);
                $containsSearch = count(array_intersect($search, $source)) === count($search);
                if ($containsSearch) {
                    $item['instock'] = $product['instock'];
                    break;
                }
            }
            if ($item['instock'] == 1) {
                $retVal[] = $item;
            }
        }
        return $retVal;        
    }

    private function getWarehouses($categoryIds, $countryId) {
        $countries = [$countryId];
        $items = WarehouseCategory::leftJoin('warehouse', 'warehouse.id', '=', 'warehouse_category.warehouse_id')
                                    ->where('warehouse.is_active', '=', 1)
                                    ->where('warehouse.type', '=', 'custom')
                                    ->whereIn('warehouse_category.category_id', $categoryIds)
                                    ->whereIn('warehouse.country_id', $countries)
                                    ->get(['warehouse_category.id', 
                                        'warehouse_category.warehouse_id', 
                                        'warehouse_category.category_id', 
                                        'warehouse.country_id']);
        $retVal = [];
        if (count($items) == 0) {
            $items = WarehouseCategory::leftJoin('warehouse', 'warehouse.id', '=', 'warehouse_category.warehouse_id')
                ->where('warehouse.is_active', '=', 1)
                ->where('warehouse.type', '=', 'custom')
                ->whereIn('warehouse_category.category_id', $categoryIds)
                ->whereIn('warehouse.country_id', [-1])
                ->get(['warehouse_category.id', 
                    'warehouse_category.warehouse_id', 
                    'warehouse_category.category_id', 
                    'warehouse.country_id']);
        }
        if (!empty($items)) {
            $ids = [];
            $countryByIds = [];
            foreach ($items as $item) {
                $ids[] = $item->id;
                $countryByIds[$item->warehouse_id] = $item->country_id;
            }
            $warehouseCategories = WarehouseCategory::with(['options', 'products' => function ($q) {
                                            $q->where('deleted', '=', 0);
                                            $q->where('instock', '=', 1);
                                        }])
                                        ->whereIn('id', $ids)
                                        ->get()->toArray();
            foreach ($warehouseCategories as $item) {
                $item['country_id'] = -2;
                if (isset($countryByIds[$item['warehouse_id']])) {
                    $item['country_id'] = $countryByIds[$item['warehouse_id']];
                }
                if (!array_key_exists($item['category_id'], $retVal)) {
                    $retVal[$item['category_id']] = [];
                }
                $retVal[$item['category_id']][] = $item;
            }
        }
        
        return $retVal;
    }

    private function getCategory($productIds) {
        $productNCategories = ProductNCategory::whereIn('product_id', $productIds)
                                                    ->orderBy('is_parent', 'ASC')
                                                    ->get(['category_id', 'product_id'])
                                                    ->toArray();
        $categoryIds = [];
        $categoryByProducts = [];
        foreach($productNCategories as $item) {
            if (!in_array($item['category_id'], $categoryIds)) {
                $categoryIds[] = $item['category_id'];
            }
            if (!array_key_exists($item['product_id'], $categoryByProducts)) {
                $categoryByProducts[$item['product_id']] = [];
            }
            $categoryByProducts[$item['product_id']][] = $item['category_id'];
        }
        return [
            'categoryIds' => $categoryIds,
            'categoryByProducts' => $categoryByProducts
        ];
    }

    private function getVariants($skuIds) {
        $items = ProductSkuValue::whereIn('sku_id', $skuIds)
                                    ->get(['sku_id', 'product_id', 'variant_id', 'variant_option_id'])
                                    ->toArray();
        $retVal = [];
        foreach ($items as $item) {
            if (!array_key_exists($item['sku_id'], $retVal)) {
                $retVal[$item['sku_id']] = [];
            }
            $retVal[$item['sku_id']][] = $item;
        }
        return $retVal;
    }

    public function orderWarning() {
        return view('warehouse-pod::system.order-warning.index', []);
    }

    public function getShippingByItemV2($items, $countryId) {
        $detectWarehouse = $this->detect($items, $countryId);
        $warehouses = [];
        if (count($detectWarehouse) > 0 && isset($detectWarehouse[0]["warehouse_ids"]) && count($detectWarehouse[0]["warehouse_ids"]) > 0) {
            $listWarehouses = Warehouse::with("warehouseShippingConfigs")
                                ->whereHas("warehouseShippingConfigs")
                                ->where('is_active', 1)
                                ->whereIn("id", $detectWarehouse[0]["warehouse_ids"])
                                ->get();
            $listById = [];
            foreach ($listWarehouses as $warehouse) {
                $listById[$warehouse->id] = $warehouse;
            }
            $newList = array_replace(array_flip($detectWarehouse[0]["warehouse_ids"]), $listById);
            foreach ($newList as $item) {
                $warehouses[] = $item;
            }
        }
        if (count($warehouses) == 0) {
            $warehouses = Warehouse::with("warehouseShippingConfigs")
                            ->whereHas("warehouseShippingConfigs")
                            ->where('is_active', 1)
                            ->where('country_id', $countryId)
                            ->where("type", "default_ship")
                            ->get();
            if (count($warehouses) == 0) {
                $warehouses = Warehouse::with("warehouseShippingConfigs")
                                ->whereHas("warehouseShippingConfigs")
                                ->where('is_active', 1)
                                ->where('country_id', -1)
                                ->where("type", "default_ship")
                                ->get();
            }
        }
        $result = [];
        try {
            foreach ($warehouses as $key => $warehouse) {
                $resultByWarehouse = [];
                for ($index = 0; $index < count($warehouse->warehouseShippingConfigs); $index++) {
                    $warehouseConfig = $warehouse->warehouseShippingConfigs[$index];
                    if ($warehouseConfig->type == 'standard') {
                        $shippingFee = $this->getShippingInfoByItem($items[0]->category_id, $warehouseConfig, $warehouse);
                        if ($shippingFee) {
                            $resultByWarehouse = $shippingFee;
                        }
                    }
                }
                $result[] = $resultByWarehouse;
            }
        } catch (\Exception $ex) {
            $errorLog = new ErrorLog();
            $errorLog->project = env('APP_NAME');
            $errorLog->url = "Error get shipping";
            $errorLog->data = json_encode($warehouses);
            $errorLog->message = $ex->getMessage();
            $errorLog->file = $ex->getFile();
            $errorLog->line = $ex->getLine();
            $errorLog->save();
        }
        return $result;
    }

    private function getShippingInfoByItem($categoryId, $warehouseConfig, $warehouse) {
        $invalidCate = explode(',', $warehouseConfig->invalid_cate);
        if (in_array($categoryId, $invalidCate)) {
            $result = false;
        } else {
            $result = [
                "warehouseConfig" => $warehouseConfig,
                "warehouse" => $warehouse,
            ];
        }
        return $result;
    }

    public function getShippingInfoByProduct($product, $countryId, $data = []) {
        $response = [
            "status" => "fail"
        ];
        $product = (object) $product;
        if (!isset($product->category_id)) {
            $productNCategory = ProductNCategory::where("product_id", $product->id)->where("is_parent", 0)->first(['category_id']);
            if ($productNCategory) {
                $categoryId = $productNCategory->category_id;
            }
        } else {
            $categoryId = $product->category_id;
        }
        $keyCache = $countryId;
        $keyCache .= "-" . $categoryId;
        if (isset($product->sku_id)) {
            $template = \DB::table('product_n_template')->where("product_id", $product->id)
                ->join("product_template_sku", "product_template_sku.template_id", "=", "product_n_template.template_id")
                ->where("is_default", 1)
                ->first(["product_template_sku.id"]);
            if ($template) {
                $keyCache .= "-" . $template->id;
            } else {
                $items = ProductSkuValue::where('sku_id', $product->sku_id)
                    ->orderBy("variant_option_id", "DESC")
                    ->get(['sku_id', 'product_id', 'variant_id', 'variant_option_id'])
                    ->toArray();
                foreach ($items as $key => $sku) {
                    $keyCache .= "-" . $sku['variant_option_id'];
                }
            }
        }
        if (Cache::has($keyCache)) {
            $result = Cache::get($keyCache);
            $response = [
                "status" => "successful",
                "standard" => $result,
            ];
        } else {
            $cartItem = new \stdClass();
            $cartItem->product_id = $product->id;
            $cartItem->category_id = $categoryId;
            $cartItem->product_sku_id = isset($product->sku_id) ? $product->sku_id : null;
            $cartItem->quantity = 1;
            $cartItem->price = $product->price;
            $cartItem->id = 1;
            $shippingInfos = [];
            $shippingInfos = $this->getShippingByItemV2([$cartItem], $countryId);
            $result = [];
            if (count($shippingInfos) > 0) {
                $shippingInfo = $shippingInfos[0];
                $shippingFee =  $this->getShippingInfoByItems($cartItem, $shippingInfo, $categoryId);
                $result["shipping_fee"] = $shippingFee["shipping_fee"];
                $result["handling_min_time"] = $shippingFee["handling_min_time"];
                $result["handling_max_time"] = $shippingFee["handling_max_time"];
                $result["delivery_min_time"] = $shippingFee["delivery_min_time"];
                $result["delivery_max_time"] = $shippingFee["delivery_max_time"];
                $response = [
                    "status" => "successful",
                    "standard" => $result,
                ];
                if (isset($product->sku_id)) {
                    Cache::put($keyCache, $result, 60 * 60);
                }
            }
        }
        return $response;
    }

    public function getShippingInfoByItems($cartItem, $shippingInfo, $categoryId) {
        $warehouseConfig = $shippingInfo["warehouseConfig"];
        $warehouseConfigItems = WarehouseConfigItem::where("warehouse_config_id", $warehouseConfig->id)->get();
        $defaultItem = [];
        foreach ($warehouseConfigItems as $typeItem) {
            if ($typeItem->type == "default") {
                $defaultItem = $typeItem;
            }
        }
        $cateGroups = [];
        $cateGroups[$categoryId] = [
            'quantity' => $cartItem->quantity,
            'category' => isset($category) && isset($categoryId) ? $categoryId : "",
            'cart_item_id' => $cartItem->id,
            'amount' => $cartItem->quantity * $cartItem->price
        ];

        $totalShippingFee = 0;
        $minShippingTime = 100000;
        $maxShippingTime = 0;
        $maxHandlingTime = 0;
        $minHandlingTime = 100000;
        $useDefault = false;
        $cateByConfigItem = [];
        $applyConfigItem = [];
        foreach ($cateGroups as $key => $cateGroup) {
            $applyItem = [];
            foreach ($warehouseConfigItems as $configItem) {
                $applyCate = [];
                if ($configItem->apply_cate) {
                    $applyCate = explode(',', $configItem->apply_cate);
                }
                if (in_array($key, $applyCate)) {
                    $applyItem = $configItem;
                }
            }
            if (!isset($applyItem->id)) {
                $applyItem = $defaultItem;
                $useDefault = true;
            }
            if (!array_key_exists($applyItem->id, $applyConfigItem)) {
                $applyConfigItem[$applyItem->id] = $applyItem;
            }
            if (!array_key_exists($applyItem->id, $cateByConfigItem)) {
                $cateByConfigItem[$applyItem->id] = [
                    'quantity' => $cateGroup['quantity'],
                    'amount' => $cateGroup['amount'],
                    'apply_item' => $applyItem,
                    'use_default' => $useDefault
                ];
            } else {
                $cateByConfigItem[$applyItem->id]['amount'] += $cateGroup['amount'];
                $cateByConfigItem[$applyItem->id]['quantity'] += $cateGroup['quantity'];
            }
        }
        foreach ($cateByConfigItem as $item) {
            $shippingFee = 0;
            $useDefault = $item['use_default'];
            $applyItem = $item['apply_item'];
            if (isset($applyItem->id)) {
                if ($applyItem->shipping_min_time && $applyItem->shipping_min_time > 0 && $minShippingTime > $applyItem->shipping_min_time) {
                    $minShippingTime = $applyItem->shipping_min_time;
                }
                if ($applyItem->shipping_max_time && $applyItem->shipping_max_time && $maxShippingTime < $applyItem->shipping_max_time) {
                    $maxShippingTime = $applyItem->shipping_max_time;
                }
                if ($applyItem->handling_min_time && $applyItem->handling_min_time > 0 && $minHandlingTime > $applyItem->handling_min_time) {
                    $minHandlingTime = $applyItem->handling_min_time;
                }
                if ($applyItem->handling_max_time && $applyItem->handling_max_time && $maxHandlingTime < $applyItem->handling_max_time) {
                    $maxHandlingTime = $applyItem->handling_max_time;
                }
                if ($applyItem->fee_limit && doubleval($applyItem->fee_limit) > 0 && $item["amount"] >= doubleval($applyItem->fee_limit)) {
                    $shippingFee += doubleval($applyItem->fee_if_limit);
                } else {
                    $shippingFee = doubleval($applyItem->default_shipping_fee);
                    if ($item["quantity"] > 1) {
                        $addingItem = [];
                        if ($applyItem->adding_item) {
                            $addingItem = explode(',', $applyItem->adding_item);
                        }
                        if (count($addingItem) > 0) {
                            $countAddingItem = count($addingItem);
                            for ($i = 0; $i < $item["quantity"] - 1; $i ++) { 
                                $shippingFee += doubleval($addingItem[$i % $countAddingItem]);
                            }
                        } else {
                            $shippingFee += ($item["quantity"] - 1) * doubleval($applyItem->default_adding_item);
                        }
                    }
                }
            }
            $totalShippingFee += round($shippingFee, 2);
        }
        if ($useDefault || $minShippingTime == 100000) {
            if (isset($defaultItem->shipping_min_time) && $minShippingTime > $defaultItem->shipping_min_time) {
                $minShippingTime = $defaultItem->shipping_min_time;
            }
        }
        if ($useDefault || $maxShippingTime == 0) {
            if (isset($defaultItem->shipping_max_time) && $maxShippingTime < $defaultItem->shipping_max_time) {
                $maxShippingTime = $defaultItem->shipping_max_time;
            }
        }
        if ($useDefault || $minHandlingTime == 100000) {
            if (isset($defaultItem->handling_min_time) && $minHandlingTime > $defaultItem->handling_min_time) {
                $minHandlingTime = $defaultItem->handling_min_time;
            }
        }
        if ($useDefault || $maxHandlingTime == 0) {
            if (isset($defaultItem->handling_max_time) && $maxHandlingTime < $defaultItem->handling_max_time) {
                $maxHandlingTime = $defaultItem->handling_max_time;
            }
        }
        $result = [
            "shipping_fee" => $totalShippingFee,
            "shipping_min_time" => $minShippingTime + ($minHandlingTime != 100000 ? $minHandlingTime : 0),
            "shipping_max_time" => $maxShippingTime + $maxHandlingTime,
            "handling_min_time" => $minHandlingTime,
            "handling_max_time" => $maxHandlingTime,
            "delivery_min_time" => $minShippingTime,
            "delivery_max_time" => $maxShippingTime,
        ];
        return $result;
    }
}
