<?php

namespace Modules\ZSearch\Controllers;

use Theme;
use Module;
use App\Utils;
use App\Utils\SlugManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Services\ElasticSearchService;
use App\Helpers\ApiClient;
use Carbon\Carbon;
use Modules\ZSearch\Services\ElasticSearchIndex;
use Modules\ZSearch\Services\ElasticSearchIndexCategory;
use Modules\ZSearch\Services\ElasticsearchIndexHistory;
use Modules\ZSearch\Services\CategoryHelper;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Models\Category;
use Illuminate\Support\Facades\Auth;
use Modules\ZSearch\Models\FilterValue;
use Modules\ZSearch\Models\ProductVariant;

class CategoryController extends Controller
{
    const CACHE_PRODUCT_FILTER = 'cache_product_n_category_top_filter::v3';
    const CACHE_PRODUCT_RESULT = 'cache_product_n_category_top_result::v3';
    const CHILD_CATEGORY_IDS = 'child_category_ids::';
    const CACHE_MINUTES = 20;
    protected $elasticService;
    protected $elasticsearchIndexCategory;
    protected $elasticSearchConfig;
    protected $filterRequest;
    protected $elasticServiceConfig;
    protected $elasticSearchIndex;
    private $type = 'product_n_category_top';
    private $configIndexKey = 'category_index';
    private $option = [
        'Size' => 1,
        'Color' => 2,
        'Type' => 5
    ];
    protected $exceptionKeyword = [
        'motley crue poster',
        'motley crue'
    ];
    const TYPES_ENABLE = [SlugManager::TYPE_TYPE, SlugManager::TYPE_COLOR];
    
    public function __construct(ElasticSearchIndexCategory $elasticsearchIndexCategory, ElasticSearchIndex $elasticSearchIndex)
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticServiceConfig = $elasticSearchConfig;
        $this->elasticService =  new ElasticSearchService($this->elasticSearchConfig);
        $this->elasticsearchIndexCategory = $elasticsearchIndexCategory;
        $this->elasticSearchIndex = $elasticSearchIndex;
    }

    /**
     * create index type
     */
    public function createType()
    {
        $result = $this->elasticService->createIndexType($this->elasticSearchConfig[$this->configIndexKey], $this->type);

        return response()->json([
            'status' => 'successful',
            'result' => $result
        ]);
    }

    /**
     * reindex data
     */
    public function reindex()
    {
        set_time_limit(3600 * 5);
        $result = $this->elasticService->reIndex($this->elasticSearchConfig['index'], $this->type, $this->elasticSearchConfig[$this->configIndexKey], $this->type);

        return response()->json([
            'status' => 'successful',
            'result' => $result
        ]);
    }

    /**
     * build product stat table
     * 
     * @return Response
     */
    public function buildProductStat()
    {
        set_time_limit(3600);
        $result = [];
        $data = $this->elasticSearchIndex->getProductsStat();

        try {
            DB::table('product_stat')->truncate();
            foreach (array_chunk($data, 200) as $chunkItem) {
                DB::table('product_stat')->insert($chunkItem);
            }

            $result['status'] = 'successful';
        } catch (\Exception $e) {
            $result['status'] = 'fail';
            $result['error'] = $e;
        }

        return response()->json($result);
    }

    /**
     * delete old document
     * 
     * @param Int $days
     * @return Response
     */
    private function deleteOldDocument($days = 7)
    {
        $result = null;
        $currentDate = Carbon::now();
        $thresholdDate = $currentDate->subDays($days)->format('Y-m-d');
        $query = [
            '_source' => false,
            'query' => [
                'bool' => [
                    'must' => [
                        [
                            'range' => [
                                'updated_at' => [
                                    'lt' => $thresholdDate
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];
        $total = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig[$this->configIndexKey])['hits']['total'];
        $documents = [];
        for ($i = 0; $i < $total; $i += 1000) {
            $query['size'] = 1000;
            $query['from'] = $i;
            $response = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig[$this->configIndexKey]);
            $documents = array_merge($documents, $response['hits']['hits']);
        }

        try {
            // delete documents
            foreach ($documents as $item) {
                $this->elasticService->deleteDocument($item['_id'], $this->type, $this->elasticSearchConfig[$this->configIndexKey]);
            }

            $result = ['status' => 'successful'];
        } catch (\Exception $e) {
            $result = $e->getMessage();
        }

        return $result;
    }

    /**
     * create index type
     * 
     * @param Request $request
     * @return Response
     */
    public function createIndexType(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '2G');
        $elasticsearch =  Config::get('z-search::elasticsearch');
        $isAll = $request->input('isAll', 0);
        $isCreateIndex = $request->input('create', 0);
        $deleteHidden = $request->input('delete_all_hidden', 0);
        $lastMin = $request->input('min', 33);
        $updatedFrom = date('Y-m-d H:i:s', time() - $lastMin * 60);
        $categoryId =  $request->input('category_id', 0);
        $startTime = time();
        if ($isCreateIndex) {
            $isAll = 1;
            //$this->elasticService->createIndex($elasticsearch['index']);
        }
        $filter = [
            'is_all' => $isAll,
            'from_id' => $request->input('from_id', -1),
            'to_id' => $request->input('to_id', -1),
            'delete_hidden' => $deleteHidden,
            'updated_from' => $updatedFrom,
            'category_id' => $categoryId,
        ];
        $this->elasticsearchIndexCategory->initMetaData();
        $result = $this->elasticsearchIndexCategory->multiIndexProductNCategory($filter, $elasticsearch);
        $result['status'] = 'successful';
        if ($isAll && !$categoryId) {
            $result['delete_result'] =  $this->deleteOldDocument();
        }
        ElasticsearchIndexHistory::logHistory($startTime, [
            'filter' => $filter,
            'result' => $result,
        ]);

        return response()->json($result);
    }

    /**
     * index new product
     * 
     * @return Response
     */
    public function indexNewProduct()
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '5G');
        $elasticsearch =  Config::get('z-search::elasticsearch');
        $startTime = time();
        $filter = [];

        $this->elasticsearchIndexCategory->initMetaData();
        $result = $this->elasticsearchIndexCategory->indexNewProduct($elasticsearch);
        $result['status'] = 'successful';
        ElasticsearchIndexHistory::logHistory($startTime, [
            'filter' => $filter,
            'result' => $result,
        ]);

        return response()->json($result);
    }

    /**
     * top 10 product
     */
    private function getTopProductCategory($categoryId)
    {
        $topProductIds = $this->getTopProductIds($categoryId);

        $topProductInformation = DB::table('product')
            ->whereIn('id', $topProductIds)
            ->get();

        return $topProductInformation;
    }

    /**
     * top 10 productIds
     */
    protected function getTopProductIds($categoryId, $limit = 10)
    {
        $topProductIds = DB::table('product_n_category_top')
            ->where('category_id', $categoryId)
            ->orderBy('sorder', 'desc')
            ->limit($limit)
            ->pluck('product_id');

        return $topProductIds;
    }

    /**
     * search category
     * 
     * @param Request $request
     * @return Response
     */
    public function searchCategory(Request $request)
    {
        $data = $request->all();
        $data['page_size'] = 60;
        $category = null;
        $this->initTimer();
        $this->cpTimer('init');
        $paramsData = CategoryHelper::parseParamsZSearch();
        $categoryId = isset($paramsData['filterParams']['category_id']) ? $paramsData['filterParams']['category_id'] : null;
        if (!$categoryId) {
            if(isset($paramsData['redirect']) && $paramsData['redirect']) {
                return redirect($paramsData['redirect'], 301);
            }
            abort(404);
        }

        if ($request->input('seller') && $request->input('seller') == 2614 && getLocale() == 'jp') {
            abort(404);
        }

        $topProductCategory = $this->getTopProductCategory($categoryId);

        // check is empty category
        $isEmptyCategory = $this->checkEmptyCategory($categoryId);
        if ($isEmptyCategory == true) {
            abort(404);
        }

        $data = array_merge($paramsData['filterParams'], $data);
        $metaDataConfig = null;
        if (\Module::isActive('Seo') && function_exists('getMetaDataConfig')) {
            $metaDataConfig = getMetaDataConfig();
        }
        $productFilters = $data;
        $this->cpTimer('before_category');
        $category = ApiClient::buildCustomRequest("category/" . $categoryId, 'GET', [], [])["result"];
        $this->cpTimer('after_category');
        $shareProduct = [];
        if ($category) {
            $categoryClone = $category;
            if ($metaDataConfig && $metaDataConfig->content_footer) {
                $categoryClone['faqs'] = $this->extractQuestionsAndAnswers($metaDataConfig->content_footer);
                $categoryClone['description'] = $this->formatFaq($metaDataConfig->content_footer);
            } else {
                $categoryClone['faqs'] = $this->extractQuestionsAndAnswers($category['description']);
                $categoryClone['description'] = $this->formatFaq($category['description']);
            }
            \View::share('raw_category', $categoryClone);
            $shareProduct['images'] = $categoryClone['image_url'];
            $shareProduct['brand'] = env('APP_NAME');
            $shareProduct['lowPrice'] = 0;
            $shareProduct['highPrice'] = 0;
            $shareProduct['offerCount'] = intval(time() / 100000);
            $shareProduct['ratingValue'] = (!empty($categoryClone['rating_value'])) ? $categoryClone['rating_value'] : 5;
            $shareProduct['reviewCount'] = (!empty($categoryClone['rating_count'])) ? $categoryClone['rating_count'] : $category['id'];
        }
        $breadcrumbs = [];
        if (isset($category['breadcrumb'])) {
            $breadcrumbs = CategoryHelper::decorAttrBreadcrumbZSearch(json_decode($category["breadcrumb"]), $paramsData);
            $shareBreadcrumb = array_map(function ($item) {
                return ['id' => $item->url, 'name' => $item->name];
            }, $breadcrumbs);
            if (!empty($shareBreadcrumb)) {
                $shareBreadcrumb[0]['name'] = env('APP_NAME');
            }
            \View::share('breadcrumbs', $breadcrumbs);
            \View::share('shareBreadcrumb', $shareBreadcrumb);
            $parseUrl = parse_url(end($shareBreadcrumb)['id']);
            $hreflangs = [];
            $currentLocale = env('APP_LOCALE');
            foreach (getModuleLocale() as $item) {
                if (!$item['enable']) continue;

                if (empty($currentLocale) || $currentLocale == 'us') {
                    if ($item['locale'] == 'us') {
                        $hreflangs[$item['locale']] = $parseUrl['path'];
                    } else {
                        $hreflangs[$item['locale']] = '/' . $item['locale'] . $parseUrl['path'];
                    }
                } else {
                    if ($item['locale'] == 'us') {
                        $hreflangs[$item['locale']] = ltrim($parseUrl['path'], '/' . $currentLocale);
                    } else {
                        $hreflangs[$item['locale']] = '/' . $item['locale'] . '/' . ltrim($parseUrl['path'], '/' . $currentLocale);
                    }
                }
            }
            \View::share('hreflangs', $hreflangs);
        }
        $isCategoryPage = true;
        if (Theme::getSetting('prerender_search', true)) {
            if (Theme::getSetting('show_pending_product', false)) {
                $data['status'] = [
                    'ACTIVE',
                    'PENDING'
                ];
            }
            if (!empty($request->header('tenancy_token'))) {
                $seller = User::where('seller_token', $request->header('tenancy_token'))->first();
                \View::share('is_tenancy', 1);
                $data['user_id'] = $seller->id;
            } else if (isset($data['seller'])) {
                $data['user_id'] = $data['seller'];
            }
            if (isset($data['size'])) {
                $data['size_variant_id'] = $data['size'];
            }
            if (isset($data['price'])) {
                $priceExplode = explode('-', $data['price']);
                $data['minPrice'] = $priceExplode[0];
                $data['maxPrice'] = $priceExplode[1];
            }
            $this->cpTimer('before_category-filter');
            $key = $this->buildKeyFromFilter($data);


            $query = $this->prepareQueryProductByCategory($data);
            $response = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig[$this->configIndexKey]);
            $result = $this->formatResponse($response, $data, $category);

//            $result = null;
//            if ($key) {
//                if (Cache::has($key)) {
//                    $result = Cache::get($key);
//                }
//
//                if (isset($_GET['clear_cache'])) {
//                    $result = null;
//                }
//            }
//            if (!$result || (isset($result['result']) && count($result['result']) == 0)) {
//                $query = $this->prepareQueryProductByCategory($data);
//                $response = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
//                $result = $this->formatResponse($response, $data, $category);
//                if (!isset($data['page_id']) || in_array($data['page_id'], [1, 2])) {
//                    Cache::put($key, $result, 10);
//                }
//            }
            // check over page count
            // if (isset($data['page_id']) && $data['page_id'] > floor($response['hits']['total'] / 60)) {
            //     $queryParams = $request->all();
            //     unset($queryParams['page_id']);

            //     return redirect($request->url() . '?' . http_build_query($queryParams), 301);
            // }
            $this->cpTimer('after_category-filter');
            if (!empty($paramId) && is_numeric($paramId) && isset($data['id'])) {
                unset($data['id']);
            }
            $products = [];
            $meta = [
                'page_id' => 0,
                'page_size' => 0,
                'page_count' => 0,
                'has_next' => false,
                'total_count' => 0,
            ];
            $tags = [];
            $searchCategories = [];
            $displayedFilters = [];
            $priceRange = [];
            $filterCategory = [];
            $users = [];
            $filterOptions = [];
            if (isset($result['status'])) {
                $products = $result['result'];
                $this->cpTimer('b_priceChangeDecor');
                priceChangeDecor($products);
                $this->cpTimer('a_priceChangeDecor');
                if (function_exists('checkDuplicateTitle')) {
                    checkDuplicateTitle($products);
                }
                if (!isset($data['order']) && !isset($data['category_id'] ) && !empty($data['q'])){
                    array_multisort(array_column($products, 'inventory'), SORT_DESC,
                                    array_column($products, 'lien_quan'),      SORT_DESC,
                                $products);
                }
                $meta = $result['meta'];
                $tags = $result['tags'];
                if (function_exists('getTagHot')) {
                    $actual_link = "$_SERVER[REQUEST_URI]";
                    $tagsConfig = getTagHot($actual_link);
                    $tags = array_merge($tagsConfig, $tags);
                }
                $searchCategories = $result['categories'];
                $priceRange = $result['priceRange'];
                if (!empty($priceRange)) {
                    $shareProduct['lowPrice'] = $priceRange[0]['to'];
                    $shareProduct['highPrice'] = $priceRange[count($priceRange) - 1]['to'];
                } else if (isset($products[0]['price'])) {
                    $shareProduct['lowPrice'] = $products[0]['price'];
                    $shareProduct['highPrice'] = $products[0]['price'];
                }
                $displayedFilters = $result['displayedFilters'];
                $filterCategory = $result['filterCategory'];

                if (array_key_exists("users", $result) && empty($request->header('tenancy_token'))) {
                    $users = $result['users'];
                }
                if (array_key_exists("filterOptions", $result)) {
                    $filterOptions = $result['filterOptions'];
                }
            }

            switch ($request->route()->getName()) {
                case 'category':
                case 'category-list':
                case 'categoryOrTag':
                    View::share('cloneCategory', $category);
                    break;
                case 'brand':
                    View::share('brand', ['name' => __("Thương hiệu")]);
                    break;

                default:
                    if ($request->has('q')) {
                        View::share('query', strip_tags($request->input('q')));
                        $keyword = $request->input('q');
                    }
                    break;
            }
            if (isset($productFilters['category_id'])) {
                if (!empty($productFilters['category']) && $productFilters['category'][0] == $productFilters['category_id']) {
                    unset($displayedFilters['category']);
                }
                unset($productFilters['category_id']);
            }

            $allFilters = [];
            $filtered = [];
            $tempFilters = [];
            $routeParameters = Route::current()->parameters();
            $this->cpTimer('displayedFilters1');
            foreach ($displayedFilters as $key => $value) {
                if(!is_array($value)) {
                    $attrItem = [
                        'type' => $key,
                        'slug' => str_slug($value)
                    ];

                    $tempFilter = [
                        'url' => SlugManager::clearFilterUrl($attrItem),
                        'checked' => true,
                        'text' => ucfirst($value),
                        'type' => $key
                    ];
                    $locale = env('APP_LOCALE', '');
                    if ($locale) {
                        $locale = '/' . $locale;
                    }
                    $checkUrl = str_replace($locale, '', $tempFilter['url']);
                    $arrCheck = explode('?', $checkUrl);
                    if (count($arrCheck) > 0 && $arrCheck[0] && $arrCheck[0] != '/') {
                        $tempFilters[] = $tempFilter;
                    }
                } else {
                    if (count($value)) {
                        $tempFilter = [
                            'url' => buildDisplayedFilterSlugUrl($routeParameters, $key, $value[0]),
                            'checked' => true,
                            'text' => ucfirst($value[0]),
                            'type' => $key
                        ];
                        $locale = env('APP_LOCALE', '');
                        if ($locale) {
                            $locale = '/' . $locale;
                        }
                        $checkUrl = str_replace($locale, '', $tempFilter['url']);
                        $arrCheck = explode('?', $checkUrl);
                        if (count($arrCheck) > 0 && $arrCheck[0] && $arrCheck[0] != '/') {
                            $tempFilters[] = $tempFilter;
                        }
                    }
                }
            }
            $this->cpTimer('displayedFilters2');
            $cloneTempFilters = [];
            foreach ($tempFilters as $tempFilter) {
                if (!empty($tempFilter['type']) && $tempFilter['type'] == 'type_variant_id') {
                    array_unshift($cloneTempFilters, $tempFilter);
                } else {
                    $cloneTempFilters[] = $tempFilter;
                }
            }
            $this->cpTimer('displayedFilters4');
            $tempFilters = $cloneTempFilters;
            $filtered = [
                'title' => __("Bộ lọc"),
                'id' => "filter",
                'filters' => $tempFilters,
                'text_filtered' => $this->getTextFiltered($tempFilters)
            ];
            $allFilters = $this->getAllFiltersCategory($paramsData, $searchCategories, $priceRange, $users, $filterOptions);
            $sorter = [
                [
                    'name' => __("Mua nhiều"),
                    'value' => 'sold',
                ],
                [
                    'name' => __("Mới nhất"),
                    'value' => 'lastest',
                ],
                [
                    'name' => __("Xem nhiều"),
                    'value' => 'view',
                ],
                [
                    'name' => __("Giá thấp"),
                    'value' => 'low_price',
                ],
                [
                    'name' => __("Giá cao"),
                    'value' => 'high_price',
                ],
                [
                    'name' => __("Giảm giá"),
                    'value' => 'sale',
                ],
            ];
            foreach ($sorter as $key => $item) {
                $attr = [
                    'type' => 'order',
                    'slug' => $item['value']
                ];
                $sorter[$key]['url'] = $this->urlFilterZSearch($paramsData, $attr);
                $sorter[$key]['checked'] = $this->isSelectedFilter($paramsData, $attr);
            }
            View::share('sorter', $sorter);
            $this->cpTimer('displayedFilters4');
            $allFilters = allFiltersOrdered($allFilters);
            View::share('allFilters', $allFilters);
            $this->cpTimer('displayedFilters5');
            if (!empty($filterCategory) && !empty($filterCategory['category_metas'])) {
                foreach ($filterCategory['category_metas'] as $item) {
                    if ($item['key'] == 'meta_description') {
                        View::share('metaDescription', $item['value']);
                    }
                    if ($item['key'] == 'meta_title') {
                        View::share('metaTitle', $item['value']);
                    }
                }
            }
            if ($metaDataConfig && $metaDataConfig->content) {
                View::share('metaDescription', $metaDataConfig->content);
            }


            View::share('category', $filterCategory);
            $strCodes = "";
            $shareItemList = [];
            $this->cpTimer('b_checkNofollowImageUrl');
            foreach ($products as $key => $item) {
                $strCodes .= $item['sku'].',';
                $products[$key]['isNofollow'] = checkNofollowImageUrl($item['image_url']);
                array_push($shareItemList, ['url' => url(clroute($item['url']))]);
            }
            $this->cpTimer('a_checkNofollowImageUrl');
            View::share('shareItemList', $shareItemList);
            View::share('shareProduct', $shareProduct);
            View::share('strCode', substr($strCodes, 0, -1));
            View::share('pageType', 'view_item_list');
            if ($request->route('id')) {
                Module::action('viewPage', [
                    'type' => 'category',
                    'data' => [
                        'product_ids' => array_column($products, 'id'),
                        'post_ids' => [],
                        'id' => $request->route('id'),
                    ]
                ]);
            } else {
                Module::action('viewPage', [
                    'type' => 'search',
                    'data' => [
                        'product_ids' => array_column($products, 'id'),
                        'post_ids' => [],
                        'id' => null,
                    ]
                ]);
            }
            View::share('filtered', $filtered);
            if ($products) {
                $this->decorProductUrl($products);
            }

            $token = getFingerprint();
            $this->cpTimer('b_wishlist');
            $wishlistItems = ApiClient::buildCustomRequest("wishlist/get-wishlist-items", "GET", ['customerToken' => $token] , []);
            $this->cpTimer('a_wishlist');
            $wishProductIds = isset($wishlistItems['wishlistItems']) ? array_column($wishlistItems['wishlistItems'], 'product_id') : [];
            View::share('wishProductIds', $wishProductIds);
            $freeShippingCate = getFreeShippingCate();
            $this->cpTimer('a_getFreeShippingCate');
            \View::share("freeShippingCate", $freeShippingCate);
            if (isset($category) && isset($category['id']) && in_array($category["id"], $freeShippingCate)) {
                foreach ($products as $key => $product) {
                    $products[$key]["is_free_shipping"] = 1;
                }
            }
            if (isset($data['page_type']) && $data['page_type'] == 'category') {
                if (isset($data['category_slug']) && $products) {
                    $this->decorProductTracking($products, $data['category_slug']);
                }
                View::share('isCatePage', true);
            }
            $this->cpTimer('a_decorProductTracking');
            $status = 200;
            if (!$products || count($products) == 0) {
                $status = 404;
                \Log::info('ZSEARCH_CATEGORY_SEARCH', [$request->fullUrl()]);
            }
            View::share('criteoProducts', array_slice($products, 0, 3));
            View::share('searchCategories', $searchCategories);
            View::share('displayedFilters', $displayedFilters);
            View::share('meta', $meta);
            $this->cpTimer('end');
            $this->persistenceTimer();
            return response()->view('search', compact(
                'products',
                'meta',
                'tags',
                'searchCategories',
                'productFilters',
                'priceRange',
                'filterCategory',
                'filtered',
                'isCategoryPage',
                'topProductCategory'
            ))->setStatusCode($status);
        } else {
            return view('search', compact('productFilters','isCategoryPage', 'topProductCategory'));
        }
    }

    private function getDisplayedFilters($filters, array $objects = [])
    {
        $result = [];
        if (isset($filters['minPrice']) || isset($filters['maxPrice'])) {
            $result['price'] = '';
        }
        if (isset($filters['minPrice']) && isset($filters['maxPrice'])) {
            $result['price'] = translate('From') . ' ' . formatPrice($filters['minPrice']) . ' ';
            $result['price'] .= translate('to') .' ' . formatPrice($filters['maxPrice']);
            unset($filters['minPrice']);
            unset($filters['maxPrice']);
        } elseif (isset($filters['minPrice']) && !isset($filters['maxPrice'])) {
            $result['price'] = translate('From') . ' ' . formatPrice($filters['minPrice']) . ' ';
            unset($filters['minPrice']);
        } elseif (!isset($filters['minPrice']) && isset($filters['maxPrice'])) {
            $result['price'] .= translate('Under') . ' ' . formatPrice($filters['maxPrice']);
            unset($filters['maxPrice']);
        }
        foreach ($filters as $key => $value) {
            if (is_array($value)) {
                $result[$key] = [];
                if (isset($objects[$key])) {
                    foreach ($value as $val) {
                        $table = $key;
                        if ($key == 'filter') {
                            $table = 'filter_value';
                        }
                        switch ($table) {
                            case 'tag':
                                $obj = DB::table($table)
                                    ->where('slug', $val)->first();
                                break;
                            case 'filter_value':
                                $obj = FilterValue::with('filter')
                                    ->where('id', $val)->first();
                                break;
                            case 'category':
                                if (count($objects[$key])) {
                                    $obj = DB::table($table)
                                        ->where('id', $val)->first();
                                }
                                break;
                            default:
                                $obj = DB::table($table)
                                    ->where('id', $val)->first();
                                break;
                        }

                        if (!empty($obj)) {
                            $display_name = isset($obj->name) ? $obj->name : (isset($obj->title) ? $obj->title : '');
                            array_push($result[$key], $display_name);
                            if (isset($obj->filter)) {
                                $result[$obj->filter->slug] = $display_name;
                            }
                        }
                    }
                }
            } else {
                if (isset($objects[$key])) {
                    foreach ($objects[$key] as $obj) {
                        if ($obj['slug'] == $value) {
                            $result[$key] = $obj['name'];
                            break;
                        }
                    }
                }
            }
        }
        return $result;
    }

    protected function getDisplayFilterVariant($data, $filterOptions) {
        $retval = [];
        $listIdVariant = [];
        foreach ($data as $key => $value) {
            if (strpos($key, 'variant_id') !== false) {
                $slug = explode("_variant_id", $key)[0];
                $option = ProductVariant::where("slug", $slug)->first();
                $listIdVariant[] = [
                    "slug" => $slug,
                    "id" => $value,
                    "name" => $option->name,
                ];
            }
        }
        foreach ($listIdVariant as $key => $value) {
            if (array_key_exists($value["name"], $filterOptions)) {
                foreach ($filterOptions[$value["name"]] as $index => $item) {        
                    if ($value["id"] == $item['id']) {
                        $retval[$value["slug"] . "_variant_id"] = $item['name'];
                    }
                }
            }
        }
        return $retval;
    }

    private function decorProductUrl(&$products) {
        foreach ($products as &$product) {
            if (!isset($product['url']) || str_contains($product['url'], '.com')) {
                continue;
            }
            $product['url'] = clroute($product['url']);
        }
    }

    private function decorProductTracking(&$products, $categorySlug) {
        foreach ($products as &$product) {
            if (!isset($product['url'])) {
                continue;
            }
            $char = str_contains($product['url'], '?') ? '&' : '?';
            $product['url'] = $product['url'] . $char . 'internal_source=category-' . $categorySlug;
        }
    }

    private function buildKeyFromFilter($filter) {
        $locale = env('APP_LOCALE', '');
        if (array_key_exists('ddebug', $filter)) {
            unset($filter['ddebug']);
        }
        $key = self::CACHE_PRODUCT_RESULT . $locale . md5(serialize($filter)) . 'v2';
        return $key;
    }

    private function buildFilterKey($categoryId, $meta) {
        $locale = env('APP_LOCALE', '');
        $key = self::CACHE_PRODUCT_FILTER . $locale . $categoryId . 'v3';
        if (isset($meta['seller'])) {
            $key .= '::seller' . $meta['seller'];
        }
        if (isset($meta['type_variant_id'])) {
            $key .= '::type' . $meta['type_variant_id'];
        }
        if (isset($meta['color_variant_id'])) {
            $key .= '::color' . $meta['color_variant_id'];
        }
        if (isset($meta['size_variant_id'])) {
            $key .= '::size' . $meta['size_variant_id'];
        }
        if (isset($meta['price'])) {
            $key .= '::price' . $meta['price'];
        }
        return $key;
    }

    /**
     * format option filter
     * 
     * @param Array $data
     * @param Int $type
     * @return Array
     */
    private function formatOptionFilter($data, $type)
    {
        $options = [];
        if ($type == $this->option['Color']) {
            foreach ($data as $dataItem) {
                foreach ($dataItem['source']['hits']['hits'][0]['_source']['variant_options'][$type]['options'] as $subItem) {
                    if ($subItem['name'] == $dataItem['key']) {
                        $options[] = $subItem['id'];
                    }
                }
            }

            $options = DB::table('product_variant_option')->whereIn('id', $options)->get(['id', 'image_url'])->keyby('id')->toArray();
        }

        $result = array_map(function($item) use ($type, $options) {
            foreach ($item['source']['hits']['hits'][0]['_source']['variant_options'][$type]['options'] as $subItem) {
                if ($subItem['name'] == $item['key']) {
                    $subItem['count'] = $item['doc_count'];
                    $subItem['variant_id'] = $type;
                    $subItem['variant_name'] = $item['source']['hits']['hits'][0]['_source']['variant_options'][$type]['variant_name']['name'];
                    $subItem['variant_slug'] = $item['source']['hits']['hits'][0]['_source']['variant_options'][$type]['variant_name']['slug'];
                    if ($type == $this->option['Color'] && !empty($options) && isset($options[$subItem['id']])) {
                        $subItem['image_url'] = $options[$subItem['id']]->image_url;
                    }
                    return $subItem;
                }
            }
        }, $data);

        return $result;
    }

    /**
     * format user filter
     * 
     * @param Array $data
     * @return Array
     */
    private function formatUserFilter($data)
    {
        $result = array_map(function($item) {
            $user = $item['source']['hits']['hits'][0]['_source']['user'];
            $user['user_id'] = $user['id'];
            $user['count'] = $item['doc_count'];

            return $user;
        }, $data);

        return $result;
    }

    /**
     * format price filter
     * 
     * @param Array $metaData
     * @param Int $total
     * @param Int $maxPrice
     * @param Int $minPrice
     * @return Array
     */
    private function formatPriceFilter($metaData, $total, $maxPrice, $minPrice)
    {
        $pageSize = isset($metaData['page_size']) ? $metaData['page_size'] : 60;
        $retVal = [];
        if ($total > $pageSize) {
            $priceRange = [];
            $unit = 4;
            $range = (int)(($maxPrice - $minPrice) / $unit);
            $length = strlen((int) $range) - 1;
            $range = round($range, -$length, PHP_ROUND_HALF_DOWN);
            $priceDiffLimit = 5 * (strlen(intval($maxPrice)) - 1) * (ceil($maxPrice / 100));
            if ($maxPrice > 0 && $range > 0 && $maxPrice != $minPrice) {
                $priceRange = [
                    [
                        'from' => 0,
                        'to' => 0,
                    ]
                ];
    
                $max = 0;
    
                while ($max < $maxPrice) {
                    if ($max <= $minPrice) {
                        $max += $range;
                        $priceRange[0]['to'] = $max;
                    } else {
                        $priceRange[] = [
                            'from' => $max,
                            'to' => $max + $range
                        ];
                        $max += $range;
                    }
                }
            }
    
            foreach ($priceRange as $key => $value) {
                if ($value['to'] == 0) {
                    unset($priceRange[$key]);
                }
            }

            $lastPriceRange = null;
            foreach ($priceRange as $key => $value) {
                if (abs($value['to'] - $value['from']) > $priceDiffLimit) {
                    if ($lastPriceRange) {
                        $retVal[] = [
                            'from' => $lastPriceRange['to'],
                            'to' => $value['to'],
                        ];
                    } else {
                        $retVal[] = $value;
                    }
                    $lastPriceRange = $value;
                } else {
                    $retVal[] = $value;
                }
            }
        }

        return $retVal;
    }

    protected function initTimer() {
        $this->timer = [
            'start_time' => microtime(true)
        ];
        $this->currentTime = microtime(true);
    }

    protected function cpTimer($name) {
        if (isset($_GET['debug_timer'])) {
            $this->timer[$name] = number_format(microtime(true) -  $this->currentTime, 6);
            $this->currentTime = microtime(true);
        }

    }

    protected function persistenceTimer() {
        if (isset($_GET['debug_timer'])) {
            $this->timer['run_time'] =  number_format(microtime(true) -  $this->timer['start_time'], 6);
            \Log::info('persistence debug Timer', $this->timer);
        }

        if (isset($_GET['show_debug_timer'])) {
            echo '<pre>';
            echo print_r($this->timer);
            echo '</pre>';
            exit;
        }

    }

    private function getTextFiltered ($tempFilters) {
        $retVal = '';
        $listText = [];
        foreach ($tempFilters as $tempFilter) {
            $listText[] = $tempFilter['text'];
        }
        if (count($listText) > 0) {
            $retVal = implode(' ', $listText);
        }
        return $retVal;
    }

    protected function checkEmptyCategory($categoryId)
    {
        $check = \DB::table('empty_category')->where('category_id', $categoryId)->first();
        if ($check) {
            return true;
        }
    }

    private function extractQuestionsAndAnswers($html_content)
    {
        $results = [];
        $allowed_tags = '<h1><h2><h3><h4><h5><h6><br><ol><ul><li><a><p><div><b><strong><i><em>';

        $pattern = '/\[faq\]\[question\](.*?)\[\/question\]\[answer\](.*?)\[\/answer\]\[\/faq\]/s';
        preg_match_all($pattern, $html_content, $matches, PREG_SET_ORDER);
        if (!empty($matches)) {
            foreach ($matches as $item) {
                $question = strip_tags($item[1], $allowed_tags);
                $answer = strip_tags($item[2], $allowed_tags);
                $results[] = [
                    'question' => $question,
                    'answer' => $answer
                ];
            }
        }
        return $results;
    }

    private function formatFaq($content)
    {
        $svg = '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" class="bi bi-chevron-down" viewBox="0 0 16 16">
                                        <path fill-rule="evenodd" d="M1.646 4.646a.5.5 0 0 1 .708 0L8 10.293l5.646-5.647a.5.5 0 0 1 .708.708l-6 6a.5.5 0 0 1-.708 0l-6-6a.5.5 0 0 1 0-.708z"/>
                                    </svg>';
        return preg_replace_callback(
            '/\[faq\]\[question\](.*?)\[\/question\]\[answer\](.*?)\[\/answer\]\[\/faq\]/s',
            function ($matches) use ($svg) {
                $question = $matches[1];
                $answer = $matches[2];

                return '<div class="section-product-page product-faq-wrapper">' .
                    '<div class="post-faq-item">' .
                    '<div class="post-faq-question flex-b align-c flex-s">' . $question . $svg . '</div>' .
                    '<div class="post-faq-answer">' .
                    '<div class="post-faq-answer-wrap">' . $answer . '</div>' .
                    '</div>' .
                    '</div>' .
                    '</div>';
            },
            $content
        );
    }

    private function getAllFiltersCategory($attrs, $childCategories, $priceRange, $users = [], $filterOptions = []) {
        $tempFilters = [];
        $allFilters = [];
        if ($filterOptions) {
            foreach ($filterOptions as $key => $variants) {
                $tempFilters = [];
                foreach ($variants as $item) {
                    $itemFilter = [
                        'text' => ucfirst($item['name']),
                        'slug' => $item['slug']
                    ];
                    $attr = [
                        'id' => $item['id'],
                        'type' => $item['variant_slug'],
                        'slug' => $item['slug']
                    ];

                    $itemFilter['url'] = $this->urlFilterZSearch($attrs, $attr);
                    $itemFilter['checked'] = $this->isSelectedFilter($attrs, $attr);
                    $itemFilter['image_url'] = $item['image_url'] ?? '';
                    $itemFilter['theme_color'] = checkThemeVariantColor($item['slug']) ?? '';
                    $tempFilters[] = $itemFilter;
                }
                $allFilters[] = [
                    'title' => __(ucfirst($key)),
                    'type' => $key,
                    'filters' => $tempFilters,
                    'id' => 'user_id',
                ];
            }
        }

        if (isset($priceRange) && count($priceRange) > 1) {
            $tempFilters = [];
            foreach ($priceRange as $item) {
                $from = formatPrice($item['from']);
                $to = formatPrice($item['to']);
                $priceAttr = SlugManager::priceAttr($item['from'], $item['to']);
                $tempFilter = [
                    'url' => $this->urlFilterZSearch($attrs, $priceAttr),
                    'checked' => $this->isSelectedFilter($attrs, $priceAttr),
                    'text' => $item['from']
                        ? __("Từ") . " $from -  $to "
                        : __("Dưới") ." $to"
                ];
                $tempFilters[] = $tempFilter;
            }
            View::share('priceRanges', $tempFilters);
            $allFilters[] = [
                'title' => __("Giá"),
                'id' => 'price',
                'filters' => $tempFilters
            ];
        }

        if ($users) {
            $tempFilters = [];
            foreach ($users as $item) {
                $attr = [
                    'id' => $item['id'],
                    'type' => 'seller',
                    'slug' => $item['slug']
                ];
                $itemFilter = [
                    'id' => $item['id'],
                    'text' => ucfirst($item['name']),
                    'count' => isset($item['count']) ? $item['count'] : false,
                ];
                $itemFilter['url'] = $this->urlFilterZSearch($attrs, $attr);
                $itemFilter['checked'] = $this->isSelectedFilter($attrs, $attr);
                $tempFilters[] = $itemFilter;
            }
            $allFilters[] = [
                'title' => __("Seller"),
                'filters' => $tempFilters,
                'id' => 'user_id',
            ];
        }

        return $allFilters;
    }

    public static function isSelectedFilter($attrs, $attr)
    {
        $pathInfo = $attrs['path_info'];
        $result = false;
        if (isset($attr['type'])
            && $attr['type'] == SlugManager::TYPE_CATEGORY
            && in_array($attr['slug'], $pathInfo['categories'])) {
            $result = true;
        } else if (isset($pathInfo['attrs'][$attr['type']])
            && $pathInfo['attrs'][$attr['type']] == $attr['slug']
        ) {
            $result = true;
        } else if (isset($attrs['query'][$attr['type']])
            && ($attrs['query'][$attr['type']] == $attr['slug']
                || (isset($attr['id']) && $attrs['query'][$attr['type']] == $attr['id'])
            )
        ) {
            $result = true;
        }
        return $result;
    }

    public static function urlFilterZSearch($attrs, $attr)
    {
        $pathInfo = $attrs['path_info'];
        if ($attr['type'] == SlugManager::TYPE_CATEGORY) {
            if (count($pathInfo['categories']) == 4) {
                $pathInfo['categories'][3] = $attr['slug'];
            } else {
                $pathInfo['categories'][] = $attr['slug'];
            }
        } else if (in_array($attr['type'], self::TYPES_ENABLE)) {
            $pathInfo['attrs'][$attr['type']] = $attr['slug'];
        } else {
            $attrs['query'][$attr['type']] = isset($attr['id']) ? $attr['id'] : $attr['slug'];
        }
        $paths = self::buildPath($pathInfo);
        $params = self::buildRouteParams($paths);
        //@todo param query
        $result = route('category-list', $params);
        if ($attrs['query']) {
            $result .= '?' . http_build_query($attrs['query']);
        }
        return $result;

    }

    protected static function buildPath($pathInfo)
    {
        $result = $pathInfo['categories'];
        foreach (self::TYPES_ENABLE as $type) {
            if (isset($pathInfo['attrs'][$type])) {
                $result[] = $pathInfo['attrs'][$type];
            }
        }
        return $result;
    }

    protected static function buildRouteParams($paths)
    {
        $paramsName = ['slug', 'slug2', 'slug3', 'slug4', 'slug5', 'slug6'];
        $result = [];
        foreach ($paramsName as $key => $param) {
            if (!isset($paths[$key])) {
                break;
            }
            $result[$param] = $paths[$key];
        }
        return $result;
    }

    /**
     * prepare query product by category
     * 
     * @param Array $data
     * @return Array
     */
    private function prepareQueryProductByCategory($data)
    {
        $query['size'] = $data['page_size'];
        if (isset($data['page_id'])) {
            $query['from'] = $data['page_id'] * $data['page_size'];
        }
        $childCategoryIds = $this->getChildCategoryIds($data['category_id']);
        $childCategoryIds = array_values(array_filter($childCategoryIds, function ($item) {
            return !is_null($item);
        }));
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'terms' => [
                            'category_id' => $childCategoryIds
                        ]
                    ]
                ]
            ]
        ];
        if (isset($data['type_variant_id'])) {
            $query['query']['bool']['must'][]['match'] = ['variant_options.5.options.id' => intval($data['type_variant_id'])];
        }
        if (isset($data['color_variant_id'])) {
            $query['query']['bool']['must'][]['match'] = ['variant_options.2.options.id' => intval($data['color_variant_id'])];
        }
        if (isset($data['size_variant_id'])) {
            $query['query']['bool']['must'][]['match'] = ['variant_options.1.options.id' => intval($data['size_variant_id'])];
        }
        if (isset($data['minPrice']) && isset($data['maxPrice'])) {
            $query['query']['bool']['filter'] = [
                'script' => [
                    'script' => [
                        'source' => 'Double.parseDouble(doc[\'price.keyword\'].value) >= params.min && Double.parseDouble(doc[\'price.keyword\'].value) <= params.max',
                        'params' => [
                            'min' => intval($data['minPrice']),
                            'max' => intval($data['maxPrice'])
                        ]
                    ]
                ]
            ];
        }
        if (isset($data['seller'])) {
            $query['query']['bool']['must'][]['term'] = ['user.id' => intval($data['seller'])];
        }
        $query['aggs']['user'] = [
            'terms' => [
                'field' => 'user.id',
                'size' => 1000
            ],
            'aggs' => [
                'source' => [
                    'top_hits' => [                
                        '_source' => ['user'],
                        'size' => 1,
                    ]
                ]
            ]
        ];
        
        $query['aggs']['min_price'] = [
            'min' => [
                'script' => [
                    'source' => 'Double.parseDouble(doc[\'price.keyword\'].value)'
                ]
            ]
        ];
        $query['aggs']['max_price'] = [
            'max' => [
                'script' => [
                    'source' => 'Double.parseDouble(doc[\'price.keyword\'].value)'
                ]
            ]
        ];
        if (isset($data['order'])) {
            switch ($data['order']) {
                case 'sold':
                    $query['sort'] = [
                        [
                            'sold' => 'desc'
                        ]
                    ];
                    break;
                case 'lastest':
                    $query['sort'] = [
                        [
                            'product_created_at.keyword' => 'desc'
                        ]
                    ];
                    break;
                case 'view':
                    $query['sort'] = [
                        [
                            'view_count' => 'desc'
                        ]
                    ];
                    break;
                case 'low_price':
                    $query['sort'] = [
                        [
                            '_script' => [
                                'type' => 'number',
                                'script' => [
                                    'source' => 'Double.parseDouble(doc[\'price.keyword\'].value)',
                                    'lang' => 'painless'
                                ],
                                'order' => 'asc'
                            ]
                        ]
                    ];
                    break;
                case 'high_price':
                    $query['sort'] = [
                        [
                            '_script' => [
                                'type' => 'number',
                                'script' => [
                                    'source' => 'Double.parseDouble(doc[\'price.keyword\'].value)',
                                    'lang' => 'painless'
                                ],
                                'order' => 'desc'
                            ]
                        ]
                    ];
                    break;
                case 'sale':
                    $query['sort'] = [
                        [
                            'percent_sale' => 'asc'
                        ]
                    ];
                    break;
            }
        }
        $query['sort'][] = [
            'score' => 'desc'
        ];
        $query['sort'][] = [
            'cr' => 'desc'
        ];

        return $query;
    }

    /**
     * get child category ids by category id
     * 
     * @param Int $categoryId
     * @return Array
     */
    private function getChildCategoryIds($categoryId)
    {
        $result = [];
        $key = $this->buildChildCategoryIdsKey($categoryId);
        if ($key) {
            if (Cache::has($key)) {
                $result = Cache::get($key);
            }
        }
        if (!$result) {
            $result = [$categoryId];
            $category = Category::find($categoryId);
            if (!empty($category)) {
                $categoryIds = Category::where('_lft', '>', $category->_lft)
                    ->where('_rgt', '<', $category->_rgt)
                    ->whereRaw('_rgt - _lft = 1')
                    ->pluck('id')
                    ->toArray();
                $result = array_merge($result, $categoryIds);
            }
            Cache::put($key, $result, self::CACHE_MINUTES);
        }

        return $result;
    }

    private function buildChildCategoryIdsKey($categoryId) {
        $locale = env('APP_LOCALE', '');
        $key = self::CHILD_CATEGORY_IDS . $locale . $categoryId;
        return $key;
    }

    /**
     * format response
     * 
     * @param Array $response
     * @param Array $meta
     * @param Array $category
     * @return Array 
     */
    private function formatResponse($response, $meta, $category)
    {
        $key = $this->buildFilterKey($category['id'], $meta);
        $filters = null;
        if ($key) {
            if (Cache::has($key)) {
                $filters = Cache::get($key);
            }
            if (isset($_GET['clear_cache'])) {
                $filters = null;
            }
        }
        if (!$filters) {
            $filters = $this->buildFilterByCategory($category, $meta, $response);
            Cache::put($key, $filters, self::CACHE_MINUTES);
        }
        $result['status'] = 'successful';
        $result['result'] = array_map(function ($item) {
            $source = $item['_source'];
            if (Auth::user()) {
                $source['horder'] = 0;
                $source['pivot_id'] = 0;
            }
            return $source;
        }, $response['hits']['hits']);
        $allFilter = $this->getDataFilter($meta);
        $result['categories'] = isset($allFilter['categories']) ? $allFilter['categories'] : [];;
        $result['displayedFilters'] = [
            'status' => []
        ];
        $result['tags'] = [];
        $result['priceRange'] = $filters['priceRange'];
        $result['meta'] = $this->getMetaData($meta, $response['hits']['total']);
        $result['filterCategory'] = $category;
        if (isset($filters['users'])) {
            $result['users'] = $filters['users'];
        } else {
            $result['users'] = [];
        }
        $result['filterOptions'] = $filters['filterOptions'];
        
        $displayedFilters = $this->getDisplayedFilters($meta);
        if (count($result['filterOptions']) > 0) {
            $displayFilterVariant = $this->getDisplayFilterVariant($meta, $result['filterOptions']);
            $displayedFilters = array_merge($displayFilterVariant, $displayedFilters);
        }
        $result['displayedFilters'] = $displayedFilters;

        return $result;
    }

    protected function getDataFilter($filter = []) {
        $categories = [];
        if (isset($filter['category_id'])) {
            $categories = $this->getChildCategories($filter['category_id']);
        }

        $result = [
            'categories' => $categories
        ];
        
        return $result;
    }

    public function getChildCategories($parentId)
    {
        $categoryQuery = \DB::table('category as c')
            ->where('c.is_hidden', '0')
            ->where('c.type', 'PRODUCT')
            ->where('c.parent_id', $parentId);
            
        $categoryQuery->join('product_n_category_top as pct', 'pct.category_id', '=', 'c.id');
        $categoryQuery->groupBy('c.id');

        $categories = $categoryQuery->get([
            'c.id', 'c.name', 'c.slug', 'c.type', 'c.image_url',
            'c.parent_id', 'c.depth', 'c._lft', 'c._rgt']);

        $categories->map(function ($category) {
            $category->url = getPrefixUrl() . "/" . $category->slug;
            return $category;
        });

        return json_decode(json_encode($categories), true);
    }

    /**
     * build filter by category
     * 
     * @param Array $category
     * @param Array $meta
     * @return Array
     */
    private function buildFilterByCategory($category, $meta, $response)
    {
        $query = $this->prepareFilterQueryByCategory($category['id']);
        $result = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig[$this->configIndexKey]);
        $result = $this->formatFilterByCategory($result, $meta, $response);

        return $result;
    }

    /**
     * prepare filter query by category
     * 
     * @param Int $categoryId
     * @return Array
     */
    private function prepareFilterQueryByCategory($categoryId)
    {
        $childCategoryIds = $this->getChildCategoryIds($categoryId);
        $childCategoryIds = array_values(array_filter($childCategoryIds, function ($item) {
            return !is_null($item);
        }));
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'terms' => [
                            'category_id' => $childCategoryIds
                        ]
                    ]
                ]
            ]
        ];
        $query['aggs']['type_filter'] = [
            'terms' => [
                'field' => 'variant_options.5.options.name.keyword',
                'size' => 1000
            ],
            'aggs' => [
                'source' => [
                    'top_hits' => [                
                        '_source' => ['variant_options.5.*'],
                        'size' => 1,
                    ]
                ]
            ]
        ];
        $query['aggs']['size_filter'] = [
            'terms' => [
                'field' => 'variant_options.1.options.name.keyword',
                'size' => 1000
            ],
            'aggs' => [
                'source' => [
                    'top_hits' => [                
                        '_source' => ['variant_options.1.*'],
                        'size' => 1,
                    ]
                ]
            ]
        ];
        $query['aggs']['color_filter'] = [
            'terms' => [
                'field' => 'variant_options.2.options.name.keyword',
                'size' => 1000
            ],
            'aggs' => [
                'source' => [
                    'top_hits' => [                
                        '_source' => ['variant_options.2.*'],
                        'size' => 1,
                    ]
                ]
            ]
        ];

        return $query;
    }

    /**
     * format filters by category
     * 
     * @param Array $data
     * @param Array $meta
     * @return Array
     */
    private function formatFilterByCategory($data, $meta, $response)
    {
        $typeResult = $this->formatOptionFilter($data['aggregations']['type_filter']['buckets'], $this->option['Type']);
        $sizeResult = $this->formatOptionFilter($data['aggregations']['size_filter']['buckets'], $this->option['Size']);
        $colorResult = $this->formatOptionFilter($data['aggregations']['color_filter']['buckets'], $this->option['Color']);
        $priceResult = $this->formatPriceFilter($meta, $data['hits']['total'], $response['aggregations']['max_price']['value'], $response['aggregations']['min_price']['value']);
        $userResult = $this->formatUserFilter($response['aggregations']['user']['buckets']);

        return [
            'priceRange' => $priceResult,
            'filterOptions' => [
                'Size' => $sizeResult,
                'Type' => $typeResult,
                'Color' => $colorResult
            ],
            'users' => $userResult
        ];
    }

    public function compareCategory(Request $request) {
        $sub = $request->input('sub', 50);
        $index = $request->input('index', $this->elasticSearchConfig['index']);
        $metaFromEls = $this->categoryMetaFromEls($index);
        $result = [];
        $categoryIds = [];
        $allCategory = DB::table('category')->get(['id', '_lft', '_rgt']);
        foreach ($allCategory as $category) {
            if ($category->_rgt -  $category->_lft > 1) {
                continue;
            }
            $countEls = isset($metaFromEls[$category->id]) ? $metaFromEls[$category->id] : 0;
            $countMysql = DB::table('product_n_category_top')
                ->where('category_id', $category->id)
                ->where('is_parent', 0)
                ->count();
            if ($countMysql > 0 && $countMysql - $countEls > $sub) {
                $result[] = [
                    'category_id' => $category->id,
                    'count_els' => $countEls,
                    'count_mysql' => $countMysql,
                    'sub' => $countMysql - $countEls,
                ];
                $categoryIds[] = $category->id;
            }

        }

        usort($result, function ($a, $b) {
            if ($a['sub'] == $b['sub']) {
                return 0;
            }
            return ($a['sub'] < $b['sub']) ? 1 : -1;
        });

        return [
            'status' => 'successful',
            'result' => $result,
            'category_ids' => $categoryIds,
        ];
    }

    protected function categoryMetaFromEls($index) {
        $result = [];
        $allData =  $this->elasticService->countByCategory($index);
        if (isset($allData['aggregations']['categories']['buckets'])) {
            foreach ($allData['aggregations']['categories']['buckets'] as $item) {
                $result[$item['key']] = $item['doc_count'];
            }
        }
        return $result;
    }

    public function checkIndexName()
    {
        return response()->json($this->elasticSearchConfig[$this->configIndexKey]);
    }
}
