<?php

namespace Modules\ZSearch\Controllers;

use App\Http\Controllers\Services\TagExploreService;
use Theme;
use Module;
use App\Utils;
use App\Utils\SlugManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Services\ElasticSearchIndex;
use Modules\ZSearch\Services\ElasticSearchIndexCategory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use DateTime;
use stdClass;
use Illuminate\Database\Query\JoinClause;

class ExploreController extends Controller
{
    const CACHE_MINUTES = 20;
    const SEARCH_TRACKING = 'qs_tr';
    protected $elasticService;
    protected $elasticsearchIndexCategory;
    protected $elasticSearchConfig;
    protected $filterRequest;
    protected $elasticSearchIndex;
    protected $tagExploreService;
    static $skuIdByProductId = [];
    static $imageUrlDefaultByProductId = [];
    static $skuTemplate = [];
    static $templateByProduct = [];
    static $flashSaleByProduct = [];
    static $flashSaleByProductIds = [];
    private static $randomFlashSales = [];
    private static $productIds = [];
    private static $templateIdsByProductIds = [];
    private static $skuTemplateDefaultByTemplateIdsColorId = [];
    private static $skuTemplateDefaultByTemplateIds = [];
    private static $templateGalleryExistByProductIds = [];
    private static $overrideImageByProductIdsSkuIds = [];
    private static $colorIdsBySkuIds = [];
    private static $skuIds = [];
    private static $designCodesByProductIdsColorId = [];
    private static $designCodesByProductIds = [];
    private static $categoryIds = [7, 8, 35, 54, 55, 48, 51];
    private $type = 'products';
    private $bestSellingKey = 'tag_page.best_selling_category';
    protected $exceptionKeyword = [
        'motley crue poster',
        'motley crue'
    ];
    const TYPES_ENABLE = [SlugManager::TYPE_TYPE, SlugManager::TYPE_COLOR];
    
    public function __construct(ElasticSearchIndexCategory $elasticsearchIndexCategory, ElasticSearchIndex $elasticSearchIndex, TagExploreService $tagExploreService)
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticService =  new ElasticSearchService($this->elasticSearchConfig);
        $this->elasticsearchIndexCategory = $elasticsearchIndexCategory;
        $this->elasticSearchIndex = $elasticSearchIndex;
        $this->tagExploreService = $tagExploreService;
    }

    /**
     * filter product by tag
     * 
     * @param Request $request
     * @return Response
     */
    public function filterProductByTag(Request $request): array
    {
        $filters = $request->all();
        $tagSlug = $filters['tag_slug'] ?? null;
        $tag = DB::table('tag')->where('slug', $tagSlug)->get(['id', 'slug', 'title', 'rating_count', 'rating_value'])->first();
        if (empty($tag->id)) {
            return redirect(route('pod::expore'), 301);
        }
        $filters['tag_id'] = $tag->id;
        $query = $this->buildProductByTagQuery($filters);
        $productsByTag = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
        if ($productsByTag['hits']['total'] < 10000) {
            $query = $this->buildProductByTagQuery($filters, true);
            $productsByTag = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
        }
        $products = $this->formatProductsByTag($productsByTag);
        $meta = $this->getMeta($filters, $productsByTag['hits']['total']);

        return [
            'status' => 'successful',
            'result' => $products,
            'meta' => $meta
        ];
    }

    /**
     * format products by tag
     * 
     * @param Array $productsByTag
     * @return Array
     */
    private function formatProductsByTag($productsByTag)
    {
        $result = [];
        foreach ($productsByTag['hits']['hits'] as $productItem) {
            $productItem = $productItem['_source'];
            $productItem = $this->buildPrice($productItem);
            if (!isset($productItem['rating_count'])) {
                $productItem['rating_count'] = 0;
            }
            if (!isset($productItem['rating_value'])) {
                $productItem['rating_value'] = 0;
            }
            if (!isset($productItem['user']) ) {
                $productItem['user'] = null;
            }
            if ($flashSaleText = $this->getFlashSaleByProduct($productItem['id'])) {
                $productItem['flashSaleText'] = $flashSaleText;
            }

            $result[] = $productItem;
        }
        $result = $this->buildSkuId($result);
        
        return $result;
    }

    /**
     * Get meta data for pagination.
     *
     * @param $data
     * @param $total
     * @return array
     */
    public function getMeta($data, $total): array
    {
        $pageId = 0;
        $pageSize = 20;
        if (!empty($data['page_size'])) {
            $pageSize = $data['page_size'];
        }

        if (!empty($data['page_id'])) {
            $pageId = $data['page_id'];
        }

        $meta = [
            'page_id' => $pageId,
            'page_size' => $pageSize,
            'has_next' => false,
            'total_count' => $total,
        ];
        $meta['page_count'] = ceil($total / $pageSize);
        if ($pageId < $meta['page_count'] - 1) {
            $meta['has_next'] = true;
        }

        return $meta;
    }

    /**
     * build product by tag query
     * 
     * @param Array $filters
     * @param Bool $sort
     * @return Array
     */
    private function buildProductByTagQuery($filters, $sort = false)
    {
        $query['size'] = isset($filters['page_size']) ? $filters['page_size'] : 20;
        if (isset($filters['page_id'])) {
            $query['from'] = $filters['page_id'] * $filters['page_size'];
        }
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $filters['tag_id']
                        ]
                    ]
                ]
            ]
        ];
        
        if ($sort) {
            $query['sort'] = [
                [
                    '_script' => [
                        'type' => 'number',
                        'script' => [
                            'lang' => 'painless',
                            'source' =>  "
                                if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                    DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                    return df.parse(doc['created_at.keyword'].value).getTime();
                                } else {
                                    return 0;
                                }
                            "
                        ],
                        'order' => 'desc'
                    ]
                ],
                [
                    'score' => 'desc'
                ]
            ];
        }

        return $query;
    }

    /**
     * build best selling by category in a week
     */
    public function buildBestSellingByCategory()
    {
        $date = new DateTime();
        $date->modify("-7 days");
        $orderByCategory = DB::table('order_item')
            ->join('order', 'order_item.order_id', '=', 'order.id')
            ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
            ->where('product_n_category.is_parent', '=', 0)
            ->where('order.payment_status', '=', 'PAID')
            ->where('order.created_at', '>', $date->format("Y-m-d H:i:s"))
            ->select('product_n_category.category_id AS id', DB::raw('COUNT(*) AS quantity'))
            ->groupBy(['id'])
            ->orderBy('quantity', 'DESC')
            ->limit(5)
            ->get();
        $categoryIds = implode(',', $orderByCategory->pluck('id')->toArray());
        $record = DB::table('option')->where('key', $this->bestSellingKey)->first();
        if ($record) {
            DB::table('option')
                ->where('key', $this->bestSellingKey)
                ->update([
                    'key' => $this->bestSellingKey,
                    'value' => $categoryIds
                ]);
        } else {
            DB::table('option')->insert([
                'key' => $this->bestSellingKey,
                'value' => $categoryIds
            ]);
        }

        return response()->json([
            'status' => 'successful'
        ]);
    }

    /**
     * explore tag
     * 
     * @param Request $request
     * @return Response
     */
    public function exploreTag(Request $request)
    {
        if (!in_array(Utils::getIP(), ['210.245.49.63', '118.70.67.73'])) {
            abort(404);
        }

        $parameters = Route::current()->parameters();
        $tagTitle = $parameters['tagTitle'] ?? null;
        if (!$tagTitle) {
            return redirect(route('pod::expore'), 301);
        }

        $keyword = str_replace('-', ' ', strip_tags($tagTitle));
        if ($this->tagExploreService->isIgnoreKeyword($keyword)) {
            abort(404);
        }

        $tag = DB::table('tag')->where('slug', $tagTitle)->get(['id', 'slug', 'title', 'rating_count', 'rating_value'])->first();
        if (empty($tag->id)) {
            return redirect(route('pod::expore'), 301);
        }

        $tagTitle = $tag->title;
        $tagId = $tag->id;
        $currentTag = ucfirst($tagTitle);
        $tagSlug = $tag->slug;
        $currentTagSlug = str_replace(' ', '-', strtolower(trim($tagTitle)));
        $breadcrumbs = $this->tagExploreService->buildBreadcrumbs($tagTitle, $currentTagSlug);
        $data = $productFilters = $this->buildParams($request);
        $data['tag_slug'] = $tag->slug;
        $data['tag_id'] = $tag->id;
        $isCategoryPage = false;
        $total = 0;
        
        $this->initTimer();
        $this->cpTimer('init');
        if (Theme::getSetting('prerender_search', true)) {
            if (Theme::getSetting('show_pending_product', false)) {
                $data['status'] = [
                    'ACTIVE',
                    'PENDING'
                ];
            }

            $locale = env("APP_LOCALE");
            if ($locale == "") {
                $locale = config('localization::module.default_locale', 'us');
            }
            $resultCacheKey = "$locale-tag-page-data::" . $tag->slug . 'v2';
            if (Cache::has($resultCacheKey)) {
                $result = Cache::get($resultCacheKey);
            } else {
                $interval = intval(date('Y')) - 2020;
                
                $this->cpTimer('buildTypeOptionQuery');

                $typeOptionquery = $this->buildTypeOptionQuery($data);
                $typeOptions = $this->elasticService->searchDocument($typeOptionquery, $this->type, $this->elasticSearchConfig['index']);
    
                $this->cpTimer('buildTypeOptionQuery1');
                
                $this->cpTimer('buildBestSellingQuery');
                for ($year = 1; $year <= $interval; $year++) {
                    $bestSellingQuery = $this->buildBestSellingQuery($data, $year);
                    $bestSellingProducts = $this->elasticService->searchDocument($bestSellingQuery, $this->type, $this->elasticSearchConfig['index']);
                    if ($bestSellingProducts['hits']['total'] > 20) {
                        break;
                    }
                }
                if ($bestSellingProducts['hits']['total'] < 500000) {
                    $bestSellingQuery = $this->buildBestSellingQuery($data, $year, true);
                    $bestSellingProducts = $this->elasticService->searchDocument($bestSellingQuery, $this->type, $this->elasticSearchConfig['index']);
                }

                if ($bestSellingProducts['hits']['total'] < 20) {
                    $additionBestSellingQuery = $this->buildAdditionBestSellingQuery($bestSellingProducts, $data);
                    $additionBestSellingProducts = $this->elasticService->searchDocument($additionBestSellingQuery, $this->type, $this->elasticSearchConfig['index']);
                    $bestSellingProducts['hits']['hits'] = array_merge($bestSellingProducts['hits']['hits'], $additionBestSellingProducts['hits']['hits']);
                    $bestSellingProducts['hits']['total'] = count($bestSellingProducts['hits']['hits']);
                    foreach ($bestSellingProducts['hits']['hits'] as $product) {
                        $this->setProductIds($product['_source']['id']);
                    }
                }
                $this->cpTimer('buildBestSellingQuery1');
                $this->cpTimer('buildShopByColorQuery');
    
                $colorOptions = $this->getVariantOptions('color', 5);
                $shopsByColor = $colorOptions;
                foreach ($colorOptions as $key => $colorOptionItem) {
                    for ($year = 1; $year <= $interval; $year++) {
                        $shopByColorQuery = $this->buildShopByColorQuery($data, $colorOptionItem, $year);
                        $shopByColor = $this->elasticService->searchDocument($shopByColorQuery, $this->type, $this->elasticSearchConfig['index']);
                        if ($shopByColor['hits']['total'] > 20) {
                            break;
                        }
                    }
                    if ($shopByColor['hits']['total'] < 500000) {
                        $shopByColorQuery = $this->buildShopByColorQuery($data, $colorOptionItem, $year, true);
                        $shopByColor = $this->elasticService->searchDocument($shopByColorQuery, $this->type, $this->elasticSearchConfig['index']);
                    }

                    $products = [];
                    foreach ($shopByColor['hits']['hits'] as $shopItem) {
                        $products[] = $shopItem['_source'];
                        $this->setProductIds($shopItem['_source']['id']);
                    }
                    $colorOptionItem->product = $products;
                    $shopsByColor[$key] = $colorOptionItem;
                    break;
                }
                $this->cpTimer('buildShopByColorQuery1');
                $this->cpTimer('buildShopByCategoryQuery');
    
                $categories = $this->getCategoryOptions();
                $shopsByCategory = $categories;
                if ($categories) {
                    foreach ($categories as $key => $categoryItem) {
                        for ($year = 1; $year <= $interval; $year++) {
                            $shopByCategoryQuery = $this->buildShopByCategoryQuery($data, $categoryItem, $year);
                            $shopByCategory = $this->elasticService->searchDocument($shopByCategoryQuery, $this->type, $this->elasticSearchConfig['index']);
                            if ($shopByCategory['hits']['total'] > 20) {
                                break;
                            }
                        }
                        if ($shopByCategory['hits']['total'] < 500000) {
                            $shopByCategoryQuery = $this->buildShopByCategoryQuery($data, $categoryItem, $year, true);
                            $shopByCategory = $this->elasticService->searchDocument($shopByCategoryQuery, $this->type, $this->elasticSearchConfig['index']);
                        }

                        $products = [];
                        foreach ($shopByCategory['hits']['hits'] as $shopItem) {
                            $products[] = $shopItem['_source'];
                            $this->setProductIds($shopItem['_source']['id']);
                        }
                        $categoryItem->product = $products;
                        $shopsByCategory[$key] = $categoryItem;
                        break;
                    }
                }
                $this->cpTimer('buildShopByCategoryQuery1');
                $this->cpTimer('setting');
    
                $this->setFlashSaleByProductIds();
                $this->setTemplateIdsByProductIds();        
                $this->setSkuTemplateDefaultByTemplateIds();
                $this->setRandomFlashSale();
                $this->setOverrideImageByProductIdsAndSkuIds();
                $this->setColorIdsBySkuIds();
                $this->setTemplateGalleryExistByProductIds();
                $this->setDesignCodeByProductIds();
                $this->cpTimer('setting1');
                $this->cpTimer('getUsers');
                $users = $this->getUsers($tagId);
                $this->cpTimer('getUsers1');
                
                $this->cpTimer('formatResponse');
                $result = $this->formatResponse($typeOptions, $bestSellingProducts, $shopsByColor, $shopsByCategory, $users);
                $result = json_decode(json_encode($result), true);

                // Cache::put($resultCacheKey, $result, 60 * 60 * 24);
            }
            
            $this->cpTimer('formatResponse1');

            $categories = [];
            $users = [];
            $colorOptions = [];
            $typeOptions = [];
            $bestSellingProducts = [];
            $total = DB::table('summary_tag_product')->where('tag_id', $tagId)->value('product_count');
            if (!$total) {
                $total = rand(1000, 2000);
            }
            $colorSlugIds = [];
            $categorySlugIds = [];
            $typeOptions = $result['typeOptions'];
            $users = $result['users'];
            $categories = $result['categories'];
            foreach($categories as $category) {
                $categorySlugIds[] = ['slug' => $category['slug'], 'id' => $category['id']];
            }
            $colorOptions = $result['colorOptions'];
            foreach($colorOptions as $colorOption) {
                $colorSlugIds[] = ['slug' => $colorOption['slug'], 'id' => $colorOption['id']];
            }
            $bestSellingProducts = $result['bestSellingProducts'];

            $strCodes = '';
            \Module::action('viewPage', [
                'type' => 'tag',
                'data' => [
                    'id' => $tagId,
                ]
            ]);

            if (\Module::isActive('Seo') && function_exists('getMetaDataConfig')) {
                $metaDataConfig = getMetaDataConfig();
                if ($metaDataConfig) {
                    if ($metaDataConfig->content_footer) {

                        if ($metaDataConfig->content_footer == strip_tags($metaDataConfig->content_footer)) {
                            $metaDataConfig->content_footer = nl2br($metaDataConfig->content_footer);
                        }

                        View::share('post', [
                            'content' => $metaDataConfig->content_footer
                        ]);
                    }

                    if ($metaDataConfig->content) {
                        View::share('metaDescription', $metaDataConfig->content);
                    }
                }
            }
            $wishProductIds = $this->tagExploreService->buildWishlistData();
            View::share('wishProductIds', $wishProductIds);
            View::share('tag', $tagId);
            $user = Auth::user();
            if ($user && isset($user->role) && in_array($user->role, ['STAFF','ADMIN']) ) {
                View::share('is_show_tagid', true);
            }

            View::share('post', $this->tagExploreService->buildPostTag($tagId));
            $slugUrlTag = '/explore/' . $currentTagSlug;
            View::share('shareBreadcrumb', $this->tagExploreService->buildShareBreadcrumb($slugUrlTag, $tagTitle));
            View::share('shareProduct', $this->tagExploreService->buildShareProduct([], $tagId));
            View::share('hreflangs', $this->tagExploreService->buildHreflangs($slugUrlTag));
            View::share('currentTagDetail', get_object_vars($tag));
            View::share('tag', ['title' => $tagTitle]);
            View::share('searchCategories', $categories);
            View::share('tagTitle', $keyword);
            View::share('strCode', substr($strCodes, 0, -1));
            View::share('pageType', 'view_item_list');
            View::share('totalCount', $total);
            $cloneTag = get_object_vars($tag);
            $this->cpTimer('end');
            $this->persistenceTimer();
            $topProductTags = [];
            return response()->view('explore.index', compact(
                'categories',
                'keyword',
                'currentTag',
                'currentTagSlug',
                'breadcrumbs',
                'cloneTag',
                'typeOptions',
                'colorOptions',
                'users',
                'total',
                'bestSellingProducts',
                'wishProductIds',
                'colorSlugIds',
                'categorySlugIds',
                'tagSlug',
                'topProductTags'
            ))->cookie('z-search-tag-page', 'new_tag', 0, '', '', false, false);
        } else {
            return view('explore.index', compact('productFilters','isCategoryPage'));
        }
    }

    /**
     * @return array
     */
    private function getUsers($tagId): array
    {
        $query = $this->buildUsersByTagQuery($tagId);
        $users = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);

        return $users;
    }

    /**
     * build users by tag query
     * 
     * @param Int $tagId
     * @return Array
     */
    private function buildUsersByTagQuery($tagId)
    {
        $query['_source'] = ['user'];
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $tagId
                        ]
                    ]
                ]
            ]
        ];
        $query['aggs'] = [
            'users' => [
                'terms' => [
                    'field' => 'user.id.keyword',
                    'order' => [
                        '_count' => 'desc'
                    ]
                ],
                'aggs' => [
                    'source' => [
                        'top_hits' => [                
                            '_source' => ['user'],
                            'size' => 1,
                        ]
                    ]
                ]
            ]
        ];

        return $query;
    }

    /**
     * Get options for the filter.
     */
    public function getVariantOptions($variantSlug, $count)
    {
        $locale = env("APP_LOCALE");
        if ($locale == "") {
            $locale = config('localization::module.default_locale', 'us');
        }

        $cacheKey = "$locale-$variantSlug-$count-by-template-tag-v4";
        if (Cache::has($cacheKey)) {
            $options = json_decode(Cache::get($cacheKey));
        } else {
            $options = DB::table('template_tag_n_variant_option as ttnvo')
                ->select(['ttnvo.variant_option_id as id', 'vo.name as name', 'ttnvo.template_id', 'vo.slug as slug', DB::raw('COUNT(sb_ttnvo.template_id) as count')])
                ->join('product_variant_option as vo', 'vo.id', '=', 'ttnvo.variant_option_id')
                ->join('product_variant as v', 'v.id', '=', 'vo.variant_id')
                ->where('v.slug', $variantSlug)
                ->groupBy(['vo.slug'])
                ->orderBy('count', 'DESC')
                ->limit($count)
                ->get();

            // Cache::put($cacheKey, json_encode($options), 60 * 60 * 24 * 7);
        }

        return $options;
    }

    /**
     * Get categories with count for the filter.
     */
    private function getCategoriesWithCount()
    {
        $categoryIds = DB::table('option')->where('key', $this->bestSellingKey)->pluck('value');
        if ($categoryIds->isNotEmpty()) {
            $categoryIds = explode(',', $categoryIds->toArray()[0]);
            $categoryIds = array_map(function ($item) {
                $obj = new stdClass();
                $obj->id = intval($item);
                return $obj;
            }, $categoryIds);

            return $categoryIds;
        }

        $locale = env("APP_LOCALE");
        if ($locale == "") {
            $locale = config('localization::module.default_locale', 'us');
        }
        $cacheKey = "$locale-sale-by-cate-v1";
        if (Cache::has($cacheKey)) {
            $orderByCategory = json_decode(Cache::get($cacheKey));
        } else {
            $date = new DateTime();
            $date->modify("-7 days");
            $orderByCategory = DB::table('order_item')
                ->join('order', 'order_item.order_id', '=', 'order.id')
                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                ->where('product_n_category.is_parent', '=', 0)
                ->where('order.payment_status', '=', 'PAID')
                ->where('order.created_at', '>', $date->format("Y-m-d H:i:s"))
                ->select('product_n_category.category_id AS id', DB::raw('COUNT(*) AS quantity'))
                ->groupBy(['id'])
                ->orderBy('quantity', 'DESC')
                ->limit(5)
                ->get();
            Cache::put($cacheKey, json_encode($orderByCategory), 60 * 60 * 24 * 7);
        }

        return $orderByCategory;
    }

    protected function buildParams($request)
    {
        $data = $request->all();
        $data['page_size'] = 20;
        if (empty($data['page_id'])) {
            $data['page_id'] = 0;
        }

        return $data;
    }

    /**
     * build type option query
     * 
     * @param Array $data
     * @return Array
     */
    private function buildTypeOptionQuery($data)
    {
        $query['size'] = isset($data['page_size']) ? $data['page_size'] : 20;
        if (isset($data['page_id'])) {
            $query['from'] = $data['page_id'] * $data['page_size'];
        }
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $data['tag_id']
                        ]
                    ]
                ]
            ]
        ];
        $query['aggs'] =[
            'type_options' => [
                'terms' => [
                    'field' => 'variant_options.5.options.name.keyword',
                    'order' => [
                        '_count' => "desc"
                    ],
                    'size' => 4
                ],
                'aggs' => [
                    'source' => [
                        'top_hits' => [                
                            '_source' => ['variant_options.5.*', 'image_url'],
                            'size' => 1,
                        ]
                    ]
                ]
            ]
        ];

        return $query;
    }

    /**
     * build best selling query
     * 
     * @param Array $data
     * @param Int $year
     * @param Bool $sort
     * @return Array
     */
    private function buildBestSellingQuery($data, $year, $sort = false)
    {
        $categories = $this->getCategoriesWithCount();
        $categoryIds = [];
        foreach ($categories as $category) {
            $categoryIds[] = $category->id;
        }
        $startDate = date('Y-m-d', time() - 60 * 60 * 24 * 365 * $year);

        $query['size'] = 20;
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $data['tag_id']
                        ]
                    ],
                    [
                        'terms' => [
                            'category_id' => $categoryIds
                        ]
                    ]
                ],
                'filter' => [
                    [
                        'script' => [
                            'script' => [
                                'lang' => 'painless',
                                'source' => "
                                    if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                        DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                        Date startDate = df.parse(params.startDate);
                                        Date docDate = df.parse(doc['created_at.keyword'].value);
                                        return docDate.getTime() > startDate.getTime();
                                    } else {
                                        return false;
                                    }
                                ",
                                'params' => [
                                    'startDate' => $startDate
                                ]
                            ]
                        ]
                    ]
                ]   
            ]
        ];
        
        if ($sort) {
            $query['sort'] = [
                [
                    '_script' => [
                        'type' => 'number',
                        'script' => [
                            'lang' => 'painless',
                            'source' => "
                                if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                    DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                    return df.parse(doc['created_at.keyword'].value).getTime();
                                } else {
                                    return 0;
                                }
                            "
                        ],
                        'order' => 'desc'
                    ]
                ],
                [
                    'score' => 'desc'
                ]
            ];
        }

        return $query;
    }

    /**
     * build shop by color query
     * 
     * @param Array $data
     * @param Object $colorOption
     * @param Int $year
     * @param Bool $sort
     * @return
     */
    private function buildShopByColorQuery($data, $colorOption, $year, $sort = false)
    {
        $query['size'] = 20;
        $startDate = date('Y-m-d', time() - 60 * 60 * 24 * 365 * $year);

        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $data['tag_id']
                        ]
                    ],
                    [
                        'term' => [
                            'variant_options.2.options.name.keyword' => $colorOption->name
                        ]
                    ]
                ],
                'filter' => [
                    [
                        'script' => [
                            'script' => [
                                'lang' => 'painless',
                                'source' => "
                                    if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                        DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                        Date startDate = df.parse(params.startDate);
                                        Date docDate = df.parse(doc['created_at.keyword'].value);
                                        return docDate.getTime() > startDate.getTime();
                                    } else {
                                        return false;
                                    }
                                ",
                                'params' => [
                                    'startDate' => $startDate
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];
        
        if ($sort) {
            $query['sort'] = [
                [
                    '_script' => [
                        'type' => 'number',
                        'script' => [
                            'lang' => 'painless',
                            'source' =>  "
                                if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                    DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                    return df.parse(doc['created_at.keyword'].value).getTime();
                                } else {
                                    return 0;
                                }
                            "
                        ],
                        'order' => 'desc'
                    ]
                ],
                [
                    'score' => 'desc'
                ]
            ];
        }

        return $query;

    }

    /**
     * build shop by category query
     * 
     * @param Array $data
     * @param Object $category
     * @param Int $year
     * @param Bool $sort
     * @return
     */
    private function buildShopByCategoryQuery($data, $category, $year, $sort = false)
    {
        $query['size'] = 20;
        $startDate = date('Y-m-d', time() - 60 * 60 * 24 * 365 * $year);

        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $data['tag_id']
                        ]
                    ],
                    [
                        'term' => [
                            'category_id' => $category->id
                        ]
                    ]
                ],
                'filter' => [
                    [
                        'script' => [
                            'script' => [
                                'lang' => 'painless',
                                'source' => "
                                    if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                        DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                        Date startDate = df.parse(params.startDate);
                                        Date docDate = df.parse(doc['created_at.keyword'].value);
                                        return docDate.getTime() > startDate.getTime();
                                    } else {
                                        return false;
                                    }
                                ",
                                'params' => [
                                    'startDate' => $startDate
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];
        
        if ($sort) {
            $query['sort'] = [
                [
                    '_script' => [
                        'type' => 'number',
                        'script' => [
                            'lang' => 'painless',
                            'source' =>  "
                                if (doc.containsKey('created_at.keyword') && !doc['created_at.keyword'].empty) {
                                    DateFormat df = new SimpleDateFormat('yyyy-MM-dd');
                                    return df.parse(doc['created_at.keyword'].value).getTime();
                                } else {
                                    return 0;
                                }
                            "
                        ],
                        'order' => 'desc'
                    ]
                ],
                [
                    'score' => 'desc'
                ]
            ];
        }

        return $query;
    }

    /**
     * build elastic query
     * 
     * @param Array $bestSellingProducts
     * @param Array $data
     * @return Array
     */
    private function buildAdditionBestSellingQuery($bestSellingProducts, $data)
    {
        $query['size'] = 20 - $bestSellingProducts['hits']['total'];
        $productIds = [];
        foreach ($bestSellingProducts['hits']['hits'] as $productItem) {
            $productIds[] = $productItem['_source']['id'];
        }
        $query['query'] = [
            'bool' => [
                'must' => [
                    [
                        'term' => [
                            'tags' => $data['tag_id']
                        ]
                    ]
                ],
                'must_not' => [
                    [
                        'terms' => [
                            'id' => $productIds
                        ]
                    ]
                ]
            ]
        ];

        return $query;
    }

    /**
     * format response
     * 
     * @param Array $response
     * @param Array $bestSellingProducts
     * @param Array $shopsByColor
     * @param Array $shopsByCategory
     * @param Array $users
     * @param Array
     */
    private function formatResponse($response, $bestSellingProducts, $shopsByColor, $shopsByCategory, $users)
    {
        $result['typeOptions'] = $this->formatTypeOptions($response['aggregations']['type_options']['buckets']);
        $result['bestSellingProducts'] = $this->formatBestSellingProducts($bestSellingProducts);
        $result['colorOptions'] = $this->formatShopsByColor($shopsByColor);
        $result['categories'] = $this->formatShopsByCategory($shopsByCategory);
        $result['users'] = $this->formatUsers($users);

        return $result;
    }

    /**
     * format users
     * 
     * @param Array $users
     * @return Array
     */
    private function formatUsers($users)
    {
        $result = [];
        $userIds = [];
        foreach ($users['aggregations']['users']['buckets'] as $userItem) {
            $userIds[] = intval($userItem['key']);
        }

        $usersData = DB::table('users')
            ->select('id', 'image_avatar')
            ->whereIn('id', $userIds)
            ->get()
            ->keyby('id')
            ->toArray();

        foreach ($users['aggregations']['users']['buckets'] as $key => $userItem) {
            $result[$key] = $userItem['source']['hits']['hits'][0]['_source']['user'];
            $userId = intval($userItem['key']);
            $result[$key]['image_avatar'] = isset($usersData[$userId]) ? $usersData[$userId]->image_avatar : null;
            $result[$key]['count'] = $userItem['doc_count'];
        }

        return $result;
    }

    /**
     * format type options
     * 
     * @param Array $typeOptions
     * @return Array
     */
    private function formatTypeOptions($typeOptions)
    {
        $result = [];
        foreach ($typeOptions as $typeItem) {
            foreach ($typeItem['source']['hits']['hits'][0]['_source']['variant_options'][5]['options'] as $subItem) {
                if ($subItem['name'] == $typeItem['key']) {
                    $result[] = [
                        'name' => $subItem['name'],
                        'slug' => $subItem['slug'],
                        'image_url' => $typeItem['source']['hits']['hits'][0]['_source']['image_url']
                    ];
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * format best selling products
     * 
     * @param Array $bestSellingProducts
     * @return Array
     */
    private function formatBestSellingProducts($bestSellingProducts)
    {
        $result = [];
        foreach ($bestSellingProducts['hits']['hits'] as $productItem) {
            $productItem = $productItem['_source'];
            $productItem = $this->buildPrice($productItem);
            if (!isset($productItem['rating_count'])) {
                $productItem['rating_count'] = 0;
            }
            if (!isset($productItem['rating_value'])) {
                $productItem['rating_value'] = 0;
            }
            if (!isset($productItem['user']) ) {
                $productItem['user'] = null;
            }
            if ($flashSaleText = $this->getFlashSaleByProduct($productItem['id'])) {
                $productItem['flashSaleText'] = $flashSaleText;
            }

            $result[] = $productItem;
        }
        $result = $this->buildSkuId($result);
        
        return $result;
    }

    /**
     * format shops by color
     * 
     * @param Array $shopsByColor
     * @return Array
     */
    private function formatShopsByColor($shopsByColor)
    {
        foreach ($shopsByColor as $key => $shopItem) {
            $products = [];
            foreach ($shopItem->product as $product) {
                $product = $this->buildPrice($product);
                $this->setDesignCodeByProductIdsAndColorId($shopItem->id);
                $this->setSkuTemplateDefaultByTemplateIdsAndColorId($shopItem->id);
                if (!isset($product['rating_count'])) {
                    $product['rating_count'] = 0;
                }
                if (!isset($product['rating_value'])) {
                    $product['rating_value'] = 0;
                }
                if (!isset($product['user']) ) {
                    $product['user'] = null;
                }
                if ($flashSaleText = $this->getFlashSaleByProduct($product['id'])) {
                    $product['flashSaleText'] = $flashSaleText;
                }
    
                $products[] = $product;
            }
            
            $products = $this->buildImageUrlByColor($products, ['color_variant_id' => $shopItem->id]);
            $products = $this->buildSkuId($products);
            $shopItem->product = $products;
            $shopsByColor[$key] = $shopItem;
            break;
        }
        
        return $shopsByColor;
    }

    /**
     * format shops by category
     * 
     * @param Array $shopsByCategory
     * @return Array
     */
    private function formatShopsByCategory($shopsByCategory)
    {
        foreach ($shopsByCategory as $key => $shopItem) {
            $products = [];
            foreach ($shopItem->product as $product) {
                $product = $this->buildPrice($product);
                if (!isset($product['rating_count'])) {
                    $product['rating_count'] = 0;
                }
                if (!isset($product['rating_value'])) {
                    $product['rating_value'] = 0;
                }
                if (!isset($product['user']) ) {
                    $product['user'] = null;
                }
                if ($flashSaleText = $this->getFlashSaleByProduct($product['id'])) {
                    $product['flashSaleText'] = $flashSaleText;
                }
    
                $products[] = $product;
            }
            
            $products = $this->buildSkuId($products);
            $shopItem->product = $products;
            $shopsByCategory[$key] = $shopItem;
            break;
        }
        
        return $shopsByCategory;
    }

    /**
     * @param $items
     * @param $filter
     * @return mixed
     */
    public function buildImageUrlByColor($items, $filter) {
        //color_variant_id
        if ($items && array_key_exists('color_variant_id', $filter) && $filter['color_variant_id'] > 0) {
            $colorVariantId = $filter['color_variant_id'];
            $pIds = $this->mapProductIds($items);
            if ($pIds) {
                $imgByColor = DB::table('product_sku_value as psv')
                    ->join('product_sku as ps', 'psv.sku_id', '=', 'ps.id')
                    ->whereIn('psv.product_id', $pIds)
                    ->where('psv.variant_option_id', $colorVariantId)
                    ->groupBy(['psv.product_id'])
                    ->pluck('ps.image_url', 'ps.product_id');
                foreach ($items as &$item) {
                    $pId = $item['id'];
                    if (str_contains($item['image_url'], 'https://sticker.liveview.printerval.com/sticker')) {
                        $item['image_url'] = str_replace('sticker-kisscut,white', 'sticker-round,kelly-green', $item['image_url']);
                        $item['image_url'] = str_replace('f9f9f9', '1cb24b', $item['image_url']);
                    }
                    if (!isset($imgByColor[$pId]) || !$imgByColor[$pId]) {
                        $imageTemplate = $this->getImageTemplate($item['id'], $colorVariantId);
                        if ($imageTemplate && !str_contains('DESIGN_ID', $imageTemplate)) {
                            $item['image_url'] = $imageTemplate;
                        }
                    } else {
                        $item['image_url'] = $imgByColor[$pId];
                    }
                }
            }
        }

        return $items;
    }

    public function getImageTemplate($productId, $colorVariantId) {
        $result = null;
        $templateId = $this->getTemplateId($productId);
        if ($templateId > 0) {
            $sku = $this->getSkuByTemplate($templateId, $colorVariantId);
            if ($sku && isset($sku->id)) {
                $result = $this->getImgBySku($productId, $sku);
            }
        }
        return $result;
    }

    public function getImgBySku($productId, $sku) {
        $result = null;
        $skuId = $sku->id;
        if (self::hasOverwriteImage($productId)) {
            $image = self::getOverwriteImage($productId, $skuId);
            if ($image && isset($image->image_url)) {
                $result = $image->image_url;
            }
        } else if (isset($sku->image_url)) {
            $colorId = self::getColorId($skuId);
            $design = self::getTemplateDesign($productId, $colorId);
            $result = self::replaceTemplateDesignImageUrl($design, $sku->image_url);
        }
        return $result;
    }

    public static function hasOverwriteImage($productId): bool
    {
        return self::$templateGalleryExistByProductIds[$productId] ?? false;
    }

    public static function getOverwriteImage($productId, $productSkuId) {
        return self::$overrideImageByProductIdsSkuIds[$productId][$productSkuId] ?? null;
    }

    public static function getColorId($skuId) {
        return self::$colorIdsBySkuIds[$skuId] ?? null;
    }

    public static function getTemplateDesign($productId, $colorId = null) {
        $designCode = null;
        if ($colorId) {
            $designCode = self::$designCodesByProductIdsColorId[$colorId] ?? null;
        }

        if (!$designCode) {
            $designCode = self::$designCodesByProductIds[$productId] ?? null;
        }

        return $designCode;
    }

    public static function replaceTemplateDesignImageUrl($design, $imageUrl) {
        if (!$design) {
            return null;
        }

        if (is_array($design)) {
            $design = reset($design);
        }

        return str_replace('[DESIGN_ID]', $design, $imageUrl);
    }

    /**
     * Build Design Code By Product Ids
     *
     * @return void
     */
    public function setDesignCodeByProductIdsAndColorId($colorId)
    {
        self::$designCodesByProductIdsColorId[$colorId] = DB::table('product_template_design_code')
            ->whereIn('product_id', array_unique(self::$productIds))
            ->where('color_id', $colorId)
            ->pluck('design_code', 'product_id')
            ->toArray();
    }

    /**
     * Build Design Code By Product Ids
     *
     * @return void
     */
    public function setDesignCodeByProductIds()
    {
        self::$designCodesByProductIds = DB::table('product_template_design_code')
            ->whereIn('product_id', array_unique(self::$productIds))
            ->pluck('design_code', 'product_id')
            ->toArray();
    }

    /**
     * Build Override Image By Product Ids And Sku Ids
     *
     * @return void
     */
    public function setOverrideImageByProductIdsAndSkuIds()
    {
         $items = DB::table('product_template_gallery_overwrite')
            ->whereIn('product_id', array_unique(self::$productIds))
            ->whereIn('product_sku_id', array_unique(self::$skuIds))
            ->where('is_primary', 1)
            ->get(['image_url', 'product_id', 'product_sku_id']);
         foreach ($items as $item) {
            self::$overrideImageByProductIdsSkuIds[$item->product_id][$item->product_sku_id] = $item->image_url;
         }
    }

    /**
     * Set Sku Ids
     *
     * @param $skuId
     * @return void
     */
    public function setSkuIds($skuId)
    {
        self::$skuIds[] = $skuId;
    }

    /**
     * Build Color Ids By Sku Ids
     *
     * @return void
     */
    public function setColorIdsBySkuIds()
    {
        self::$colorIdsBySkuIds = DB::table('product_sku_value')
            ->whereIn('sku_id', array_unique(self::$skuIds))
            ->where('variant_id', 2)
            ->pluck('variant_option_id', 'sku_id')
            ->toArray();
    }

    public function getSkuByTemplate($templateId, $colorVariantId) {
        return self::$skuTemplateDefaultByTemplateIdsColorId[$templateId][$colorVariantId] ?? null;
    }

    /**
     * Build Template Gallery Exist By Product Ids
     *
     * @return void
     */
    public function setTemplateGalleryExistByProductIds()
    {
        self::$templateGalleryExistByProductIds = DB::table('product_template_gallery_overwrite')
            ->whereIn('product_id', array_unique(self::$productIds))
            ->pluck('id', 'product_id')
            ->toArray();
    }

    /**
     * Build Flash Sale by Product Ids
     *
     * @return void
     */
    public function setFlashSaleByProductIds()
    {
        $startDate = new DateTime();
        $flashSales = DB::table('flash_sale_item')
            ->join('flash_sale', 'flash_sale.id', '=', 'flash_sale_item.flash_sale_id')
            ->whereIn('product_id', array_unique(self::$productIds))
            ->where('status', 'ACTIVE')
            ->where('end_at', '>=', $startDate->format('Y-m-d H:i:s'))
            ->get(['flash_sale_item.product_id', 'flash_sale.id', 'flash_sale.sale_percent', 'flash_sale.end_at', 'flash_sale.name', 'flash_sale.title']);

        foreach ($flashSales as $flashSale) {
            self::$flashSaleByProductIds[$flashSale->product_id] = $flashSale;
        }
    }

    /**
     * Set Product Ids
     *
     * @param $productId
     * @return void
     */
    public function setProductIds($productId)
    {
        self::$productIds[] = $productId;
    }

    /**
     * Set random flash sale
     *
     * @return void
     */
    public function setRandomFlashSale()
    {
        $startDate = new DateTime();
        self::$randomFlashSales = DB::table('flash_sale')
            ->where('start_at', '<=', $startDate->format('Y-m-d H:i:s'))
            ->where('end_at', '>=', $startDate->format('Y-m-d H:i:s'))
            ->where('status', 'ACTIVE')
            ->limit(10)
            ->get()
            ->toArray();
    }

    /**
     * @param $productId
     * @return array|string|string[]
     */
    private function getFlashSaleByProduct($productId)
    {
        if (array_key_exists($productId, self::$flashSaleByProduct)) {
            return self::$flashSaleByProduct[$productId];
        }
        try {
            $startDate = new DateTime();
            $flashSale = self::$flashSaleByProductIds[$productId] ?? false;
            if (!$flashSale) {
                $flashSales = self::$randomFlashSales;
                if ($flashSales) {
                    $random_keys = array_rand($flashSales);
                    $flashSale = $flashSales[$random_keys];
                    $oldFlashSale = DB::table('flash_sale_item')
                        ->where('product_id', $productId)
                        ->first();
                    if ($oldFlashSale) {
                        DB::table('flash_sale_item')->where('id', $oldFlashSale->id)->update([
                            'flash_sale_id' => $flashSale->id,
                            'updated_at' => new \Datetime(),
                        ]);
                    } else {
                        DB::table('flash_sale_item')->insert([
                            'flash_sale_id' => $flashSale->id,
                            'product_id' => $productId,
                            'created_at' => new \Datetime(),
                            'updated_at' => new \Datetime(),
                        ]);
                    }
                }
            }
            $flashSaleText = '';
            if ($flashSale) {
                $flashSaleText = $flashSale->title ?? '';
                $percentFlashSale = $flashSale->sale_percent ?? 30;
                $endTime = new DateTime($flashSale->end_at);
                $diff = $startDate->diff($endTime);
                $flashSaleRemain = ($diff->days * 24 + $diff->h) != 0 ? ($diff->days * 24 + $diff->h) : 0;
                $monthNames = [
                    __('January #date'), __('February #date'), __('March #date'), __('April #date'), __('May #date'), __('June #date'),
                    __('July #date'), __('August #date'), __('September #date'), __('October #date'), __('November #date'), __('December #date')
                ];
                if ($flashSaleText == "") {
                    if ($flashSaleRemain > 24) {
                        $text = __('#percent% off sale #flashSaleName ends at #date');
                        $flashSaleText = str_replace('#percent', $percentFlashSale, $text);
                        $flashSaleText = str_replace('#date', str_replace('#date', $endTime->format('d'), $monthNames[$endTime->format('m') - 1]), $flashSaleText);
                    } else {
                        $text = __('#percent% off sale #flashSaleName for the next #hour');
                        $flashSaleText = str_replace('#percent', $percentFlashSale, $text);
                        if ($flashSaleRemain == 0) {
                            $flashSaleText = str_replace('#hour', $diff->i <= 1 ? __('1 minute') : $diff->i . " " . __('minutes'), $flashSaleText);
                        } else {
                            $flashSaleText = str_replace('#hour', $flashSaleRemain == 1 ? __('1 hour') : $flashSaleRemain . " " . __('hours'), $flashSaleText);
                        }
                    }
                    $flashSaleText = str_replace('#flashSaleName', $flashSale->name, $flashSaleText);
                } else {
                    if ($flashSaleRemain > 24) {
                        $text = __('ends at #date');
                        $flashSaleText .= " " . str_replace('#date', str_replace('#date', $endTime->format('d'), $monthNames[$endTime->format('m') - 1]), $text);
                    } else {
                        if ($flashSaleRemain == 0) {
                            $text = __("for the next #hour");
                            $flashSaleText .= " " . str_replace('#hour', $diff->i <= 1 ? __('1 minute') : $diff->i . " " . __('minutes'), $text);
                        } else {
                            $text = __('for the next #hour');
                            $flashSaleText .= " " . str_replace('#hour', $flashSaleRemain == 1 ? __('1 hour') : $flashSaleRemain . " " . __('hours'), $text);
                        }
                    }
                }
            }
            
            self::$flashSaleByProduct[$productId] = $flashSaleText;
            return $flashSaleText;
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * build price
     * 
     * @param Array $product
     * @return Array
     */
    private function buildPrice($product)
    {
        if (floatval($product['high_price']) > 0) {
            $product['sale_off'] = (floatval($product['high_price']) - floatval($product['price'])) * 100 / floatval($product['high_price']);
        } else {
            $product['sale_off'] = 0;
        }
        $product['sale_percent'] = '-' . $product['percent_sale'] . '%';
        $product['price'] = floatval($product['price']);
        $product['high_price'] = floatval($product['high_price']);
        $product['display_drop_price'] = formatPrice($product['high_price'] - $product['price']);

        return $product;
    }

    /**
     * build sku id
     * 
     * @param Array $products
     * @return Array
     */
    private function buildSkuId($products)
    {
        if ($products) {
            $colorVariantId = $filter['color_variant_id'] ?? null;
            $pIds = $this->mapProductIds($products);
            $query = DB::table('product_sku_value as psv')
                ->join('product_sku as ps', 'psv.sku_id', '=', 'ps.id')
                ->whereIn('psv.product_id', $pIds);
            if ($colorVariantId) {
                $query->where('psv.variant_option_id', $colorVariantId);
            } else {
                $query->where('ps.is_default', 1);
            }

            $skuIds = $query->groupBy(['psv.product_id'])
                ->pluck('ps.id', 'ps.product_id');
            foreach ($products as &$item) {
                $pId = $item['id'];
                if (array_key_exists($pId, self::$skuIdByProductId)) {
                    $item['sku_id'] = self::$skuIdByProductId[$pId];
                } else {
                    if (!isset($skuIds[$pId]) || !$skuIds[$pId]) {
                        $skuTemplate = $this->getSkuTemplate($item['id'], $colorVariantId);
                        if ($skuTemplate) {
                            $item['sku_id'] = $skuTemplate->id;
                        } else {
                            $item['sku_id'] = $item['id'];
                        }
                    } else {
                        $item['sku_id'] = $skuIds[$pId];
                    }

                    self::$skuIdByProductId[$pId] = $item['sku_id'];
                }

                $this->setSkuIds($item['sku_id']);
            }
        }

        return $products;
    }

    protected function mapProductIds($items): array
    {
        $pIds = [];
        if ($items) {
            foreach ($items as $item) {
                $pIds[] = $item['id'];
            }
        }

        return $pIds;
    }

    /**
     * Build Sku Template Default By Template Ids And Color Id
     *
     * @param $colorId
     * @return void
     */
    public function setSkuTemplateDefaultByTemplateIdsAndColorId($colorId)
    {
        $items = DB::table('product_template_sku as ps')
            ->join('product_template_sku_value as psv', 'psv.sku_id', '=', 'ps.id')
            ->whereIn('ps.template_id', array_unique(array_values(self::$templateIdsByProductIds)))
            ->where('psv.variant_option_id', $colorId)
            ->get(['ps.id', 'ps.image_url', 'ps.template_id']);

        foreach ($items as $item) {
            $result = new \stdClass();
            $result->id = $item->id;
            $result->image_url = $item->image_url;
            self::$skuTemplateDefaultByTemplateIdsColorId[$item->template_id][$colorId] = $result;
        }
    }

    public function getSkuTemplate($productId, $colorVariantId) {
        $result = null;
        $templateId = $this->getTemplateId($productId);
        if ($templateId > 0) {
            if ($colorVariantId) {
                $sku = self::$skuTemplateDefaultByTemplateIdsColorId[$templateId][$colorVariantId] ?? null;
            } else {
                $sku = self::$skuTemplateDefaultByTemplateIds[$templateId] ?? null;
            }

            if ($sku && isset($sku->id)) {
                $result = $sku;
            }
        }

        return $result;
    }

    /**
     * Build Sku Template Default By Template Ids
     *
     * @return void
     */
    public function setSkuTemplateDefaultByTemplateIds()
    {
        $items = DB::table('product_template_sku as ps')
            ->join('product_template_sku_value as psv', 'psv.sku_id', '=', 'ps.id')
            ->whereIn('ps.template_id', array_unique(array_values(self::$templateIdsByProductIds)))
            ->where('ps.is_default', 1)
            ->get(['ps.id', 'ps.image_url', 'ps.template_id']);

        foreach ($items as $item) {
            $result = new \stdClass();
            $result->id = $item->id;
            $result->image_url = $item->image_url;
            self::$skuTemplateDefaultByTemplateIds[$item->template_id] = $result;
        }
    }

    /**
     * Build template ids by product ids
     *
     * @return void
     */
    public function setTemplateIdsByProductIds()
    {
        self::$templateIdsByProductIds = DB::table('product_n_template')
            ->whereIn('product_id', array_unique(self::$productIds))
            ->pluck('template_id', 'product_id')
            ->toArray();
    }

    protected function getTemplateId($productId) {
        return self::$templateIdsByProductIds[$productId] ?? 0;
    }

    /**
     * Get Category Options
     *
     * @return array
     */
    public function getCategoryOptions(): array
    {
        $keys = array_rand(self::$categoryIds, 5);
        $categoryIds = [];
        foreach ($keys as $key) {
            $categoryIds[] = self::$categoryIds[$key];
        }

        return DB::table('category')
            ->whereIn('id', $categoryIds)
            ->get(['id', 'name', 'slug'])
            ->toArray();
    }

    protected function initTimer() {
        $this->timer = [
            'start_time' => microtime(true)
        ];
        $this->currentTime = microtime(true);
    }

    protected function cpTimer($name) {
        if (isset($_GET['debug_timer'])) {
            $this->timer[$name] = number_format(microtime(true) -  $this->currentTime, 6);
            $this->currentTime = microtime(true);
        }

    }

    protected function persistenceTimer() {
        if (isset($_GET['debug_timer'])) {
            $this->timer['run_time'] =  number_format(microtime(true) -  $this->timer['start_time'], 6);
            \Log::info('persistence debug Timer', $this->timer);
        }

        if (isset($_GET['show_debug_timer'])) {
            echo '<pre>';
            echo print_r($this->timer);
            echo '</pre>';
            exit;
        }

    }
}
