<?php

namespace Modules\ZSearch\Controllers;

use App\Modules\ZSearch\Services\ElasticSearchIndexLite;
use App\Utils;
use App\Utils\SlugManager;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use Modules\ZSearch\Services\CampaignService;
use Modules\ZSearch\Services\ElasticSearchIndex;
use Modules\ZSearch\Services\ElasticsearchIndexHistory;
use Theme;
use Module;
use Illuminate\Http\Request;
use Modules\ZSearch\Models\Tag;
use Modules\ZSearch\Models\Post;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Models\Option;
use Modules\ZSearch\Models\Product;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Redis;
use Modules\ZSearch\Models\Category;
use Modules\ZSearch\Controllers\Controller;
use Modules\ZSearch\Services\AmazonDataDecor;
use Modules\ZSearch\Services\SearchDataDecor;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Services\EtsyDataDecor;
use Modules\ZSearch\Services\TagService;

class HomeController extends Controller
{
    const SEARCH_TRACKING = 'qs_tr';
    protected $elasticServiceConfig;
    protected $amazonDataDecor;
    protected $etsyDataDecor;
    protected $elasticService;
    protected $campaignService;
    protected $searchDataDecor;
    protected $tagService;
    protected $productsStat;
    protected $filterRequest;
    protected $users;
    protected $productVariantOption;
    protected $productVariant;
    protected $isAdmin;
    protected $timer = [];
    protected $currentTime = 0;
    const TYPE_PRODUCTS = 'products';
    const TYPE_PRODUCTS_TOP = 'products_top';
    const TYPE_PRODUCTS_SELLER = 'products_seller';
    protected $exceptionKeyword = [
        'motley crue poster',
        'motley crue'
    ];

    public function __construct()
    {
        $user = Auth::user();
        $this->isAdmin = false;
        if ($user && isset($user->role) && in_array($user->role, ['STAFF','ADMIN'])) {
            View::share('isAdmin', true);
            $this->isAdmin = true;
        } else {
            View::share('isAdmin', false);
        }
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticServiceConfig = $elasticSearchConfig;
        $this->elasticService = new ElasticSearchService($elasticSearchConfig);
        $this->searchDataDecor = new SearchDataDecor();
        $this->tagService = new TagService();
        $this->campaignService = new CampaignService();
        $this->etsyDataDecor = new EtsyDataDecor();
    }

    public function index($tagTitle = null, Request $request) {
        $tagId = null;
        $this->initTimer();
        $parameters = Route::current()->parameters();
        $routeName = Route::currentRouteName();
        Log::info('search params', [$parameters, $request->all(), request()->header('user-agent'), Utils::getIP()]);
        if (!in_array($routeName, ['search', 'merch'])) {
            if (!empty($parameters['id'])) {
                $tagId = $parameters['id'];
            }
            $tagTitle = isset($parameters['tagTitle']) ? $parameters['tagTitle'] : $tagTitle;
            if (!empty($tagTitle)) {
                $tags = Tag::where('slug', '=', $tagTitle)->first();
                if (empty($tags) && $tagId) {
                    $tags = Tag::find($tagId);
                }
                if (!empty($tags)) {
                    $tagTitle = $tags->title;
                    $tagId = $tags->id;
                }
            }
        }
        if (!$tagId && $routeName == 'tag::explore') {
            return redirect(route('pod::expore'), 301);
        }
        if ($tagId && isset($tags)) {
            $this->tagId = $tagId;
            View::share('currentTagDetail', $tags);
        }

        $pageType = (!empty($tagId)) ? 'tag' : 'search';
        $keyword = null;

        if ($request->has('q')) {
            $keyword = trim(strip_tags($request->input('q')));
            $keyword = trim(preg_replace('/\s+/', ' ', $keyword));
            View::share('query', $keyword);

        } else if (isset($parameters['keyword']) && $routeName == 'merch') {
            $keyword = $parameters['keyword'];
            if ($keyword) {
                $keyword = str_replace('-', ' ', strip_tags($keyword));
            }
        } else if (!empty($tagTitle)) {
            $keyword = str_replace('-', ' ', strip_tags($tagTitle));
            View::share('tagTitle', $keyword);
        }
        if ($this->isIgnoreKeyword($keyword)) {
            abort(404);
        }
        $data = $request->all();
        $keys = [
            'size_variant_id',
            'color_variant_id',
            'type_variant_id',
        ];
        $isAutoContent = false;
        $data['keyword'] = $keyword;
        $data['origin_keyword'] = $keyword;
        $data['page_size'] = 60;
        if (empty($data['page_id'])) {
            $data['page_id'] = 0;
        }

        if (isset($data['category']) && $data['category']) {
            $data['category'] = explode(',', $data['category']);
        }
        if (isset($data['category_id']) && $data['category_id']) {
            $data['category'] = explode(',', $data['category_id']);
        }
        if (isset($data['tag']) && $data['tag']) {
            $data['tag'] = explode(',', $data['tag']);
        }
        if (isset($data['filter']) && $data['filter']) {
            $data['filter'] = explode(',', $data['filter']);
        }
        if (!isset($data['order'])) {
            $data['order'] = '';
            if ($keyword && strpos(strtolower($keyword), 'custom') !== false) {
                $data['order'] = 'sold';
            }
        }
        if (isset($data['category'])) {
            View::share('selectedCategory', $data['category']);
        }
        foreach ($data as $index => $itemFilter) {
            if (is_array($itemFilter)
                && isset($itemFilter['type'])
                && isset($itemFilter['slug'])) {
                if ($itemFilter['type'] == 'size' && strtolower($keyword) != strtolower($itemFilter['slug'])) {
                    $indexKey = $itemFilter['type'] . '_variant_id';
                    $data[$indexKey] = $itemFilter['id'];
                    unset($data[$index]);
                } else if ($itemFilter['type'] == 'category') {
                    unset($data[$index]);
                }
            }
        }
        $bkKeyword = $keyword;
        $isDetachOk = false;
        $this->cpTimer('cp1');
        if (!isset($data['category'])) {
            $detachCategoryData = $this->searchDataDecor->detachCategory($keyword);
            if ($detachCategoryData && array_key_exists('category_ids', $detachCategoryData)) {
                $isDetachOk = true;
                $data['category_detach'] = $detachCategoryData['category_ids'];
                $keyword = isset($detachCategoryData['q']) && $detachCategoryData['q'] ? $detachCategoryData['q'] : $keyword;
                $data['keyword'] = $keyword;
            }
        }
        $this->cpTimer('cp2');

        $data['campaign'] = $this->campaignService->detachCampaign($keyword, $data);
        $productFilters = $data;
        $productFilters['is_customer'] = $request->header('is-customer');
        $this->setFilterRequest($data);
        $this->cpTimer('cp2.1');
        if (Theme::getSetting('prerender_search', true)) {
            if (Theme::getSetting('show_pending_product', false)) {
                $data['status'] = [
                    'ACTIVE',
                    'PENDING'
                ];
            }
            $minimumShouldMatchGlobal = isset($_GET['order']) && $_GET['order'];
            $minimumShouldStr = $minimumShouldMatchGlobal ? 1 : 0;
            $keyCache = 'cache::search::' . env('APP_LOCALE') . $keyword . $minimumShouldStr;
            $isCache = $this->isBot() || in_array(strtolower($keyword), ['cafepress', 'teepublic', 'spreadshirt']);
            if ($this->isExceptionKeyword($keyword)) {
                $isCache = true;
            }
            $result = null;
            if ($isCache) {
                $result = Cache::get($keyCache);
            }
            $this->cpTimer('cp2.2');
            if (!$result) {
                Log::info('search params cache miss', [request()->header('user-agent'), $keyword, $isCache, Utils::getIP()]);
                $result = $this->searchProduct($keyword, $minimumShouldMatchGlobal, false, $productFilters);
                if ($isCache) {
                    //Cache::put($keyCache, $result, 60);
                }
            } else {
                Log::info('search params cache hit', [$keyword]);
            }
            $this->cpTimer('cp3');
            $products = [];
            $meta = [
                'page_id' => 0,
                'page_size' => 0,
                'page_count' => 0,
                'has_next' => false,
                'total_count' => 0,
            ];

            $brands = [];
            $tags = [];
            $searchCategories = [];
            $filters = [];
            $displayedFilters = [];
            $priceRange = [];
            $filterCategory = [];
            $filterOptions = [];
            $relatedItems = [];
            if (!empty($result['hits']['hits'])) {
                $this->searchDataDecor->decorHits($result['hits']['hits']);
                $hits = $result['hits']['hits'];
                $result = $this->searchDataDecor->buildFilter($hits, $data);
                $items = $this->searchDataDecor->filter($hits, $productFilters);
                $this->cpTimer('cp4');
                if (!$items && $isDetachOk) {
                    $items = $hits;
                } else if (!$items && isset($productFilters['category'])) {
                    $relatedFilter = $productFilters;
                    unset($relatedFilter['category']);
                    $relatedItems = $this->searchDataDecor->filter($hits, $relatedFilter);
                    if ($relatedItems) {
                        $relatedItems = $this->searchDataDecor->sort($relatedItems, array_merge($data, [
                            'keyword' => $keyword,
                            'q' => $keyword,
                        ]));
                        $relatedItems = array_slice($relatedItems, 0, 20);
                        $relatedItems = $this->decorSearchItems($relatedItems, array_merge($data, [
                            'page_type' => $pageType
                        ]));
                        priceChangeDecor($relatedItems);
                    }
                }
                $this->cpTimer('cp5');
                $items = $this->searchDataDecor->sort($items, array_merge($data, [
                    'keyword' => $keyword,
                    'q' => $keyword,
                ]));
                $this->cpTimer('cp6');
                $items = $this->searchDataDecor->handleDuplicateDesign($items, $productFilters);
                $this->cpTimer('cp7');

                $tags = $this->tagService->getTagData($data, $tagTitle);
                $this->cpTimer('cp8');
                if (!$tags || !count($tags)) {
                    $tags = $this->searchDataDecor->buildTag(array_slice($items, 0, 50));
                }

                if (function_exists('getTagHot')) {
                    $actual_link = "$_SERVER[REQUEST_URI]";
                    $tagsConfig = getTagHot($actual_link);
                    $tags = array_merge($tagsConfig, $tags);
                }
                $result['tags'] = $tags;
                $meta = $this->getMetaData($data, count($items));
                if (isset($this->filterRequest['campaign']) && $this->filterRequest['campaign']) {
                    $items = $this->campaignService->distribute($this->filterRequest['campaign'], $items, $meta);
                }
                $this->cpTimer('cp9');
                $products = array_slice($items, $data['page_id'] * $data['page_size'], $data['page_size']);
                if (isset($this->filterRequest['campaign']) && $this->filterRequest['campaign'] && $products) {
                    $products = $this->campaignService->sortData($products, 'campaign_sort');
                }
                $this->cpTimer('cp10');
                $products = $this->decorSearchItems($products, array_merge($data, [
                    'page_type' => $pageType
                ]));
                $this->cpTimer('cp11');
                if (function_exists('checkDuplicateTitle')) {
                    checkDuplicateTitle($products);
                }
                $brands = $result['brands'];
                $searchCategories = $result['categories'];
                $priceRange = $result['priceRange'];
                $displayedFilters = $result['displayedFilters'];
                $filterOptions = $result['filterOptions'];
            }
            if (isset($productFilters['category_id'])) {
                unset($productFilters['category_id']);
            }
            $allFilters = [];
            $filtered = [];
            $tempFilters = [];

            $isTagPage = ((!empty($tagTitle)) || $tagId || $request->has('is_tag_page')) && Route::currentRouteName() != "search";
            $filteredCategory = '';
            $this->cpTimer('cp12');
            foreach ($displayedFilters as $key => $value) {
                if(!is_array($value)) {
                    if($key == 'price') {
                        $url = buildFilterUrl('minPrice', false, 'maxPrice', false);
                        if ($pageType == 'tag') {
                            $url = SlugManager::clearFilterUrl([
                                'type' => $key
                            ]);
                        }
                        $tempFilters[] = [
                            'url' => $url,
                            'checked' => true,
                            'text' => $value,
                            'type' => $key
                        ];
                    } else {
                        $url = buildFilterUrl($key, false);
                        $valueArr = explode('::', $value);
                        $strSlug = isset($valueArr[1]) ? $valueArr[1] : $valueArr[0];
                        $url = buildFriendlyFilterUrl($strSlug, true, $key);
                        if ($key == 'category') {
                            $filteredCategory = $strSlug;
                        }
                        $tempFilters[] = [
                            'url' => $url,
                            'checked' => true,
                            'text' => $valueArr[0],
                            'slug' => $strSlug,
                            'type' => $key
                        ];
                    }
                } else {
                    $urls = buildFilterUrlArray($key);
                    if(count($urls)) {
                        foreach($urls as $key => $url) {
                            $tempFilters[] = [
                                'url' => $url,
                                'checked' => true,
                                'text' => isset($value[$key]) ? $value[$key] : '',
                                'type' => $key
                            ];
                        }
                    }
                }
            }
            $this->cpTimer('cp13');
            $cloneTempFilters = [];
            foreach ($tempFilters as $tempFilter) {
                if (!empty($tempFilter['type']) && $tempFilter['type'] == 'type_variant_id') {
                    array_unshift($cloneTempFilters, $tempFilter);
                } else {
                    $cloneTempFilters[] = $tempFilter;
                }
            }
            $tempFilters = $cloneTempFilters;
            $filtered = [
                'title' => __("Bộ lọc"),
                'id' => "filter",
                'filters' => $tempFilters,
                'text_filtered' => $this->getTextFiltered($tempFilters)
            ];
            if ($pageType == 'tag') {
                $dispatchesAttrs = request()->input('dispatches_attrs');
                if ($dispatchesAttrs) {
                    $result = [];
                    $keys = array_keys($dispatchesAttrs);
                    if ($keys && count($keys) >= 2 && $keys[0] == 'category' && $keys[1] == 'tag') {
                        $keyTmp = $keys[0];
                        $keys[0] = $keys[1];
                        $keys[1] = $keyTmp;
                    }
                    foreach ($keys as $key) {
                        $attr = $dispatchesAttrs[$key];
                        $attrName = null;
                        if ($key == 'category') {
                            $category = Category::find($attr['id'], ['name']);
                            $attrName = $category && isset($category->name) && $category->name ? $category->name : $attrName;
                        } else if ($key == 'price') {
                            $minPrice = isset($attr['min_price']) ? $attr['min_price'] : 0;
                            $maxPrice = isset($attr['max_price']) ? $attr['max_price'] : 0;

                            if ($maxPrice > 0) {
                                $minPrice = formatPrice($minPrice);
                                $maxPrice = formatPrice($maxPrice);
                                $attrName = $minPrice
                                    ? __("Từ") . " $minPrice -  $maxPrice "
                                    : __("Dưới") ." $minPrice";
                            }
                        } else {
                            $tag = Tag::where("slug", $attr['slug'])->first();
                            if ($tag) {
                                $attrName = ucfirst($tag->title);
                            } else {
                                $attrName = ucfirst(str_replace('-', ' ', $attr['slug']));
                            }
                        }
                        $result[] = $attrName;
                    }
                    if ($result) {
                        $filtered['text_filtered'] = implode(' ', $result);
                    }
                }

            }
            $tempFilters = [];
            $categoryItem = [];
            $this->cpTimer('cp14');
            foreach ($searchCategories as $item) {
                $itemSlug = ($item['slug']);
                $isChecked = false;
                if (isset($item['id']) && isset($_GET['category']) && $item['id'] == $_GET['category']) {
                    $categoryItem = $item;
                }
                if (isset($item['id']) && isset($data['category']) && is_array($data['category']) && in_array($item['id'], $data['category'])) {
                    $categoryItem = $item;
                }
                if (!empty($parameters) && isset($parameters['slug'])) {
                    $attr = [
                        'id' => $item['id'],
                        'type' => 'category',
                        'slug' => $item['slug']
                    ];
                    $url = SlugManager::url ($attr);
                    $isChecked = SlugManager::isSelected($attr);
                } else {
                    $isChecked = false;
                    foreach ($filtered['filters'] as $fItem) {
                        $fSlug = isset($fItem['slug']) ? $fItem['slug'] : '';
                        if ($fSlug == $itemSlug && $fItem['type'] == 'category') {
                            $isChecked = true;
                            break;
                        }
                    }
                    $url = buildFilterUrl('category', $item['id']);
                }
                $tempFilters[] = [
                    'url' => $url,
                    'checked' => $isChecked,
                    'image_url' => isset($item['image_url']) ? $item['image_url'] : '',
                    'text' => $item['name'],
                    'count' => isset($item['count']) ? $item['count'] : false,
                ];
            }

            if ($isDetachOk) {
                $topCategoryKeywords = $this->searchDataDecor->getSuggestionCategoryKeywords($data['keyword'], $tempFilters);
                View::share('topCategoryKeywords', $topCategoryKeywords);
            }

            $allFilters[] = [
                'title' => __("Danh mục"),
                'id' => "category",
                'filters' => $tempFilters
            ];
            $this->cpTimer('cp15');
            if (isset($priceRange) && count($priceRange) > 1) {
                $tempFilters = [];
                foreach ($priceRange as $item) {
                    $from = formatPrice($item['from']);
                    $to = formatPrice($item['to']);
                    $url = buildFilterUrl('minPrice', $item['from'], 'maxPrice', $item['to']);
                    $isChecked = isSelectedFilter('minPrice', $item['from'], 'maxPrice', $item['to']);
                   

                    $tempFilter = [
                        'url' => $url,
                        'checked' => $isChecked,
                        'text' => $item['from']
                            ? __("Từ") . " $from -  $to "
                            : __("Dưới") ." $to"
                    ];
                    $tempFilters[] = $tempFilter;
                }
                View::share('priceRanges', $tempFilters);
                $allFilters[] = [
                    'title' => __("Giá"),
                    'id' => 'price',
                    'filters' => $tempFilters
                ];
            }

            if ($brands) {
                $tempFilters = [];
                foreach ($brands as $item) {
                    $tempFilters[] = [
                        'url' => buildFilterUrl('brand', $item['slug']),
                        'text' => $item['name'],
                        'count' => (!empty($item['count'])) ? $item['count'] : 0,
                        'checked' => isSelectedFilter('brand', $item['slug']),
                    ];
                }
                $allFilters[] = [
                    'title' => __("Thương hiệu"),
                    'filters' => $tempFilters,
                    'id' => 'brand',
                ];

            }
            if (count($filters)) {
                foreach($filters as $filter){
                    $tempFilters = [];
                    foreach ($filter['values'] as $filterValues) {
                        $tempFilters[] = [
                            'url' => buildFilterUrl($filter['slug'], $filterValues['slug']),
                            'text' => $filterValues['name'],
                            'count' => $filterValues['count'],
                        ];

                    }
                    $allFilters[] = [
                        'title' => $filter['display_name'],
                        'id' => $filterValues['slug'],
                        'filters' => $tempFilters
                    ];
                }
            }

            if ($filterOptions) {
                foreach ($filterOptions as $key => $variants) {
                    $tempFilters = [];
                    foreach ($variants as $item) {
                        if (!empty($parameters) && isset($parameters['slug'])) {
                            $attr = [
                                'id' => $item['id'],
                                'type' => $item['variant_slug'],
                                'slug' => $item['slug']
                            ];
                            $url = SlugManager::url ($attr);
                            $isChecked = SlugManager::isSelected($attr);
                        } else {
                            $url = buildFilterUrl($item['variant_slug'] . '_variant_id', $item['id']);
                            $isChecked = isSelectedFilter($item['variant_slug'] . '_variant_id', $item['id']);
                        }
                        $tempFilters[] = [
                            'id' => $item['id'],
                            'url' => $url,
                            'slug' => $item['slug'],
                            'text' => ucfirst($item['name']),
                            'checked' => $isChecked,
                            'image_url' => $item['image_url'] ?? '',
                            'theme_color' => checkThemeVariantColor($item['slug']) ?? ''
                        ];
                    }
                    $allFilters[] = [
                        'title' => __(ucfirst($key)),
                        'type' => $key,
                        'filters' => $tempFilters,
                        'id' => 'user_id',
                    ];
                }
            }
            $sorter = [
                [ 'name' => __("Mua nhiều"), 'value' => 'sold' ],
                [ 'name' => __("Mới nhất"), 'value' => 'latest' ],
                [ 'name' => __("Xem nhiều"), 'value' => 'view' ],
                [ 'name' => __("Giá thấp"), 'value' => 'low_price'],
                [ 'name' => __("Giá cao"), 'value' => 'high_price'],
                [ 'name' => __("Giảm giá"), 'value' => 'sale' ],
            ];
            $this->cpTimer('cp16');
            foreach ($sorter as $key => $item) {
                $sorter[$key]['url'] = buildFilterUrl('order', $item['value']);
                $sorter[$key]['checked'] = isSelectedFilter('order', $item['value']);
            }
            $allFilters = allFiltersOrdered($allFilters);
            View::share('sorter', $sorter);
            View::share('allFilters', $allFilters);
          
            if (!empty($filterCategory) && !empty($filterCategory['category_metas'])) {
                foreach ($filterCategory['category_metas'] as $item) {
                    if ($item['key'] == 'meta_description') {
                        View::share('metaDescription', $item['value']);
                    }
                    if ($item['key'] == 'meta_title') {
                        View::share('metaTitle', $item['value']);
                    }
                }
            }

            View::share('category', $filterCategory);
            View::share('categoryFilter', $categoryItem);
            $strCodes = "";
            $amzUrls = [];
            $shareProduct = ['images' => []];
            $shareItemList = [];
            foreach ($products as $item) {
                $strCodes .= $item['sku'] . ',';
                if (array_key_exists('amz_url', $item) && $item['amz_url']) {
                    $amzUrls[] = $item['amz_url'];
                }
                if (count($shareProduct['images']) < 10 && !empty($item['image_url'])) {
                    array_push($shareProduct['images'], $item['image_url']);
                }
                if (!empty($item['url'])) {
                    array_push($shareItemList, ['url' => url(clroute($item['url']))]);
                }
            }
            View::share('shareItemList', $shareItemList);
            View::share('strCode', substr($strCodes, 0, -1));
            View::share('pageType', 'view_item_list');

            Module::action('viewPage', [
                'type' => 'search',
                'data' => [
                    'product_ids' => array_column($products, 'id'),
                    'post_ids' => [],
                    'id' => null,
                ]
            ]);

            /** for tags page */
            if (!empty($tagTitle)) {
                View::share('tag', ['title' => $tagTitle]);
                if ($isAutoContent) {
                    View::share('cloneTag', [ 'title' => $tagTitle, 'slug' => str_replace(' ', '-', strtolower($tagTitle)) ]);
                }
                foreach ($tags as $itemTag) {
                    if (isset($itemTag['id']) && $itemTag['id'] == $tagId) {
                        View::share('tag', $itemTag);
                        if ($isAutoContent) {
                            View::share('cloneTag', $itemTag);
                        }
                        $user = Auth::user();
                        if ($user && isset($user->role) && in_array($user->role, ['STAFF','ADMIN']) ) {
                            View::share('is_show_tagid', true);
                        }
                        break;
                    }
                }
            }
            
            $keyword = $data['origin_keyword'];
            if (!$isAutoContent && !empty($tagTitle) && $tagId) {
                $post = Post::whereHas('postTag', function ($query) use ($tagId) {
                    $query->where('tag.id', $tagId);
                })->where('status', 'PENDING')->first();

                if (!empty($post)) {
                    view::share('post', $post->toArray());
                }
            }
            $metaDataConfig = null;
            if (\Module::isActive('Seo') && function_exists('getMetaDataConfig')) {
                $metaDataConfig = getMetaDataConfig();
                if ($metaDataConfig) {
                    if ($metaDataConfig->content_footer) {

                        if (($metaDataConfig->content_footer == strip_tags($metaDataConfig->content_footer))) {
                            $metaDataConfig->content_footer = nl2br($metaDataConfig->content_footer);
                        }

                        view::share('post', [
                            'content' => $metaDataConfig->content_footer
                        ]);
                    }
                    if ($metaDataConfig->content) {
                        view::share('metaDescription', $metaDataConfig->content);
                    }
                }
            }
            $this->cpTimer('cp17');
            priceChangeDecor($products);
            $this->cpTimer('cp17.1');
            View::share('filtered', $filtered);
            $currentTag = !empty($tagTitle) && $tagId ? ucfirst($tagTitle) : '';
            $currentTagSlug = !empty($tagTitle) && $tagId ? str_replace(' ', '-', strtolower(trim($tagTitle))) : '';
            
            $categoryByKeyword = Category::where('type', 'PRODUCT')
                ->where('slug', str_slug($keyword))
                ->select('slug', 'id')
                ->first();
            $this->cpTimer('cp17.2');
            $routeName = Route::currentRouteName();
            if ($routeName == "search") {
                if ($request->has('q')) {
                    $canonicalUrl = $request->url() . '?' . http_build_query($request->only(['q']));
                    $searchUrl = '/search?' . http_build_query($request->only(['q']));
                    $hreflangs = [];
                    $currentLocale = env('APP_LOCALE');
                    foreach (getModuleLocale() as $item) {
                        if (!$item['enable']) continue;

                        if ($item['locale'] == 'us') {
                            $hreflangs[$item['locale']] = $searchUrl;
                        } else {
                            $hreflangs[$item['locale']] = '/' . $item['locale'] . $searchUrl;
                        }
                    }
                    View::share('hreflangs', $hreflangs);
                } else {
                    $canonicalUrl = \URL::full();
                    if ($categoryByKeyword) {
                        $canonicalUrl = route('categoryOrTag', ['slug' => $categoryByKeyword->slug]);
                    } else {
                        $tagByKeyword = Tag::where('slug', str_slug($keyword))
                            ->select('slug', 'id')
                            ->first();
                        if ($tagByKeyword) {
                            $canonicalUrl = route('categoryOrTag', ['slug' => $tagByKeyword->slug]);
                        }
                    }

                }
                View::share('canonical_url', $canonicalUrl);
                if (!$request->has('q')) {
                    View::share('robots_index', 'noindex, follow');
                } 
            }
            $this->cpTimer('cp17.3');
            if (!empty($tagTitle) && $tagId){
                $slugUrlTag = '/explore/' . $currentTagSlug;
                $shareBreadcrumb = [
                    [ 'id' => url(clroute('/')), 'name' => env('APP_NAME') ],
                    [ 'id' => url(clroute('/explore')), 'name' => __('Explore') ],
                    [ 'id' => url(clroute($slugUrlTag)), 'name' => $tagTitle ],
                ];
                $shareProduct['brand'] = env('APP_NAME');
                $shareProduct['lowPrice'] = 0;
                $shareProduct['highPrice'] = 0;
                if (!empty($priceRange)) {
                    $shareProduct['lowPrice'] = $priceRange[0]['to'];
                    $shareProduct['highPrice'] = $priceRange[count($priceRange) - 1]['to'];
                }
                $shareProduct['offerCount'] = intval(time() / 100000);
                $shareProduct['ratingValue'] = 5;
                $shareProduct['reviewCount'] = $tagId;
                $hreflangs = [];
                $currentLocale = env('APP_LOCALE');
                foreach (getModuleLocale() as $item) {
                    if (!$item['enable']) continue;

                    if ($item['locale'] == 'us') {
                        $hreflangs[$item['locale']] = $slugUrlTag;
                    } else {
                        $hreflangs[$item['locale']] = '/' . $item['locale'] . $slugUrlTag;
                    }
                }
                View::share('shareBreadcrumb', $shareBreadcrumb);
                View::share('shareProduct', $shareProduct);
                View::share('hreflangs', $hreflangs);
            }
            $this->cpTimer('cp17.4');
            View::share('relatedItems', $relatedItems);
            $this->cpTimer('cp18');
            if (function_exists("getFreeShippingCate")) {
                $freeShippingCate = getFreeShippingCate();
                foreach ($products as $key => $product) {
                    if (isset($product["categories"]) && count($product["categories"]) > 0) {
                        $countCate = count($product["categories"]);
                        if (isset($product["categories"][$countCate - 1]["id"]) && in_array($product["categories"][$countCate - 1]["id"], $freeShippingCate)) {
                            $products[$key]["is_free_shipping"] = 1;
                        }
                    }
                }
            }
            $this->cpTimer('cp19');
            if ($pageType == 'search' && $keyword && str_contains(strtolower($keyword), 'smokey the bear')) {
                $products = [];
                $meta = $this->getMetaData([], 0);
            }
            $status = 200;
            if (!$products || count($products) == 0) {
                $status = 404;
            }
            View::share('criteoProducts', array_slice($products, 0, 3));
            $breadcrumbs = [];
            if (!empty($tagTitle) && $tagId) {
                $breadcrumbs = self::buildBreadcrumbs($tagTitle, $currentTagSlug);
            }
            $this->cpTimer('cp20');
            $this->persistenceTimer();
            View::share('meta', $meta);
            $countPickProduct = 0;
            return response()->view((!empty($tagTitle)) && $tagId ? 'tag' : 'search', compact(
                'products',
                'meta',
                'brands',
                'tags',
                'filters',
                'searchCategories',
                'productFilters',
                'priceRange',
                'filterCategory',
                'displayedFilters',
                'filtered',
                'keyword',
                'currentTag',
                'filteredCategory',
                'currentTagSlug',
                'breadcrumbs'
            ))->setStatusCode($status);
        } else {
            return view('search', compact('productFilters'));
        }
    }

    protected function initTimer() {
        $this->timer = [
            'start_time' => microtime(true)
        ];
        $this->currentTime = microtime(true);
    }

    protected function cpTimer($name) {
        if (isset($_GET['debug_timer'])) {
            $this->timer[$name] = number_format(microtime(true) -  $this->currentTime, 6);
            $this->currentTime = microtime(true);
        }

    }

    protected function persistenceTimer() {
        if (isset($_GET['debug_timer'])) {
            $this->timer['run_time'] =  number_format(microtime(true) -  $this->timer['start_time'], 6);
            Log::info('persistence debug Timer', $this->timer);
        }

        if (isset($_GET['show_debug_timer'])) {
            echo '<pre>';
            echo print_r($this->timer);
            echo '</pre>';
            exit;
        }

    }
    
    public static function decorProducts($products , $keyWord = null) {
        $etsyDataDecor = new EtsyDataDecor();
        $products = $etsyDataDecor->decor($products);
        priceChangeDecor($products);
        return $products;
    }

    public static function buildBreadcrumbs($tagTitle, $tagSlug = null) {
        return [
            [
                'name' => __('Explore'),
                'url' => clroute('explore')
            ],
            [
                'name' => ucwords($tagTitle),
                'url' => route('tag::explore', [
                    'tagTitle' => (!empty($tagSlug)) ? $tagSlug : $tagTitle
                ]),
            ],
        ];

    }

    public static function getTagSchemaData($currentTagDetail)
    {
        // get description
        $storeName = getOption('general.store_name');
        $routeName = 'tag';

        if (isset(getOption($routeName)->description) && getOption($routeName)->description) {
            $description = getOption($routeName)->description;
        } else {
            $description = $currentTagDetail->title;
        }

        if (isset(getOption($routeName)->title) && getOption($routeName)->title) {
            $title = getOption($routeName)->title;
        } else {
            $title = $currentTagDetail->title;
        }

        $title = str_replace(['{name}', '{store_name}'], [$currentTagDetail->title, $storeName], $title);
        $description = str_replace(['{name}', '{store_name}'], [$currentTagDetail->title, $storeName], $description);

        $tagMetas = DB::table('tag_meta')
            ->where('tag_id', $currentTagDetail->id)
            ->limit(10)
            ->get();

        $imageUrls = [];
        $highPrice = 0;
        $lowPrice = 0;
        $ratingValue = 0;
        $ratingCount = 0;

        foreach ($tagMetas as $meta) {
            switch ($meta->key) {
                case 'top_product_images' : $imageUrls = json_decode($meta->string_value); break;
                case 'high_price' : $highPrice = $meta->number_value; break;
                case 'low_price' : $lowPrice = $meta->number_value; break;
                case 'rating_value' : $ratingValue = $meta->number_value; break;
                case 'rating_count' : $ratingCount = $meta->number_value; break;
            }
        }

        if (!$lowPrice) {
            $lowPrice = $highPrice;
        }
        if (!$highPrice) {
            $highPrice = $lowPrice;
        }

        if (!$ratingCount) {

            $ratingValue = rand(4, 5);
            if ($ratingValue < 5) {
                $ratingValue += rand(0, 10) / 10;
            }
            $ratingCount = rand(800, 2000);

            self::tagMetaUpdateOrCreate($currentTagDetail->id, 'rating_value', $ratingValue, 'number_value');
            self::tagMetaUpdateOrCreate($currentTagDetail->id, 'rating_count', $ratingCount, 'number_value');
        }

        return [
            'title' => $title,
            'description' => $description,
            'imageUrls' => $imageUrls,
            'images' => $description,
            'highPrice' => $highPrice,
            'lowPrice' => $lowPrice,
            'ratingCount' => $ratingCount,
            'ratingValue' => $ratingValue,
        ];
    }

    public function getProductByFilter($keyword, $dataFilter)
    {
        $result = $this->searchProduct($keyword, false);
        $hits = $result['hits']['hits'];

        $items = $this->searchDataDecor->filter($hits, $dataFilter);
        $items = $this->searchDataDecor->sort($items, array_merge($dataFilter, [
            'keyword' => $keyword,
            'q' => $keyword,
        ]));
        $items = $this->searchDataDecor->handleDuplicateDesign($items, $dataFilter);

        $products = array_slice($items, $dataFilter['page_id'] * $dataFilter['page_size'], $dataFilter['page_size']);
        $products = $this->decorSearchItems($products, $dataFilter);

        return $products;
    }

    protected function searchProduct($keyword, $minimumShouldMatchGlobal = false, $useOnlyPhasePrefix = false, $filter = []) {
        $routeName = Route::currentRouteName();
        $isUnique = false;
        $dispatchesAttrs = request()->input('dispatches_attrs');
        if ($useOnlyPhasePrefix) {
            $filterData = $this->_buildPhrasePrefixFilterData($keyword, $minimumShouldMatchGlobal);
        } else { 
            $filterData = $this->_buildFilterData($keyword, $minimumShouldMatchGlobal, $dispatchesAttrs);
        }
        $this->cpTimer('cp2.3');
        $result = $this->elasticService->searchDocument($filterData, 'products', $this->elasticServiceConfig['index']);
        $this->cpTimer('cp2.4');
        if (isset($this->filterRequest['category_detach']) && (!isset($result['hits']['hits']) || !$result['hits']['hits'])) {
            unset($this->filterRequest['category_detach']);
            $filterData = $this->_buildFilterData($keyword, $minimumShouldMatchGlobal, $dispatchesAttrs);
            $result = $this->elasticService->searchDocument($filterData, 'products', $this->elasticServiceConfig['index']);
        } else if (isset($this->filterRequest['category']) && $this->filterRequest['category']) {
            $this->decorCategory($result, $filterData);

        }

        $this->cpTimer('cp2.5');
        if ($minimumShouldMatchGlobal && (!isset($result['hits']['hits']) || !$result['hits']['hits'])) {
            if ($useOnlyPhasePrefix) {
                $filterData = $this->_buildPhrasePrefixFilterData($keyword);
            } else { 
                $filterData = $this->_buildFilterData($keyword, false, $dispatchesAttrs);
            }
            $result = $this->elasticService->searchDocument($filterData, 'products', $this->elasticServiceConfig['index']);
        }
        $this->cpTimer('cp2.6');
        if ($this->isExistsApostrophes($keyword)) {
            $keyword = str_replace('\'', '', $keyword);
            $otherResult = $this->searchProduct($keyword, $minimumShouldMatchGlobal, $useOnlyPhasePrefix);
            if (isset($result['hits']['hits']) && isset($otherResult['hits']['hits'])) {
                $result['hits']['hits'] = array_merge($result['hits']['hits'], $otherResult['hits']['hits']);
                $isUnique = true;
            }
        }
        $this->cpTimer('cp2.7');
        if (!$useOnlyPhasePrefix && $routeName != 'tag::explore') {
            $bestProduct = $this->searchBestProduct($keyword, $minimumShouldMatchGlobal);
            if ($bestProduct) {
                $result['hits']['hits'] = array_merge($bestProduct, $result['hits']['hits']);
                $isUnique = true;
            }
        }

        if ($isUnique) {
            $result['hits']['hits'] = $this->uniqueArray($result['hits']['hits']);
        }
        $this->cpTimer('cp2.8');
        return $result;
    }

    protected function decorCategory(&$result, $filterData) {
        $ids = [];
        if ($result['hits']['hits']) {
            foreach ($result['hits']['hits'] as $item) {
                $ids[] = $item['_source']['id'];
            }
        }
        $category = DB::table('category')->where('id', $this->filterRequest['category'][0])->first(['_lft', '_rgt']);
        if ($category) {
            $categoryIds = DB::table('category')->where('_lft', '>=', $category->_lft)->where('_rgt', '<=', $category->_rgt)->pluck('id')->toArray();
            $categoryFilter = $filterData;
            $categoryFilter['query']['bool']['must_not']['terms']['id'] = $ids;
            $categoryFilter['query']['bool']['filter']['terms']['category_id'] = $categoryIds;
            $categoryFilter['size'] = 120;
            $categoryData = $this->elasticService->searchDocument($categoryFilter, 'products', $this->elasticServiceConfig['index']);
            if ($result['hits']['hits'] && isset($categoryData['hits']['hits'])) {
                $result['hits']['hits'] = array_merge($categoryData['hits']['hits'], $result['hits']['hits']);
            }
        }
    }

    protected function isExceptionKeyword($keyword) {
        $result = false;
        if ($this->exceptionKeyword && in_array(trim(strtolower($keyword)), $this->exceptionKeyword)) {
            $result = true;
        }
        return $result;
    }

    protected function searchBestProduct($keyword, $minimumShouldMatchGlobal) {
        $result = [];
        $query = $this->_buildFilterData($keyword, $minimumShouldMatchGlobal);
        $query['size'] = 40;
        $data = $this->elasticService->searchDocument($query, ElasticSearchIndex::TYPE_PRODUCTS_TOP, $this->elasticServiceConfig['index']);
        if (isset($data['hits']['hits']))  {
            $result = $this->searchDataDecor->filterBestProduct($data['hits']['hits'], $keyword);
        }
        return $result;
    }

    protected function uniqueArray($items) {
        $result = [];
        $flagIds = [];
        foreach ($items as $item) {
            $id = isset($item['_source']['id']) ? $item['_source']['id'] : null;
            if ($id && !array_key_exists($id, $flagIds)) {
                $result[] = $item;
                $flagIds[$id] = 1;
            }
        }
        return $result;
    }

    protected function isExistsApostrophes($keyword) {
        return str_contains($keyword, '\'');
    }

    private function isBot()
    {
        return preg_match("/rambler|abacho|acoi|accona|aspseek|altavista|estyle|scrubby|lycos|geona|ia_archiver|alexa|sogou|skype|facebook|twitter|pinterest|linkedin|naver|bing|google|yahoo|duckduckgo|yandex|baidu|teoma|xing|java\/1.7.0_45|bot|crawl|slurp|spider|mediapartners|\sask\s|\saol\s/i", request()->header('user-agent',''));
    }


    private function getTextFiltered ($tempFilters) {
        $retVal = '';
        $listText = [];
        foreach ($tempFilters as $tempFilter) {
            $listText[] = $tempFilter['text'];
        }
        if (count($listText) > 0) {
            $retVal = implode(' ', $listText);
        }
        return $retVal;
    }

    private function miniKeyword($keyword) {
        if ($keyword) {
            $keyword = substr($keyword, 0, 64);
        }
        return $keyword;
    }

    private function _buildFilterData($keyword, $minimumShouldMatchGlobal = false, $dispatchesAttrs = []) {
        $keyword = $this->miniKeyword($keyword);
        $size = $this->elasticServiceConfig['defaultSize'];
        if ($this->isExceptionKeyword($keyword)) {
            $size = 4000;
        }
        $routeName = Route::currentRouteName();
        $fields = $this->elasticServiceConfig['type']['products']['searchFields'];
        if ($routeName == 'search') {
            $fields = ['name'];
        } else if ($routeName == 'tag::explore') {
            $fields = ['tag.title'];
            $size = $size * 2;
        }
        $elasticParams = [
            'from' => $this->elasticServiceConfig['defaultFrom'],
            'size' => $size,
            '_source' => ['id', 'sku', 'name', 'slug', 'image_url',
                'status', 'brand_id', 'inventory',
                'display_price', 'display_high_price',
                'view_count', 'sold', 'price', 'high_price',
                'total_click', 'total_sale', 'cr', 'user', 'random_image',
                'category_id', 'tags',
                'variant', 'template_id', 'porder', 'position', 'url'
            ],
        ];
        if ($minimumShouldMatchGlobal) {
            $elasticParams['query']['bool']['minimum_should_match'] = $this->elasticServiceConfig['type']['products']['shouldMatch'];
        }
        if ($keyword) {
            $shouldQuery = [
                [
                    'multi_match' => [
                        'query' => $keyword,
                        'fields' => $fields,
                        'analyzer' => 'synonym_filter'
                    ]
                ],
            ];
            
            if (config('app.locale') == 'jp') {
                array_unshift($shouldQuery, [
                    'multi_match' => [
                        'query' => $keyword,
                        'fields' => $this->elasticServiceConfig['type']['products']['searchFields'],
                        'analyzer' => 'kuromoji_normalize'
                    ]
                ]);
            }
            if ($this->filterRequest) {
                if (isset($this->filterRequest['category_detach']) && $this->filterRequest['category_detach']) {

                    $elasticParams['query']['bool']['must'] = [
                        [
                            'terms' => [
                                "category_id" =>  $this->filterRequest['category_detach'],
                            ],
                        ],
                    ];
                }

                if (isset($this->filterRequest['campaign'])
                    && isset($this->filterRequest['campaign']['extend_keywords'])
                    && count($this->filterRequest['campaign']['extend_keywords']) > 0) {
                   foreach ($this->filterRequest['campaign']['extend_keywords'] as $extendKeyword) {
                       $shouldQuery[] = [
                           'multi_match' => [
                               'query' => $extendKeyword,
                               'fields' => ['name'],
                               'analyzer' => 'synonym_filter'
                            ]
                       ];
                   }
                }
            }
            if ($shouldQuery) {
                $elasticParams['query']['bool']['should'] = $shouldQuery;
            }

        }

        return $elasticParams;
    }

    private function _buildPhrasePrefixFilterData($keyword, $minimumShouldMatchGlobal = false) {
        $elasticParams = [
            'from' => $this->elasticServiceConfig['defaultFrom'],
            'size' => $this->elasticServiceConfig['defaultSize'],
            'query' => [
                'bool' => [
                    'must' => [
                        'match' => [
                            'status' => 'ACTIVE',
                        ]
                    ],
                ],

            ]
        ];
        if ($minimumShouldMatchGlobal) {
            $elasticParams['query']['bool']['minimum_should_match'] = $this->elasticServiceConfig['type']['products']['shouldMatch'];
        }

        if (!empty($keyword)) {
            $shouldQuery = [
                [
                    'multi_match' => [
                        'query' => $keyword,
                        'fields' => $this->elasticServiceConfig['type']['products']['searchFields'],
                        'type' => 'phrase_prefix',
                        'minimum_should_match' => $this->elasticServiceConfig['type']['products']['shouldMatch']
                    ]
                ]
            ];
            
            if (config('app.locale') == 'jp') {
                array_unshift($shouldQuery, [
                    'multi_match' => [
                        'query' => $keyword,
                        'fields' => $this->elasticServiceConfig['type']['products']['searchFields'],
                        'analyzer' => 'kuromoji_normalize'
                    ]
                ]);
            }
            $elasticParams['query']['bool']['should'] = $shouldQuery;
            
        }
        return $elasticParams;
    }

    public function trending(Request $request) {
        $from = $request->input('from');
        $trends = $this->getTrendingData();
        $response = $from != 'api' ? $trends : [
            'status' => 'successful',
            'trends' => $trends,
        ];
        return response()->json($response, 200, [], JSON_NUMERIC_CHECK);
    }

    protected function getTrendingData() {
        $defaultCache = 'cache::trending::' . env('APP_LOCALE');
        $trends = Cache::get($defaultCache);
        if (empty($trends)) {
            $trends = DB::table('search_tracking')
                ->whereBetween('created_at', [date('Y-m-d', strtotime(' -7 day')), date('Y-m-d')])
                ->select('keyword', DB::raw('SUM(quantity) AS quantity'))
                ->groupBy('keyword')
                ->orderBy('quantity', 'DESC')
                ->orderBy('id', 'ASC')
                ->limit(100)
                ->get()
                ->toArray();
            if ($trends) {
                shuffle($trends);
                $trends = array_slice($trends, 0, 7);
                foreach ($trends as $item) {
                    if (!isset($item->keyword)) {
                        continue;
                    }
                   $item->keyword = html_entity_decode($item->keyword);
                }
            }

            if (count($trends) > 0) {
                Cache::forget($defaultCache);
                Cache::put($defaultCache, $trends, 3600);
            }
        }
        return $trends;
    }

    public function suggestOne($keyword, $filter = []) {
        $data = $this->productByKeyword($keyword, $filter);
        return isset($data[0]) ? $data[0] : null;
    }

    public function productByKeyword($keyword, $filter =[]) {
        $keyword = trim(strip_tags($keyword));
        $keyword = trim(preg_replace('/\s+/', ' ', $keyword));
        $result = [];
        if (!empty($keyword)) {
            $filter['origin_keyword'] = $keyword;
            if (!isset($filter['category'])) {
                $detachCategoryData = $this->searchDataDecor->detachCategory($keyword);
                if ($detachCategoryData && array_key_exists('category_ids', $detachCategoryData)) {
                    $filter['category_detach'] = $detachCategoryData['category_ids'];
                    $keyword = isset($detachCategoryData['q']) && $detachCategoryData['q'] ? $detachCategoryData['q'] : $keyword;
                }
            }
            $this->setFilterRequest($filter);
            $data = $this->elasticService->searchDocument($this->_buildFilterData($keyword), 'products', $this->elasticServiceConfig['index']);
            if (!empty($data['hits']['hits'])) {
                $hits = $data['hits']['hits'];
                $filter['keyword'] = $keyword;
                if (!isset($filter['page_size'])) {
                    $filter['page_size'] = 1;
                }
                if (empty($filter['page_id'])) {
                    $filter['page_id'] = 0;
                }
                $items = $this->searchDataDecor->filter($hits, $filter);
                $items = $this->searchDataDecor->sort($items, $filter);
                $items = $this->searchDataDecor->handleDuplicateDesign($items, $filter);
                $products = array_slice($items, $filter['page_id'] * $filter['page_size'], $filter['page_size']);
                $result = $this->decorSearchItems($products, $filter);
            }
        }
        return $result;
    }


    public function suggestion(Request $request) {
        $keyword = $request->input('q', null);
        $keyword = trim(strip_tags($keyword));
        $keyword = trim(preg_replace('/\s+/', ' ', $keyword));
        $retVal['products'] = [];
        $retVal['categories'] = [];
        $data = [];
        if ($request->ip() != '127.0.0.1') {
            return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
        }
        if (!empty($keyword)) {
            $data['origin_keyword'] = $keyword;
            if (!isset($data['category'])) {
                $detachCategoryData = $this->searchDataDecor->detachCategory($keyword);
                if ($detachCategoryData && array_key_exists('category_ids', $detachCategoryData)) {
                    $isDetachOk = true;
                    $data['category_detach'] = $detachCategoryData['category_ids'];
                    $keyword = isset($detachCategoryData['q']) && $detachCategoryData['q'] ? $detachCategoryData['q'] : $keyword;
                }
            }
            $this->setFilterRequest($data);
            $keyword = $this->miniKeyword($keyword);
            $minimumShouldMatchGlobal = isset($_GET['order']) && $_GET['order'];
            $result = $this->searchProduct($keyword, $minimumShouldMatchGlobal, $request->has('use_phase_prefix'));
            if ($this->isIgnoreKeyword($keyword)) {
                $result = [];
            }
            if (!empty($result['hits']['hits'])) {
                $parentCategories = Category::whereNull('parent_id')->where('type', '=', 'PRODUCT')->pluck('id')->toArray();
                $topCategories = [];
                $hits = $result['hits']['hits'];
                $data['keyword'] = $keyword;
                $data['page_size'] = $request->input('s', 10);
                if (empty($data['page_id'])) {
                    $data['page_id'] = 0;
                }
                if ($request->has('min_score')) {
                    $data['min_score'] = $request->get('min_score');
                }
                foreach ($hits as $index => $hit) {
                    if ($index + 1 > 50) break;
                    if (!empty($hit['_source']['categories'])) {
                        foreach ($hit['_source']['categories'] as $category) {
                            if (!in_array($category['id'], $parentCategories) && !in_array($category['id'], $topCategories)) {
                                $topCategories[] = $category['id'];
                            }
                        }
                    }
                }

                $result = $this->searchDataDecor->buildFilter($hits, $data);
                $items = $this->searchDataDecor->filter($hits, $data);
                $items = $this->searchDataDecor->sort($items, array_merge($data, [
                    'keyword' => $keyword,
                    'q' => $keyword,
                ]));
                $items = $this->searchDataDecor->handleDuplicateDesign($items, $data);
                $products = array_slice($items, $data['page_id'] * $data['page_size'], $data['page_size']);
                $products = $this->decorSearchItems($products, $data);
                $productsRebuild = []; 
                foreach ($products as $product) {
                    $productItem = [
                        'id' => $product['id'],
                        'name' => $product['name'],
                        'slug' => $product['slug'],
                        'image_url' => getImageCdn($product['image_url'], 0, 250, true),
                        'url' => $product['url'],
                        'price' => formatPrice($product['price'])
                    ];
                    if (!empty($product['high_price']) && $product['high_price'] != 0 && $product['high_price'] > $product['price']) {
                        $productItem['high_price'] = formatPrice($product['high_price']);
                    }
                    array_push($productsRebuild, $productItem);
                }

                $retVal['products'] = $productsRebuild;
                
                foreach ($result['categories'] as $index => $category) {
                    if (in_array($category['id'], $topCategories)) {
                        $result['categories'][$index]['url'] = route('search', ['q' => $keyword, 'category' => $category['id']]);
                    } else {
                        unset($result['categories'][$index]);
                    }
                }

                $categories = collect($result['categories'])->sortByDesc('count')->values()->splice(0, 6)->all();
                if (count($categories) > 1) {
                    $retVal['categories'] = $categories;
                } else {
                    $retVal['categories'] = [];
                }
            }
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function init(Request $request) {

        set_time_limit(3600 * 5);
        $elasticsearch =  Config::get('z-search::elasticsearch');
        $elsIndexController = new IndexController();
        $newProductStatus = $elsIndexController->indexNewProduct($request);
        ini_set('memory_limit', '1000M');
        $isAll = $request->input('isAll', 0);
        $isCreateIndex = $request->input('create', 0);
        $updateAllColumn = $request->input('update_all', 0);
        $deleteHidden = $request->input('delete_all_hidden', 0);
        $lastMin = $request->input('min', 33);
        $updatedFrom = date('Y-m-d H:i:s', time() - $lastMin * 60);
        $startTime = time();
        if ($isCreateIndex) {
            $isAll = 1;
            //$this->elasticService->createIndex($elasticsearch['index']);
        }
        $filter = [
            'is_all' => $isAll,
            'from_id' => $request->input('from_id', -1),
            'to_id' => $request->input('to_id', -1),
            'delete_hidden' => $deleteHidden,
            'updated_from' => $updatedFrom,
            'created_from' => $updatedFrom,
        ];


        if ($updateAllColumn) {
            $elsIndexService = new ElasticSearchIndex();
            $elsIndexService->initMetaData();
            $result = $elsIndexService->multiIndexProduct($filter, $elasticsearch);
        } else {
            $elsIndexService = new ElasticSearchIndexLite();
            $elsIndexService->initProductVariantOption();
            $result = $elsIndexService->multiUpdateProduct($filter, $elasticsearch);
        }
        $result['status'] = 'successful';
        $result['new_product'] = $newProductStatus;
        ElasticsearchIndexHistory::logHistory($startTime, [
            'filter' => $filter,
            'result' => $result,
        ]);
        return response()->json($result);
    }

    public function handleHiddenProduct(Request $request) {
        set_time_limit(3600 * 4);
        ini_set('memory_limit', '2248M');
        $elasticsearch =  Config::get('z-search::elasticsearch');
        $lastMin = $request->input('min', 35);
        $updatedFrom = date('Y-m-d H:i:s', time() - $lastMin * 60);
        $filter = [
            'from_id' => $request->input('from_id', -1),
            'to_id' => $request->input('to_id', -1),
            'updated_from' => $updatedFrom,
        ];
        $elsIndexService = new ElasticSearchIndex();
        $result = $elsIndexService->handleHidden($filter, $elasticsearch);
        $result['status'] = 'successful';
        return response()->json($result);
    }

    public function position(Request $request) {
        $productId = $request->input('product_id');
        $keyword = $request->input('keyword');
        $position = $request->input('position', 0);
        if ($productId && $keyword && $position >= 0) {
            $this->createOrUpdatePosition($keyword, [
                $productId => $position
            ]);
        }
        $result = [
            'status' => 'successful',
        ];
        return response()->json($result);
    }

    protected function createOrUpdatePosition($keyword, $data)
    {
        $key = SearchDataDecor::buildKeyPosition($keyword);
        $option = $this->getOption($key);
        if ($option && isset($option->id)) {
            $positionData = json_decode($option->value, true);
            foreach ($data as $productId => $position) {
                $positionData[$productId] = $position;
            }
            $option->value = json_encode($positionData);
            $option->save();
        } else {
            Option::insert([
                'name' => $keyword,
                'key' => $key,
                'value' => json_encode($data),
                'type' => 'json'
            ]);
        }
    }

    protected function getOption($key) {
        return Option::where('key', $key)->first();
    }

    public function searchTracking(Request $request) {
        if (!$request->has('keyword') || !$request->has('image_url')) {
            return response()->json(['status' => 'fail', 'message' => 'keyword, image_url required params!']);
        }

        $imageUrl = $request->input('image_url');
        $keyword = strip_tags($request->input('keyword'));
        $keyword = trim(preg_replace('/\s+/', ' ', strtolower($keyword)));
        $currentDate = date('Y-m-d');

        $existsKeyword = DB::table('search_tracking')
            ->select(['id', 'quantity'])
            ->where('keyword', '=', $keyword)
            ->where('created_at', '=', $currentDate)
            ->first();

        if ($existsKeyword) {
            DB::table('search_tracking')
                ->where('id', $existsKeyword->id)
                ->update([
                    'quantity' => $existsKeyword->quantity + 1,
                    'image_url' => $imageUrl
                ]);
        } else {
            try {
                DB::table('search_tracking')->insert([
                    'keyword' => $keyword,
                    'image_url' => $imageUrl,
                    'created_at' => $currentDate,
                    'updated_at' => $currentDate
                ]);
            } catch (\Exception $ex) {}
        }
        return response()->json(['status' => 'successful', 'message' => 'successful']);
    }

    public function decorAmazonItems($items, $filter = []) {
        $termHighItem = config('z-search::amazon.term_high_item', 0);
        if ($termHighItem > 0) {
            $items = $this->amazonDataDecor->decor($items, $filter);
        }

        return $items;
    }

    public function decorSearchItems($items, $filter = []) {
        $authors = $this->getAuthors($items);
        foreach ($items as &$item) {
            if (!isset($item['url'])) {
                continue;
            }

            $url = $this->clRoute($item['url']);
            $keyword = isset($filter['origin_keyword']) ? $filter['origin_keyword'] : '';
            $pageType = array_key_exists('page_type', $filter) ? $filter['page_type'] : 'search';
            $item['url'] = $url;
            if ($pageType && $pageType == 'search') {
                $char = str_contains($url, '?') ? '&' : '?';
                $url .= $char . self::SEARCH_TRACKING . '=' . urlencode($keyword);
                $item['url'] = $url;
            }
            $authorId = isset($item['user']['id']) ? $item['user']['id'] : 0;
            if ($authors && $authorId && array_key_exists($authorId, $authors)) {
                $item['user']['name'] = $authors[$authorId]->name;
                $item['user']['slug'] = $authors[$authorId]->slug;
            }
        }
        $items = $this->searchDataDecor->buildImageByColor($items, $filter);
        $items = $this->searchDataDecor->decorProductAttr($items, ['rating_count', 'rating_value', 'price', 'high_price', 'sold']);
        //priceChangeDecor($items);
        return $items;
    }

    protected function getAuthors($items) {
        $result = [];
        $userIds = $this->mappingUserIds($items);
        $users = [];
        if ($userIds && count($userIds) > 0) {
            $users = DB::table('users')
                ->whereIn('id', $userIds)
                ->get(['id', 'name', 'slug']);
        }

        if ($users && count($users) > 0) {
            foreach ($users as $user) {
                $result[$user->id] = $user;
            }
        }
        return $result;
    }



    protected function mappingUserIds($items) {
        $userIds = [];
        foreach ($items as $item) {
            if (isset($item['user']['id'])) {
                $userIds[] = $item['user']['id'];
            }
        }
        return array_unique($userIds);
    }


    protected function replaceImage($imageUrl) {
        if ($imageUrl) {
            $imageUrl = str_replace('storage.googleapis.com/printerval-central', 'assets.artworkcraft.com', $imageUrl);
            $imageUrl = str_replace('gdn.printerval.com', 'gdn.artworkcraft.com', $imageUrl);
            $imageUrl = str_replace('cdn.printerval.com', 'cdn.artworkcraft.com', $imageUrl);
            $imageUrl = str_replace('liveview.printerval.com', 'liveview.artworkcraft.com', $imageUrl);
            $imageUrl = str_replace('assets.printerval.com', 'assets.artworkcraft.com', $imageUrl);
            $imageUrl = str_replace('storage.googleapis.com', 'assets.artworkcraft.com', $imageUrl);
            $imageUrl = str_replace('printerval.com', 'assets.artworkcraft.com', $imageUrl);
        }
        return $imageUrl;
    }

    protected function searchColumnIndex($search, $items, $column) {
        $retval  = -1;
        if (!empty($column)) {
            foreach ($items as $key => $value) {
                if (isset($value[$column]) && $value[$column] == $search) {
                    $retval = $key;
                    break;
                }
            }
        }
        return $retval;
    }

    public function indexPaymentStat(Request $request) {
        set_time_limit(3 * 3600);
        ini_set('memory_limit', '4448M');
        $days = $request->input('days', 60);
        $elsIndexService = new ElasticSearchIndex();
//        $elsIndexService->deleteAllProductTop();
        sleep(10);
        $query = DB::table(DB::raw("sb_order as sb_o force INDEX (sa_inoutput_created_at)"))
            ->join('order_item as oi', 'o.id', '=', 'oi.order_id')
            ->where('o.created_at', '>', date('Y-m-d 00:00:00', time() - $days * 86400))
            ->where('o.payment_status', '=', 'PAID')
            ->groupBy('oi.product_id');
        $pids = $query->pluck('oi.product_id')->toArray();
        $elsIndexService->initMetaData();
        $data = [];
        if ($pids) {
            foreach (array_chunk($pids, 200)  as $partIds) {
                $data[] = $elsIndexService->multiIndexTopProduct([
                    'ids' => $partIds,
                    'is_all' => 1,
                ], $this->elasticServiceConfig);

            }
            foreach (array_chunk($pids, 200)  as $partIds) {
                $data[] = $elsIndexService->multiIndexProduct([
                    'ids' => $partIds,
                    'is_all' => 1,
                ], $this->elasticServiceConfig);
            }

        }
        $result = [
            'count' => $pids ? count($pids) : 0,
            'result' => $data,
        ];
        return response()->json($result);
    }

    public function indexFromQueue(Request $request)
    {
        set_time_limit(3 * 3600);
        ini_set('memory_limit', '4448M');
        $elsIndexService = new ElasticSearchIndex();
        $elsIndexService->initMetaData();
        $fromId = $request->input('from_id', 1);
        $toId = $request->input('to_id', 1);
        $pids = DB::table('product_zsearch_index')
            ->where('id', '>=', $fromId)
            ->where('id', '<', $toId)
            ->pluck('product_id')
            ->toArray();
        $data = [];
        if ($pids && count($pids) > 0) {
            $elsIndexService = new ElasticSearchIndex();
            $elsIndexService->initMetaData();
            foreach (array_chunk($pids, 200)  as $partIds) {
                $data[] = $elsIndexService->multiIndexProduct([
                    'ids' => $partIds,
                    'is_all' => 1,
                ], $this->elasticServiceConfig);
            }
        }
        $result = [
            'count' => $pids ? count($pids) : 0,
            'result' => $data,
            'input' => $request->all(),
        ];
        \Log::info('indexFromQueue', $result);
        return response()->json($result);

    }

    public function buildTagSchema()
    {
        ini_set('memory_limit', '1024M');
        set_time_limit(1800);

        $client = new Client();

        $perPage = 1000;
        $maxId = DB::table('tag')->max('id');
        $pageCount = ceil($maxId / $perPage);
        $queuedTags = [];

        for ($i = 0; $i < $pageCount; $i ++) {
            $start = $i * $perPage;
            $stop = $start + $perPage;

            $tags = DB::table('tag')->select(['id', 'title'])
                ->where('id', '>=', $start)
                ->where('id', '<', $stop)
                ->get();

            foreach ($tags as $tag) {

                // check has tag meta (top_product_images key)
                $hasValidMeta = DB::table('tag_meta')
                    ->select(['id'])
                    ->where('tag_id', $tag->id)
                    ->where('key', 'top_product_images')
                    ->where('updated_at', '>', date("Y-m-d", strtotime("-30 day")))
                    ->first();

                if (!$hasValidMeta) {
                    $prefixLocale = env('APP_LOCALE') !== '' ? '/' . env('APP_LOCALE') . '/' : '/';
                    $restqOrigin = config('restq.origin');

                    $urlParams = http_build_query([
                        'id' => $tag->id,
                        'title' => $tag->title,
                        'site' => request()->getHost()
                    ]);

                    if (request()->input('debug')) {
                        echo $prefixLocale . 'z-search/make-tag-meta-schema?' . $urlParams . "\n";
                        $queuedTags[] = [
                            'id' => $tag->id,
                            'title' => $tag->title
                        ];
                    } else {
                        $client->post($restqOrigin . $prefixLocale . 'z-search/make-tag-meta-schema?' . $urlParams, [
                            'form_params' => [
                                'id' => $tag->id,
                                'title' => $tag->title,
                            ]
                        ]);
                        $queuedTags[] = [
                            'id' => $tag->id,
                            'title' => $tag->title
                        ];
//                        usleep(100);
                    }
                }

            }
        }

        return [
            'count' => count($queuedTags),
            'queued_tag' => $queuedTags
        ];
    }

    public function makeTagMeta()
    {
        ini_set('memory_limit', '1024M');

        $id = request()->input('id');
        $title = request()->input('title');

        $dataFilters = [
            "q" => $title,
            "keyword" => $title,
            "origin_keyword" => $title,
            "page_size" => 60,
            "page_id" => 0,
            "order" => "",
        ];

        try {

            \Log::info('makeTagMeta', [$id, $title]);

            $result = $this->searchProduct($title, false);
            $hits = $result['hits']['hits'];
            $items = $this->searchDataDecor->filter($hits, $dataFilters);
            $items = $this->searchDataDecor->sort($items, array_merge($dataFilters, [
                'keyword' => $title,
                'q' => $title,
            ]));

            $imageUrls = [];
            $minPrice = 9999;
            $maxPrice = 0;

            foreach ($items as $item) {
                // get images top product in tag
                if (isset($item['image_url']) && count($imageUrls) < 6) {
                    $imageUrls[] = $item['image_url'];
                }

                // get low price - high price
                if ($item['price'] > $maxPrice) $maxPrice = $item['price'];
                if ($item['price'] < $minPrice) $minPrice = $item['price'];
            }

            $this->tagMetaUpdateOrCreate($id, 'top_product_images', json_encode($imageUrls), 'string_value');
            $this->tagMetaUpdateOrCreate($id, 'high_price', $maxPrice, 'number_value');
            $this->tagMetaUpdateOrCreate($id, 'low_price', $minPrice, 'number_value');

            return [
                'images' => $imageUrls,
                'high_price' => $maxPrice,
                'low_price' => $minPrice,
            ];

        } catch (\Exception $e) {

            \Log::info('makeTagMeta Error:', [$id, $title, [
                'message' => $e->getMessage(),
                'src' => $e->getFile() . ':' . $e->getLine()
            ]]);

            return null;
        }
    }

    private static function tagMetaUpdateOrCreate($id, $key, $value, $valueType = 'string_value')
    {
        $metaExist = DB::table('tag_meta')
            ->where('tag_id', $id)
            ->where('key', $key)
            ->count();

        if ($metaExist) {
            DB::table('tag_meta')
                ->where('tag_id', $id)
                ->where('key', $key)
                ->update([
                    $valueType => $value,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
        } else {
            DB::table('tag_meta')
                ->insert([
                    'tag_id' => $id,
                    'key' => $key,
                    $valueType => $value,
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
        }
    }

    public function apiSearch(Request $request) {
        $response = [
            'status' => 'successful',
        ];
        try {
            $data = $this->apiSearchData($request);
            $response['meta'] = $data['meta'];
            unset($data['meta']);
            $response['result'] = $data;
        } catch (\Exception $e) {
            $response = [
                'status' => 'fail',
            ];
        }

        return response()->json($response);

    }

    protected function apiSearchData($request) {
        $keyword = null;
        $tag = null;
        $pageType = 'search';
        if ($request->has('q')) {
            $keyword = trim(strip_tags($request->input('q')));
            $keyword = trim(preg_replace('/\s+/', ' ', $keyword));
        } elseif ($request->has('tag_id')) {
            $tagId = $request->input('tag_id');
            $tag = Tag::find($tagId);
            $keyword = str_replace('-', ' ', strip_tags($tag->title));
            $pageType = 'tag';
        }
        $filter = $request->all();
        $filter['keyword'] = $keyword;
        $filter['origin_keyword'] = $keyword;
        $filter['page_size'] = $request->input('page_size', 40);
        $filter['page_type'] = $pageType;
        $keys = [
            'size_variant_id',
            'color_variant_id',
            'type_variant_id',
        ];
        foreach ($keys as $key) {
            if ($request->input($key)) {
                $filter[$key] = $request->input($key);
            }
        }

        if (empty($filter['page_id'])) {
            $filter['page_id'] = 0;
        }
        if (isset($filter['category_id']) && $filter['category_id']) {
            $filter['category'] = explode(',', $filter['category_id']);
        }
        if (!isset($filter['order'])) {
            $filter['order'] = '';
        }
        $isDetachOk = false;
        if (!isset($filter['category'])) {
            $detachCategoryData = $this->searchDataDecor->detachCategory($keyword);
            if ($detachCategoryData && array_key_exists('category_ids', $detachCategoryData)) {
                $isDetachOk = true;
                $filter['category_detach'] = $detachCategoryData['category_ids'];
                $keyword = isset($detachCategoryData['q']) && $detachCategoryData['q'] ? $detachCategoryData['q'] : $keyword;
                $filter['keyword'] = $keyword;
            }
        }
        $minimumShouldMatchGlobal = isset($filter['order']) && $filter['order'];
        $minimumShouldStr = $minimumShouldMatchGlobal ? 1 : 0;
        $this->setFilterRequest($filter);
        $result = $this->searchProduct($keyword, $minimumShouldMatchGlobal);
        $products = [];
        $meta = [
            'page_id' => 0,
            'page_size' => 0,
            'page_count' => 0,
            'has_next' => false,
            'total_count' => 0,
        ];

        $brands = [];
        $tags = [];
        $searchCategories = [];
        $filters = [];
        $displayedFilters = [];
        $priceRange = [];
        $filterCategory = [];
        $filterOptions = [];
        $relatedItems = [];
        if ($this->isIgnoreKeyword($keyword)) {
            $result = [];
        }

        if (!empty($result['hits']['hits'])) {
            $hits = $result['hits']['hits'];
            $this->searchDataDecor->decorHits($hits);
            $result = $this->searchDataDecor->buildFilter($hits, $filter);
            $items = $this->searchDataDecor->filter($hits, $filter);
            if (!$items && $isDetachOk) {
                $items = $hits;
            }
            $items = $this->searchDataDecor->sort($items, array_merge($filter, [
                'keyword' => $keyword,
                'q' => $keyword,
            ]));
            $items = $this->searchDataDecor->handleDuplicateDesign($items, $filter);
            $tags = $this->searchDataDecor->buildTag(array_slice($items, 0, 50));
            if (function_exists('getTagHot')) {
                $actual_link = "$_SERVER[REQUEST_URI]";
                $tagsConfig = getTagHot($actual_link);
                $tags = array_merge($tagsConfig, $tags);
            }
            $result['tags'] = $tags;
            $products = array_slice($items, $filter['page_id'] * $filter['page_size'], $filter['page_size']);
            $products = $this->decorSearchItems($products, array_merge($filter, [
                'page_type' => $pageType
            ]));
            $meta = $this->getMetaData($filter, count($items));
            $brands = $result['brands'];
            $searchCategories = $result['categories'];
            $priceRange = $result['priceRange'];
            $displayedFilters = $result['displayedFilters'];
            $filterOptions = $result['filterOptions'];
        }
        if (isset($filterOptions['Color'])) {
            $filterOptions['Color']  = $this->searchDataDecor->buildColorCode($filterOptions['Color']);
        }
        if (isset($productFilters['category_id'])) {
            unset($productFilters['category_id']);
        }
        $allFilters = [];
        $filtered = [];
        $tempFilters = [];

        $isTagPage = $filter['page_type'] == 'tag';
        $filteredCategory = '';

        foreach ($displayedFilters as $key => $value) {
            if(!is_array($value)) {
                if($key == 'price') {
                    $url = buildFilterUrl('minPrice', false, 'maxPrice', false);
                    if ($pageType == 'tag') {
                        $url = SlugManager::clearFilterUrl([
                            'type' => $key
                        ]);
                    }
                    $tempFilters[] = [
                        'url' => $url,
                        'checked' => true,
                        'text' => $value,
                        'type' => $key
                    ];
                } else {
                    $url = buildFilterUrl($key, false);
                    $valueArr = explode('::', $value);
                    $strSlug = isset($valueArr[1]) ? $valueArr[1] : $valueArr[0];
                    $url = buildFriendlyFilterUrl($strSlug, true, $key);
                    if ($key == 'category') {
                        $filteredCategory = $strSlug;
                    }
                    $tempFilters[] = [
                        'url' => $url,
                        'checked' => true,
                        'text' => $valueArr[0],
                        'slug' => $strSlug,
                        'type' => $key
                    ];
                }
            } else {
                $urls = buildFilterUrlArray($key);
                if(count($urls)) {
                    foreach($urls as $key => $url) {
                        $tempFilters[] = [
                            'url' => $url,
                            'checked' => true,
                            'text' => isset($value[$key]) ? $value[$key] : '',
                            'type' => $key
                        ];
                    }
                }
            }
        }
        $cloneTempFilters = [];
        foreach ($tempFilters as $tempFilter) {
            if (!empty($tempFilter['type']) && $tempFilter['type'] == 'type_variant_id') {
                array_unshift($cloneTempFilters, $tempFilter);
            } else {
                $cloneTempFilters[] = $tempFilter;
            }
        }
        $tempFilters = $cloneTempFilters;
        $filtered = [
            'title' => __("Bộ lọc"),
            'id' => "filter",
            'filters' => $tempFilters,
            'text_filtered' => $this->getTextFiltered($tempFilters)
        ];

        $tempFilters = [];
        $categoryItem = [];

        foreach ($searchCategories as $item) {
            $itemSlug = ($item['slug']);
            $isChecked = false;
            if (isset($item['id']) && isset($_GET['category']) && $item['id'] == $_GET['category']) {
                $categoryItem = $item;
            }
            if (isset($item['id']) && isset($filter['category']) && is_array($filter['category']) && in_array($item['id'], $filter['category'])) {
                $categoryItem = $item;
            }
            if (!empty($parameters) && isset($parameters['slug'])) {
                $attr = [
                    'id' => $item['id'],
                    'type' => 'category',
                    'slug' => $item['slug']
                ];
                $url = SlugManager::url ($attr);
                $isChecked = SlugManager::isSelected($attr);
            } else {
                $isChecked = false;
                foreach ($filtered['filters'] as $fItem) {
                    $fSlug = isset($fItem['slug']) ? $fItem['slug'] : '';
                    if ($fSlug == $itemSlug && $fItem['type'] == 'category') {
                        $isChecked = true;
                        break;
                    }
                }
                $url = buildFilterUrl('category', $item['id']);
            }
            $tempFilters[] = [
                'url' => $url,
                'checked' => $isChecked,
                'text' => $item['name'],
                'count' => isset($item['count']) ? $item['count'] : false,
            ];
        }
        $allFilters[] = [
            'title' => __("Danh mục"),
            'id' => "category",
            'filters' => $tempFilters
        ];

        if (isset($priceRange) && count($priceRange) > 1) {
            $tempFilters = [];
            foreach ($priceRange as $item) {
                $from = formatPrice($item['from']);
                $to = formatPrice($item['to']);
                $url = buildFilterUrl('minPrice', $item['from'], 'maxPrice', $item['to']);
                $isChecked = isSelectedFilter('minPrice', $item['from'], 'maxPrice', $item['to']);
                if ($pageType == 'tag') {
                    $priceAttr = SlugManager::priceAttr($item['from'], $item['to']);
                    $url = SlugManager::url($priceAttr);
                    $isChecked = SlugManager::isSelected($priceAttr);
                }

                $tempFilter = [
                    'url' => $url,
                    'checked' => $isChecked,
                    'text' => $item['from']
                        ? __("Từ") . " $from -  $to "
                        : __("Dưới") ." $to"
                ];
                $tempFilters[] = $tempFilter;
            }
            $allFilters[] = [
                'title' => __("Giá"),
                'id' => 'price',
                'filters' => $tempFilters
            ];
        }

        if ($brands) {
            $tempFilters = [];
            foreach ($brands as $item) {
                $tempFilters[] = [
                    'url' => buildFilterUrl('brand', $item['slug']),
                    'text' => $item['name'],
                    'count' => (!empty($item['count'])) ? $item['count'] : 0,
                    'checked' => isSelectedFilter('brand', $item['slug']),
                ];
            }
            $allFilters[] = [
                'title' => __("Thương hiệu"),
                'filters' => $tempFilters,
                'id' => 'brand',
            ];

        }
        if (count($filters)) {
            foreach($filters as $filter){
                $tempFilters = [];
                foreach ($filter['values'] as $filterValues) {
                    $tempFilters[] = [
                        'url' => buildFilterUrl($filter['slug'], $filterValues['slug']),
                        'text' => $filterValues['name'],
                        'count' => $filterValues['count'],
                    ];

                }
                $allFilters[] = [
                    'title' => $filter['display_name'],
                    'id' => $filterValues['slug'],
                    'filters' => $tempFilters
                ];
            }
        }

        if ($filterOptions) {
            foreach ($filterOptions as $key => $variants) {
                $tempFilters = [];
                foreach ($variants as $item) {
                    if (!empty($parameters) && isset($parameters['slug'])) {
                        $attr = [
                            'id' => $item['id'],
                            'type' => $item['variant_slug'],
                            'slug' => $item['slug']
                        ];
                        $url = SlugManager::url ($attr);
                        $isChecked = SlugManager::isSelected($attr);
                    } else {
                        $url = buildFilterUrl($item['variant_slug'] . '_variant_id', $item['id']);
                        $isChecked = isSelectedFilter($item['variant_slug'] . '_variant_id', $item['id']);
                    }
                    $fItem = [
                        'id' => $item['id'],
                        'url' => $url,
                        'slug' => $item['slug'],
                        'text' => ucfirst($item['name']),
                        'checked' => $isChecked,
                    ];
                    if (array_key_exists('image_url', $item)
                        && array_key_exists('color_code', $item)) {
                        $fItem['color_code'] = $item['color_code'];
                        $fItem['image_url'] = $item['image_url'];
                    }
                    $tempFilters[] = $fItem;
                }
                $allFilters[] = [
                    'title' => __(ucfirst($key)),
                    'type' => $key,
                    'filters' => $tempFilters,
                    'id' => 'user_id',
                ];
            }
        }
        $sorter = [
            [ 'name' => __("Mua nhiều"), 'value' => 'sold' ],
            [ 'name' => __("Mới nhất"), 'value' => 'latest' ],
            [ 'name' => __("Xem nhiều"), 'value' => 'view' ],
            [ 'name' => __("Giá thấp"), 'value' => 'low_price'],
            [ 'name' => __("Giá cao"), 'value' => 'high_price'],
            [ 'name' => __("Giảm giá"), 'value' => 'sale' ],
        ];
        foreach ($sorter as $key => $item) {
            $sorter[$key]['url'] = buildFilterUrl('order', $item['value']);
            $sorter[$key]['checked'] = isSelectedFilter('order', $item['value']);
        }
        $allFilters = allFiltersOrdered($allFilters);
        $strCodes = "";
        priceChangeDecor($products);
        return compact(
            'products',
            'meta',
            'brands',
            'tags',
            'filters',
            'searchCategories',
            'priceRange',
            'filterCategory',
            'displayedFilters',
            'filtered',
            'keyword',
            'sorter',
            'filteredCategory',
            'allFilters'
        );
    }


    protected function isIgnoreKeyword($keyword) {
        $result = false;
        $ignoreKeywords = [
            'georgia bulldogs',
            'miami hurricanes',
            'kentucky wildcats',
            'wisconsin badgers',
            'oregon ducks',
            'them dawgs is hell',
        ];
        if ($ignoreKeywords && $keyword && in_array(strtolower($keyword), $ignoreKeywords) && env('ELASTICSEARCH_INDEX') == 'printerval') {
            $result = true;
        }
        return $result;
    }

    public function updatePrice(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '4048M');
        $elasticsearch =  Config::get('z-search::elasticsearch');
        $isAll = $request->input('isAll', 0);
        $lastMin = $request->input('min', 33);
        $updatedFrom = date('Y-m-d H:i:s', time() - $lastMin * 60);

        $filter = [
            'is_all' => $isAll,
            'from_id' => $request->input('from_id', -1),
            'to_id' => $request->input('to_id', -1),
            'updated_from' => $updatedFrom,
        ];

        $elsIndexService = new ElasticSearchIndex();
        $result = $elsIndexService->multiUpdatePrice($filter, $elasticsearch);
        $result['status'] = 'successful';
        $result['input'] = $request->all();
        return response()->json($result);
    }

    public function buildTrendingRelatedTags()
    {
        $insertCount = $this->tagService->buildTrendingRelatedTags();

        return [
            'status' => 'successful',
            'insert_count' => $insertCount
        ];
    }

    public function buildProductTrendingRelatedTags()
    {
        $insertCount = $this->tagService->buildProductTrendingRelatedTags();

        return [
            'status' => 'successful',
            'insert_count' => $insertCount
        ];
    }

    protected function setFilterRequest($filter) {
        $this->filterRequest = $filter;
    }

    public function warningElsIndex(Request $request) {
        $intervalMin = $request->input('interval', 30);
        $from = $request->input('from', 24);
        $to = $request->input('to', 2);
        $filter = [
            'from_time' => date('Y-m-d H:i:s', time() - $from * 3600),
            'to_time' => date('Y-m-d H:i:s', time() - $to * 3600),
            'interval_time' => $intervalMin * 60,
        ];
        $elsIndexHistory = new ElasticsearchIndexHistory();
        $data = $elsIndexHistory->warning($filter);
        return [
            'status' => 'successful',
            'data' => $data
        ];
    }


}
