<?php

namespace Modules\ZSearch\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Services\ElasticSearchIndex;

class IgnoreProductController extends Controller
{
    public function store(Request $request)
    {
        $pIds = $request->input('product_ids', []);
        $status = 'fail';
        if ($pIds && ElasticSearchIndex::tableExists('search_ignore_product')) {
            $status = 'successful';
            $items = $this->buildData($pIds);
            foreach (array_chunk($items, 200) as $chunkItems) {
                $this->insert($chunkItems);
            }
        }
        return [
            'status' => $status
        ];
    }

    public function storeIgnoreDuplicated(Request $request) {
        set_time_limit(3600);

        $products = $request->input('products', []);
        $pIds = [];
        foreach($products as $product) {
            $pIds[] = $product['id'];
        }

        $status = 'fail';
        $ignoreItems = DB::table('product')->whereIn('id', $pIds)->whereNull('deleted_at')->where('sold', 0)->orderBy('view_count', 'desc')->get(['id', 'actor_id', 'view_count'])->toArray();
        $sellerIds = DB::table('users')->where('role', 'SELLER')->whereNotNull('seller_token')->get(['id'])->keyBy('id')->toArray();
        foreach ($ignoreItems as $item) {
            if (array_key_exists($item->actor_id, $sellerIds)) {
                $item->sort = '0' . $item->view_count;
            } else {
                $item->sort = '1' . $item->view_count;
            }
        }

        usort($ignoreItems, function ($itemA, $itemB)  {
            $aValue = isset($itemA->sort) ? $itemA->sort : $itemA->view_count;
            $bValue = isset($itemB->sort) ? $itemB->sort : $itemB->view_count;
            $result = 0;
            if ($aValue > $bValue) {
                $result = -1;
            } else if ($aValue < $bValue) {
                $result = 1;
            }

            return $result;
        });

        $ignoreIds = [];
        foreach ($ignoreItems as $item) {
            $ignoreIds[] = $item->id;
        }

        $keepId = array_shift($ignoreIds);
        $existedIds = DB::table('search_ignore_product')->whereIn('product_id', $ignoreIds)->pluck('product_id')->toArray();
        $ignoreIds = array_diff($ignoreIds, $existedIds);
        if (count($ignoreIds) || ElasticSearchIndex::tableExists('search_ignore_product')) {
            \Log::useDailyFiles(storage_path() . '/logs/ignore-search-duplicate.log');
            $status = 'successful';
            $items = $this->buildData($ignoreIds);
            foreach (array_chunk($items, 200) as $chunkItems) {
                $this->insert($chunkItems);
            }
            $logData = [];
            foreach ($products as $item) {
                $logData[] = [
                    'product_id' => $item['id'],
                    'image_url' => $item['image_url'],
                ];
            }
            $locale = env('APP_LOCALE', 'us');
            if (!$locale) {
                $locale = 'us';
            }
            \Log::info('search_ignore_data', [
                'locale' => $locale,
                'keep_id' => $keepId,
                'remove_ids' => $ignoreIds,
                'keyword' => $request->get('keyword'),
                'data' => $logData,
            ]);

        }

        return [
            'status' => $status
        ];
    }

    protected function buildData($pIds)
    {
        $result = [];
        $date = date('Y-m-d H:i:s');
        foreach ($pIds as $pId) {
            $result[] = [
                'product_id' => $pId,
                'created_at' => $date,
            ];
        }
        return $result;
    }

    protected function insert($items)
    {
        DB::table('search_ignore_product')->insert($items);
    }

    public function deleteByUpdated(Request $request)
    {
        $start = time();
        $elasticService = new ElasticSearchIndex();
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $maxId =  DB::table('search_ignore_product')->max('id');
        $min = $request->input('min', 60);
        $pageSize = 500;
        $count = 0;
        $countNew = 0;
        for ($fromId = 1; $fromId < $maxId; $fromId += $pageSize) {
            $pIds =  $this->getPIdsByUpdatedAt($fromId, $fromId + $pageSize, $min);
            if ($pIds && count($pIds) > 0) {
                $count += count($pIds);
                $this->deleteByIds($pIds, $elasticService, $elasticSearchConfig);
            }
        }

        $newIds = $this->deleteNewPIds($min);
        if ($newIds && count($newIds) > 0) {
            $countNew = count($newIds);
            $this->deleteByIds($newIds, $elasticService, $elasticSearchConfig);
        }


        return [
            'status' => 'successful',
            'runtime' => time() - $start,
            'result' => [
                'count' => $count,
                'countNew' => $countNew,
            ],
        ];

    }

    protected function deleteByIds($pIds, $elasticService, $elasticSearchConfig) {
        foreach (array_chunk($pIds, 200) as $ids) {
            $items = $this->buildDeleteData($ids);
            $elasticService->deleteList($items, $elasticSearchConfig['index'], ElasticSearchIndex::TYPE_PRODUCTS);
            $elasticService->deleteList($items, $elasticSearchConfig['index'], ElasticSearchIndex::TYPE_PRODUCTS_TOP);
        }
    }

    protected function buildDeleteData($pIds) {
        $result = [];
        foreach ($pIds as $id) {
            $result[] = [
                'id' => $id
            ];
        }
        return $result;
    }

    protected function deleteNewPIds($min)
    {
        return  DB::table('search_ignore_product')
            ->where('created_at', '>', date('Y-m-d H:i:s', time() - $min * 60))
            ->pluck('product_id')
            ->toArray();

    }

    protected function getPIdsByUpdatedAt($fromId, $toId, $min) {
        $query = DB::table('search_ignore_product')
            ->join('product', 'product.id', '=', 'search_ignore_product.product_id')
            ->where('search_ignore_product.id', '>=', $fromId)
            ->where('search_ignore_product.id', '<', $toId)
            ->where('product.updated_at', '>', date('Y-m-d H:i:s', time() - $min * 60));
        $query->from(DB::raw("sb_search_ignore_product USE INDEX (PRIMARY)"));
        return $query->pluck('search_ignore_product.product_id')
            ->toArray();
    }

    public function rebuildAllProductIgnore(Request $request)
    {
        $elasticService = new ElasticSearchIndex();
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $items = $this->getItems();
        if ($items) {
            foreach (array_chunk($items, 200) as $chunkItems) {
                $elasticService->deleteList($chunkItems, $elasticSearchConfig['index'], ElasticSearchIndex::TYPE_PRODUCTS);
                $elasticService->deleteList($chunkItems, $elasticSearchConfig['index'], ElasticSearchIndex::TYPE_PRODUCTS_TOP);
            }
        }
        return [
            'status' => 'successful',
            'result' => count($items),
        ];

    }


    protected function getItems() {
        $result = [];
        $items = DB::table('search_ignore_product')
            ->orderBy('id', 'desc')
            ->get(['product_id']);
        foreach ($items as $item) {
            $result[] = [
                'id' => $item->product_id
            ];
        }
        return $result;
    }



}