<?php

namespace Modules\ZSearch\Controllers;

use Modules\ZSearch\Models\Option;
use App\Modules\ZSearch\Services\ElasticSearchIndexLite;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Services\ElasticsearchIndexHistory;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Services\ElasticSearchIndexSeller;
use Modules\ZSearch\Services\SearchDataDecor;
use Modules\ZSearch\Services\ElasticSearchService;

class IndexController extends Controller
{
    protected $elasticServiceConfig;
    protected $elasticServiceIndex;
    protected $etsyDataDecor;
    protected $elasticService;
    protected $sellerService;
    protected $searchDataDecor;
    protected $tagService;
    protected $users;
    protected $productVariant;
    protected $isAdmin;
    protected $timer = [];
    protected $currentTime = 0;
    const TYPE_PRODUCTS = 'products';
    const TYPE_PRODUCTS_TOP = 'products_top';
    const TYPE_PRODUCTS_SELLER = 'products_seller';
    const KEY_INDEX = 'elasticsearch_product_id';


    public function __construct()
    {
        set_time_limit(3600 * 2);
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticServiceConfig = $elasticSearchConfig;
        $this->elasticService = new ElasticSearchService($elasticSearchConfig);
        $this->elasticServiceIndex = new ElasticSearchIndexLite();
        $this->searchDataDecor = new SearchDataDecor();
        $this->sellerService = new ElasticSearchIndexSeller();
    }

    public function indexNewProduct(Request $request) {
        ini_set('memory_limit', '500M');
        $startTime = time();
        $result['status'] = 'successful';
        $range = $this->getRange($request);
        $this->handleIndexByRange([
            'type' => 'init',
            'is_all' => 1,
        ], $range['from_id'], $range['to_id']);
        if ($range['set_start_id']) {
            Option::setStartId(self::KEY_INDEX, $range['to_id']);
        }
        $this->handleIndexOption($range['from_id'], $range['to_id']);
        $this->handleIndexSku($range['from_id'], $range['to_id']);
        $this->indexSellerByRange($range['from_id'], $range['to_id']);
        $result['data'] = $range;
        ElasticsearchIndexHistory::logHistory($startTime, [
            'result' => $result,
            'page' => 'indexNewProduct',
        ]);
        return response()->json($result);
    }

    protected function getRange($request) {
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $setStartId = false;
        if (!$toId) {
            $toId = DB::table('product')->max('id');
            $setStartId = true;
        }
        if (!$fromId) {
            $fromId = Option::getStartId(self::KEY_INDEX);
        }
        if ($fromId < 2) {
            $item = DB::table('product')
                ->where('created_at', '>', date('Y-m-d H:i:s', time() - 1860))
                ->orderBy('created_at', 'asc')
                ->first(['id']);
            if (isset($item->id)) {
                $fromId = $item->id;
            } else {
                $fromId = $toId;
            }
        }
        return [
            'from_id' => $fromId,
            'to_id' => $toId,
            'set_start_id' => $setStartId,
        ];
    }

    public function indexByRange(Request $request)
    {
        set_time_limit(3600);
        ini_set('memory_limit', '500M');
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $filterSold = $request->input('sold');
        $startTime = time();
        $result['status'] = 'successful';
        if (!$fromId || !$toId) {
            return $result;
        }
        $this->elasticServiceIndex->initProductVariantOption();
        $filter = [
            'type' => 'init',
            'is_all' => 1,
        ];
        if ($filterSold) {
            $filter['sold'] = 1;
        }
        $this->handleIndexByRange($fromId, $toId, $filter);
        $result['run_time'] = time() - $startTime;
        ElasticsearchIndexHistory::logHistory($startTime, [
            'filter' => [
                'init' => 'indexByRange',
                'from' => $fromId,
                'to' => $toId,
            ],
            'result' => $result,
        ]);
        return response()->json($result);
    }

    protected function handleIndexByRange($filter, $fromId, $toId) {
        $pageSize = 1000;
        for ($startId = $fromId; $startId <= $toId; $startId += $pageSize) {
            $endId = $startId + $pageSize;
            if ($endId > $toId) {
                $endId = $toId;
            }
            $this->elasticServiceIndex->indexByRange(array_merge([
                'from_id' => $startId,
                'to_id' => $endId,
            ], $filter), $this->elasticServiceConfig);
        }
    }

    public function indexOptionByRange(Request $request)
    {
        // tag + category_id + seller + score + templateid + variant options in template
        set_time_limit(3600 * 2);
        ini_set('memory_limit', '500M');
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $pageSize = $request->input('page_size', 200);
        $startTime = time();
        $result['status'] = 'fail';
        if (!$fromId || !$toId) {
            return $result;
        }
        $result['status'] = 'successful';
        $this->elasticServiceIndex->initMetaData();
        $this->handleIndexOption($fromId, $toId, $pageSize);
        $result['run_time'] = time() - $startTime;
        ElasticsearchIndexHistory::logHistory($startTime, [
            'filter' => [
                'init' => 'indexOptionByRange',
                'from' => $fromId,
                'to' => $toId,
            ],
            'result' => $result,
        ]);
        return response()->json($result);
    }

    protected function handleIndexOption($fromId, $toId, $pageSize = 200) {
        for ($startId = $fromId; $startId < $toId; $startId += $pageSize) {
            $endId = $startId + $pageSize;
            if ($endId > $toId) {
                $endId = $toId;
            }
            $this->elasticServiceIndex->updateByRange($startId, $endId);
        }
    }

    public function indexSkuByRange(Request $request)
    {
        // tag + category_id + seller + score + templateid + variant options in template
        set_time_limit(3600 * 5);
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $pageSize = $request->input('page_size', 20);
        $startTime = time();
        $result['status'] = 'fail';
        if (!$fromId || !$toId) {
            return $result;
        }
        $result['status'] = 'successful';
        $this->elasticServiceIndex->initProductVariantOption();
        $this->handleIndexSku($fromId, $toId, $pageSize);
        $result['run_time'] = time() - $startTime;
        ElasticsearchIndexHistory::logHistory($startTime, [
            'filter' => [
                'init' => 'indexSkuByRange',
                'from' => $fromId,
                'to' => $toId,
            ],
            'result' => $result,
        ]);
        return response()->json($result);
    }

    protected function handleIndexSku($fromId, $toId, $pageSize = 200) {
        for ($startId = $fromId; $startId < $toId; $startId += $pageSize) {
            $endId = $startId + $pageSize;
            if ($endId > $toId) {
                $endId = $toId;
            }
            $this->elasticServiceIndex->updateSkuByRange($startId, $endId);
        }
    }

    public function indexProductCustom() {
        set_time_limit(3600 * 1);
        $startTime = time();
        $this->elasticServiceIndex->updateCustomProduct();
        $result['status'] = 'successful';
        $result['run_time'] = time() - $startTime;
        return response()->json($result);

    }

    public function indexSeller(Request $request)
    {
        set_time_limit(3600);
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $startTime = time();
        $this->indexSellerByRange($fromId, $toId);
        $result['status'] = 'successful';
        $result['run_time'] = time() - $startTime;
        return response()->json($result);
    }

    public function indexSellerByRange($fromId, $toId) {
        if ($fromId && $toId  &&  $toId > $fromId) {
            $this->sellerService->multiIndexProductSeller(range($fromId, $toId));
        }
    }


}
