<?php

namespace Modules\ZSearch\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Models\TagRefer;
use Modules\ZSearch\Models\User;
use Modules\ZSearch\Models\Product;

class ProductController extends Controller {

    private $elasticService;
    const TYPE_NAME = 'products_lite';

    public function __construct() {
        $this->elasticService = new ElasticSearchService(Config::get('z-search::elasticsearch'));
    }

    public function index(Request $request) {
        $input = $request->all();
        
        $input['from_time'] = date('Y-m-d H:00:00', strtotime('-1 hours'));
        $input['to_time'] = date('Y-m-d H:59:59', strtotime('-1 hours'));
        if ($request->input('type') == 'created_at') {
            $product = DB::table('product')
                            ->whereBetween('created_at', [ $input['from_time'], $input['to_time'] ])
                            ->select([DB::raw('MIN(id) AS min_id'), DB::raw('MAX(id) AS max_id')])
                            ->first();
            if (!empty($product->min_id) && !empty($product->max_id)) {
                $input['from_id'] = $product->min_id;
                $input['to_id'] = $product->max_id;
            }
            unset($input['from_time']);
            unset($input['to_time']);
        }
        
        // push to queue
        $locale = env('APP_LOCALE');
        if (!empty($locale)) {
            $locale = '/' . $locale;
        }

        $url = env('RESTQ_ORIGIN', 'https://q.printerval.com') . $locale . '/z-search/product/init?' . http_build_query($input);
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $result = curl_exec($ch);

        return response()->json([ 'url' => $url, 'data' => $input, 'result' => json_decode($result, true) ]);
    }

    public function init(Request $request) {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '2048M');
        $isAll = $request->input('isAll', 0);
        $fromId = $request->input('from_id', 0);
        $toId = $request->input('to_id', 0);
        $retVal = [ 'status' => 'successful', 'affectedRows' => 0];

        $baseQuery = DB::table('product')->select('id')->orderBy('id', 'asc');
        if ($isAll) {
            $products = $baseQuery->get();
            $retVal['affectedRows'] = $this->_indexing($products);
        } else if ($request->has('from_time') && $request->has('to_time')) {
            $products = DB::table('product')
                            ->whereBetween($request->input('type'), [$request->input('from_time'), $request->input('to_time')])
                            ->select('id')->get();
            $retVal['affectedRows'] = $this->_indexing($products);
        } else if ($fromId > 0 && $toId >= $fromId) {
            $step = 500;
            for ($productId = $fromId; $productId <= $toId; $productId = $productId + $step) {
                $newToId = $productId + $step;
                if ($productId + $step > $toId) {
                    $newToId = $toId;
                }

                $newQuery = clone $baseQuery;
                $products = $newQuery->whereBetween('id', [$productId, $newToId])->get();
                $retVal['affectedRows'] += $this->_indexing($products);
            }
        } 

        return response()->json($retVal);
    }

    public function search(Request $request) {
        $retVal = [ 'status' => 'successful', 'result' => [], 'meta' => [] ];
        $input = $request->all();
        $query = $this->_buildFilterData($input);
        $products = $this->elasticService->searchDocument($query, self::TYPE_NAME, Config::get('z-search::elasticsearch.index'));

        if (empty($products['hits']['hits']) && !empty($input['keyword']) && is_numeric($input['keyword'])) {
            $items = DB::table('product')->where('id', $input['keyword'])->select('id')->get();
            if (count($items) == 0) {
                $items = DB::table('product')->where('sku', $input['keyword'])->select('id')->get();
            }
            if (count($items) != 0) {
                $this->_indexing($items);
                $products = $this->elasticService->searchDocument($query, self::TYPE_NAME, Config::get('z-search::elasticsearch.index'));
            }
        }

        $productIds = [];
        $userIds = [];
        foreach ($products['hits']['hits'] as $product) {
            $product = $product['_source'];
            if (!empty($product['created_at'])) {
                $product['created_at'] = date('Y-m-d H:i:s', $product['created_at']);
            }
            if (!empty($product['updated_at'])) {
                $product['updated_at'] = date('Y-m-d H:i:s', $product['updated_at']);
            }
            array_push($retVal['result'], $product);
            array_push($productIds, $product['id']);
            if (!empty($product['actor_id'])) {
                array_push($userIds, $product['actor_id']);
            }
            if (!empty($product['updater_id'])) {
                array_push($userIds, $product['updater_id']);
            }
        }
        
        if (!empty($input['columns']) && $input['columns'] == 'id') { // only return product ids
            $retVal['result'] = $productIds;
        } else {
            $users = User::query()->whereIn('id', $userIds)->pluck('name', 'id')->toArray();
            $meta = $this->_getMetaData($productIds);
            foreach ($retVal['result'] as &$product) {
                if (isset($meta[$product['id']])) {
                    $product['meta'] = $meta[$product['id']];
                }
                if (!empty($product['actor_id']) && isset($users[$product['actor_id']])) {
                    $product['creater'] = $users[$product['actor_id']];
                }
                if (!empty($product['updater_id']) && isset($users[$product['updater_id']])) {
                    $product['modifier'] = $users[$product['updater_id']];
                }
            }
            if (!empty($products['hits']['total'])) {
                $retVal['meta']['total_count'] = $products['hits']['total'];
                $retVal['meta']['page_id'] = (int) $input['page_id'];
                $retVal['meta']['page_size'] = (int) $input['page_size'];
                $retVal['meta']['page_count'] = ceil($products['hits']['total'] / $input['page_size']);
                $retVal['meta']['has_next'] = ($input['page_id'] < $retVal['meta']['page_count'] - 1) ? true : false;
            }
        }
        
        return response()->json($retVal);
    }

    private function _indexing($products) {
        $affectedRows = 0;
        $insertData = [];

        foreach ($products as $product) {
            $productData = $this->_getProductData($product->id);
            if (empty($productData)) continue;
            
            $insertData['body'][] = [
                'index' => [
                    '_index' => Config::get('z-search::elasticsearch.index'),
                    '_type' => self::TYPE_NAME,
                    '_id' => $product->id,
                ]
            ];
            $insertData['body'][] = $productData;

            if (count($insertData['body']) >= 500) {
                $this->elasticService->bulkAction($insertData);
                $affectedRows += count($insertData['body']) / 2;
                $insertData = [];
            }
        }
        if (!empty($insertData['body'])) { 
            $this->elasticService->bulkAction($insertData);
            $affectedRows += count($insertData['body']) / 2;
            $insertData = [];
        }

        return $affectedRows;
    }

    private function _buildFilterData(&$input) {
        if (empty($input['page_id'])) {
            $input['page_id'] = Config::get('z-search::elasticsearch.defaultFrom');
        }
        if (empty($input['page_size'])) {
            $input['page_size'] = Config::get('z-search::elasticsearch.defaultSize');;
        }

        $retVal['from'] = (int) $input['page_id'] * (int) $input['page_size'];
        $retVal['size'] = $input['page_size'];
        $retVal['sort'][] = [ 
            'id' => [
                'order' => 'desc'
            ]
        ];
        $isExactly = false;
        if (!empty($input['keyword'])) {
            $retVal['query']['bool']['should'][] = [
                'multi_match' => [
                    'query' => $input['keyword'],
                    'fields' => Config::get('z-search::elasticsearch.type.products.searchFields'),
                ]
            ];
            if (is_numeric($input['keyword'])) {
                $retVal['query']['bool']['should'][] = [
                    'term' => [
                        'id' => $input['keyword']
                    ]
                ];
                $retVal['query']['bool']['should'][] = [
                    'term' => [
                        'sku' => $input['keyword']
                    ]
                ];
                $isExactly = true;
            }
            $retVal['query']['bool']['minimum_should_match'] = 1;
        }
        if (!$isExactly && !empty($input['created_from']) && !empty($input['created_to'])) {
            $retVal['query']['bool']['must'][] = [
                'range' => [
                    'created_at' => [
                        'gte' => strtotime($input['created_from']),
                        'lte' => strtotime($input['created_to'])
                    ]
                ]
            ];
        }
        if (!empty($input['status'])) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'status' => $input['status']
                ]
            ];
        }
        if (key_exists('category_id', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'categories.id' => $input['category_id']
                ]
            ];
        }
        if (key_exists('seller_id', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'seller_id' => $input['seller_id']
                ]
            ];
        }
        if (key_exists('is_violation', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'is_violation' => $input['is_violation']
                ]
            ];
        }
        if (key_exists('is_trademark', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'is_trademark' => $input['is_trademark']
                ]
            ];
        }
        if (key_exists('approve_advertising', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'approve_advertising' => $input['approve_advertising']
                ]
            ];
        }
        if (key_exists('actor_id', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'actor_id' => $input['actor_id']
                ]
            ];
        }
        if (key_exists('updater_id', $input)) {
            $retVal['query']['bool']['must'][] = [
                'term' => [
                    'updater_id' => $input['updater_id']
                ]
            ];
        }
        if (!empty($input['columns'])) {
            $retVal['_source'] = explode(',', $input['columns']);
        }

        return $retVal;
    }

    private function _getProductData($productId) {
        $retVal = null;
        $product = Product::find($productId);
        if ($product) {
            $categories = DB::table('product_n_category')
                            ->join('category', 'category.id', '=', 'product_n_category.category_id')
                            ->where('product_n_category.product_id', $productId)
                            ->select('category.id', 'category.name')
                            ->get();
            $sellers = DB::table('product_n_user')
                            ->join('users', 'users.id', '=', 'product_n_user.user_id')
                            ->where('product_n_user.product_id', $productId)
                            ->where('users.role', User::ROLE_SELLER)
                            ->select('users.id', 'users.name')
                            ->get();
            $retVal = [
                'id' => $product->id,
                'sku' => $product->sku,
                'barcode' => $product->barcode,
                'name' => $product->name,
                'slug' => $product->slug,
                'url' => $product->url,
                'price' => $product->price,
                'high_price' => $product->high_price,
                'display_price' => $product->display_price,
                'display_high_price' => $product->display_high_price,
                'image_url' => $product->image_url,
                'brand_id' => $product->brand_id,
                'status' => $product->status,
                'sold' => $product->sold,
                'view_count' => $product->view_count,
                'is_trademark' => $product->is_trademark,
                'is_violation' => $product->is_violation,
                'is_hidden' => $product->is_hidden,
                'is_always_on_ads' => $product->is_always_on_ads,
                'status_out_stock' => $product->status_out_stock,
                'pod_parent_id' => $product->pod_parent_id,
                'approve_advertising' => $product->approve_advertising,
                'created_at' => strtotime($product->created_at),
                'updated_at' => strtotime($product->updated_at),
                'gtin' => $product->gtin,
                'actor_id' => $product->actor_id,
                'updater_id' => $product->updater_id,   
                'categories' => json_decode($categories, true),
                'sellers' => json_decode($sellers, true)
            ];
        }

        return $retVal;
    }

    private function _getMetaData($productIds) {
        $retVal = [];
        $items = DB::table('product_meta')
                    ->whereIn('product_id', $productIds)
                    ->select('product_id', 'key', 'value')
                    ->get();
        foreach ($items as $item) {
            if (!isset($retVal[$item->product_id])) {
                $retVal[$item->product_id] = [];
            }
            $decorValue = $item->value;
            if (is_numeric($item->value)) {
                $decorValue = $item->value + 0;
            } else {
                $newValue = json_decode($decorValue);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $decorValue = $newValue;
                }
            }
            $retVal[$item->product_id][$item->key] = $decorValue;
        }
        return $retVal;
    }
}