<?php

namespace Modules\ZSearch\Controllers;

use App\Modules\ZSearch\Controllers\Feature\MainFeature;
use Module;
use Modules\ZSearch\Controllers\Feature\OtherFeature;
use Modules\ZSearch\Services\ElasticsearchIndexHistory;
use Modules\ZSearch\Services\ProductDetailService;
use Theme;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Services\ElasticSearchService;
use Illuminate\Support\Facades\DB;

use App\Modules\ZSearch\Controllers\Feature\BulkFeature;

class ProductDetailController extends Controller
{
    const OPTION_KEY = 'build_detail_product_id';
    const OPTION_UPDATE_KEY = 'update_build_detail_product_id';
    private $type = 'product_detail';


    protected $elasticService;
    protected $elasticSearchConfig;
    protected $elasticServiceConfig;
    protected $mainFeature;
    protected $bulkFeature;
    protected $otherFeature;
    protected $productDetailService;



    public function __construct(
    ) {
        $elasticSearchConfig = Config::get('z-search::elasticsearch-product');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticService =  new ElasticSearchService($this->elasticSearchConfig);
        $this->mainFeature = new MainFeature();
        $this->bulkFeature = new BulkFeature();
        $this->otherFeature = new OtherFeature();
        $this->productDetailService = new ProductDetailService();
    }

    /**
     * build detail product
     * 
     * @param Request $request
     * @return Response
     */
    public function buildDetailProduct(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '1G');
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        if (!$fromId || !$toId) {
            return;
        }
        $joinMerchant = $request->input('join_merchant_product');
        $neverIndexed = $request->input('never_indexed');
        $startTime = time();
        $productIds = $this->productDetailService->getProductIds([
            'from_id' => $fromId,
            'to_id' => $toId,
            'join_merchant_product' => $joinMerchant,
            'never_indexed' => $neverIndexed
        ]);
        $error = $this->indexProducts($productIds);
        $responseData = [
            'index' => $this->getDetailIndex(),
            'count' => $productIds ? count($productIds) : 0,
            'params' => $request->all(),
        ];
        ElasticsearchIndexHistory::logHistory($startTime, $responseData);
        return [
            'status' => 'successful',
            'error' => $error,
            'data' => $responseData,
        ];
    }


    public function updateFlashSale(Request $request) {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '1G');
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        if (!$fromId || !$toId) {
            return;
        }
        $startTime = time();
        $items = $this->productDetailService->getFlashSaleData([
            'from_id' => $fromId,
            'to_id' => $toId,
        ]);
        if ($items) {
            $this->updateProducts($items, $this->getDetailIndex(), $this->type);
        }
        $responseData = [
            'index' => $this->getDetailIndex(),
            'count' => $items ? count($items) : 0,
            'params' => $request->all(),
        ];
        ElasticsearchIndexHistory::logHistory($startTime, $responseData);
        return [
            'status' => 'successful',
            'data' => $responseData,
        ];
    }

    public function syncElasticsearchProductIndex(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '200M');
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $pageSize = 5000;
        if (!$fromId || !$toId) {
            return [];
        }
        $startTime = time();
        $count = 0;
        for ($start = $fromId; $start <= $toId; $start += $pageSize) {
            $items = $this->getElsData($start, $pageSize);
            if (!$items) {
                continue;
            }
            $ids = [];
            foreach ($items as $item) {
                $ids[] = $item['_source']['id'];
            }
            $ids = array_unique($ids);
            $count += $this->storeFlagData($ids);
        }

        return [
            'status' => 'successful',
            'count' => $count,
            'runtime' => time() - $startTime,
            'params' => $request->all(),
        ];
    }

    protected function indexProducts($productIds) {
        $error = [];
        foreach (array_chunk($productIds, 200) as $pIds) {
            $items = $this->mainFeature->handleChunk($pIds);
            $this->bulkFeature->handleChunk($pIds, $items);
            $this->otherFeature->handle($items);
            try {
                $this->elasticService->bulkAction($this->elasticService->bulkData($items, $this->getDetailIndex(), $this->type));
                $this->storeFlagData($pIds);
            } catch (\Exception $e) {
                $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
            }
        }
        return $error;
    }


    protected function storeFlagData($pIds) {
        if (!$pIds) {
            return 0;
        }
        $count = 0;
        foreach (array_chunk($pIds, 200) as $chunkPids) {
            $insertData = [];
            $existsIds = DB::table('elasticsearch_product_index')
                ->whereIn('id', $chunkPids)
                ->pluck('id', 'id')
                ->toArray();
            foreach ($chunkPids as $id) {
                if (array_key_exists($id, $existsIds)) {
                    continue;
                }
                $insertData[] = [
                    'id' => $id
                ];
            }
            if ($insertData) {
                $count += count($insertData);
                DB::table('elasticsearch_product_index')->insert($insertData);
            }
        }
        return $count;
    }

    protected function parseAndSave($logFile) {
        $file = fopen($logFile, 'r');
        $items = [];
        while (($line = fgets($file)) !== false) {
            $explode = explode('Query Executed:', $line);
            $line = trim($explode[1]);
            $explode2 = explode(' ', $line);
            array_pop($explode2);
            $executionTime = array_pop($explode2);
            $explode2[count($explode2) - 1] = str_replace('in', '', $explode2[count($explode2) - 1]);
            $items[] = [
                'query' => implode(' ', $explode2),
                'runtime' => $executionTime,
            ];
        }
        foreach (array_chunk($items, 500) as $xItems) {
            DB::table('mysql_query')->insert($xItems);

        }
    }

    protected function getDetailIndex() {
        return isset($this->elasticSearchConfig['detail_index']) ? $this->elasticSearchConfig['detail_index'] : 'product_detail';
    }


    protected function getElsData($fromId, $pageSize) {
        $body = [
            'from' => 0,
            'size' => $pageSize,
            '_source' => ['id'],
            //'_source' => false,
            'query' => [
                'range' => [
                    'id' => [
                        'gte' => $fromId,
                        'lte' => $fromId + $pageSize
                    ]
                ]
            ]
        ];
        $response = $this->elasticService->searchDocument($body, $this->type, $this->getDetailIndex());
        return  ($response && isset($response['hits']['hits'])) ? $response['hits']['hits'] : [];
    }


    private function updateProducts($products, $index, $type) {
        foreach (array_chunk($products, 200) as $chunkProducts) {
            $params = $this->elasticService->dataForUpdate($chunkProducts, $index, $type);
            $this->elasticService->bulkAction($params);
        }
    }

    protected function logQuery() {
        if (isset($_GET['debugg'])) {
            DB::listen(function ($query) {
                $bindings = array_map(function ($binding) {
                    return $binding instanceof \DateTime ? $binding->format('Y-m-d H:i:s') : $binding;
                }, $query->bindings);
                \Log::info('Query Executed: ' .str_replace_array('?', $bindings, $query->sql) . ' in ' . $query->time . ' ms');
            });
        }
    }

}