<?php

namespace Modules\ZSearch\Controllers;


use Illuminate\Http\Request;
use Modules\ZSearch\Models\Option;
use Modules\ZSearch\Models\SmallDesign;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Services\ElasticSearchService;
use Illuminate\Support\Facades\DB;

class SmallDesignController extends Controller
{
    protected $elasticService;
    protected $elasticSearchConfig;
    protected $maxSize = 10000;
    private $type = 'small_design';
    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticService =  new ElasticSearchService($this->elasticSearchConfig);
    }

    private function indexDesign($design)
    {
        $this->elasticService->saveDocument($design, $this->type, $this->elasticSearchConfig['index']);
    }

    public function indexDesignById(Request $request) {
        $totalIndex = 0;
        $errors = [];
        if ($request->has('ids')) {
            $ids = explode(',', $request->get('ids'));
            $smallDesigns = SmallDesign::join('product_n_design as pnd', 'pnd.design_id', '=', 'small_design.design_id')
                                ->join('product_n_user as pnu', 'pnu.product_id', '=', 'pnd.product_id')
                                ->whereIn('small_design.id', $ids)
                                ->select('pnu.user_id', 'small_design.id', 'small_design.name', 'small_design.small_design_url', 'small_design.design_id', 'small_design.token')
                                ->get()->keyBy('id')->toArray();
            if (!empty($smallDesigns)) {
                $this->addMultiIndex($smallDesigns, $errors);
                $totalIndex = count($smallDesigns);
            }
        }
        return response()->json([
            'totalIndex' => $totalIndex,
            'errors' =>  $errors
        ], 200);
    }

    public function multiIndexDesign(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '2048M');
        $filters = $request->all();
        $errors = [];
        $ranges = $this->getRangeIds($filters);
        $totalItem = 0;
        $firstDesign = [];
        if (!empty($ranges)) {
            foreach ($ranges as $range) {
                $range = array_merge($filters, $range);
                $designs = $this->getDesigns($range);
                if (empty($firstDesign)) {
                    $firstDesign = current($designs);
                    $firstDesign['token'] = 'is string';
                    $this->indexDesign($firstDesign);
                }
                $totalItem += count($designs);
                $this->addMultiIndex($designs, $errors);
            }
        }
        return response()->json([
            'totalIndex' => $totalItem,
            'errors' =>  $errors
        ], 200);
    }

    public function searchDesign(Request $request)
    {
        $filters = $request->all();
        $query = $this->prepareFilters($filters);
        try {
            $result = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
        } catch (\Exception $e) {
            $result = $e->getMessage();
        }
        $return = $this->prepareOutPut($result, $filters);
        return response()->json($return, 200);
    }

    public function searchDesignImageSearch(Request $request)
    {
        $filters = $request->all();
        $query = $this->prepareFiltersImageSearch($filters);
        try {
            $result = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
        } catch (\Exception $e) {
            $result = $e->getMessage();
        }
        $return = $this->prepareOutPut($result, $filters);
        return response()->json($return, 200);
    }

    protected function prepareOutPut($result, $filters)
    {
        $return = parent::prepareOutPut($result, $filters);
        if (!empty($return['result'])) {
            $this->getDesignPrices($return['result']);
            $return['result'] = $this->filterDesign($return['result']);
        }
        return $return;
    }

    private function filterDesign($items) {
        $result = [];
        $smallIds = array_values(array_column($items, 'id'));
        $designs = DB::table('small_design')
            ->whereIn('id', $smallIds)
            ->pluck('id', 'id')
            ->toArray();
        if ($items) {
            foreach ($items as $item) {
                if (array_key_exists($item['id'], $designs)) {
                    $result[] = $item;
                }
            }
        }

        return $result;
    }

    private function getDesignPrices(&$results)
    {
        $designIds = array_values(array_column($results, 'design_id'));
        $designs = DB::table('design')->select('price', 'id')->whereIn('id', $designIds)->get()->keyBy('id')->toArray();
        if ($designs) {
            foreach ($results as $key => $item) {
                $item['price'] = $item['display_price'] = 0;
                if (!empty($item['design_id']) && 
                    !empty($designs[$item['design_id']]) &&
                    !empty($designs[$item['design_id']]->price)    
                    ) {
                    $item['price'] = $designs[$item['design_id']]->price;
                    $item['display_price'] = formatPrice(doubleval($designs[$item['design_id']]->price));
                }
                $results[$key] = $item;
            }
        }
    }

    private function prepareFilters(&$filters)
    {
        $pageSize = !empty($filters['page_size']) ? $filters['page_size'] : 20;
        $pageId = !empty($filters['page_id']) ? $filters['page_id'] : 1;
        $filters['from'] = ($pageId - 1) * $pageSize;
        $query = [
            'from' => !empty($filters['from']) ? $filters['from'] : 0,
            'size' => $pageSize,
            'sort' => [
                ['id' => ['order' => 'desc']],
            ],
        ];
        if (!empty($filters['keyword'])) {
            $query['query'] = [
                'bool' => [
                    'must' => [
                        [
                            'bool' => [
                                'should' => [
                                    [
                                        'multi_match' => [
                                            'query' => $filters['keyword'],
                                            'type' => 'phrase_prefix',
                                            'minimum_should_match' => '90%',
                                            'fields' => ['name']
                                        ]
                                    ],
                                    [
                                        'multi_match' => [
                                            'query' => $filters['keyword'],
                                            'operator' => 'and',
                                            'minimum_should_match' => '90%',
                                            'fuzziness' => 0.8,
                                            'fields' => ['name']
                                        ]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ];
        }
        foreach (['user_id', 'design_id'] as $field) {
            if (!empty($filters[$field])) {
                $query['query']['bool']['must'][] = [
                    'term' => [
                        $field => $filters[$field]
                    ]
                ];
            }
        }
        if (isset($filters['seller'])) {
                $query['query']['bool']['must'][] = [
                    'term' => [
                        'real_seller' => $filters['seller'] == 'true'
                    ]
                ];
        }

        return $query;
    }

    private function prepareFiltersImageSearch(&$filters)
    {
        $pageSize = !empty($filters['page_size']) ? $filters['page_size'] : 200;
        $pageId = !empty($filters['page_id']) ? $filters['page_id'] : 1;
        $filters['from'] = ($pageId - 1) * $pageSize;
        $query = [
            'from' => !empty($filters['from']) ? $filters['from'] : 0,
            'size' => $pageSize,
            'sort' => [
                ['id' => ['order' => 'desc']],
            ],
        ];
        if (!empty($filters['keyword'])) {
            $query['query'] = [
                'bool' => [
                    'must' => [
                        [
                            'bool' => [
                                'should' => [
                                    [
                                        'multi_match' => [
                                            'query' => $filters['keyword'],
                                            'type' => 'phrase_prefix',
                                            'minimum_should_match' => '90%',
                                            'fields' => ['name']
                                        ]
                                    ],
                                    [
                                        'multi_match' => [
                                            'query' => $filters['keyword'],
                                            'operator' => 'and',
                                            'minimum_should_match' => '90%',
                                            'fuzziness' => 0.8,
                                            'fields' => ['name']
                                        ]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ];
        }
        foreach (['user_id', 'design_id'] as $field) {
            if (!empty($filters[$field])) {
                $query['query']['bool']['must'][] = [
                    'term' => [
                        $field => $filters[$field]
                    ]
                ];
            }
        }
        if (isset($filters['design_ids'])) {
            $query['query']['bool']['must'][] = [
                'terms' => [
                    'design_id' => explode(',', $filters['design_ids'])
                ]
            ];
        }
        if (isset($filters['seller'])) {
                $query['query']['bool']['must'][] = [
                    'term' => [
                        'real_seller' => $filters['seller'] == 'true'
                    ]
                ];
        }

        return $query;
    }

    public function findDesign($id)
    {
        try {
            $result = $this->elasticService->findDocument($id, $this->type, $this->elasticSearchConfig['index']);
        } catch (\Throwable $e) {
            $result = $e->getMessage();
        }
        return response()->json($result, 200);
    }

    public function deleteIndexDesign($id)
    {
        if ($this->elasticService->existsDocument($id, $this->type, $this->elasticSearchConfig['index'])) {
            $this->elasticService->deleteDocument($id, $this->type, $this->elasticSearchConfig['index']);
        }
    }

    public function deleteIndex()
    {
        return $this->elasticService->deleteIndex($this->elasticSearchConfig['index']);
    }

    private function addMultiIndex($groupDesigns, &$errors)
    {
        foreach ($groupDesigns as $design) {
            try {
                $this->indexDesign($design);
            } catch (\Exception $e) {
                $message = $e->getMessage() . ' ' . $design['id'] . ' ' . $e->getFile() . '-' . $e->getLine();
                $errors[] = $message;
            }
        }
    }

    private function getNearestIndexId($filters, $key = 'nearestIndexDesignId')
    {
        $nearestIndexId = 1;
        if (!empty($filters['min_id'])) {
            $nearestIndexId = (int) $filters['min_id'];
        } else if ($nearestIndex = Option::where('key', $key)->value('value')) {
            $nearestIndexId = $nearestIndex;
        }
        return $nearestIndexId;
    }

    private function getRangeIds($filters)
    {
        $min = $this->getNearestIndexId($filters);
        $max = $this->getMaxIdDesignIndex($filters);
        $ranges = $this->buildRangerIds($min, $max, $this->maxSize);

        if (!array_key_exists('max_id', $filters)) {
            $this->setNearestIndexDesignId($max);
        }
        return $ranges;
    }

    private function getMaxIdDesignIndex($filters)
    {   
        $maxId = 1;
        if (!empty($filters['max_id'])) {
            $maxId = (int) $filters['max_id'];
        } else if($latestDesign = SmallDesign::orderBy('id', 'desc')->first()) {
            $maxId = $latestDesign->id;
        }
        return $maxId;
    }

    private function setNearestIndexDesignId($max)
    {
        Option::updateOrCreate(
            ['key' => 'nearestIndexDesignId'],
            [
                'value' => $max,
                'key' => 'nearestIndexDesignId'
            ]
        );
    }

    private function getDesigns($filters)
    {
        $query = SmallDesign::select('id', 'name', 'small_design_url', 'design_id', 'token')->orderBy('id');
        if (!empty($filters['id'])) {
            $query->where('id', '=', $filters['id']);
        }
        if (!empty($filters['min_id'])) {
            $query->where('id', '>=',  $filters['min_id']);
        }
        if (!empty($filters['max_id'])) {
            $query->where('id', '<',  $filters['max_id']);
        }
        return $query->get()->toArray();
    }

    public function indexSmallDesign(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '2048M');
        $total = 0;
        $maxId = $this->getMaxIdDesignIndex($request);
        $minId = $this->getNearestIndexId($request, 'init_small_design:id');
        $ranges = $this->buildRangerIds($minId, $maxId, $request->get('chuck', 2000));
        if (!empty($ranges)) {
            foreach ($ranges as $range) {
                $total += $this->buildOwnerInfo($range);
            }
        }
        Option::updateOrCreate(
            ['key' => 'update_small_design_owner:id'],
            [
                'value' => $maxId,
                'key' => 'update_small_design_owner:id'
            ]
        );
        return response()->json([
            'status' => 'successful',
            'result' => [
                'max_id' =>$maxId,
                'min_id' => $minId,
                'total' => $total
            ]
        ]);
    }

    protected function buildOwnerInfo($range) {
        $total = 0;
        if (!empty($range['min_id']) && !empty($range['max_id'])) {
            $result = \DB::table('small_design as sd')
                        ->leftjoin('product_n_design as pnd', 'pnd.design_id', '=', 'sd.design_id')
                        ->leftjoin('product_n_user as pnu', 'pnu.product_id', '=', 'pnd.product_id')
                        ->where('sd.id', '>=', $range['min_id'])
                        ->where('sd.id', '<=', $range['max_id'])
                        ->select('pnu.user_id', 'sd.id', 'sd.name', 'sd.small_design_url', 'sd.design_id', 'sd.token')
                        ->get()->keyBy('id');

            if (!empty($result)) {
                foreach ($result->chunk(200) as $items) {
                    $updateData = [];
                    foreach ($items as $smallDesign) {
                        $updateData['body'][] = [
                            'index' => [
                                '_index' => $this->elasticSearchConfig['index'],
                                '_type' => $this->type,
                                '_id' => $smallDesign->id
                            ],
                        ];
                        $updateData['body'][] = $smallDesign;
                    }
                    if (!empty($updateData)) {
                        $this->elasticService->bulkAction($updateData);
                    }
                }
                $total = count($result);
            }
        }
        return $total;
    }

    protected function buildRangerIds($minId, $maxId, $chunk = 2000) {
        $rangeIds = [];
        for ($i = ($minId - 1); $i < $maxId; $i+= $chunk) { 
            $rangeIds[] = [
                'min_id' => $i + 1,
                'max_id' => (($i + $chunk) > $maxId) ? $maxId : ($i + $chunk)
            ];
        }
        return $rangeIds;
    }

    public function indexSeller(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '4048M');
        $startId = $request->input('from_id', 1);
        $maxId = $request->input('to_id', 1000000);
        $step = 1000;
        $startTime  = time();
        $elasticSearchConfig =  Config::get('z-search::elasticsearch');

        $count = 0;
        for ($head = $startId; $head < $maxId; $head += $step) {
            $smallDesigns = \DB::table('small_design as sd')
                ->join('product_n_design as pnd', 'pnd.design_id', '=', 'sd.design_id')
                ->join('product_n_user as pnu', 'pnu.product_id', '=', 'pnd.product_id')
                ->join('users as u', 'pnu.user_id', '=', 'u.id')
                ->whereIn('u.role', ['SELLER', 'STAFF'])
                ->where('sd.id', '>=', $head)
                ->where('sd.id', '<', $head + $step)
                ->select('pnu.user_id', 'sd.id', 'u.seller_token')
                ->get();
            if ($smallDesigns->isNotEmpty()) {
                $data = [];
                foreach ($smallDesigns as $smallDesignItem) {
                        $data[] = [
                            'id' => $smallDesignItem->id,
                            'user_id' => $smallDesignItem->user_id,
                            'real_seller' => $smallDesignItem->seller_token == null ? false : true,
                        ];
                }
                foreach (array_chunk($data, 200) as $partData) {
                    $count += count($partData);
                    $res = $this->bulkActionSmallDesign($partData, $elasticSearchConfig['index']);
                    if ($res['error']) {
                        \Log::info('ZSEARCH - multiUpdateInfoProduct ERROR', [$res['error']]);
                        $error[] = $res['error'];
                    }
                }
            }
        }

        return response()->json([
            'status' => 'successful',
            'data' => [
                'count' => $count,
            ],
            'run_time' => time() - $startTime
        ]);
    }

    public function bulkActionSmallDesign($smallDesigns, $index) {
        $error = [];
        $params = [];
        $type = $this->type;
        foreach (array_chunk($smallDesigns, 200) as $chunkSmallDesigns) {
            try {
                foreach ($chunkSmallDesigns as $smallDesign) {
                    $params['body'][] = [
                        'update' => [
                            '_index' => $index,
                            '_type' => $type,
                            '_id' => $smallDesign['id']
                        ],
                    ];
                    unset($smallDesign['id']);
                    $params['body'][] = [
                        'doc' => $smallDesign
                    ];
                }
                $this->elasticService->bulkAction($params);
            } catch (\Exception $e) {
                $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
            }
        }
        return [
            'error' => $error
        ];
    }
}
