<?php

namespace Modules\ZSearch\Controllers;


use Illuminate\Http\Request;
use Modules\ZSearch\Services\SuggestDataService;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Models\Option;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Models\SearchSuggest;

class SuggestController extends Controller
{

    protected $optionKey = 'last_insert_suggest_id';
    private $type = 'search_suggest';
    protected $elasticService;
    protected $elasticSearchConfig;

    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticService =  new ElasticSearchService($this->elasticSearchConfig);
    }

    public function find(Request $request) {
        $suggestDataService = new SuggestDataService();
        $limit = $request->input('limit', 10);
        $keyword = $request->input('keyword');

        $suggest = [];
        $trending = [];
        $status = 'fail';
        try {
            if ($keyword) {
                $retval = $this->search([
                    'keyword' => $keyword,
                    'page_size' => $limit * 3,
                    'fields' => ['keyword']
                ]);
                if (!empty($retval) && !empty($retval['result'])) {
                    $suggest = $retval['result'];
                } else {
                    $suggest = $suggestDataService->suggest([
                        'limit' => $limit * 3,
                        'keyword' => $keyword,
                    ]);
                }
                if (!empty($suggest)) {
                    $suggest = $this->prepareSuggest($suggest, $limit);
                }
            }
            $trending = $suggestDataService->findTrending([
                'limit' => $limit
            ]);
            $status = 'successful';
        } catch (\Exception $e) {
           //@todo on error

        }

        $result = [
            'status' => $status,
            'suggest' =>  $this->filterKeyword($suggest),
            'trending' =>  $this->filterKeyword($trending),
        ];
        return response()->json($result);
    }

    protected function prepareSuggest($suggest, $limit) {
        $items = array_unique(array_column($suggest, 'keyword'));
        $retval = [];
        foreach (array_slice($items, 0, $limit) as $item) {
            $retval[] = ['keyword' => $item];
        }
        return $retval;
    }

    protected function filterKeyword($items) {
        $result = [];
        if ($items) {
            foreach ($items as $item) {
                $keyword = is_array($item) ?  $item['keyword'] : $item->keyword;
                if ($keyword > 50) {
                    continue;
                }
                $keyword = html_entity_decode($keyword);
                if (is_array($item)) {
                    $item['keyword'] = $keyword;
                } else {
                    $item->keyword = $keyword;
                }
                $result[] = $item;
            }
        }
        return $result;
    }

    public function cronBuildData(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '4048M');
        $filter = $this->buildFilter($request);
        $date = new \DateTime();
        $suggestDataService = new SuggestDataService();
        $trackingData = $suggestDataService->getSearchTrackingData($filter);
        $insertData = [];

        foreach ($trackingData as $item) {
            $insertData[] = [
                'search' => $item->quantity,
                'conversion' => $item->conversion,
                'keyword' => html_entity_decode($item->keyword),
                'created_at' => $date,
            ];
        }
        if (!empty($insertData)) {
            $suggestDataService->insertAll($insertData);
            $suggestDataService->delete($date);
        }

        $response = array(
            "status" => 'successful',
            "items" => count($trackingData)
        );
        return response()->json($response);
    }

    private function buildFilter($request) {
        $day = $request->input('day', 60);
        $fromDate = date('Y-m-d', time() - 86400 * $day);
        return [
            'from_date' => $fromDate
        ];
    }

    private function indexSuggest($item)
    {
        $this->elasticService->saveDocument($item, $this->type, $this->elasticSearchConfig['index']);
    }

    public function multiIndex(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '2048M');
        $filter = $request->all();
        $errors = [];
        if (!empty($filter['id'])) {
            $finalId = $startId = $filter['id'];
        } else {
            $startId = $this->getStartId($filter);
            $finalId = $this->getFinalId($filter, $startId);
        }
        $totalItem = 0;
        $firstItem = [];
        $range = Config::get('z-search::suggest.index_range', 1000);
        if (isset($startId) && !empty($finalId) && $finalId >= $startId) {
            for ($i = $startId; $i <= $finalId; $i += $range) {
                if (($i + $range) > $finalId) {
                    $maxId = $finalId + 1;
                } else {
                    $maxId = $i + $range;
                }
                $suggestFilter = [
                    'min_id' => $i,
                    'max_id' => $maxId
                ];
                $suggestSearchs = $this->getSuggests($suggestFilter);
                if (empty($firstItem)) {
                    $firstItem = current($suggestSearchs);
                    $firstItem['token'] = 'is string';
                    $this->indexSuggest($firstItem);
                }
                $this->addMultiIndex($suggestSearchs, $errors);
                $totalItem += count($suggestSearchs);
            }
            if (empty($filter['id'])) {
                $this->setNearestIndexId($finalId);
            }
            $this->removeOldDocuments();
        }

        return response()->json([
            'totalIndex' => $totalItem,
            'errors' =>  $errors
        ], 200);
    }

    protected function removeOldDocuments() {
        $suggestDataService = new SuggestDataService();
        $maxId = $suggestDataService->getCurrentMinId();
        $deleteQuery = [
            'range' => [
                'id' => [
                    'lt' => $maxId
                ]
            ]
        ];
        $this->elasticService->deleteItemsByQuery($this->elasticSearchConfig['index'], $this->type, $deleteQuery);
    }

    protected function getStartId($filter) {
        if (!empty($filter['start_id'])) {
            $startId = $filter['start_id'];
        } else if (empty($filter['reset'])) {
            $lastId = \DB::table('option')->where('key', $this->optionKey)->value('value');
            $lastId++;
            if (isset($lastId)) {
                $startId = $lastId;
            }
        }

        if (empty($startId)) {
            $startId = \DB::table('search_suggest')->min('id');
        }
        
        return $startId;
    }

    protected function getFinalId($filter, $startId) {
        if (!empty($filter['final_id'])) {
            $finalId = $filter['final_id'];
        } else if (!empty($filter['limit'])) {
            $finalId = \DB::table('search_suggest')->where('id', '>=', $startId)->limit($filter['limit'])->get()->max('id');
        } else {
            $finalId = \DB::table('search_suggest')->max('id');
        }
        return $finalId;
    }

    private function setNearestIndexId($id)
    {
        Option::updateOrCreate(
            ['key' => $this->optionKey],
            [
                'value' => $id
            ]
        );
    }

    protected function getSuggests($filter)
    {
        $maxLength = Config::get('z-search::suggest.max_length', 100);
        return SearchSuggest::where('id', '>=', $filter['min_id'])
            ->where('id', '<', $filter['max_id'])
            ->select('id', 'keyword', 'search')
            ->get()->map(function ($item) use($maxLength) {
                if (isset($item->keyword) && strlen($item->keyword) > $maxLength) {
                    $item->keyword = html_entity_decode(substr($item->keyword, 0, ($maxLength - 1)));
                }
                return $item;
            })->toArray();
    }

    private function addMultiIndex($items, &$errors)
    {
        foreach ($items as $item) {
            try {
                $this->indexSuggest($item);
            } catch (\Exception $e) {
                $message = $e->getMessage() . ' ' . $item['id'] . ' ' . $e->getFile() . '-' . $e->getLine();
                $errors[] = $message;
            }
        }
    }

    public function search($filters)
    {
        $query = $this->prepareFilters($filters);
        try {
            $result = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
        } catch (\Exception $e) {
            $result = $e->getMessage();
        }
        $return = $this->prepareOutPut($result, $filters);
        return $return;
    }

    private function prepareFilters(&$filters)
    {
        $pageSize = !empty($filters['page_size']) ? $filters['page_size'] : 20;
        $pageId = !empty($filters['page_id']) ? $filters['page_id'] : 1;
        $filters['from'] = ($pageId - 1) * $pageSize;
        $query = [
            'from' => !empty($filters['from']) ? $filters['from'] : 0,
            'size' => $pageSize,
            'sort' => [
                ['id' => ['order' => 'desc']],
            ],
        ];
        if (!empty($filters['keyword'])) {
            $query['query'] = [
                'match' => [
                    'keyword' => [
                        'query' => $filters['keyword'],
                        'fuzziness' => 1
                    ]
                ]
            ];
        }
        if (!empty($filters['fields'])) {
            $query['_source'] = $filters['fields'];
        }
        return $query;
    }
}
