<?php

namespace Modules\ZSearch\Controllers;


use Illuminate\Http\Request;
use Modules\ZSearch\Models\Option;
use Illuminate\Support\Facades\Config;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Models\Tag;

class TagController extends Controller
{
    protected $elasticService;
    protected $elasticSearchConfig;
    protected $maxSize = 10000;
    private $type = 'tags';
    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticService =  new ElasticSearchService($this->elasticSearchConfig);
    }

    public function index(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '2048M');
        $total = 0;
        $maxId = $this->getMaxId($request);
        $minId = $this->getNearestIndexId($request, 'index_tags:id');
        $ranges = $this->buildRangerIds($minId, $maxId, $request->get('chuck', 2000));
        if (!empty($ranges)) {
            foreach ($ranges as $range) {
                $total += $this->indexTags($range);
            }
        }
        Option::updateOrCreate(
            ['key' => 'index_tags:id'],
            [
                'value' => $maxId,
                'key' => 'index_tags:id'
            ]
        );
        return response()->json([
            'status' => 'successful',
            'result' => [
                'max_id' =>$maxId,
                'min_id' => $minId,
                'total' => $total
            ]
        ]);
    }

    private function getNearestIndexId($filters, $key = 'index_tags:id')
    {
        $nearestIndexId = 1;
        if (!empty($filters['min_id'])) {
            $nearestIndexId = (int) $filters['min_id'];
        } else if ($nearestIndex = Option::where('key', $key)->value('value')) {
            $nearestIndexId = $nearestIndex;
        }
        return $nearestIndexId;
    }

    private function getMaxId($filters)
    {   
        $maxId = 1;
        if (!empty($filters['max_id'])) {
            $maxId = (int) $filters['max_id'];
        } else if($latestItem = Tag::orderBy('id', 'desc')->first()) {
            $maxId = $latestItem->id;
        }
        return $maxId;
    }

    protected function indexTags($range) {
        $total = 0;
        if (!empty($range['min_id']) && !empty($range['max_id'])) {
            $result = Tag::where('id', '>=', $range['min_id'])
                        ->where('id', '<=', $range['max_id'])
                        ->get()->toArray();
            if (!empty($result)) {
                $tags = array_chunk($result, 200);
                foreach ($tags as $items) {
                    $params = $this->elasticService->bulkData($items, $this->elasticSearchConfig['index'], $this->type);
                    $this->elasticService->bulkAction($params);
                }
                $total = count($result);
            }
        }
        return $total;
    }

    protected function buildRangerIds($minId, $maxId, $chunk = 2000) {
        $rangeIds = [];
        for ($i = ($minId - 1); $i < $maxId; $i+= $chunk) { 
            $rangeIds[] = [
                'min_id' => $i + 1,
                'max_id' => (($i + $chunk) > $maxId) ? $maxId : ($i + $chunk)
            ];
        }
        return $rangeIds;
    }

    public function search(Request $request)
    {
        $filters = $request->all();
        $query = $this->prepareFilters($filters);
        try {
            $result = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
        } catch (\Exception $e) {
            $result = $e->getMessage();
        }
        $return = $this->prepareOutPut($result, $filters);
        return response()->json($return, 200);
    }

    private function prepareFilters(&$filters)
    {
        $pageSize = !empty($filters['page_size']) ? $filters['page_size'] : 20;
        $pageId = !empty($filters['page_id']) ? $filters['page_id'] : 1;
        $filters['from'] = ($pageId - 1) * $pageSize;
        $query = [
            'from' => !empty($filters['from']) ? $filters['from'] : 0,
            'size' => $pageSize,
            'sort' => [
                ['id' => ['order' => 'desc']],
            ],
        ];
        if (!empty($filters['keyword'])) {
            $query['query'] = [
                'bool' => [
                    'must' => [
                        [
                            'bool' => [
                                'should' => [
                                    [
                                        'multi_match' => [
                                            'query' => $filters['keyword'],
                                            'type' => 'phrase_prefix',
                                            'minimum_should_match' => '90%',
                                            'fields' => ['title']
                                        ]
                                    ],
                                    [
                                        'multi_match' => [
                                            'query' => $filters['keyword'],
                                            'operator' => 'and',
                                            'minimum_should_match' => '90%',
                                            'fuzziness' => 0.8,
                                            'fields' => ['title']
                                        ]
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ];
        }
        return $query;
    }

    public function indexById(Request $request)
    {
        $totalIndex = 0;
        $errors = [];
        if ($request->has('ids')) {
            $ids = explode(',', $request->get('ids'));
            $tags = Tag::whereIn('id', $ids)->get()->toArray();
            if (!empty($tags)) {
                foreach ($tags as $tag) {
                    $this->elasticService->saveDocument($tag, $this->type, $this->elasticSearchConfig['index']);
                }
                $totalIndex = count($tags);
            }
        }
        return response()->json([
            'totalIndex' => $totalIndex,
            'errors' =>  $errors
        ], 200);
    }
}
