<?php

namespace Modules\ZSearch\Controllers;

use App\Utils;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Models\Product;
use Modules\ZSearch\Services\ElasticSearchIndexSeller;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Services\TrademarkService;

class TrademarkController extends Controller
{
    protected $trademarkService = null;
    private $type = 'trademark';
    protected $elasticService;

    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticService = new ElasticSearchService($elasticSearchConfig);
    }

    public function search(Request $request) {
        $response = [
            'status' => 'fail'
        ];
        try {
            $filter = $this->buildSearchFilter($request);
            $data = [];
            if ($filter['keyword']) {
                $data = $this->trademarkService()->search($filter);
            }
            $response = [
                'status' => 'successful',
                'meta' => $this->buildMeta($data, $filter),
                'result' => $this->trademarkService()->buildSearchData($data),
            ];
        } catch (\Exception $e) {
        }
        return response()->json($response);

    }

    public function index(Request $request) {
        set_time_limit(3600 * 5);
        $start = time();
        $filter = $this->buildIndexFilter($request);
        $pIds = Product::getProductIds($filter);
        Product::$withoutAppends = true;
        $this->trademarkService()->index($pIds, $filter);
        $result = [
            'status' => 'successful',
            'run_time' => time() - $start,
            'result' => count($pIds),
            'input' => $request->all(),
        ];
        return response()->json($result);
    }

    protected function buildIndexFilter($request) {
        $isAll = $request->input('isAll', 0);
        $lastMin = $request->input('min', 33);
        $filter = [
            'init' => 1,
            'is_all' => $isAll,
            'from_id' => $request->input('from_id', -1),
            'to_id' => $request->input('to_id', -1),
            'updated_from' => date('Y-m-d H:i:s', time() - $lastMin * 60),
        ];
        return $filter;
    }

    protected function trademarkService() {
        if (!$this->trademarkService) {
            $this->trademarkService = new TrademarkService();
        }
        return $this->trademarkService;
    }

    protected function buildMeta($data, $filter) {
        $total = isset($data['hits']['total']) ? $data['hits']['total'] : 0;
        return $this->getMetaData($filter, $total);
    }

    protected function buildSearchFilter($request) {
        $filter = [
            'keyword' => $request->input('q'),
            'page_size' => $request->input('page_size', 40),
            'from' => $request->input('page_id', 0) * $request->input('page_size', 40),
        ];
        if ($request->has('from_id')) {
            $filter['from_id'] = $request->input('from_id');
        }
        if ($request->has('c')) {
            $categoryId = $request->input('c');
            $filter['categories_id'] = explode(',', $categoryId);
        }
        return $filter;
    }

    /**
     * index trademark
     * 
     * @param Request $request
     * @return Response
     */
    public function indexTradeMark(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '4048M');
        $startId = $request->input('from_id', 1);
        $maxId = $request->input('to_id', 10000);
        $step = 1000;
        $startTime  = time();
        $elasticSearchConfig =  Config::get('z-search::elasticsearch');
        $error = [];
        $count = 0;

        for ($head = $startId; $head <= $maxId; $head += $step) {
            $trademarks = DB::table('trademarks')
                ->whereIn('type', ['trademark', 'violation'])
                ->where('status', 'ACTIVE')
                ->where('id', '>=', $head)
                ->where('id', '<', $head + $step)
                ->get()->toArray();
            foreach(array_chunk($trademarks, 200) as $trademarkChunk) {
                $data = $this->buildTrademarkData($trademarkChunk);
                $count += count($data);

                try {
                    $this->elasticService->bulkAction($this->elasticService->bulkData($data, $elasticSearchConfig['index'], $this->type));
                } catch (\Exception $e) {
                    $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
                }
            }
        }

        return [
            'count' => $count,
            'error' => $error,
            'rumtime' => time() - $startTime
        ];
    }

    /**
     * build trademark data
     * 
     * @param Array $trademarks
     * @return Array
     */
    private function buildTrademarkData($trademarks)
    {
        $result = [];
        foreach ($trademarks as $trademarkItem) {
            $result[] = json_decode(json_encode($trademarkItem), true);
        }

        return $result;
    }
}