<?php

namespace Modules\ZSearch\Controllers;

use App\Utils;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Models\Product;
use Modules\ZSearch\Services\ElasticSearchIndexSeller;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Services\TrademarkService;
use PhpOffice\PhpSpreadsheet\IOFactory;

class TrademarkKeywordController extends Controller
{
    protected $trademarkService = null;
    private $type = 'trademark_keyword';
    protected $elasticService;
    protected $elasticSearchConfig;

    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticSearchConfig = $elasticSearchConfig;
        $this->elasticService = new ElasticSearchService($elasticSearchConfig);
    }


    /**
     * search trademark
     * 
     * @param Request $request
     * @return Response
     */
    public function check(Request $request) {
        $retval = [
            'status' => 'successful',
            'keywords' => []
        ];
        if ($request->has('keywords')) {
            $retval['keywords'] = $this->detectKeyword($request->input('keywords'), $request->all());
        }
        return response()->json($retval);
    }

    protected function detectKeyword($title, $params = []) {
        if (!empty($params['is_explode']) && $params['is_explode'] == 1) {
            $keywords = explode(' ', $title);
        }
        $keywords[] = $title;
        $retval = [];
        $multiScore = !empty($params['multi_score']) ? (int) $params['multi_score'] : 9;
        $singleScore = !empty($params['single_score']) ? (int) $params['single_score'] : 5.5;
        foreach ($keywords as $keyword) {
            $minScore = (count(explode(' ', $keyword)) > 1) ? $multiScore : $singleScore;
            $query = $this->prepareFilters($keyword);
            try {
                $result = $this->elasticService->searchDocument($query, $this->type, $this->elasticSearchConfig['index']);
                if ($result['hits']['total'] > 0) {
                    $record = $result['hits']['hits'][0];
                    
                    if ($minScore <= $record['_score']) {
                        $countWord = substr_count($record['_source']['keyword'], ' ');
                        if (substr_count($keyword, ' ') == 0 && $countWord == substr_count($keyword, ' ')) {
                            $retval[$keyword] = [
                                'trademark_keyword' => $record['_source']['keyword'],
                                'score' => $record['_score'],
                                'min_score' => $minScore
                            ];
                        } else if (substr_count($keyword, ' ') >= 1 && $countWord >= 1) {
                            $retval[$keyword] = [
                                'trademark_keyword' => $record['_source']['keyword'],
                                'score' => $record['_score'],
                                'min_score' => $minScore
                            ];

                        }
                    }
                }
            } catch (\Throwable $th) {
                //throw $th;
            }
            
        }
        return $retval;
    }
   

    /**
     * index trademark
     * 
     * @param Request $request
     * @return Response
     */
    public function index(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '4048M');
        $startId = $request->input('from_id', 1);
        $maxId = $request->input('to_id', 10000);
        $step = 1000;
        $startTime  = time();
        $elasticSearchConfig =  Config::get('z-search::elasticsearch');
        $error = [];
        $count = 0;

        for ($head = $startId; $head <= $maxId; $head += $step) {
            $trademarks = DB::table('trademarks')
                ->whereIn('type', ['trademark', 'violation'])
                ->where('status', 'ACTIVE')
                ->where('id', '>=', $head)
                ->where('id', '<', $head + $step)
                ->get(['id', 'keyword', 'type', 'status'])->toArray();
            foreach(array_chunk($trademarks, 200) as $trademarkChunk) {
                $data = $this->buildTrademarkData($trademarkChunk);
                $count += count($data);

                try {
                    $this->elasticService->bulkAction($this->elasticService->bulkData($data, $elasticSearchConfig['index'], $this->type));
                } catch (\Exception $e) {
                    $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
                }
            }
        }

        return [
            'count' => $count,
            'error' => $error,
            'rumtime' => time() - $startTime
        ];
    }

    /**
     * build trademark data
     * 
     * @param Array $trademarks
     * @return Array
     */
    private function buildTrademarkData($trademarks)
    {
        $result = [];
        foreach ($trademarks as $trademarkItem) {
            $result[] = (array) $trademarkItem;
        }

        return $result;
    }

    private function prepareFilters($keyword)
    {
        $query = [
            'size' => 1,
        ];
        $keyword = strtolower($keyword);
        if (!empty($keyword)) {
            if (count(explode(' ', $keyword)) > 1) {
                $query['query'] = [
                    'match' => [
                        'keyword' => [
                            'query' => $this->removeDuplicateWords($keyword),
                            'minimum_should_match' => '70%'
                        ]
                    ]
                ];
            } else {
                $query['query'] = [
                    'fuzzy' => [
                        'keyword' => [
                            'value' => $keyword,
                            'fuzziness' => 2,
                            'prefix_length' => 1
                        ]
                    ]
                ];
            }
        }
        return $query;
    }

    function readKeywordsFromExcel($filePath) {
        $keywords = [];
    
        try {
            // Load the Excel file
            $spreadsheet = IOFactory::load($filePath);
    
            // Get the active sheet
            $sheet = $spreadsheet->getActiveSheet();
    
            // Iterate through each row in column A
            foreach ($sheet->getColumnIterator('B', 'B') as $column) {
                foreach ($column->getCellIterator() as $cell) {
                    if (!is_null($cell)) {
                        // Add the cell value to the keywords array
                        $keywords[] = [
                            'colA' => $sheet->getCell('A' . $cell->getRow())->getValue(),
                            'colB' => $cell->getValue()
                        ];
                    }
                    
                }
            }
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            die('Error loading file: ' . $e->getMessage());
        }
    
        return $keywords;
    }

    function importExcel(Request $request) {
        $keywords = $this->readKeywordsFromExcel($request->file('file')->getPathname());
        $retval = [];
        if (!empty($keywords)) {
            foreach ($keywords as $key => $keyword) {
                if (empty($keyword['colA'])) {
                    continue;
                }
                $retval[$key] = [
                    'keyword' => $keyword['colA'],
                    'desire' => $keyword['colB'],
                    'retval' => 'true'
                ];
                $trademarkData = $this->detectKeyword($keyword['colA']);
                if (!empty($trademarkData)) {
                    $retval[$key]['retval'] = false;
                    $retval[$key]['trademark_keyword'] = $trademarkData;
                }
            }
        }
        
        return response()->json([
            'total' => count($retval),
            'data' => $retval
        ]);
    }

    /**
     * Find and remove duplicate words in the keyword
     * 
     * @param string $keyword
     * @return string
     */
    private function removeDuplicateWords($keyword)
    {
        $words = explode(' ', $keyword);
        $uniqueWords = array_unique($words);
        $result = implode(' ', $uniqueWords);
        return $result;
    }

    public function removeAllRecords()
    {
        try {
            $this->elasticService->deleteAllItems($this->elasticSearchConfig['index'], $this->type);
            return 'All records of type ' . $this->type . ' have been deleted.';
        } catch (\Exception $e) {
            return 'Error deleting records: ' . $e->getMessage();
        }
    }
}
