<?php
namespace Modules\ZSearch\Models;

use App\Modules\ZSearch\Models\ProductGallery;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Builder;
use Modules\ZSearch\Models\Traits\BaseTrait;
use Modules\ZSearch\Models\Traits\CustomAppend;
use Illuminate\Database\Query\JoinClause;


class Product extends Model
{
    use SoftDeletes;
    use BaseTrait;
    use CustomAppend;


    protected $table = 'product';
    protected $guarded = ['id'];

    protected $fillable = [
        'sku',
        'barcode',
        'name',
        'slug',
        'image_url',
        'price',
        'high_price',
        'add_shipping_fee',
        'weight',
        'status',
        'description',
        'content',
        'note',
        'inventory',
        'brand_id',
        'status_out_stock',
        'pod_parent_id',
        "approve_advertising",
        'created_at',
        'updated_at',
        'gtin',
        'actor_id'
    ];
    protected $variantDefault = null;

    protected $appends = [
        'url', 'display_price', 'display_high_price'
//        'url', 'editUrl', 'variant_default', 'display_price', 'display_high_price',
//        'sale_percent', 'display_drop_price', 'brand', 'attributes'
    ];

    public function getUrlAttribute () {
        $retval = getPrefixUrl();
        return $retval . "/" . (!empty($this->slug) ? $this->slug : "san-pham") . "-p" . $this->id;
    }


    public function tag() {
        return $this->belongsToMany(Tag::class, 'tag_refer' ,'refer_id', 'tag_id')
            ->where('refer_type', TagRefer::REFER_PRODUCT);
    }
    public function tags() {
        return $this->belongsToMany(Tag::class, 'tag_refer' ,'refer_id', 'tag_id')
            ->where('refer_type', TagRefer::REFER_PRODUCT);
    }

    public function galleries()
    {
        return $this->hasMany(ProductGallery::class, 'product_id', 'id')->where('type', 'PRODUCT');
    }

    public function categories()
    {
        return $this->belongsToMany(Category::class, 'product_n_category', 'product_id', 'category_id');
    }


    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    public function getDisplayPriceAttribute() {
        if ($this->price > 0) {
            $this->price = doubleval($this->price);
            return formatPrice($this->price);
        } else {
            return null;
        }
    }

    public function getDisplayHighPriceAttribute() {
        if ($this->high_price > 0) {
            $this->high_price = doubleval($this->high_price);
            return formatPrice($this->high_price);
        }
        return null;
    }

    public static function displayPrice($price) {
        if ($price > 0) {
            $price = doubleval($price);
            return formatPrice($price);
        } else {
            return null;
        }
    }

    public static function displayHighPrice($price)
    {
        if ($price > 0) {
            $price = doubleval($price);
            return formatPrice($price);
        }
        return null;
    }

    public static function productUrl($id, $slug) {
        $retval = getPrefixUrl();
        return $retval . "/" . (!empty($slug) ? $slug : "san-pham") . "-p" . $id;
    }

    public static function getProductQuery($filter = [])
    {
        $type = array_key_exists('type', $filter) ? $filter['type'] : '';
        $query = Product::query();
        if (isset($filter['using_query_builder'])) {
            $query = DB::table('product');
        }
        $isAll = array_key_exists('is_all', $filter) ? $filter['is_all'] : 0;
        $fromId = array_key_exists('from_id', $filter) ? $filter['from_id'] : -1;

        if (!$isAll
            && $fromId < 1
            && !array_key_exists('ids', $filter)
        ) {
            $query->from(DB::raw("sb_product USE INDEX (updatedAt)"));
        }
        if ($type == 'delete') {
            $query = Product::onlyTrashed();
        } else if ($type == 'init') {
            $query->where('price', '>', 0);
            $query->where('status', '=', 'ACTIVE');
            $query->where('is_hidden', '!=', 1);
            $query->whereNull('deleted_at');
            if (array_key_exists('created_from', $filter)) {
                $query->where('created_at', '<' , $filter['created_from']);
            }
        } else if ($type == 'hidden') {
            $query->where(function($where) {
                $where->where('is_hidden', '=', 1)
                    ->orWhere('status', '=', 'PENDING');
            });
        } else if ($type == 'seller-hidden') {
            $query->where(function($where) {
                $where->whereNotNull('deleted_at')
                    ->orWhere('status', '!=', 'ACTIVE');
            });
        }
        if (!array_key_exists('is_all', $filter) || !$filter['is_all']) {
            if (array_key_exists('updated_from', $filter)) {
                $query->where('updated_at', '>=' , $filter['updated_from']);
            } else {
                $query->where('updated_at', '>' , date('Y-m-d H:i:s', time() - 2000));
            }
        }
        if (array_key_exists('from_id', $filter) && $filter['from_id'] > 0) {
            $query->from(DB::raw("sb_product USE INDEX (PRIMARY)"));
            $query->where('id', '>=' , $filter['from_id']);
        }
        if (array_key_exists('to_id', $filter) && $filter['to_id'] > 0) {
            $query->where('id', '<=' , $filter['to_id']);
        }
        if (array_key_exists('ids', $filter)) {
            $query->from(DB::raw("sb_product USE INDEX (PRIMARY)"));
            $query->whereIn('id', $filter['ids']);
        }

        if (array_key_exists('sold', $filter)) {
            $query->where('sold', $filter['sold']);
        }
        return $query;
    }

    public static function getProductNCategoryQuery($filter = [])
    {
        $type = array_key_exists('type', $filter) ? $filter['type'] : '';
        $query = Product::query();
        if (isset($filter['using_query_builder'])) {
            $query = DB::table('product');
        }
        $isAll = array_key_exists('is_all', $filter) ? $filter['is_all'] : 0;
        $fromId = array_key_exists('from_id', $filter) ? $filter['from_id'] : -1;

        if (!$isAll
            && $fromId < 1
            && !array_key_exists('ids', $filter)
        ) {
            $query->from(DB::raw("sb_product USE INDEX (updatedAt)"));
        }
        if ($type == 'delete') {
            $query = Product::onlyTrashed();
        } else if ($type == 'init') {
            $query->where('product.price', '>', 0);
            $query->where('product.status', '=', 'ACTIVE');
            $query->where('product.is_hidden', '!=', 1);
            $query->whereNull('product.deleted_at');
        } else if ($type == 'hidden') {
            $query->where(function($where) {
                $where->where('product.is_hidden', '=', 1)
                    ->orWhere('product.status', '=', 'PENDING');
            });
        } else if ($type == 'seller-hidden') {
            $query->where(function($where) {
                $where->whereNotNull('product.deleted_at')
                    ->orWhere('product.status', '!=', 'ACTIVE');
            });
        }
        if (!array_key_exists('is_all', $filter) || !$filter['is_all']) {
            if (array_key_exists('updated_from', $filter)) {
                $query->where('product.updated_at', '>=' , $filter['updated_from']);
            } else {
                $query->where('product.updated_at', '>' , date('Y-m-d H:i:s', time() - 2000));
            }
        }
        if (array_key_exists('from_id', $filter) && $filter['from_id'] > 0) {
            $query->from(DB::raw("sb_product USE INDEX (PRIMARY)"));
            $query->where('product.id', '>=' , $filter['from_id']);
        }
        if (array_key_exists('to_id', $filter) && $filter['to_id'] > 0) {
            $query->where('product.id', '<=' , $filter['to_id']);
        }
        if (array_key_exists('ids', $filter)) {
            $query->from(DB::raw("sb_product USE INDEX (PRIMARY)"));
            $query->whereIn('product.id', $filter['ids']);
        }
        $query->join('product_n_category_top as pnct', function (JoinClause $join) {
            return $join->on('product.id', '=', 'pnct.product_id')
                ->where('pnct.is_parent', 0);
        });
        return $query;
    }

    public static function getProductIds($filter = []) {
        $filter['using_query_builder'] = true;
        $query = Product::getProductQuery($filter)->whereNull('deleted_at');
        return $query->pluck('id')->toArray();
    }

    public static function getTemplateId($productId) {
        $result = null;
        if (class_exists('\App\Utils\SkuTemplateHelper')) {
            $item = DB::table('product_n_template')
                ->where('product_id', $productId)
                ->first(['template_id']);
            $result = $item && isset($item->template_id) ? $item->template_id : null;
        }
        return $result;

    }

    private function getDefaultVariant($productId, $templateId) {
        $retVal = [];
        if ($templateId) {
            $retVal = $this->getDefaultTemplateSku($productId, $templateId);
        } else {
            $retVal = DB::table('product_sku')->where('product_id', '=', $productId)
                ->where('is_default', 1)
                ->first(['id', 'sku', 'price', 'high_price', 'image_url', 'status']);
        }

        return $retVal;
    }


    private function getDefaultTemplateSku($productId, $templateId) {
        $retVal = [];
        try {
            $defaultSku = null;
            $hasDefaultOverwrite = ProductTemplateSku::hasDefaultOverwrite($productId);

            if ($hasDefaultOverwrite) {
                $defaultSku = ProductTemplateSku::getDefaultWithOverwrite($productId, $templateId);
                $defaultSku = $defaultSku->first();
            }
            if (!$defaultSku){
                $defaultSku = ProductTemplateSku::getDefault($templateId);
                $defaultSku = $defaultSku->first();
            }
            $sku = null;
            if (!$defaultSku) {
                $defaultSku = ProductTemplateSku::where('template_id', $templateId)->orderBy('price', 'asc')->first();
            }

            if(!$hasDefaultOverwrite && ProductTemplateSku::isIgnoreDefault($productId, $templateId)) {
                $key = 'default_sku_' . $productId;
                if (!array_key_exists($key, self::$cache)) {
                    $ignoreColors = ProductTemplateSku::getIgnoreColorsByProduct($productId);
                    $ignoreSkuIds = ProductTemplateSkuValue::whereIn('variant_option_id', $ignoreColors)->pluck('sku_id')->toArray();
                    if ($defaultSku) {
                        $optionIds = ProductTemplateSkuValue::where('sku_id', $defaultSku->id)->where('variant_id', 5)->get()->pluck('variant_option_id');
                        $sameTypesSkuIds = ProductTemplateSkuValue::whereIn('variant_option_id', $optionIds)->pluck('sku_id')->toArray();
                        $sku = ProductTemplateSku::where('template_id', $templateId)
                            ->whereIn('id', array_diff($sameTypesSkuIds, $ignoreSkuIds))
                            ->orderBy('price', 'asc')
                            ->first();
                    }
                    if (!$sku) {
                        $sku = ProductTemplateSku::where('template_id', $templateId)
                            ->whereNotIn('id', $ignoreSkuIds)
                            ->orderBy('price', 'asc')
                            ->first();
                    }
                    if ($sku) {
                        self::$cache[$key] = $sku;
                    }
                } else {
                    $sku = self::$cache[$key];
                }
            }

            if (!$sku && $defaultSku) {
                $sku = $defaultSku->toArray();
            }

            if ($sku && isset($sku['id'])) {
                $sku['product_id'] = $productId;
                $sku['image_url'] = Utils::decoreTemplateImageUrl($productId, $sku);
                $retVal = $sku;
            }
        } catch (\Exception $e) {
            \Log::error('getDefautVariant ', [$e->getMessage() . " Line: " . $e->getLine(), $productId]);
        }

        return $retVal;
    }




}
