<?php

namespace Modules\ZSearch\Services;


use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CampaignService extends ElasticSearchIndex
{
    const ZSEARCH_CAMPAIGN_MAPPING = 'zsearch_campaign_mapping';
    const ZSEARCH_CAMPAIGN_SORT_KEY = 'campaign_sort';
    public function detachCampaign($keyword, $filter = []) {
        if ($filter && is_array($filter)
            && (array_key_exists('category', $filter) || array_key_exists('category_detach', $filter))) {
            return null;
        }
        $result = null;
        $campaigns = $this->getCampaigns();
        if ($campaigns) {
            foreach ($campaigns as $campaign) {
                if (isset($campaign['keywords'])
                    && $campaign['keywords']
                    && str_contains($keyword, $campaign['keywords'])) {
                    $result = $campaign;
                    Log::info('ZSEARCH detachCampaign', [
                        'campaign' => $campaign,
                        'keyword' => $keyword,
                    ]);
                    break;
                }
            }
        }
        return $result;
    }

    public function distribute($campaign, $items, $meta) {
        if (!isset($campaign['categories']) || count($campaign['categories']) == 0) {
            return $items;
        }
        try {
            $specialLength = 8;
            $result = $this->specialItems($items, $specialLength);
            $groupData = $this->groupByCategory(array_slice($items, $specialLength), $campaign, $specialLength);
            return array_merge($result, $this->distributeAllPage($groupData, $campaign, $meta, $specialLength));
        } catch (\Exception $e) {
            Log::info('ZSEARCH distribute', [
                'message' => $e->getMessage() . ' file ' . $e->getFile() . ' line ' . $e->getLine(),
            ]);
            return $items;
        }

    }

    public function distributeAllPage($groupData, $campaign, $meta, $specialLength) {
        $result = [];
        $logMeta = [];
        for($page = 0; $page <= $meta['page_id']; $page++) {
            $pageSize = $page > 0 ? $meta['page_size'] : $meta['page_size'] - $specialLength;
            $pageData = [];
            $capacityPageSize =  $pageSize;
            foreach ($campaign['categories'] as $category) {
                if (!isset($groupData[$category['id']]) || count($groupData[$category['id']]) == 0) {
                    continue;
                }
                $size = ceil(($pageSize * $category['percent'])/100);
                $categoryData = array_splice($groupData[$category['id']], 0, $size);
                if ($categoryData && count($categoryData) > 0) {
                    $pageData = array_merge($pageData, $categoryData);
                    $capacityPageSize -= count($categoryData);
                    $logMeta[$page][$category['id']] = count($categoryData);
                }
            }

            if ($capacityPageSize > 0 && isset($groupData['other']) && count($groupData['other']) > 0) {
                $logMeta[$page]['other'] = $capacityPageSize;
                $pageData = array_merge($pageData, array_splice($groupData['other'], 0, $capacityPageSize));
            }
            $result = array_merge($result, $pageData);
        }
        Log::info('ZSEARCH distributeAllPage', [
            'logMeta' => $logMeta,
        ]);
        return $result;
    }

    public function sortData($items, $col) {
        usort($items, function ($itemA, $itemB) use ($col) {
            $aValue = isset($itemA[$col]) ? $itemA[$col] : '';
            $bValue = isset($itemB[$col]) ? $itemB[$col] : '';
            $result = 0;
            if ($aValue > $bValue) {
                $result = 1;
            } else if ($aValue < $bValue) {
                $result = -1;
            }
            return $result;
        });
        return $items;
    }



    protected function specialItems($items, $limit) {
        $result = [];
        $items = array_slice($items, 0, $limit);
        foreach ($items as $key => $item) {
            $item[self::ZSEARCH_CAMPAIGN_SORT_KEY] = $key;
            $result[] = $item;
        }
        return $result;
    }

    protected function groupByCategory($items, $campaign, $specialLength) {
        $result = [];
        $categoryIds = [];
        foreach ($campaign['categories'] as $category) {
            $categoryIds[] = $category['id'];
        }
        if (count($items) > 0) {
            foreach ($items as $key => $item) {
                $categoryId = $this->getCategoryId($item);
                if (!in_array($categoryId, $categoryIds)) {
                    $categoryId = 'other';
                }
                if (!array_key_exists($categoryId, $result)) {
                    $result[$categoryId] = [];
                }
                $item[self::ZSEARCH_CAMPAIGN_SORT_KEY] = $key + $specialLength;
                $result[$categoryId][] = $item;
            }
        }
        return $result;
    }

    protected function getCategoryId($item) {
        $category = $this->getLastCategory($item);
        return $category && isset($category['id']) ? $category['id'] : 0;
    }

    protected function getLastCategory($item) {
        $result = null;
        if (isset($item['categories']) && count($item['categories']) > 0) {
            $result = $item['categories'][count($item['categories']) - 1];
        }
        return $result;
    }

    protected function getCampaigns() {
        $result = [];
        $option = DB::table('option')->where('key', self::ZSEARCH_CAMPAIGN_MAPPING)->first();
        if (isset($option->value) && $option->value) {
            $result = json_decode($option->value, true);
        }
        return $result;

    }

    public function getData($keyword, $campaign, $size = 400) {
        $result = [];
        if (isset($campaign['categories'])
            && isset($campaign['extend_keywords'])
            && $campaign['categories']
            && $campaign['extend_keywords']
        ) {
            //@todo impl size categories >  extend_keywords
            $size = $size/2;
        }
        if (isset($campaign['categories']) && $campaign['categories']) {
            $result['categories'] = $this->searchCategories($keyword, $campaign, $size);
        }

        if (isset($campaign['extend_keywords']) && $campaign['extend_keywords']) {
            $result['extend_keyword'] = $this->searchExtendKeyword($campaign, $size);
        }

        return $result;
    }

    protected function searchExtendKeyword($campaign, $size) {
        $size = ceil($size/count($campaign['extend_keywords']));
        $result = [];
        foreach ($campaign['extend_keywords'] as $keyword) {
            $filter = [
                'from' => 0,
                'size' => $size,
            ];
            $data = $this->search($keyword, $filter);
            $result = $this->merge($result, $data);
        }
        return $result;
    }

    protected function searchCategories($keyword, $campaign, $size = 400)
    {
        $result = [];
        foreach ($campaign['categories'] as $category) {
            $size = ceil(($size * $category['percent'])/100);
            $filter = [
                'category_id' => $category['id'],
                'from' => 0,
                'size' => $size,
            ];
            $data = $this->search($keyword, $filter);
            $result = $this->merge($result, $data);

        }
        return $result;
    }

    protected function merge($result, $items) {

        if (isset($items['hits']['hits'])) {
            $result = $this->merge($result, $items['hits']['hits']);
        }

        return $result;
    }

    public function search($keyword, $filter) {
        $params = $this->buildElasticsearchParams($keyword, $filter);
        return $this->elasticService->searchDocument($params, ElasticSearchIndex::TYPE_PRODUCTS, $this->elasticServiceConfig['index']);
    }

    public function buildElasticsearchParams($keyword, $filter = []) {
        $elasticParams = [
            'from' => $filter['from'],
            'size' => $filter['size'],
        ];
        $elasticParams['query']['bool']['should'] =  [
            [
                'match' => ['name' => $keyword]
            ],
        ];
        $elasticParams['query']['bool']['must'] = [
            'must' => [
                'match' => [
                    'status' => 'ACTIVE',
                ]
            ],
        ];
        if (array_key_exists('category_id', $filter)) {
            $elasticParams['query']['bool']['must'][] = [
                'match' =>  [
                    "categories.id" => $filter['category_id'],
                ],
            ];
        }
        return $elasticParams;
    }

}
