<?php

namespace Modules\ZSearch\Services;

use App\Utils\SlugManager;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Route;


class CategoryHelper
{

    static $cacheUrl = [];
    const CATEGORY_PARAMS = [
        'slug', 'slug2', 'slug3'
    ];

    const TYPE_PRODUCT = 'PRODUCT';
    const STATUS_ACTIVE = 'ACTIVE';
    const VALIDATE_CATEGORY = 'category fail';

    const TYPE_INCLUDE = [
        [SlugManager::TYPE_CATEGORY],
        [SlugManager::TYPE_CATEGORY, SlugManager::TYPE_COLOR, SlugManager::TYPE_TYPE],
        [SlugManager::TYPE_CATEGORY, SlugManager::TYPE_COLOR, SlugManager::TYPE_TYPE],
        [SlugManager::TYPE_CATEGORY, SlugManager::TYPE_COLOR, SlugManager::TYPE_TYPE],
        [SlugManager::TYPE_COLOR, SlugManager::TYPE_TYPE],
        [SlugManager::TYPE_COLOR],
    ];
    const TYPES_ENABLE = [SlugManager::TYPE_TYPE, SlugManager::TYPE_COLOR];

    public function __construct()
    {
    }

    public static function parseParamsZSearch()
    {
        $params = Route::current()->parameters();
        if (isset($params['locale'])) {
            unset($params['locale']);
        }
        $key = decorCacheKey('parseParamsv6::' . serialize($params));
        $result = cacheGet($key);
        if (!$result) {
            $slugParams = self::buildParams($params);
            $categoriesData = self::parseCategories($slugParams['category']);
            $filterParams = [];
            $redirect = '';
            $validateData = self::validateParams($params, $categoriesData['categories'], $slugParams['attrs']);
            if ($validateData['isOk']) {
                $slugManager = SlugManager::getInstance();
                $filterParams = $slugManager->buildRequestParams(array_merge([$categoriesData['category']], $slugParams['attrs']));
            } else if ($validateData['message'] == self::VALIDATE_CATEGORY
                && $categoriesData && isset($categoriesData['category']->slug)) {
                $redirect = CategoryHelper::urlZSearch($categoriesData['category']->slug);
            }
            $result = [
                'filterParams' => $filterParams,
                'categoriesData' => $categoriesData,
                'attrs' => $slugParams['attrs'],
                'path_info' => self::parseUrl($categoriesData, $slugParams['attrs']),
                'redirect' => $redirect,
            ];
            Cache::put($key, $result, 10);
        }

        $result['query'] = request()->all();
        return $result;
    }

    public static function parseData($params)
    {

    }

    protected static function parseUrl($categoriesData, $attrs)
    {
        $result = [];

        if ($categoriesData['slugs']) {
            $result['categories'] = $categoriesData['slugs'];
        }

        if ($attrs) {
            foreach ($attrs as $attr) {
                $result['attrs'][$attr->type] = $attr->slug;
            }
        }
        return $result;
    }


    public static function buildParams($params, $filter = [])
    {
        $slugManager = SlugManager::getInstance();
        $result = [
            'category' => [],
            'attrs' => [],
        ];
        $params = array_values($params);
        foreach ($params as $i => $paramValue) {
            $filter['type_include'] = self::TYPE_INCLUDE[$i];
            $item = $slugManager->getAttr($paramValue, $filter);

            if ($item) {
                $type = $item->type == 'category' ? $item->type : 'attrs';
                $item->index = $i;
                if ($type == 'category') {
                    $cat =  self::getCategory($item->slug);
                    if ($cat && $item->id != $cat->id) {
                        $item->id = $cat->id;
                        $item->target_id = $cat->id;
                    }
                }
                $result[$type][] = $item;
            }
        }
        return $result;
    }

    public static function parseCategories($items)
    {
        $category = null;
        $categories = [];
        $slugs = [];
        if ($items) {
            foreach ($items as $item) {
                $category = self::getCategory($item->slug);
                if ($category) {
                    $item->_lft = $category->_lft;
                    $item->_rgt = $category->_rgt;
                    $item->parent_id = $category->parent_id;
                    $categories[] = $item;
                    $slugs[] = $item->slug;
                    $category = $item;
                }
            }
        }
        return [
            'categories' => $categories,
            'category' => $category,
            'slugs' => $slugs,
        ];;

    }

    protected static function getCategory($slug, $columns = ['id', 'slug', '_lft', '_rgt', 'parent_id'])
    {
        return \DB::table('category')
            ->where('slug', $slug)
            ->where('type', self::TYPE_PRODUCT)
            ->first($columns);
    }

    public static function validateParams($inputs, $categories, $attrs)
    {
        $isOk = true;
        $countParams = $attrs ? count($categories) + count($attrs) : count($categories);
        $message = '';
        if ($categories && !self::validateCategory($categories)) {
            $isOk = false;
            $message = self::VALIDATE_CATEGORY;
        } else if (count($inputs) != $countParams) {
            $isOk = false;
            $message = 'countParams';
        } else if (!self::validateAttrs($attrs)) {
            $message = 'validateAttrs';
            $isOk = false;
        }
        return [
            'isOk' => $isOk,
            'message' => $message,
        ];
    }

    public static function validateAttrs($items)
    {
        $isOk = true;
        if ($items && count($items) > 1) {
            //duplicate
            if ($items[0]->type == $items[1]->type) {
                $isOk = false;
            } else if ($items[0]->type == SlugManager::TYPE_COLOR) {
                $isOk = false;
            }
        }
        return $isOk;

    }

    public static function validateCategory($categories)
    {
        if (!$categories) {
            return false;
        }
        $category = $categories[count($categories) - 1];
        $countParent = self::buildQueryTree($category)->count();
        if ($countParent + 1 != count($categories)) {
            return false;
        }
        $isOk = true;
        $parent = null;
        foreach ($categories as $category) {
            if (!$category || $category->type != SlugManager::TYPE_CATEGORY) {
                $isOk = false;
                break;
            }
            if ($parent
                && ($parent->_lft > $category->_lft
                    || $parent->_rgt < $category->_rgt
                    || $category->index - $parent->index > 1)) {
                $isOk = false;
                break;
            } else if (!$parent && $category->parent_id != 0) {
                $isOk = false;
                break;
            }
            $parent = $category;
        }
        return $isOk;
    }

    public static function urlFilter($attrs, $attr)
    {
        $pathInfo = $attrs['path_info'];
        if ($attr['type'] == SlugManager::TYPE_CATEGORY) {
            if (count($pathInfo['categories']) == 4) {
                $pathInfo['categories'][3] = $attr['slug'];
            } else {
                $pathInfo['categories'][] = $attr['slug'];
            }
        } else if (in_array($attr['type'], self::TYPES_ENABLE)) {
            $pathInfo['attrs'][$attr['type']] = $attr['slug'];
        } else {
            $attrs['query'][$attr['type']] = isset($attr['id']) ? $attr['id'] : $attr['slug'];
        }
        $paths = self::buildPath($pathInfo);
        $params = self::buildRouteParams($paths);
        //@todo param query
        $result = route('category-list', $params);
        if ($attrs['query']) {
            $result .= '?' . http_build_query($attrs['query']);
        }
        return $result;

    }

    public static function isSelectedFilter($attrs, $attr)
    {
        $pathInfo = $attrs['path_info'];
        $result = false;
        if (isset($attr['type'])
            && $attr['type'] == SlugManager::TYPE_CATEGORY
            && in_array($attr['slug'], $pathInfo['categories'])) {
            $result = true;
        } else if (isset($pathInfo['attrs'][$attr['type']])
            && $pathInfo['attrs'][$attr['type']] == $attr['slug']
        ) {
            $result = true;
        } else if (isset($attrs['query'][$attr['type']])
            && ($attrs['query'][$attr['type']] == $attr['slug']
                || (isset($attr['id']) && $attrs['query'][$attr['type']] == $attr['id'])
            )
        ) {
            $result = true;
        }
        return $result;
    }

    protected static function buildRouteParams($paths)
    {
        $paramsName = ['slug', 'slug2', 'slug3', 'slug4', 'slug5', 'slug6'];
        $result = [];
        foreach ($paramsName as $key => $param) {
            if (!isset($paths[$key])) {
                break;
            }
            $result[$param] = $paths[$key];
        }
        return $result;
    }

    protected static function buildPath($pathInfo)
    {
        $result = $pathInfo['categories'];
        foreach (self::TYPES_ENABLE as $type) {
            if (isset($pathInfo['attrs'][$type])) {
                $result[] = $pathInfo['attrs'][$type];
            }
        }
        return $result;
    }

    public static function urlZSearch($slug, $attrs = [])
    {
        $result = self::urlBySlugZSearch($slug);
        if ($attrs) {
            foreach (self::TYPES_ENABLE as $type) {
                if (isset($attrs[$type])) {
                    $result .= '/' . $attrs[$type];
                }
            }
        }
        return $result;
    }

    public static function urlBySlugZSearch($slug)
    {
        $result = isset(self::$cacheUrl[$slug]) ? self::$cacheUrl[$slug] : null;
        if (!$result) {
            $key = decorCacheKey('category::url' . $slug);
            $result = cacheGet($key);
        }
        if (!$result) {
            $category = self::getCategory($slug);
            $paths = [];
            if ($category && $category->parent_id != 0) {
                $paths = self::buildQueryTree($category)
                    ->orderBy('_lft', 'asc')
                    ->pluck('slug')
                    ->toArray();
                $paths[] = $category->slug;
            } else if (isset($category->slug)) {
                $paths = [$category->slug];
            }
            $params = self::buildRouteParams($paths);
            if ($params) {
                $result = route('category-list', $params);
                //$result = clroute('/c/' . implode('/', $paths));
                self::$cacheUrl[$slug] = $result;
                Cache::put($slug, $result, 5);
            }
        }

        if (!$result) {
            $result = clroute('/' . $slug);
        }
        return $result;
    }

    protected static function buildQueryTree($category) {
        return DB::table('category')
            ->where('_lft', '<', $category->_lft)
            ->where('_rgt', '>', $category->_rgt)
            ->where('type', self::TYPE_PRODUCT);
    }

    public static function decorAttrBreadcrumbZSearch($breadcrumb, $paramsData)
    {
        $locale = env('APP_LOCALE');
        if (empty($locale)) {
            $locale = 'us';
        } 
        $lastCategory = null;
        $result = [];
        $default = [
            [
                'name' => __('Home'),
                'url' => route('home'),
            ]
            ,
            [
                'name' => __('All Categories'),
                'url' => route('all-category'),
            ],

        ];
        foreach ($default as $item) {
            $result[] = (object)$item;
        }
        if ($breadcrumb) {
            foreach ($breadcrumb as $item) {
                $item->url = self::urlBySlugZSearch($item->slug);
                $lastCategory = $item;
                $result[] = $item;
            }
        }

        if ($paramsData['attrs'] && $lastCategory) {
            $attrs = [];
            $names = [];
            foreach ($paramsData['attrs'] as $key => $item) {
                $attrs[$item->type] = $item->slug;
                if ($item->type === 'type' && in_array($locale, ['en', 'us', 'uk', 'ca', 'au'])) {
                    $slugName = ucwords(str_replace('-', ' ', $item->slug . "'s"));
                } else {
                    $slugName = ucwords(str_replace('-', ' ', $item->slug));
                }
                

                $names[] = $slugName;
                $breadcrumbItem = new \stdClass();
                $breadcrumbItem->url = self::urlZSearch($lastCategory->slug, $attrs);
                $breadcrumbItem->name = implode(' ', $names) . ' ' . $lastCategory->name;
                $result[] = $breadcrumbItem;

            }
        }
        return $result;
    }


}
