<?php
namespace Modules\ZSearch\Services;

use Carbon\Carbon;
use Elasticsearch\ClientBuilder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\View;
use Modules\ZSearch\Models\Product;

class ElasticSearchIndex {

    /** https://github.com/elastic/elasticsearch-php/tree/5.x */
    private $client;

    public $elasticServiceConfig;
    protected $amazonDataDecor;
    public $elasticService;
    public $searchDataDecor;
    public $productsStat = [];
    public $productsCustom;
    protected $users = [];
    protected $tags = [];
    protected $productVariantOption;
    protected $productVariant;
    protected $productNDesign;
    protected $isAdmin;
    const TYPE_PRODUCTS = 'products';
    const TYPE_PRODUCTS_TOP = 'products_top';
    const TYPE_PRODUCTS_SELLER = 'products_seller';
    const TYPE_PRODUCTS_TRADEMARK = 'products_trademark';
    const TYPE_PRODUCTS_N_CATEGORY = 'product_n_category_top';
    static $cacheDefaultStyleTemplate = [];
    static $cacheCategory = [];
    static $cacheOptionTemplate = [];
    static $cache = [];
    static $productsTop = [];
    static $ignoreProducts = [];
    static $hasTableProductStat = false;


    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticServiceConfig = $elasticSearchConfig;
        $this->elasticService = new ElasticSearchService($elasticSearchConfig);
        $this->searchDataDecor = new SearchDataDecor();
    }

    public function multiIndexProduct($filter, $elasticsearchConfig) {
        $start = time();
        $productIds = $this->getProductIds(array_merge($filter, ['type' => 'init']));
        \Log::info('ZSEARCH - multiIndexProduct count', [count($productIds)]);
        $count = 0;
        $error = [];
        if ($productIds) {
            foreach (array_chunk($productIds, 200) as $partProductIds) {
                if (self::tableExists('product_stat')) {
                    $this->buildProductStat($partProductIds);
                }
                $count += count($partProductIds);
                $products = $this->getProducts(array_merge($filter, [
                    'type' => 'init',
                    'ids' => $partProductIds
                ]))->toArray();
                $data = $this->indexMainProducts($products, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
                if ($data['error']) {
                    $error[] = $data['error'];
                }
            }
            $this->indexSeller($productIds, $filter);
        }
        $this->indexSellerHidden($filter);
        $hiddenData = $this->handleHidden($filter, $elasticsearchConfig);
        $ignoreData = $this->handleIgnoreData($productIds, $elasticsearchConfig);
        return [
            'count' => $count,
            'error' => $error,
            'hidden' => $hiddenData,
            'runtime' => time() - $start,

        ];
    }

    protected function indexSeller($productIds, $filter) {
        $elsSeller = new ElasticSearchIndexSeller();
        $elsSeller->multiIndexProductSeller($productIds, $filter);
    }

    protected function indexSellerHidden($filter) {
        $elsSeller = new ElasticSearchIndexSeller();
        $hiddenProductsIds = $this->getProductIds(array_merge($filter, ['type' => 'seller-hidden']));
        $elsSeller->multiIndexProductSeller($hiddenProductsIds);
    }

    public function deleteAllProductTop() {
        return $this->elasticService->deleteAllItems($this->elasticServiceConfig['index'], self::TYPE_PRODUCTS_TOP);
    }

    public function multiIndexTopProduct($filter, $elasticsearchConfig) {
        $products = $this->getProducts(array_merge($filter, ['type' => 'init']))->toArray();
        $data = $this->indexMainProducts($products, self::TYPE_PRODUCTS_TOP, $elasticsearchConfig['index']);
        $deleteProductTop = $this->deleteProduct($filter, self::TYPE_PRODUCTS_TOP, $elasticsearchConfig['index']);
        $hiddenProductTop = $this->hiddenProduct($filter, self::TYPE_PRODUCTS_TOP, $elasticsearchConfig['index']);
        return [
            'count' => $products ? count($products) : 0,
            'error' => $data['error'],
            'deleteProductTop' => $deleteProductTop ? count($deleteProductTop) : 0,
            'hiddenProductTop' => $hiddenProductTop ? count($hiddenProductTop) : 0,
        ];
    }


    public function indexMainProducts($products, $type, $index) {
        $error = [];
        foreach (array_chunk($products, 200) as $chunkProducts) {
            try {
                $items = $this->decorProducts($chunkProducts);
                $this->elasticService->bulkAction($this->elasticService->bulkData($items, $index, $type));
            } catch (\Exception $e) {
                $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
            }
        }
        return [
            'error' => $error
        ];
    }


    protected function decorProducts($products) {
        $result = [];
        if ($products) {
            foreach ($products as $product) {
                $result[] = $this->decorProduct($product);
            }
        }
        return $result;
    }

    public function handleHidden($filter, $elasticsearchConfig) {
        $types = $this->getAllTypeEls();
        $deletes = $this->deleteProducts($filter, $types, $elasticsearchConfig['index']);
        $hiddens = $this->hiddenProducts($filter, $types, $elasticsearchConfig['index']);
        return [
            'deletes' => $deletes ? count($deletes) : 0,
            'hiddens' => $hiddens ? count($hiddens) : 0,
        ];
    }

    public function getAllTypeEls() {
        return [
            self::TYPE_PRODUCTS, self::TYPE_PRODUCTS_TOP,
            self::TYPE_PRODUCTS_TRADEMARK,
            self::TYPE_PRODUCTS_SELLER, self::TYPE_PRODUCTS_N_CATEGORY
        ];
    }

    protected function getProductIds($filter = []) {
        return Product::getProductIds($filter);
    }

    protected function getProducts($filter = [], $customColumns = []) {
        $query = Product::getProductQuery($filter);
        $columns = [
            'id', 'sku', 'name', 'slug', 'image_url', 'brand_id',
            'status', 'price', 'high_price', 'inventory', 'view_count', 'sold', 'created_at'
        ];
        $columns = $customColumns ? array_merge($columns, $customColumns) : $columns;
        return $query->get($columns);
    }

    public function indexProduct($product, $type, $index) {
        $product = $this->decorProduct($product);
        $this->elasticService->saveDocument($product, $type, $index);
    }

    public function deleteProducts($filter, $types, $index) {
        $items = $this->getDeleteItems($filter);
        return $this->deleteByItems($items, $types, $index);
    }

    public function deleteByItems($items, $types, $index) {
        $result = [];
        if ($items) {
            foreach ($types as $type) {
                $result = $this->deleteList($items, $index, $type);
            }

        }
        return $result;
    }

    public function deleteProduct($filter, $type, $index) {
        $result = [];
        $items = $this->getDeleteItems($filter);
        if ($items) {
            $result = $this->deleteList($items, $index, $type);
        }
        return $result;
    }

    protected function getDeleteItems($filter) {
        $filter['type'] = 'delete';
        return Product::getProductQuery($filter)
            ->get(['id', 'deleted_at'])
            ->toArray();
    }

    public function hiddenProducts($filter, $types, $index) {
        $items =  $this->getHiddenItems($filter);
        return $this->deleteByItems($items, $types, $index);

    }

    public function hiddenProduct($filter, $type, $index) {
        $items =  $this->getHiddenItems($filter);
        $result = [];
        if ($items) {
            $result = $this->deleteList($items, $index, $type);
        }
        return $result;
    }

    protected function getHiddenItems($filter) {
        $filter['type'] = 'hidden';
        if (array_key_exists('delete_hidden', $filter) && $filter['delete_hidden']) {
            $filter['is_all'] = true;
        }
        $query = Product::getProductQuery($filter);
        return $query->get(['id'])->toArray();
    }

    public function deleteList($items, $index, $type = 'products') {
        $result = [];
        foreach (array_chunk($items, 500) as $chunkItems) {
            $params = [];
            foreach ($chunkItems as $item) {
                $result[] = $item['id'];
                $params['body'][] = [
                    'delete' => [
                        '_index' => $index,
                        '_type' => $type,
                        '_id' => $item['id'],
                    ]
                ];
            }
            $this->elasticService->bulkAction($params);
        }
        return $result;
    }


    protected function decorProduct($product) {
        $productId = $product['id'];
        $templateId = $this->getTemplateId($productId);
        if (\Module::isActive('Seo')) {
            $product['url'] = rtrim($product['url'], '.html');
        }
        $percentSale = $product['high_price'] > 0 ? ($product['high_price'] - $product['price'])/$product['high_price'] : 0;
        $percentSale = $percentSale > 0 ? ceil($percentSale * 100) : 0;
        $product['percent_sale'] = $percentSale;
        $referTag = $this->tagsFromProduct($productId);
        $product['tag'] = [];
        $product['tags'] = $referTag;
        $categories = $this->getCategoryFromProduct($productId);
        $product['categories'] = $categories ?: [];
        $product['category_id'] = $this->getParentCategory($productId);
        $product['total_click'] = 0;
        $product['total_sale'] = 0;
        $product['cr'] = 0;
        $product['design_id'] = 0;
        if ($this->productsStat && array_key_exists($productId, $this->productsStat)) {
            $stat = $this->productsStat[$productId];
            $clicks = isset($stat['total_click']) ?  $stat['total_click'] : 0;
            $sales = isset($stat['total_sale']) ?  $stat['total_sale'] : 0;
            $cr = 0;
            if ($clicks > 0) {
                $cr = $sales/$clicks;
            }
            $product['total_click'] = $stat['total_click'];
            $product['total_sale'] = $stat['total_sale'];
            $product['cr'] = $cr;
        }
        if ($this->productsCustom && array_key_exists($productId, $this->productsCustom)) {
            $product['is_custom_design'] = 1;
        }
        $pnu = $this->getPNU($productId);
        if ($pnu && isset($pnu->user_id) && $this->users &&  array_key_exists($pnu->user_id, $this->users)) {
            $userId = $pnu->user_id;
            $product['user'] = [
                'id' => $userId, //user_id
                'name' => $this->users[$userId]->name,
                'slug' => $this->users[$userId]->slug,
            ];
        }
        $product['variant_options'] = $templateId ? [] : $this->buildVariantOption($productId, $templateId);
        $product['variant'] = $templateId ? [] : $this->buildVariant($productId, $templateId);
        $product['template_id'] = $templateId;
        $product['random_image'] = $this->getRandomImage($productId, $templateId);
        $product['updated_at'] = Carbon::now()->format('Y-m-d');
        $product['score'] = $this->getProductScore($productId);
        return $product;
    }

    private function getProductScore($productId)
    {
        $result = DB::table('score_product')
            ->where('id', $productId)
            ->pluck('score');
        return $result->isNotEmpty() ? $result[0] : 0;
    }

    protected function tagsFromProduct($productId) {
        return DB::table('tag_refer')
            ->where('refer_id', $productId)
            ->where('refer_type', 'PRODUCT')
            ->pluck('tag_id')
            ->toArray();
    }

    protected function tagsFromProducts($items) {
        $result = [];
        foreach ($items as $item) {
            $result[$item->product_id] = [];
        }
        $productIds = array_keys($result);
        return $this->tagsFromProductIds($productIds);
    }

    protected function tagsFromProductIds($productIds) {
        $result = [];
        foreach (array_chunk($productIds, 50) as $chunkPIds) {
            $items =  DB::table('tag_refer')
                ->whereIn('refer_id', $chunkPIds)
                ->where('refer_type', 'PRODUCT')
                ->get(['refer_id', 'tag_id']);
            if ($items && count($items) > 0) {
                foreach ($items as $item) {
                    $result[$item->refer_id][] = $item->tag_id;
                }
            }
        }
        return $result;
    }

    protected function getPNU($productId) {
        $userId = null;
        if (self::tableExists('product_n_user')) {
            $pnu = DB::table('product_n_user')
                ->where('product_id', $productId)
                ->first(['user_id']);
        } else {
            $pnu = DB::table('product_meta')
                ->where("key", "user_id")
                ->where("product_id", $productId)
                ->first(['value as user_id']);
        }
        return $pnu;
    }

    public static function tableExists($table) {
        if (!array_key_exists($table, self::$cache)) {
            self::$cache[$table] = Schema::hasTable($table);
        }
        return array_key_exists($table, self::$cache) ? self::$cache[$table] : false;
    }

    protected function buildVariantOption($productId, $templateId) {
        $key = null;
        if ($templateId) {
            $key = decorCacheKey('buildVariantOption' . $templateId);
            $variantOptions = $this->getCache($key);
            if ($variantOptions) {
                return $variantOptions;
            }
        }

        $variantOptions = [];
        $variantOptionIds = $this->getOptionFromProduct($productId, $templateId);
        foreach ($variantOptionIds as $optionId) {
            if (!isset($this->productVariantOption[$optionId]) || !isset($this->productVariantOption[$optionId]['variant_id'])) {
                continue;
            }
            $optionItem = $this->productVariantOption[$optionId];
            $variantId = $optionItem['variant_id'];
            if (!isset($variantOptions[$variantId])) {
                $variantOptions[$variantId] = [
                    'variant_id' => $variantId,
                    'variant_name' => $optionItem['variant_name'],
                    'options' => [],
                ];
            }
            $variantOptions[$variantId]['options'][] = [
                'id' => $optionId,
                'name' => $optionItem['name'],
                'slug' =>  $optionItem['slug'],
            ];

        }
        if ($templateId) {
            $this->pushCache($key, $variantOptions);
        }
        return $variantOptions;
    }

    protected function buildVariant($productId, $templateId) {
        $key = null;
        if ($templateId) {
            $key = decorCacheKey('buildVariant::' . $templateId);
            $variantOptions = $this->getCache($key);
            if ($variantOptions) {
                return $variantOptions;
            }
        }
        $variantOptions = [];
        $variantOptionIds = $this->getOptionFromProduct($productId, $templateId);
        foreach ($variantOptionIds as $optionId) {
            if (!isset($this->productVariantOption[$optionId]) || !isset($this->productVariantOption[$optionId]['variant_id'])) {
                continue;
            }
            $optionItem = $this->productVariantOption[$optionId];
            $variantId = $optionItem['variant_id'];
            if (!isset($variantOptions[$variantId])) {
                $variantOptions[$variantId] = [
                    'variant_id' => $variantId,
                    'variant_name' => $optionItem['variant_name'],
                    'options' => [],
                ];
            }
            $variantOptions[$variantId]['options'][] = $optionId;

        }
        if ($templateId) {
            $this->pushCache($key, $variantOptions);
        }
        return $variantOptions;
    }

    protected function pushCache($key, $variantOptions) {
        self::$cache[$key] = $variantOptions;
        Cache::put($key, $variantOptions, 600);
    }

    protected function getCache($key) {
        if (array_key_exists($key, self::$cache)) {
            return self::$cache[$key];
        }
        $variantOptions = Cache::get($key);
        if ($variantOptions) {
            self::$cache[$key] = $variantOptions;
            return $variantOptions;
        }
        return [];
    }

    protected function getOptionFromProduct($productId, $templateId) {
        $variantOptionIds = [];

        if ($templateId) {
            $variantOptionIds = $this->getOptionFromTemplate($templateId);
        } else {
            $variantOptionIds = DB::table('product_sku_value as psv')
                ->where('psv.product_id', $productId)
                ->groupBy('psv.variant_option_id')
                ->pluck('psv.variant_option_id', 'psv.variant_option_id')
                ->toArray();
        }
        return $variantOptionIds;
    }


    protected function getOptionFromTemplate($templateId) {
        if (!array_key_exists($templateId, self::$cacheOptionTemplate)) {
            $skuIds = DB::table('product_template_sku')
                ->where('template_id', $templateId)
                ->pluck('id')->toArray();
            $options = [];
            if ($skuIds) {
                $options = DB::table('product_template_sku_value')
                    ->whereIn('sku_id', $skuIds)
                    ->pluck('variant_option_id', 'variant_option_id')
                    ->toArray();
            }
            self::$cacheOptionTemplate[$templateId] = $options;
        }
        return self::$cacheOptionTemplate[$templateId];
    }



    protected function getCategoryFromProduct($productId) {
        $result = [];
        $categoryIds = DB::table('product_n_category')
            ->where('product_id', $productId)
            ->pluck('category_id')
            ->toArray();
        foreach ($categoryIds as $categoryId) {
            $cat = $this->getCategory($categoryId);
            if ($cat) {
                $result[] = $cat;
            }
        }
        return $result;
    }

    protected function getParentCategory($productId) {
        $item = DB::table('product_n_category')
            ->where('product_id', $productId)
            ->where('is_parent', 0)
            ->first(['category_id']);

        return isset($item->category_id) ? $item->category_id : 0;
    }

    protected function getCategory($id) {
        if (!array_key_exists($id, self::$cacheCategory)) {
            self::$cacheCategory[$id] = DB::table('category')
                ->where('id', $id)
                ->first(['category.id', 'category.name', 'category.slug', 'category.image_url']);
        }
        return self::$cacheCategory[$id];

    }

    public function getRandomImage($productId, $templateId) {
        $result = null;
        if ($templateId) {
            $img = $this->getRandImgFromSkuTemplate($templateId);
            $result = \App\Utils\SkuTemplateHelper::buildImageUrl($productId, $img);
        } else {
            $result = $this->getRandImgFromSku($productId);
        }
        return $result;
    }

    protected function getTemplateId($productId) {
        return Product::getTemplateId($productId);

    }

    protected function getRandImgFromSkuTemplate($templateId) {
        $defaultStyle = self::getDefaultTemplateStyle($templateId);
        $query = DB::table('product_template_sku as ps')
            ->where('ps.template_id', $templateId)
            ->orderBy(DB::raw('RAND()'));
        if ($defaultStyle) {
            $query->join('product_template_sku_value as psv', 'ps.id', '=', 'psv.sku_id')
                ->where('psv.variant_option_id', $defaultStyle);
        }
        $item = $query->first(['ps.image_url']);
        return $item && $item->image_url ? $item->image_url : null;
    }


    public static function getDefaultTemplateStyle($templateId) {
        if (!array_key_exists($templateId, self::$cacheDefaultStyleTemplate)) {
            $defaultSku = DB::table('product_template_sku')
                ->where('template_id', $templateId)
                ->where('is_default', 1)
                ->first(['id']);
            if ($defaultSku && isset($defaultSku->id)) {
                $item = DB::table('product_template_sku_value')
                    ->where('sku_id', $defaultSku->id)
                    ->where('variant_id', 7)
                    ->first(['variant_option_id']);
                self::$cacheDefaultStyleTemplate[$templateId] = $item && isset($item->variant_option_id) ? $item->variant_option_id : null;
            }
        }

        return array_key_exists($templateId, self::$cacheDefaultStyleTemplate) ? self::$cacheDefaultStyleTemplate[$templateId] : null;
    }


    protected function getRandImgFromSku($productId) {
        $defaultStyle = $this->getDefaultStyle($productId);
        $query = DB::table('product_sku as ps')
            ->where('ps.product_id', $productId)
            ->orderBy(DB::raw('RAND()'));
        if ($defaultStyle) {
            $query->join('product_sku_value as psv', 'ps.id', '=', 'psv.sku_id')
                ->where('psv.variant_option_id', $defaultStyle);
        }
        $item = $query->first(['ps.image_url']);
        return $item && $item->image_url ? $item->image_url : null;
    }

    protected function getDefaultStyle($productId) {
        $defaultSku = DB::table('product_sku')
            ->where('product_id', $productId)
            ->where('is_default', 1)
            ->first(['id']);
        $result = null;
        if ($defaultSku && isset($defaultSku->id)) {
            $item = DB::table('product_sku_value')
                ->where('sku_id', $defaultSku->id)
                ->where('variant_id', 7)
                ->first(['variant_option_id']);
            $result = $item && isset($item->variant_option_id) ? $item->variant_option_id : null;
        }
        return $result;
    }


    public function initMetaData() {
        $variantIds =  Config::get('z-search::default.filter_variant_id', [1,2,5]);
        if (!self::tableExists('product_stat')) {
            $this->productsStat = $this->getProductsStat();
        }
        $this->productsCustom = null;
        $this->productVariant = $this->buildProductVariant($variantIds);
        $this->productVariantOption = $this->buildProductVariantOption($variantIds);
        $this->users = $this->getUsers();
    }

    public function initAllUser() {
        $this->users = $this->getUsers();
    }

    public function initProductVariantOption() {
        $variantIds =  Config::get('z-search::default.filter_variant_id', [1,2,5]);
        $this->productVariant = $this->buildProductVariant($variantIds);
        $this->productVariantOption = $this->buildProductVariantOption($variantIds);
    }

    public function multiUpdatePrice($filter, $elasticsearchConfig) {
        $start = time();
        $productIds = $this->getProductIds(array_merge($filter, ['type' => 'init']));
        \Log::info('ZSEARCH - multiUpdatePrice count', [count($productIds)]);
        $count = 0;
        $error = [];
        if ($productIds) {
            foreach (array_chunk($productIds, 200) as $partProductIds) {
                $count += count($partProductIds);
                $products = $this->getProducts(array_merge($filter, [
                    'type' => 'init',
                    'ids' => $partProductIds
                ]))->toArray();
                \Log::info('ZSEARCH - multiIndexProduct pids', $partProductIds);
                $data = $this->changePriceProducts($products, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
                if ($data['error']) {
                    \Log::info('ZSEARCH - multiIndexProduct ERROR', [$data['error']]);
                    $error[] = $data['error'];
                }
            }
        }
        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,

        ];
    }

    public function changePriceProducts($products, $type, $index) {
        $error = [];
        $params = [];
        foreach (array_chunk($products, 200) as $chunkProducts) {
            try {
                foreach ($chunkProducts as $product) {
                    $params['body'][] = [
                        'update' => [
                            '_index' => $index,
                            '_type' => $type,
                            '_id' => $product['id']
                        ],

                    ];
                    $params['body'][] = [
                        'doc' => [
                            'price' => $product['price'],
                            'high_price' => $product['high_price'],
                        ]
                    ];
                }
                $this->elasticService->bulkAction($params);
            } catch (\Exception $e) {
                $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
            }
        }
        return [
            'error' => $error
        ];
    }

    protected function getProductsCustom() {
        $keyCache = decorCacheKey('cache::zsearch::getProductsCustom::' . env('APP_LOCALE'));
        $result = Cache::get($keyCache);
        if (!$result) {
            $result =  \DB::table('product_custom')
                ->whereIn('type', ['event', 'upload'])
                ->pluck('product_id', 'product_id')
                ->toArray();
            Cache::put($keyCache, $result, 3 * 24 * 60);
        }
        return $result;

    }

    protected function getProductsStatOld() {
        $keyCache = 'cache::product-stat::' . env('APP_LOCALE');
        $crFromDay = Config::get('z-search::sort.cr_from_day', 15);
        $result = Cache::get($keyCache);
        if (!$result) {
            $from = date('Y-m-d', time() - $crFromDay * 86400);
            $query = DB::table('payment_stats')
                ->where('date', '>', $from)
                ->whereIn('type', ['sale', 'click'])
                ->groupBy('product_id');
            $items = $query->get([
                DB::raw('SUM(click) as total_click, SUM(sale) as total_sale'),
                'product_id'
            ]);
            if ($items && count($items) > 0) {
                foreach ($items as $item) {
                    $result[$item->product_id] = (array) $item;
                }
            }
            Cache::put($keyCache, $result, 24 * 60 - 10);
        }
        return $result;
    }

    public function getProductsStat($day = null) {
        $keyCache = decorCacheKey('cache::product-statt::' . env('APP_LOCALE'));
        if (!$day) {
            $day = Config::get('z-search::sort.cr_from_day', 15);
        }

        $startTime = time() - $day * 86400;
        $result = Cache::get($keyCache);
        $now = time();
        if (!$result) {
            $result = [];
            while ($startTime < $now) {
                $toTime = $startTime + 86400 * 10;
                $fromDate = date('Y-m-d', $startTime);
                $toDate = date('Y-m-d', $toTime);
                $this->mergeProductStat($result, $this->getProductStatByTime($fromDate, $toDate));
                $startTime = $toTime;
            }
            Cache::put($keyCache, $result, 24 * 60 - 10);
        }
        return $result;
    }


    protected function mergeProductStat(&$result, $items) {
        if($items && count($items) > 0) {
            foreach ($items as $item) {
                if (!array_key_exists($item->product_id, $result)) {
                    $result[$item->product_id] = $this->defaultProductStat($item->product_id);
                }
                $result[$item->product_id]['total_click'] += $item->total_click;
                $result[$item->product_id]['total_sale'] += $item->total_sale;
            }
        }
    }

    protected function defaultProductStat($pId) {
        return [
            'product_id' => $pId,
            'total_click' => 0,
            'total_sale' => 0,
        ];
    }

    protected function getProductStatByTime($from, $to) {
        return DB::table('payment_stats')
            ->where('date', '>', $from)
            ->where('date', '<=', $to)
            ->whereIn('type', ['sale', 'click'])
            ->groupBy('product_id')->get([
                DB::raw('SUM(click) as total_click, SUM(sale) as total_sale'),
                'product_id'
            ]);
    }


    protected function buildProductVariantOption($variantIds) {
        $result = [];
        $items = DB::table('product_variant_option')
            ->whereIn('variant_id', $variantIds)
            ->get(['id', 'variant_id', 'name', 'slug']);
        foreach ($items as $item) {
            $result[$item->id] = [
                'name' => $item->name,
                'slug' => $item->slug,
                'variant_id' => $item->variant_id,
                'variant_name' => isset($this->productVariant[$item->variant_id]) ? $this->productVariant[$item->variant_id] : $item->variant_id,
            ];
        }
        return $result;
    }

    protected function buildProductVariant($variantIds) {
        return self::getProductVariant($variantIds);
    }

    public static function getProductVariant($variantIds) {
        $result = [];
        $items = DB::table('product_variant')
            ->whereIn('id', $variantIds)
            ->get(['id', 'slug', 'name', 'type']);
        if ($items) {
            foreach ($items as $item) {
                $result[$item->id] = [
                    'name' => $item->name,
                    'slug' => $item->slug,
                    'type' => $item->type,
                ];
            }
        }
        return $result;
    }

    protected function getUsers($ids = []) {
        $result = [];
        $query = DB::table('users');
        if ($ids) {
            $query->whereIn('id', $ids);
        }
        $users = $query->get(['name', 'id', 'slug'])
            ->toArray();
        foreach ($users as $user) {
            $result[$user->id] = $user;
        }
        return $result;
    }

    protected function getTagById($tagId) {
        if (!array_key_exists($tagId, $this->tags)) {
            $this->tags[$tagId] = (array) DB::table('tag')
                ->where('id', $tagId)
                ->first(['title', 'id', 'slug']);
        }
        return $this->tags[$tagId];

    }
    protected function getTag() {
        $result = [];
        $tags = DB::table('tag')
            ->get(['title', 'id', 'slug'])
            ->toArray();
        foreach ($tags as $tag) {
            $result[$tag->id] = [
                'id' => $tag->id,
                'slug' => $tag->slug,
                'title' => $tag->title,
            ];
        }
        return $result;
    }

    protected function getMinPrice($productId) {
        return DB::table('product_sku')
            ->where('product_id', $productId)
            ->min('price');
    }


    public function multiUpdateSeller($items, $elasticsearchConfig) {
        $start = time();
        \Log::info('ZSEARCH - multiUpdateSeller count', [count($items)]);
        $count = 0;
        $error = [];
        $data = [];
        if ($items) {
            foreach ($items as $item) {
                if (isset($item->user_id)
                    && $this->users
                    &&  array_key_exists($item->user_id, $this->users)) {
                    $data[] = [
                        'id' => $item->product_id,
                        'user' => [
                            'id' => $item->user_id,
                            'name' => $this->users[$item->user_id]->name,
                            'slug' => $this->users[$item->user_id]->slug,
                        ]
                    ];
                }
            }
            foreach (array_chunk($data, 200) as $partData) {
                $count += count($partData);
                $res = $this->updateProducts($partData, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
                if ($res['error']) {
                    \Log::info('ZSEARCH - multiIndexProduct ERROR', [$res['error']]);
                    $error[] = $res['error'];
                }
            }
        }
        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,

        ];
    }

    public function multiUpdateInfoProduct($items, $elasticsearchConfig) {
        $start = time();
        \Log::info('ZSEARCH - multiUpdateInfoProduct count', [count($items)]);
        $count = 0;
        $error = [];
        $data = [];
        if ($items) {
            foreach ($items as $item) {
                $data[$item->product_id] = [
                    'id' => $item->product_id,
                    'design_id' => $item->design_id ?? 0,
                ];
            }
            $data = array_values($data);
            foreach (array_chunk($data, 200) as $partData) {
                $count += count($partData);
                $res = $this->updateProducts($partData, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
                if ($res['error']) {
                    \Log::info('ZSEARCH - multiUpdateInfoProduct ERROR', [$res['error']]);
                    $error[] = $res['error'];
                }
            }

        }
        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,

        ];
    }

    public function multiUpdatePaymentStatProduct($items, $elasticsearchConfig) {
        $start = time();
        \Log::info('ZSEARCH - multiUpdatePaymentStatProduct count', [count($items)]);
        $count = 0;
        $error = [];
        $data = [];


        if ($items) {
            foreach ($items as $item) {
                $cr = 0;
                if ($item->total_click > 0) {
                    $cr = $item->total_sale/ $item->total_click;
                }

                $data[] = [
                    'id' => $item->product_id,
                    'total_click' => $item->total_click,
                    'total_sale' => $item->total_sale,
                    'cr' => $cr,
                ];
            }
            foreach (array_chunk($data, 200) as $partData) {
                $count += count($partData);
                $res = $this->updateProducts($partData, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
                if ($res['error']) {
                    \Log::info('ZSEARCH - multiUpdateInfoProduct ERROR', [$res['error']]);
                    $error[] = $res['error'];
                }
            }

        }
        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,

        ];
    }

    public function indexTag($items, $elasticsearchConfig) {
        $start = time();
        \Log::info('ZSEARCH - multiUpdateInfoProduct count', [count($items)]);
        $count = 0;
        $error = [];
        $data = [];
        if ($items) {
            foreach ($items as $item) {
                $data[] = [
                    'id' => $item->product_id,
                    'tags' => $this->tagsFromProduct($item->product_id),
                    'rating_count' => isset($item->rating_count) ? $item->rating_count : 0,
                    'rating_value' => isset($item->rating_value) ? $item->rating_value : 0,
                ];
            }
            foreach (array_chunk($data, 200) as $partData) {
                $count += count($partData);
                $res = $this->updateProducts($partData, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
                if ($res['error']) {
                    \Log::info('ZSEARCH - multiUpdateInfoProduct ERROR', [$res['error']]);
                    $error[] = $res['error'];
                }
            }

        }
        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,

        ];
    }

    public function updateProducts($products, $type, $index) {
        $error = [];
        $params = [];
        foreach (array_chunk($products, 200) as $chunkProducts) {
            try {
                foreach ($chunkProducts as $product) {
                    $params['body'][] = [
                        'update' => [
                            '_index' => $index,
                            '_type' => $type,
                            '_id' => $product['id']
                        ],
                    ];
                    unset($product['id']);
                    $params['body'][] = [
                        'doc' => $product
                    ];
                }
                $this->elasticService->bulkAction($params);
            } catch (\Exception $e) {
                $error[] = $e->getMessage() . ' '  . $e->getFile() . '-' . $e->getLine();
            }
        }
        return [
            'error' => $error
        ];
    }

    public function buildProductStat($productIds)
    {
        foreach (array_chunk($productIds, 100) as $chunkProductIds) {
            $data = DB::table('product_stat')
                ->whereIn('product_id', $chunkProductIds)
                ->get(['product_id', 'total_click', 'total_sale']);
            if ($data) {
                foreach ($data as $item) {
                    $this->productsStat[$item->product_id] = [
                        'product_id' => $item->product_id,
                        'total_click' =>  $item->total_click,
                        'total_sale' =>  $item->total_sale,
                    ];
                }
            }
        }
    }

    public function buildProductProductNDesign($productIds)
    {
        $this->productNDesign = [];
        foreach (array_chunk($productIds, 100) as $chunkProductIds) {
            $data = DB::table('product_n_design')
                ->whereIn('product_id', $chunkProductIds)
                ->get(['product_id', 'design_id', 'is_primary']);
            if ($data) {
                foreach ($data as $item) {
                    if (!array_key_exists($item->product_id, $this->productNDesign)
                        ||  $item->is_primary > $this->productNDesign[$item->product_id]['is_primary']) {
                        $this->productNDesign[$item->product_id] = [
                            'design_id' =>  $item->design_id,
                            'is_primary' =>  $item->is_primary,
                        ];
                    }

                }
            }
        }


    }

    public function handleIgnoreData($productIds, $elasticsearchConfig) {
        $items = [];
        if (!$productIds) {
            return;
        }
        $this->initIgnoreProduct();
        if (!self::$ignoreProducts) {
            return;
        }
        foreach ($productIds as $productId) {
            if (!array_key_exists($productId, self::$ignoreProducts)) {
                continue;
            }
            $items[] = [
                'id' => $productId
            ];
        }
        if ($items) {
            $this->deleteByItems($items, [self::TYPE_PRODUCTS, self::TYPE_PRODUCTS_TOP], $elasticsearchConfig['index']);
        }
        return $items;
    }

    protected function initIgnoreProduct()
    {
        if (self::tableExists('search_ignore_product') && !self::$ignoreProducts) {
            $ignoreProducts = DB::table('search_ignore_product')
                ->pluck('product_id')
                ->toArray();
            self::$ignoreProducts = array_fill_keys($ignoreProducts, 1);
        }

    }

    public function multiUpdateProductTradeMark($items, $elasticsearchConfig) {
        $start = time();
        \Log::info('ZSEARCH - multiUpdateProductTrademark count', [count($items)]);
        $count = 0;
        $error = [];
        if ($items) {
            foreach (array_chunk($items, 200) as $partData) {
                $count += count($partData);
                $res = $this->updateProducts($partData, self::TYPE_PRODUCTS_TRADEMARK, $elasticsearchConfig['index']);
                if ($res['error']) {
                    \Log::info('ZSEARCH - multiUpdateProductTrademark ERROR', [$res['error']]);
                    $error[] = $res['error'];
                }
            }
        }

        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,
        ];
    }
}

?>