<?php
namespace App\Modules\ZSearch\Services;

use Carbon\Carbon;
use Elasticsearch\ClientBuilder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\View;
use Modules\ZSearch\Models\Product;
use Modules\ZSearch\Services\ElasticSearchIndex;
use Modules\ZSearch\Services\ElasticSearchIndexSeller;
use Modules\ZSearch\Services\ElasticSearchService;
use Modules\ZSearch\Services\SearchDataDecor;

class ElasticSearchIndexLite extends ElasticSearchIndex {

    /** https://github.com/elastic/elasticsearch-php/tree/5.x */
    private $client;

    public $elasticServiceConfig;
    protected $amazonDataDecor;
    public $elasticService;
    public $searchDataDecor;
    public $productsStat = [];
    public $productsCustom;
    protected $users = [];
    protected $tags = [];
    protected $tagNProduct = [];
    protected $productVariantOption;
    protected $productVariant;
    protected $isAdmin;
    const TYPE_PRODUCTS = 'products';
    const TYPE_PRODUCTS_TOP = 'products_top';
    const TYPE_PRODUCTS_SELLER = 'products_seller';
    const TYPE_PRODUCTS_TRADEMARK = 'products_trademark';
    const TYPE_PRODUCTS_N_CATEGORY = 'product_n_category_top';
    static $cacheDefaultStyleTemplate = [];
    static $cacheCategory = [];
    static $cacheOptionTemplate = [];
    static $cache = [];
    static $productsTop = [];
    static $ignoreProducts = [];
    static $hasTableProductStat = false;


    public function __construct()
    {
        $elasticSearchConfig = Config::get('z-search::elasticsearch');
        $this->elasticServiceConfig = $elasticSearchConfig;
        $this->elasticService = new ElasticSearchService($elasticSearchConfig);
        $this->searchDataDecor = new SearchDataDecor();
    }

    public function indexByRange($filter, $elasticsearchConfig) {
        $start = time();
        $count = 0;
        $error = [];
        $products = $this->getProducts($filter)->toArray();
        $data = $this->indexMainProducts($products, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
        if ($data['error']) {
            \Log::error('ZSEARCH - indexByRange ERROR', [$data['error']]);
            $error[] = $data['error'];
        }
        return [
            'count' => $count,
            'error' => $error,
            'runtime' => time() - $start,

        ];
    }

    protected function decorProduct($product) {
        $productId = $product['id'];
        //$templateId = $this->getTemplateId($productId);
        $percentSale = $product['high_price'] > 0 ? ($product['high_price'] - $product['price'])/$product['high_price'] : 0;
        $percentSale = $percentSale > 0 ? ceil($percentSale * 100) : 0;
        $product['percent_sale'] = $percentSale;
        $product['tag'] = [];
        $product['tags'] = [];
        //$referTag = $this->tagsFromProduct($productId);
        //$categories = $this->getCategoryFromProduct($productId);
        $product['categories'] = [];
        $product['category_id'] = 0;
        $product['total_click'] = 0;
        $product['total_sale'] = 0;
        $product['cr'] = 0;
        $product['is_custom_design'] = 0;
        $product['user'] = [];
        $product['variant_options'] = [];
        $product['variant'] = [];
        $product['template_id'] = 0;
        $product['random_image'] = null;
        $product['updated_at'] = Carbon::now()->format('Y-m-d');
        $product['score'] = 0;
        return $product;
    }

    protected function getProductsByRange($filter = []) {
        $query = Product::getProductQuery($filter);
        $columns = [
            'id', 'sku', 'name', 'slug', 'image_url', 'brand_id',
            'status', 'price', 'high_price', 'inventory', 'view_count', 'sold', 'created_at'
        ];
        return $query->get($columns);
    }

    public function updateSkuByRange($fromId, $toId) {
        $result = [];
        $products = $this->fetchProductsWithoutTemplateInRange($fromId, $toId);
        if ($products && count($products) > 0) {
            $updateDate = [];
            foreach ($products as $item) {
                $product = [
                    'id' => $item->id,
                ];
                $product['variant_options'] = $this->buildVariantOption($item->id, null);
                $product['variant'] = $this->buildVariant($item->id, null);
                $updateDate[] = $product;
                $result[] = $product['id'];
            }
            $this->updateProducts($updateDate, self::TYPE_PRODUCTS, $this->elasticServiceConfig['index']);
        }
        \Log::info('ZSEARCH - updateSkuByRange', $result);
        return $result;
    }

    public function updateByRange($fromId, $toId) {
        $products = $this->fetchProductsInRange($fromId, $toId);
        $tags = $this->groupTags($this->fetchTagsInRange($fromId, $toId));
        $data = $this->buildUpdateData($products, $tags);
        foreach (array_chunk($data, 200) as $partData) {
            $res = $this->updateProducts($partData, self::TYPE_PRODUCTS, $this->elasticServiceConfig['index']);
            if ($res['error']) {
                \Log::info('ZSEARCH - updateByRange ERROR', [$res['error']]);
            }
        }
    }

    protected function buildUpdateData($products, $tags) {
        $data = [];
        foreach ($products as $item) {
            $templateId = $item->template_id ?: 0;
            $product = [
                'id' => $item->id,
                'score' => $item->score,
                'template_id' => $templateId,
                'category_id' => $item->category_id ?: 0,
            ];
            if ($item->user_id && array_key_exists($item->user_id, $this->users)) {
                $product['user'] = [
                    'id' => $item->user_id, //user_id
                    'name' => $this->users[$item->user_id]->name,
                    'slug' => $this->users[$item->user_id]->slug,
                ];
            }
            if ($tags && array_key_exists($item->id, $tags)) {
                $product['tags'] = $tags[$item->id];
            }
            if ($templateId) {
                $product['variant_options'] = $this->buildVariantOption($item->id, $templateId);
                $product['variant'] = $this->buildVariant($item->id, $templateId);
            }
            $data[] = $product;
        }
        return $data;
    }

    public function fetchProductsInRange($fromId, $toId)
    {
        $query = $this->buildQueryFetchProducts()
            ->whereNull('p.deleted_at')
            ->where('p.status', 'ACTIVE')
            ->where('p.id', '>=', $fromId)
            ->where('p.id', '<', $toId);
        return $query->get(['p.id', 'sp.score', 'pnu.user_id', 'pnt.template_id', 'pnc.category_id']);
    }


    protected function buildQueryFetchProducts() {
        return DB::table('product as p')
            ->from(DB::raw("sb_product as sb_p USE INDEX (PRIMARY)"))
            ->leftJoin('score_product as sp', 'p.id', '=', 'sp.id')
            ->leftJoin('product_n_user as pnu', 'pnu.product_id', '=', 'p.id')
            ->leftJoin('product_n_template as pnt', 'pnt.product_id', '=', 'p.id')
            ->leftJoin('product_n_category as pnc', function ($join) {
                $join->on('pnc.product_id', '=', 'p.id')
                    ->where('pnc.is_parent', '=', 0);
            });
    }

    public function fetchProductsWithoutTemplateInRange($fromId, $toId)
    {
        $query = DB::table('product as p')
            ->from(DB::raw("sb_product as sb_p USE INDEX (PRIMARY)"))
            ->leftJoin('product_n_template as pnt', 'pnt.product_id', '=', 'p.id')
            ->whereNull('p.deleted_at')
            ->where('p.status', 'ACTIVE')
            ->where('p.id', '>=', $fromId)
            ->where('p.id', '<', $toId)
            ->whereNull('pnt.template_id');
        return $query->get(['p.id']);
    }


    public function groupTags($items)
    {
        $tags = [];
        foreach ($items as $item) {
            if (!array_key_exists($item->refer_id, $tags)) {
                $tags[$item->refer_id] = [];
            }
            $tags[$item->refer_id][] =  $item->id;
        }
        return $tags;
    }

    public function fetchTagsInRange($fromReferId, $toReferId)
    {
        return DB::table('tag_refer as tr')
            ->select('tr.refer_id', 'tr.tag_id as id')
            ->where('tr.refer_id', '>=', $fromReferId)
            ->where('tr.refer_id', '<', $toReferId)
            ->get();
    }

    public function updateCustomProduct() {
        $result = [];
        $products = $this->getProductsCustom();
        if ($products && count($products) > 0) {
            $updateDate = [];
            foreach ($products as $productId) {
                $product = [
                    'id' => $productId,
                    'is_custom_design' => 1,
                ];
                $updateDate[] = $product;
                $result[] = $product['id'];
            }
            $this->updateProducts($updateDate, self::TYPE_PRODUCTS, $this->elasticServiceConfig['index']);
        }
        return $result;
    }

    public function multiUpdateProduct($filter, $elasticsearchConfig) {
        $start = time();
        $productIds = $this->getProductIds(array_merge($filter, ['type' => 'init']));
        $error = [];
        if ($productIds) {
            $this->updateByList($productIds, $elasticsearchConfig);
            $this->indexSeller($productIds, $filter);
        }
        $hiddenData = $this->handleHidden($filter, $elasticsearchConfig);
        $this->indexSellerHidden($filter);
        $ignoreData = $this->handleIgnoreData($productIds, $elasticsearchConfig);
        return [
            'count' => $productIds ? count($productIds) : 0,
            'error' => $error,
            'hidden' => $hiddenData,
            'runtime' => time() - $start,

        ];
    }

    public function updateByList($productIds, $elasticsearchConfig) {
        foreach (array_chunk($productIds, 200) as $partProductIds) {
            $products = $this->buildProductsData($partProductIds);
            $this->updateProducts($products, self::TYPE_PRODUCTS, $elasticsearchConfig['index']);
        }
    }

    public function buildProductsData($productIds) {
        $result = [];
        $products = $this->buildQueryFetchProducts()
            ->whereIn('p.id', $productIds)
            ->get([
                'p.id', 'p.sku', 'p.name', 'p.slug', 'p.image_url', 'p.brand_id',
                'p.status', 'p.price', 'p.high_price', 'p.inventory',
                'p.view_count', 'p.sold', 'p.created_at',
                'sp.score', 'pnu.user_id', 'pnt.template_id', 'pnc.category_id'
            ]);
        if ($products && count($products) > 0) {
            $this->tagNProduct = $this->tagsFromProductIds($productIds);
            $this->buildProductStat($productIds);
            $this->buildProductProductNDesign($productIds);
            $this->getUserFromProduct($products);
            foreach ($products as $product) {
                $result[] = $this->buildUpdateProduct($product);
            }
        }
        return $result;
    }

    protected function getUserFromProduct($products) {
        if (!$products || count($products) == 0) {
            return;
        }
        $userIds = [];
        foreach ($products as $product) {
            if (!$this->users || !array_key_exists($product->user_id, $this->users)) {
                continue;
            }
            $userIds[] = $product->user_id;
        }
        $userIds = array_values(array_unique($userIds));
        foreach (array_chunk($userIds, 100) as $chunkUserId) {
            $this->users = $this->users + $this->getUsers($chunkUserId);
        }
    }

    protected function buildUpdateProduct($product) {
        $result = [];
        $keys = ['id', 'name', 'sku', 'slug', 'price', 'high_price', 'view_count', 'sold', 'image_url'];
        foreach ($keys as $key) {
            if (isset($product->{$key})) {
                $result[$key] = $product->{$key};
            }
        }
        $percentSale = $product->high_price > 0 ? ($product->high_price  - $product->price)/$product->high_price : 0;
        $percentSale = $percentSale > 0 ? ceil($percentSale * 100) : 0;
        $result['percent_sale'] = $percentSale;
        $result['category_id'] = $product->category_id;
        if ($this->tagNProduct && array_key_exists($product->id, $this->tagNProduct)) {
            $result['tags'] = $this->tagNProduct[$product->id];
        }

        if ($product && isset($product->user_id) && $this->users &&  array_key_exists($product->user_id, $this->users)) {
            $result['user'] = [
                'id' => $product->user_id, //user_id
                'name' => $this->users[$product->user_id]->name,
                'slug' => $this->users[$product->user_id]->slug,
            ];
        }
        if ($this->productsStat && array_key_exists($product->id, $this->productsStat)) {
            $stat = $this->productsStat[$product->id];
            $result['total_click'] = $stat['total_click'];
            $result['total_sale'] = $stat['total_sale'];
            $result['cr'] = $stat['total_click'] > 0 ? $stat['total_sale']/$stat['total_click'] : 0;
        }

        if ($this->productNDesign && array_key_exists($product->id, $this->productNDesign)) {
            $result['design_id'] = $this->productNDesign[$product->id]['design_id'];
        }

        $result['variant_options'] = $this->buildVariantOption($product->id, $product->template_id);
        $result['variant'] = $product->template_id ? [] : $this->buildVariant($product->id, $product->template_id);
        $result['template_id'] = $product->template_id;
        $result['updated_at'] = Carbon::now()->format('Y-m-d');
        $result['score'] = $product->score ?: 0;
        return $result;
    }


}

?>