<?php

namespace Modules\ZSearch\Services;

use Illuminate\Support\Facades\DB;

class ElasticSearchIndexSeller extends ElasticSearchIndex
{
    protected $users = [];


    public function multiIndexProductSeller($productIds, $filter = [])
    {

        $sellerService = new ElasticSearchIndexSeller();
        $productIds = $sellerService->filterActor($productIds);
        if (!$productIds) {
            return;
        }
        $filter['is_all'] = 1;
        \Log::info('ZSEARCH - multiIndexProductSeller count', [count($productIds)]);
        foreach (array_chunk($productIds, 200) as $chunkProductIds) {
            try {
                \Log::info('ZSEARCH - multiIndexProductSeller pids', $chunkProductIds);
                $this->indexProductSeller($chunkProductIds, $filter);
            } catch (\Exception $e) {
                \Log::error('ZSEARCH - multiIndexProductSeller ERROR', $chunkProductIds);
                \Log::error('ZSEARCH - multiIndexProductSeller ERROR', [$e->getMessage() . ' - ' . $e->getFile() . ' ' . $e->getLine()]);
            }
        }

    }

    public function search($filter) {
        $params = $this->buildSearchParams($filter);
        return $this->elasticService->searchDocument($params, self::TYPE_PRODUCTS_SELLER, $this->elasticServiceConfig['index']);
    }

    public function buildSearchData($data) {
        $result = [];
        if (isset($data['hits']['hits'])) {
            $searchDecor =  new SearchDataDecor();
            $result = $searchDecor->mergeData([], $data['hits']['hits']);
        }
        return $result;
    }

    protected function buildSearchParams($filter) {
        $elasticParams = [
            'from' => $filter['from'],
            'size' => $filter['page_size'],
        ];
        $elasticParams['query']['bool']['should'] = $this->getSearchMatch($filter);
        $elasticParams['query']['bool']['filter'] =  $this->getSearchFilter($filter);

        return $elasticParams;
    }

    protected function getSearchMatch($filter) {
        $result = [
            ['match' => ['name' => $filter['keyword']]],
            ['match' => ['sku' => $filter['keyword']]],
        ];
        if (isset($params['status'])) {
            $result[] = [
                ['match' => ['status' => $filter['status']]],
            ];
        }
        return $result;
    }

    protected function getSearchFilter($params) {
        $columnsFilter = ['is_violation', 'is_trademark'];
        $result = [];
        $filter = [
            'actor_id' => $params['seller']->id,
        ];
        foreach ($columnsFilter as $column) {
            if (array_key_exists($column, $params)) {
                $filter[$column] = $params[$column];
            }
        }
        if (isset($params['category'])) {
            $filter['categories.id'] = $params['category'];
        }
        foreach ($filter as $column => $value) {
            $result[] = [
                'term' => [
                    $column => $value
                ]
            ];
        }
        return $result;
    }


    protected function indexProductSeller($productIds, $filter = [])
    {
        $products = $this->getProducts(array_merge($filter, [
            'ids' => $productIds
        ]), ['is_violation', 'is_trademark', 'actor_id'])->toArray();

        $productsSeller = $this->decorProductsSeller($products);
        if ($productsSeller && count($productsSeller) > 0) {
            $params = $this->elasticService->bulkData($productsSeller, $this->elasticServiceConfig['index'], self::TYPE_PRODUCTS_SELLER);
            $this->elasticService->bulkAction($params);
        }
    }

    protected function decorProductsSeller($products)
    {
        $result = [];
        if ($products) {
            foreach ($products as $product) {
                $result[] = $this->decorProductSeller($product);
            }
        }

        return $result;
    }

    protected function decorProductSeller($product)
    {
        $ignoreColumns = ['brand_id', 'inventory'];
        foreach ($ignoreColumns as $col) {
            if(isset($product[$col])) {
                unset($product[$col]);
            }
        }
        $categories = $this->getCategoryFromProduct($product['id']);
        $product['categories'] = $categories ? $categories : [];
        return $product;
    }

    public function filterActor($productIds)
    {
        $result = [];
        if ($productIds) {
            foreach (array_chunk($productIds, 200) as $partIds) {
                $result = array_merge($result, $this->chunkFilterActor($partIds));
            }
        }
        return $result;
    }


    protected function chunkFilterActor($productIds)
    {
        $result = [];
        $products = $this->actorProducts($productIds, ['id', 'actor_id']);
        if ($products) {
            foreach ($products as $product) {
                if ($this->checkActor($product->actor_id)) {
                    $result[] = $product->id;
                }
            }
        }
        return $result;
    }

    public function checkActor($id, $token = null)
    {
        $isOk = false;
        $user = $this->getUser($id, ['id', 'seller_token']);
        if ($user && isset($user->seller_token) && $user->seller_token) {
            $isOk = !$token || $user->seller_token == $token;
        }
        return $isOk;
    }

    protected function actorProducts($ids, $columns = ['*'])
    {
        return DB::table('product')
            ->whereIn('id', $ids)
            ->get($columns);
    }

    protected function getUser($id, $columns = ['*'])
    {
        if (!array_key_exists($id, $this->users)) {
            $this->users[$id] = DB::table('users')
                ->where('id', $id)
                ->first($columns);
        }
        return $this->users[$id];
    }

    public function getUserByToken($token, $columns = ['*'])
    {
        return $token ? DB::table('users')->where('seller_token', $token)->first($columns)
                      : null;
    }


}