<?php
namespace Modules\ZSearch\Services;
use Elasticsearch\ClientBuilder;

class ElasticSearchService {

    /** https://github.com/elastic/elasticsearch-php/tree/5.x */
    private $client;

    public function __construct($config) {
        $this->config = $config;
        $elasticAuth = $this->config['auth'];
        $this->client = ClientBuilder::create()->setHosts(["{$elasticAuth['scheme']}://{$elasticAuth['host']}:{$elasticAuth['port']}"])->build();
    }

    public function searchDocument(array $query, $type, $indexName = 'zSearch') {
        $params = [
            'index' => $indexName,
            'type' => $type,
            'body' => $query
        ];
        return $this->client->search($params);
    }

    public function saveDocument($data, $type, $indexName = 'zSearch') {
        $params = [
            'index' => $indexName,
            'type' => $type,
            'body' => $data
        ];
        if (!empty($data['id'])) {
            $params['id'] = $data['id'];
        }
        return $this->client->index($params);
    }

    public function findDocument($id, $type, $indexName = 'zSearch') {
        $params = [
            'index' => $indexName,
            'type' => $type,
            'id' => $id
        ];
        return $this->client->get($params);
    }

    public function existsDocument($id, $type, $indexName = 'zSearch') {
        $params = [
            'index' => $indexName,
            'type' => $type,
            'id' => $id
        ];
        return $this->client->exists($params);
    }

    public function deleteDocument($id, $type, $indexName = 'zSearch') {
        $params = [
            'index' => $indexName,
            'type' => $type,
            'id' => $id
        ];
        return $this->client->delete($params);
    }

    public function bulkAction($params) {
        /* actions item example
        [
            'delete' => [
                '_index' => 'er',
                '_type' => 'state',
                '_id' => $i
            ]
        ];
        */
        return $this->client->bulk($params);
    }

    public function deleteIndex($indexName = 'zSearch') {
        $params = [
            'index' => $indexName
        ];
        return $this->client->indices()->delete($params);
    }

    public function deleteAllItems($index, $type) {
        $params = [
            'index' => $index,
            'type' => $type,
            'body' => [],
        ];
        return $this->client->deleteByQuery($params);
    }

    public function deleteItemsByQuery($index, $type, $query) {
        $params = [
            'index' => $index,
            'type' => $type,
            'body' => [
                'query' => $query
            ],
            'conflicts' => 'proceed'
        ];
        return $this->client->deleteByQuery($params);
    }

    public function createIndex($indexName = 'zSearch') {
        $params = [
            'index' => $indexName
        ];
        if ($this->client->indices()->exists($params)){
            $this->client->indices()->delete($params);
        }

        $types = $this->config['type'];
        $mappings = [];
        foreach ($types as $key => $type) {
            $properties = isset($type['mappingFields']) ? $type['mappingFields'] : [];
            $mappings[$key]['properties'] = $properties;
        }

        $params = [
            'index' => $indexName,
            'body' => [
                'settings' => $this->config['settings'],
                'mappings' => $mappings
            ]
        ];
        
        return $this->client->indices()->create($params);
    }

    public function bulkData($items, $index, $type) {
        $params = [];
        foreach ($items as $item) {
            if ($type == 'product_n_category_top' && isset($item['variant'])) {
                unset($item['variant']);
            }
            $params['body'][] = [
                'index' => [
                    '_index' => $index,
                    '_type' => $type,
                    '_id' => $item['id']
                ]
            ];
            $params['body'][] = $item;
        }
        return $params;
    }

    public function dataForUpdate($products, $index, $type) {
        $result = [];
        if (count($products) > 0) {
            foreach ($products as $product) {
                $result['body'][] = [
                    'update' => [
                        '_index' => $index,
                        '_type' => $type,
                        '_id' => $product['id']
                    ],
                ];
                unset($product['id']);
                $result['body'][] = [
                    'doc' => $product
                ];
            }
        }

        return $result;
    }



    public function countByCategory( $index, $type = 'product_n_category_top') {
        $params = [
            'index' => $index,
            'type'  => $type,
            'size'  => 0,
            'body'  => [
                'aggs' => [
                    'categories' => [
                        'terms' => [
                            'field' => 'category_id',
                            'size' => 10000000
                        ]
                    ]
                ]
            ]
        ];

        return $this->client->search($params);

    }

    public function createIndexType($indexName, $type) {
        $types = $this->config['type'];
        $mappings = [];
        $mappings[$type]['properties'] = isset($types[$type]) && isset($types[$type]['mappingFields']) ? $types[$type]['mappingFields'] : [];

        $params = [
            'index' => $indexName,
            'body' => [
                'settings' => $this->config['settings'],
                'mappings' => $mappings
            ]
        ];

        return $this->client->indices()->create($params);
    }

    public function reIndex($oldIndex, $oldType, $newIndex, $newType)
    {
        $params = [
            'body' => [
                'source' => [
                    'index' => $oldIndex,
                    'type' => $oldType,
                ],
                'dest' => [
                    'index' => $newIndex,
                    'type' => $newType
                ]
            ]
        ];

        return $this->client->reindex($params);
    }
}

?>