<?php

namespace Modules\ZSearch\Services;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Models\Option;
use Illuminate\Support\Facades\Cache;
use Modules\ZSearch\Models\Product;
use Illuminate\Support\Facades\Redis;

class SearchDataDecor
{

    const STATUS_PINED = 'pined';
    const STATUS_POSITION = 'position';
    const STATUS_CONTAINS = 'contains';
    const CACHE_PRODUCT_FILTER = 'cache_product_filters::';
    const CACHE_MINUTES = 600;

    static $skuByTemplate = [];
    static $colorBySku = [];
    static $categories = [];
    static $options = [];
    static $variants = [];
    static $templates = [];

    public function sort($items, $filter = [])
    {
        $sort = array_key_exists('order', $filter) ? $filter['order'] : '';
        $result = [];
        $sortConfig = [
            'sold' => [
                'col' => 'sold',
                'type' => 'desc',
            ],
            'lastest' => [
                'col' => 'id',
                'type' => 'desc',
            ],
            'latest' => [
                'col' => 'id',
                'type' => 'desc',
            ],
            'low_price' => [
                'col' => 'price',
                'type' => 'asc',
            ],
            'high_price' => [
                'col' => 'price',
                'type' => 'desc',
            ],
            'sale' => [
                'col' => 'percent_sale',
                'type' => 'desc',
            ],
        ];
        $sort = $sort && array_key_exists($sort, $sortConfig) ? $sortConfig[$sort] : '';
        if ($sort) {
            $result = $this->mergeData($result, $this->sortData($items, $sort));
        } else {
            //@todo disable $sortFirstPageConfig
            $sortFirstPageConfig = config('z-search::sort.sort_first_page_number', 0);
            $dataByStatus = $this->buildByStatus($items, $filter);
            $activeItems = $dataByStatus['ACTIVE'];
            $sortData = $this->sortByRating($activeItems, $filter);
            $result = $this->mergeData($result, $sortData);
            if ($sortFirstPageConfig > 0 && count($result) > 0) {
                $firstPageData = array_splice($result, 0, $sortFirstPageConfig);
                $firstPageSortData = $this->sortFirstPage($firstPageData);
                $result = $this->mergeData($firstPageSortData, $result);
            }
            $result = $this->mergeSpecialData($result, $dataByStatus);
        }
        return $result;
    }

    protected function sortFirstPage($firstPageData)
    {
        $result = [];
        if ($firstPageData) {
            $highScoreData = [];
            $lowScoreData = [];
            $firstItem = $firstPageData[0];
            $maxScore = isset($firstItem['_score']) ? $firstItem['_score'] : 0;
            $termScore = $this->buildTermScore($maxScore);
            $flagScore = $maxScore - $termScore;
            foreach ($firstPageData as $item) {
                $score = isset($item['_score']) ? $item['_score'] : 0;
                if ($score > $flagScore) {
                    $highScoreData[] = $item;
                } else {
                    $lowScoreData[] = $item;
                }
            }
            $result = $this->sortData($highScoreData, [
                'col' => 'cr',
                'type' => 'desc',
            ]);
            $result = $this->mergeData($result, $this->sortData($lowScoreData, [
                'col' => 'cr',
                'type' => 'desc',
            ]));
        }

        return $result;
    }

    protected function mergeSpecialData($resultData, $dataByStatus) {
        $result = [];
        if ($dataByStatus['PIN'] || $dataByStatus['POSITION']) {
            if ($dataByStatus['PIN']) {
                $pinData = $this->sortData($dataByStatus['PIN'], [
                    'col' => 'porder',
                    'type' => 'desc',
                ]);
                $result = $this->mergeData($result, $pinData);
            }

            if ($dataByStatus['POSITION']) {
                $positionData = $this->sortData($dataByStatus['POSITION'], [
                    'col' => 'position',
                    'type' => 'desc',
                ]);
                $result = $this->mergeData($result, $positionData);
            }

            $result = $this->mergeData($result, $resultData);
        } else {
            $result = $resultData;
        }
        return $result;
    }

    protected function sortByRating($activeItems, $filter)
    {
        $result = [];
        if ($activeItems) {
            $count = count($activeItems);
            $firstItem = $activeItems[0];
            $lastItem = $activeItems[$count - 1];
            $maxScore = $firstItem['_score'];
            $minScore = $lastItem['_score'];
            $notDecorNumberCheck = config('z-search::sort.term_contain_not_decor', 12);
            $keywordFrom = config('z-search::sort.keyword_length_from', 3);
            $termScore = $this->buildTermScore($maxScore - $minScore);
            $sort = [
                'col' => 'cr',
                'type' => 'desc',
            ];

            $q = isset($filter['keyword']) ? $filter['keyword'] : request()->input('q');
            $q = $q ? strtolower($q) : $q;
            $decorContainData = [];
            $notDecorContainData = [];
            $tmpContainSearch = [];
            $countContains = 0;
            $tmpData = [];
            $qCombinations = $this->combinations($q);
            foreach ($activeItems as $item) {
                $isContainsKeyWord = $this->itemContainSearch($item, $qCombinations);
                if ($isContainsKeyWord) {
                    $item['_source']['zstatus'] = self::STATUS_CONTAINS;
                    $decorContainData[] = $item;
                    $id = isset($item['_source']['id']) ? $item['_source']['id'] : 0;
                    $tmpContainSearch[$id] = $id;
                }
            }
            if ($decorContainData) {
                $result = $this->mergeData($result, $this->sortBySaleLevel($decorContainData, $sort));
            }
            if (count($result) < $notDecorNumberCheck && $q && strlen($q) > $keywordFrom) {
                foreach ($activeItems as $item) {
                    $id = isset($item['_source']['id']) ? $item['_source']['id'] : 0;
                    $isContainsKeyWord = array_key_exists($id, $tmpContainSearch);
                    if ($isContainsKeyWord) {
                        continue;
                    }
                    $isContainsKeyWord = $this->itemContainSearch($item, $qCombinations, false);
                    if ($isContainsKeyWord) {
                        $item['_source']['zstatus'] = self::STATUS_CONTAINS;
                        $notDecorContainData[] = $item;
                        $tmpContainSearch[$id] = $id;
                    }
                }
                if ($notDecorContainData) {
                    $result = $this->mergeData($result, $this->sortBySaleLevel($notDecorContainData, $sort));
                }
            }

            foreach ($activeItems as $key => $item) {
                $id = isset($item['_source']['id']) ? $item['_source']['id'] : 0;
                $isContainsKeyWord = array_key_exists($id, $tmpContainSearch);
                if ($isContainsKeyWord) {
                    $countContains++;
                    //truong hop search chi 1 hoac 2 ky tu, sp cuoi cung itemContainSearch = true
                    if ($key + 1 == $count && $tmpData) {
                        $result = $this->mergeData($result, $this->sortBySaleLevel($tmpData, $sort));
                    }
                    continue;
                }
                $score = $item['_score'];
                $tmpData[] = $item;
                if (($score + $termScore <= $maxScore) || $key + 1 == $count) {
                    $maxScore = $score + 0.1;
                    $result = $this->mergeData($result, $this->sortBySaleLevel($tmpData, $sort));
                    $tmpData = [];
                }
            }
            if (isset($_GET['zdebug']) && $_GET['zdebug'] == 1) {
                echo count($activeItems);
                echo '<br>';
                echo count($result);
                echo '<br>';
                echo $countContains;
                echo '<br>tmpContainSearch';
                echo '<pre>';
                echo print_r($tmpContainSearch);
                echo '</pre>';
                echo '<br>decorContainData';
                echo '<pre>';
                echo print_r($decorContainData);
                echo '</pre>';
                echo '<br>notDecorContainData';
                echo '<pre>';
                echo print_r($notDecorContainData);
                echo '</pre>';
                echo '<br>';
                echo '<pre>';
                echo print_r(array_slice($activeItems, 0, 1000));
                echo '</pre>';
                exit;
            }
        }
        return $result;
    }

    protected function sortBySaleLevel($items, $sort) {
        $result = [];
        $keys = ['from800', 'to800', 'to400', 'to200', 'to100', 'to50', 'to10', 'to5', 'to0'];
        $groupData = $this->groupBySaleLevel($items);
        foreach ($keys as $key) {
            if ($key == 'to0') {
                $sort = [
                    'col' => 'category_id',
                    'type' => 'asc',
                ];
            }
            if (isset($groupData[$key]) && count($groupData[$key]) > 0) {
                $result = $this->mergeData($result, $this->sortData($groupData[$key], $sort));
            }

        }
        return $result;
    }

    protected function groupBySaleLevel($items) {
        $data = [];
        foreach ($items as $item) {
            $totalSale = isset($item['_source']['total_sale']) ? $item['_source']['total_sale'] : 0;
            $key = 'to0';
            if ($totalSale > 800) {
                $key = 'from800';
            }  else if ($totalSale > 400) {
                $key = 'to800';
            }  else if ($totalSale > 200) {
                $key = 'to400';
            } else if ($totalSale > 100) {
                $key = 'to200';
            } else if ($totalSale > 50) {
                $key = 'to100';
            } else if ($totalSale > 10) {
                $key = 'to50';
            } else if ($totalSale > 5) {
                $key = 'to10';
            } else if ($totalSale >= 1) {
                $key = 'to5';
            }

            if (!array_key_exists($key, $data)) {
                $data[$key] = [];
            }
            $data[$key][] = $item;
        }
        return $data;
    }

    protected function buildTermScore($maxScore) {
        $term = config('z-search::sort.term_decor', 10);
        $termScore = $maxScore / $term;
        if ($termScore < 5) {
            $termScore = 5;
        }
        return $termScore;
    }

    protected function itemContainSearch($item, $qCombinations, $decor = true)
    {
        $name = $item['_source']['name'];
        $name = strtolower($name);
        $result = false;
        if ($qCombinations) {
            foreach ($qCombinations as $qItem) {
                $result = $this->strContains($name, $qItem, $decor);
                if ($result) {
                    break;
                }
            }
        }

        return $result;
    }

    public function combinations($q)
    {
        if (!$q) {
            return [];
        }
        $qExplode = explode(' ', $q);
        $count = count($qExplode);
        $qSet = [
            $q
        ];
        if ($count == 2) {
            $qSet[] = implode(' ', [$qExplode[1], $qExplode[0]]);
        } else if ($count == 3) {
            $qSet[] = implode(' ', [$qExplode[0], $qExplode[2], $qExplode[1]]);
            $qSet[] = implode(' ', [$qExplode[1], $qExplode[0], $qExplode[2]]);
            $qSet[] = implode(' ', [$qExplode[1], $qExplode[2], $qExplode[0]]);
            $qSet[] = implode(' ', [$qExplode[2], $qExplode[0], $qExplode[1]]);
            $qSet[] = implode(' ', [$qExplode[2], $qExplode[1], $qExplode[0]]);
        }
        if (str_contains($q, '\'')) {
            $qSet = array_merge($qSet, $this->combinations(str_replace('\'', '', $q)));
        }
        return $qSet;
    }

    protected function decorS($q) {
        return $q . 's';
    }



    protected function strContains($name, $q, $decor = true) {
        $isContainsKeyWord = false;
        if ($name) {
            if ($decor) {
                $name = ' ' . $name . ' ';
                $q = ' ' . $q . ' ';
            }
            if ($q && strpos($name, $q) !== false) {
                $isContainsKeyWord = true;
            }
        }
        return $isContainsKeyWord;
    }

    protected function sortByTerm($activeItems)
    {
        $term = config('z-search::sort.term_decor', 10);
        $count = count($activeItems);
        $perTerm = ceil($count / $term);
        if ($perTerm == 0) {
            $perTerm = 5;
        }

        $i = 1;
        $result = [];
        $tmpData = [];
        $sort = [
            'col' => 'sold',
            'type' => 'desc',
        ];
        if ($activeItems) {
            foreach ($activeItems as $item) {
                $tmpData[] = $item;
                if ($i % $perTerm == 0 || $i == $count) {
                    $result = $this->mergeData($result, $this->sortData($tmpData, $sort));
                    $tmpData = [];
                }
                $i++;
            }
        }
        return $result;
    }

    public function filter($items, $filter)
    {
        $result = [];
        $variantMapping = [
            'size_variant_id' => 1,
            'color_variant_id' => 2,
            'type_variant_id' => 5,
        ];
        foreach ($items as $item) {
            $tmpItem = $item['_source'];
            $price = $tmpItem['price'];
            $categoryIsOk = !array_key_exists('category', $filter) || $this->colInArray($tmpItem['categories'], 'id', $filter['category']);
            $categoryDetachIsOk = !array_key_exists('category_detach', $filter) || $this->colInArray($tmpItem['categories'], 'id', $filter['category_detach']);
            $variantSizeIsOk = !array_key_exists('size_variant_id', $filter) ;
            if (array_key_exists('size_variant_id', $filter) &&
                isset($tmpItem['variant_options'][$variantMapping['size_variant_id']]['options']))  {
                $variantSizeIsOk = $this->colInArray($tmpItem['variant_options'][$variantMapping['size_variant_id']]['options'], 'id', $filter['size_variant_id']);
            }

            $colorSizeIsOk = !array_key_exists('color_variant_id', $filter);
            if (array_key_exists('color_variant_id', $filter) &&
                isset($tmpItem['variant_options'][$variantMapping['color_variant_id']]['options']))  {
                $colorSizeIsOk = $this->colInArray($tmpItem['variant_options'][$variantMapping['color_variant_id']]['options'], 'id', $filter['color_variant_id']);
            }

            $typeSizeIsOk = !array_key_exists('type_variant_id', $filter);
            if (array_key_exists('type_variant_id', $filter) &&
                isset($tmpItem['variant_options'][$variantMapping['type_variant_id']]['options']))  {
                $typeSizeIsOk = $this->colInArray($tmpItem['variant_options'][$variantMapping['type_variant_id']]['options'], 'id', $filter['type_variant_id']);
            }
            $minPriceIsOk = !array_key_exists('minPrice', $filter) || $price > $filter['minPrice'];
            $maxPriceIsOk = !array_key_exists('maxPrice', $filter) || $price < $filter['maxPrice'];

            $minScoreIsOk = !array_key_exists('min_score', $filter) || $item['_score'] >= $filter['min_score'];

            if ($categoryIsOk && $minPriceIsOk && $maxPriceIsOk
                && $categoryDetachIsOk && $variantSizeIsOk
                && $colorSizeIsOk && $typeSizeIsOk && $minScoreIsOk) {
                if (isset($tmpItem['categories'])
                    && count($tmpItem['categories']) > 0
                    && !isset($item['_source']['category_id'])) {
                    $category = end($tmpItem['categories']);
                    if (isset($category['id']) && $item['_source']) {
                        $item['_source']['category_id'] = $category['id'];
                    }
                }
                $result[] = $item;
            }
        }
        return $result;
    }

    protected function colInArray($items, $col, $value)
    {
        $result = false;
        foreach ($items as $item) {
            if (isset($item[$col])) {
                $result = is_array($value) ? in_array($item[$col], $value) : $item[$col] == $value;
                if ($result == true) {
                    break;
                }
            }
        }
        return $result;

    }

    protected function buildKeyFromFilter($filter) {
        $key = '';
        $locale = env('APP_LOCALE', '');
        if (array_key_exists('category_id', $filter)) {
            $key = self::CACHE_PRODUCT_FILTER . $locale . $filter['category_id'];
        } else if (array_key_exists('user_id', $filter)) {
            $key = self::CACHE_PRODUCT_FILTER . $locale . 'user_id' . $filter['user_id'];
        }
        return $key;
    }

    public function buildFilter($items, $filter = [])
    {
        $retVal = [];
        $categories = [];
        $displayFilter = [];
        if (!$retVal) {
            $minPrice = 0;
            $maxPrice = 0;
            $tags = [];
            $brands = [];
            $productCount = 0;
            $filterOptions = [];
            $filterOptionsFlag = [];

            foreach ($items as $item) {
                if (array_key_exists('_source', $item)) {
                    $item = $item['_source'];
                }
                if (!isset($item['price'])) {
                    continue;
                }
                $productCount++;
                $price = $item['price'];
                if ($price < $minPrice || $minPrice == 0) {
                    $minPrice = $price;
                }
                if ($price > $maxPrice || $minPrice == 0) {
                    $maxPrice = $price;
                }
                $item['categories'] = [];
                if (isset($item['category_id'])) {
                    $item['categories'] = self::getTreeCategory($item['category_id']);
                }
                $categories = $this->generateAttFilter($categories, $item['categories']);
                $filterOptionsFlag = $this->groupFilterOption($filterOptionsFlag, $item);

            }
            $filterOptions = $this->buildFilterOption($filterOptionsFlag);
            $priceRange = $this->getPriceRange($filter, $productCount, $maxPrice, $minPrice);
            $retVal = [
                'categories' => array_values($categories),
                'tags' => [],
                'brands' => array_values($brands),
                'priceRange' => $priceRange,
                'filterOptions' => $filterOptions,
                'displayedFilters' => $this->generateDisplayFilter($filter, [
                    'categories' => $categories,
                    'brands' => array_values($brands),
                    'priceRange' => $priceRange,
                    'filterOptions' => $filterOptions,
                ])
            ];
        }



        return $retVal;
    }

    protected function groupFilterOption($filterOptionsFlag, $item) {
        if (isset($item['variant_options']) && count($item['variant_options']) > 0) {
            foreach ($item['variant_options'] as $variant) {
                if (!isset($variant['variant_id']) || !isset($variant['variant_name']['name']) || !isset($variant['options'])) {
                    continue;
                }
                $variantId = $variant['variant_id'];
                if (!isset($filterOptionsFlag[$variantId])) {
                    $filterOptionsFlag[$variantId] = [
                        'variant_id' => $variantId,
                        'variant_name' => $variant['variant_name']['name'],
                        'variant_slug' => $variant['variant_name']['slug'],
                        'options' => []
                    ];
                }
                $filterOptionsFlag[$variantId]['options'] = $this->generateAttFilter($filterOptionsFlag[$variantId]['options'], $variant['options']);
            }
        }
        return $filterOptionsFlag;
    }

    protected function buildFilterOption($filterOptionsFlag) {
        $filterOptions = [];
        foreach ($filterOptionsFlag as $variantData) {
            $options = $this->sortArray(array_values($variantData['options']), 'count');
            $options = array_slice($options, 0, 20);
            foreach ($options as $option) {
                $variantName = $variantData['variant_name'];
                $option['variant_id'] = $variantData['variant_id'];
                $option['variant_name'] =$variantName;
                $option['variant_slug'] = $variantData['variant_slug'];
                if (!isset($filterOptions[$variantName])) {
                    $filterOptions[$variantName] = [];
                }
                $filterOptions[$variantName][] = $option;
            }
        }
        return $filterOptions;
    }



    public function buildTag($items) {
        $result = [];
        if ($items) {
            foreach ($items as $item) {
                if (!isset($item['tag'])) {
                    continue;
                }
                $result = $this->generateAttFilter($result, $item['tag']);
            }

            if ($result) {
                $result = array_values($result);
                $result = $this->sortArray($result, 'count');
                $result = array_slice($result, 0, 20);
            }

        }
        return $result;
    }

    public function sortArray($items, $col) {
        usort($items, function ($itemA, $itemB) use ($col) {
            $aValue = isset($itemA[$col]) ? $itemA[$col] : '';
            $bValue = isset($itemB[$col]) ? $itemB[$col] : '';
            $result = 0;
            if ($aValue > $bValue) {
                $result = -1;
            } else if ($aValue < $bValue) {
                $result = 1;
            }
            return $result;
        });
        return $items;
    }

    protected function generateDisplayFilter($filter, $data) {
        $result = [];
        $minPrice = array_key_exists('minPrice', $filter) ? $filter['minPrice'] : 0;
        $maxPrice = array_key_exists('maxPrice', $filter) ? $filter['maxPrice'] : $minPrice;
        if ($minPrice > 0 || $maxPrice > 0) {
            $minPrice = formatPrice($minPrice);
            $maxPrice = formatPrice($maxPrice);
            $result['price'] = $minPrice
                ? __("Từ") . " $minPrice -  $maxPrice "
                : __("Dưới") ." $maxPrice";
        }
        if (array_key_exists('category', $filter) && $filter['category']) {
            $categoryFilter = $filter['category'];
            $categories = $data['categories'];
            if ($categories && count($categoryFilter) > 0) {
                $result['category'] = [];
                foreach ($categoryFilter as $categoryId) {
                    if(array_key_exists($categoryId, $categories)) {
                        $categoryItem = $categories[$categoryId];
                        if ($categoryItem && isset($categoryItem['name'])) {
                            $result['category'] = $categoryItem['name'] . '::' . $categoryItem['slug'];
                        }
                    }
                }
            }
        }

        if (array_key_exists('brand', $filter) && $filter['brand']) {
            $result['brand'] = ucfirst($filter['brand']);
        }
        if (isset($data['filterOptions'])) {
            $filterOptions = $data['filterOptions'];
            $options = config("z-search::default.filter_option", []);
            foreach ($filter as $itemFilter) {
                if (is_array($itemFilter) && isset($itemFilter['type']) && $itemFilter['type'] == 'size') {
                    $indexKey = $itemFilter['type'] . '_variant_id';
                    $filter[$indexKey] = $itemFilter['id'];
                    break;
                }
            }
            foreach ($options as $option) {
                $slug = str_slug($option);
                $key = $slug . '_variant_id';
                if (array_key_exists($key, $filter) && $filter[$key]) {
                    $variants = isset($filterOptions[$option]) ? $filterOptions[$option] : [];
                    foreach ($variants as $variant) {
                        if ($variant['id'] == $filter[$key]) {
                            $result[$key] = $variant['name'] . '::' . $variant['slug'];
                        }
                    }
                }
            }
        }
        return $result;
    }

    protected function generateAttFilter($filter, $items)
    {
        if ($items) {
            foreach ($items as $item) {
                if (!$item || !isset($item['id'])) {
                    continue;
                }
                $id = $item['id'];
                if (!array_key_exists($id, $filter)) {
                    $item['count'] = 1;
                    $filter[$id] = $item;
                } else {
                    $filter[$id]['count']++;
                }
            }
        }
        return $filter;
    }


    public function mergeData($result, $mergeData)
    {
        foreach ($mergeData as $item) {
            if (isset($item['_source'])) {
                $score = isset($item['_score']) ? $item['_score'] : 0;
                $item = $item['_source'];
                $item['_score'] = $score;
            }
            $result[] = $item;
        }
        return $result;
    }


    protected function buildByStatus($items, $filter)
    {
        $result = [
            'ACTIVE' => [],
            'PENDING' => [],
            'PIN' => [],
            'POSITION' => [],
        ];
        if ($items) {
            $productPined = $this->getProductPined();
            $productPosition = $this->getProductPosition($filter);
            $avgScore = $this->avgScore($items);
            foreach ($items as $item) {
                $status = isset($item['_source']['status']) ? $item['_source']['status'] : 'ACTIVE';
                $id = $item['_source']['id'];
                $score = $item['_score'];
                $pOrder = 0;
                $position = 0;
                if ($productPined && $score >= $avgScore && array_key_exists($id, $productPined)) {
                    $status = 'PIN';
                    $pOrder = $productPined[$id];
                    $item['_source']['zstatus'] = self::STATUS_PINED;

                } else if ($productPosition && array_key_exists($id, $productPosition) && $productPosition[$id]) {
                    $status = 'POSITION';
                    $position = $productPosition[$id];
                    $item['_source']['zstatus'] = self::STATUS_POSITION; 

                }
                $item['_source']['porder'] = $pOrder;
                $item['_source']['position'] = $position;
                if (!isset($result[$status])) {
                    $result[$status] = [];
                }
                $result[$status][] = $item;

            }
        }
        return $result;
    }

    protected function avgScore($items)
    {
        $firstItem = $items[0];
        $maxScore = $firstItem['_score'];
        $result = ceil($maxScore) / 2;
        if ($result < 5) {
            $result = 2;
        }
        return $result;
    }


    public function sortData($items, $sort, $usingScore = true, $usingDesign = true)
    {
        if (!$sort) {
            return $items;
        }
        usort($items, function ($itemA, $itemB) use ($sort, $usingScore, $usingDesign) {
            $result = 0;
            $col = $sort['col'];
            $type = $sort['type'];
            $aValue = isset($itemA['_source'][$col]) ? $itemA['_source'][$col] : '';
            $bValue = isset($itemB['_source'][$col]) ? $itemB['_source'][$col] : '';

            if ($col == 'cr' && !$aValue && !$bValue) {
                $aValue = isset($itemA[$col]) ? $itemA[$col] : '';
                $bValue = isset($itemB[$col]) ? $itemB[$col] : '';
            }
            if ($aValue > $bValue) {
                $result = -1;
            } else if ($aValue < $bValue) {
                $result = 1;
            }
            if ($type == 'asc') {
                $result *= -1;
            }
            if ($aValue == $bValue && $usingScore) {
                $aScore = isset($itemA['_score']) ? $itemA['_score'] : 1;
                $bScore = isset($itemB['_score']) ? $itemB['_score'] : 1;
                if ($aScore > $bScore) {
                    $result = -1;
                } else if ($aScore < $bScore) {
                    $result = 1;
                } else {
                    $aValue = isset($itemA['_source']['design_id']) ? 1 : 0 ;
                    $bValue = isset($itemB['_source']['design_id']) ? 1 : 0;
                    if ($aValue > $bValue) {
                        $result = -1;
                    } else if ($aValue < $bValue) {
                        $result = 1;
                    }
                }
            }
            return $result;
        });
        return $items;
    }

    public function getPriceRange($filter, $total, $maxPrice, $minPrice)
    {
        $pageSize = isset($filter['page_size']) ? $filter['page_size'] : 20;
        $retVal = [];
        if ($total > $pageSize) {
            $priceRange = [];
            $unit = 4;
            $range = (int)(($maxPrice - $minPrice) / $unit);
            $length = strlen((int) $range) - 1;
            $range = round($range, -$length, PHP_ROUND_HALF_DOWN);
            $priceDiffLimit = 5 * (strlen(intval($maxPrice)) - 1) * (ceil($maxPrice / 100));
            if ($maxPrice > 0 && $range > 0 && $maxPrice != $minPrice) {
                $priceRange = [
                    [
                        'from' => 0,
                        'to' => 0,
                    ]
                ];
    
                $max = 0;
    
                while ($max < $maxPrice) {
                    if ($max <= $minPrice) {
                        $max += $range;
                        $priceRange[0]['to'] = $max;
                    } else {
                        $priceRange[] = [
                            'from' => $max,
                            'to' => $max + $range
                        ];
                        $max += $range;
                    }
                }
            }
    
            foreach ($priceRange as $key => $value) {
                if ($value['to'] == 0) {
                    unset($priceRange[$key]);
                }
            }

            $lastPriceRange = null;
            foreach ($priceRange as $key => $value) {
                if (abs($value['to'] - $value['from']) > $priceDiffLimit) {
                    if ($lastPriceRange) {
                        $retVal[] = [
                            'from' => $lastPriceRange['to'],
                            'to' => $value['to'],
                        ];
                    } else {
                        $retVal[] = $value;
                    }
                    $lastPriceRange = $value;
                } else {
                    $retVal[] = $value;
                }
            }
        }

        return $retVal;
    }

    protected function getProductPined()
    {
        try {
            return DB::table('product_pined')->pluck('order', 'product_id')->toArray();
        } catch (\Exception $e) {
            return [];
        }
    }

    protected function getProductPosition($filter)
    {
        $q = isset($filter['origin_keyword']) ? $filter['origin_keyword'] : request()->input('q');
        $key = self::buildKeyPosition($q);
        return $this->getOptionValue($key);
    }

    private function getOptionValue($key) {
        $result = [];
        try {
            $option = Option::where('key', $key)->first();
            if ($option && isset($option->id) && $option->id > 0) {
                $result = json_decode($option->value, true);
            }
        } catch (\Exception $e) {
            //@todo
        }
        return $result;
    }

    public function detachCategory($q) {
        $mappingData = $this->getMappingKeyword();
        $originKeyword = $q;
        $q = strtolower($q);
        $result = [];
        $categoryIds = [];
        if ($mappingData) {
            foreach ($mappingData as $word => $listCategory) {
                $isContain = $this->strContains($q, $word);
                if ($isContain && $q && $word) {
                    $q = $this->minifyKeyword($q, $word);
                    $categoryIds = array_merge($categoryIds, $listCategory);
                }
            }
            if ($categoryIds) {
                $listCate = [];
                $categoryIds = array_unique($categoryIds);
                foreach ($categoryIds as $id) {
                    if (!is_numeric($id)) {
                        continue;
                    }
                    $listCate[] = $id;
                }
                $result = [
                    'category_ids' => $listCate,
                    'q' => $q && strlen($q) > 1 ? $q : $originKeyword,
                ];
            }
        }

        return $result;
    }

    protected function getMappingKeyword() {
        $keyCache = decorCacheKey('zSearch::getMappingKeywordv2');
        $result = Cache::get($keyCache);
        if (!$result) {
            $result = [];
            $configData = $this->getOptionValue(config('z-search::sort.mapping_keyword', 'mapping_keyword'));
            if ($configData) {
                foreach ($configData as $categoryId => $words) {
                    foreach ($words as $word) {
                        $word = strtolower(trim($word));
                        if ($word && !array_key_exists($word, $result)) {
                            $result[$word] = [];
                        }
                        $result[$word][] = $categoryId;
                    }
                }
                Cache::put($keyCache, $result, 30);
            }
        }
        return $result;
    }

    public function minifyKeyword($q, $word) {
        $qLastDecor = $q . ' ';
        $wordLastDecor = $word . ' ';
        $qFirstDecor = ' ' . $q;
        $wordFirstDecor = ' ' . $word;
        if ($q && strpos($qLastDecor, $wordLastDecor) !== false) {
            $q = substr_replace($qLastDecor, '',  strpos($qLastDecor, $wordLastDecor), strlen($wordLastDecor));
        } else if ($q && strpos($qFirstDecor, $wordFirstDecor) !== false) {
            $q = substr_replace($qFirstDecor, '',  strpos($qFirstDecor, $wordFirstDecor), strlen($wordFirstDecor));
        }
        if ($q) {
            $q = trim($q);
        }
        return $q;
    }

    public static function buildKeyPosition($keyword) {
        $key = str_slug($keyword);
        return 'search::' . $key;
    }


    public function buildImageByColor($items, $filter) {
        //color_variant_id
        if ($items && array_key_exists('color_variant_id', $filter) && $filter['color_variant_id'] > 0 && $items) {
            $colorVariantId = $filter['color_variant_id'];
            $pIds = $this->mapProductIds($items);
            if ($pIds) {
                $imgByColor = DB::table('product_sku_value as psv')
                    ->join('product_sku as ps', 'psv.sku_id', '=', 'ps.id')
                    ->whereIn('psv.product_id', $pIds)
                    ->where('psv.variant_option_id', $colorVariantId)
                    ->groupBy('psv.product_id')
                    ->pluck('ps.image_url', 'ps.product_id');
                foreach ($items as &$item) {
                    $pId = $item['id'];
                    if (str_contains($item['image_url'], 'https://sticker.liveview.printerval.com/sticker')) {
                        $item['image_url'] = str_replace('sticker-kisscut,white', 'sticker-round,kelly-green', $item['image_url']);
                        $item['image_url'] = str_replace('f9f9f9', '1cb24b', $item['image_url']);
                    }
                    if (!isset($imgByColor[$pId]) || !$imgByColor[$pId]) {
                        $imageTemplate = $this->getImageTemplate($item['id'], $colorVariantId);
                        if ($imageTemplate && !str_contains('DESIGN_ID', $imageTemplate)) {
                            $item['image_url'] = $imageTemplate;
                        }
                    } else {
                        $item['image_url'] = $imgByColor[$pId];
                    }
                }
            }
        }

        return $items;
    }

    public function decorProductAttr($items, $attrs) {
        if ($items) {
            $pIds = $this->mapProductIds($items);
            $isAdmin = $this->isAdmin();
            if ($isAdmin) {
                $attrs = array_merge($attrs, ['is_violation', 'is_trademark']);
            }
            $products = $this->getProductsByAttr($pIds, $attrs);
            foreach ($items as &$item) {
                $pId = $item['id'];
                $item['rating_count'] = isset($products[$pId]) && isset($products[$pId]->rating_count) ? $products[$pId]->rating_count : 0;
                $item['rating_value'] = isset($products[$pId]) && isset($products[$pId]->rating_value) ? $products[$pId]->rating_value : 0;
                $item['sold'] = isset($products[$pId]) && isset($products[$pId]->sold) ? $products[$pId]->sold : 0;
                if (isset($item['price']) && isset($products[$pId]) && isset($products[$pId]->price) && $item['price'] != $products[$pId]->price) {
                    $item['price'] = $products[$pId]->price;
                    $item['high_price'] = isset($products[$pId]->high_price) ? $products[$pId]->high_price :  $item['price'] + 6;
                    $item['display_price'] = formatPrice($item['price']);
                    $item['display_high_price'] = formatPrice($item['high_price']);
                }
                if ($isAdmin) {
                    $item['is_violation'] = isset($products[$pId]) && isset($products[$pId]->is_violation) ? $products[$pId]->is_violation : null;
                    $item['is_trademark'] = isset($products[$pId]) && isset($products[$pId]->is_trademark) ? $products[$pId]->is_trademark : null;
                }
            }
        }
        return $items;
    }

    protected function isAdmin() {
        $user = \Illuminate\Support\Facades\Auth::user();
        $result = false;
        if($user && isset($user->role) && in_array($user->role, ['STAFF','ADMIN']) ) {
            $result = true;
        }
        return $result;
    }

    protected function mapProductIds($items) {
        $pIds = [];
        if ($items) {
            foreach ($items as $item) {
                $pIds[] = $item['id'];
            }
        }

        return $pIds;
    }

    protected function getProductsByAttr($pIds, $attrs) {
        $result = [];
        $attrs[] = 'id';
        $items = DB::table('product')
            ->whereIn('id', $pIds)
            ->get($attrs);
        if ($items) {
            foreach ($items as $item) {
                $result[$item->id] = $item;
            }
        }
        return $result;
    }

    private function getFilterOptions($productIds) {
        return [];
    }

    public function handleDuplicateDesign($items, $productFilters, $limit = 200) {
        $pnDesign = $this->productNDesign($this->getByColumn($items, 'id', $limit, 'design_id'));
        $this->decorDesignId($items, $pnDesign, $limit);
        $removeProductIds = $this->getRemoveList($items, $productFilters, $limit);
        if (!$removeProductIds) {
            return $items;
        }
        $result = [];
        foreach ($items as $item) {
            if (array_key_exists($item['id'], $removeProductIds)) {
                continue;
            }
            $result[] = $item;
        }
        return $result;
    }

    public function filterCustomProduct($items) {
        $result = [];
        if ($items) {
            foreach ($items as $item) {
                if (isset($item['is_custom_design']) && $item['is_custom_design'] == 1) {
                    continue;
                }
                $result[] = $item;
            }
        }
        return $result;
    }

    protected function getRemoveList($items, $productFilters, $limit) {
        $result = [];
        $duplicateData = $this->getDuplicateDesign($items, $limit);
        $categoryId = isset($productFilters['category']) ? $productFilters['category'] : 0;
        if ($categoryId == 0 &&  isset($productFilters['category_detach'])) {
            $categoryId = $productFilters['category_detach'];
        }

        foreach ($duplicateData as $designId => $duplicateItems) {
            $bestItem = $this->findBestItems($duplicateItems, $categoryId);
            foreach ($duplicateItems as $item) {
               if ($item['id'] == $bestItem['id']) {
                  continue;
               }
                $result[$item['id']] = $item['id'];
            }
        }
        return $result;
    }

    protected function findBestItems($duplicateItems, $categoryId) {
        $duplicateItems = $this->sortArray($duplicateItems, 'cr');
        $result = $duplicateItems[0];
        if ($categoryId == 0
            && isset($duplicateItems[0])
            && $duplicateItems[0]['cr']  == 0) {
            $categoryId = 7;
        }
        if ($categoryId > 0) {
            foreach ($duplicateItems as $item) {
                if ($this->isExistsCategory($item['categories'], $categoryId)) {
                    $result = $item;
                    break;
                }
            }
        }
        return $result;
    }

    protected function isExistsCategory($categories, $categoryId) {
        $result = false;
        if ($categories) {
            foreach ($categories as $cat) {
                if ($cat['id'] == $categoryId) {
                    $result = true;
                    break;
                }
            }
        }

        return $result;
    }

    protected function getDuplicateDesign($items, $limit) {
        $counter = [];
        $result = [];
        $i = 0;
        foreach ($items as $index => $item) {
            $i++;
            if ($i > $limit) {
                break;
            }
            $designId = $this->getValue($item, 'design_id');
            if (!$designId) {
                continue;
            }
            $isDuplicate = false;
            if (!array_key_exists($designId, $counter)) {
                $counter[$designId] = [];
            } else {
                $isDuplicate = true;
            }
            $item['index'] = $index;
            $counter[$designId][] = $item;
            if ($isDuplicate) {
                $result[$designId] = $counter[$designId];
            }
        }
        return $result;
    }

    protected function decorDesignId(&$items, $pnDesign, $limit) {
        $i = 0;
        foreach ($items as &$item) {
            $i++;
            if ($i > $limit) {
                break;
            }
            $item = isset($item['_source']) ? $item['_source'] : $item;
            $productId = $this->getValue($item, 'id');
            $designId = $productId && $pnDesign && array_key_exists($productId, $pnDesign) ? $pnDesign[$productId] : 0;
            if ($designId) {
                $this->decorItem($item, 'design_id', $designId);
            }
        }
    }

    public function productNDesign($pIds) {
        $result = [];
        if ($pIds) {
            $items = DB::table('product_n_design as pnd')
                ->whereIn('product_id', $pIds)
                ->where('is_primary', 1)
                ->orderBy('design_id' , 'asc')
                ->get(['design_id', 'product_id']);
            if ($items) {
                foreach ($items as $item) {
                    if (array_key_exists($item->product_id, $result)) {
                        continue;
                    }
                    $result[$item->product_id] = $item->design_id;
                }
            }
        }
        return $result;
    }

    protected function getByColumn($items, $column, $limit, $filterColumn = null){
        $result = [];
        if ($items) {
            $i = 0;
            foreach ($items as $item) {
                $i++;
                if ($i > $limit) {
                    break;
                }
                $item = isset($item['_source']) ? $item['_source'] : $item;
                //validate exists design_id
                if ($filterColumn) {
                    $filterValue = $this->getValue($item, $filterColumn);
                    if ($filterValue) {
                        continue;
                    }
                }
                if (!$item) {
                    continue;
                }
                $value = $this->getValue($item, $column);
                if ($value) {
                    $result[] = $value;
                }
            }
        }
        return $result;
    }

    protected function getValue($item, $column) {
        $value = null;
        if (is_array($item) && isset($item[$column])) {
            $value = $item[$column];
        } else if (is_object($item) && isset($item->{$column})) {
            $value = $item->{$column};
        }
        return $value;
    }

    protected function decorItem(&$item, $column, $value) {
        if (is_array($item)) {
            $item[$column] = $value;
        } else if (is_object($item)) {
            $item->{$column} = $value;
        }
    }

    public function filterBestProduct($items, $keyword) {
        $result = [];
        $keyword = strtolower($keyword);
        $qCombinations = $this->combinations($keyword);
        foreach ($items as $item) {
            $isContainsKeyWord = $this->itemContainSearch($item, $qCombinations);
            if ($isContainsKeyWord) {
                $result[] = $item;
            }
        }
        return $result;
    }

    public function getImageTemplate($productId, $colorVariantId) {
        $result = null;
        $templateId = Product::getTemplateId($productId);
        if ($templateId > 0) {
            $sku = $this->getSkuByTemplate($templateId, $colorVariantId);
            if ($sku && isset($sku->id)) {
                $result = $this->getImgBySku($productId, $sku);
            }
        }
        return $result;

    }

    public function getSkuByTemplate($templateId, $colorVariantId) {
        $key = 'getSkuByTemplate::' . $templateId . '-' . $colorVariantId;
        if (!array_key_exists($key, self::$skuByTemplate)) {
            $defaultStyle = ElasticSearchIndex::getDefaultTemplateStyle($templateId);
            $query = DB::table('product_template_sku as ps')
                ->join('product_template_sku_value as psv', 'psv.sku_id', '=', 'ps.id')
                ->where('ps.template_id', $templateId)
                ->where('psv.variant_option_id', $colorVariantId);
            if ($defaultStyle) {
                $query->join('product_template_sku_value as psv2', 'ps.id', '=', 'psv2.sku_id')
                    ->where('psv2.variant_option_id', $defaultStyle);
            }
            self::$skuByTemplate[$key]  = $query->first(['ps.*']);
        }
        return isset(self::$skuByTemplate[$key]) ? self::$skuByTemplate[$key] : null;
    }

    public function getImgBySku($productId, $sku) {
        $result = null;
        $skuId = $sku->id;
        if (self::hasOverwriteImage($productId)) {
            $image = self::getOverwriteImage($productId, $skuId, 1);
            if ($image && isset($image->image_url)) {
                $result = $image->image_url;
            }
        } else if (isset($sku->image_url)) {
            $colorId = self::getColorId($skuId);
            $design = self::getTemplateDesign($productId, $colorId);
            $result = self::replaceTemplateDesignImageUrl($design, $sku->image_url);
        }
        return $result;
    }

    public static function replaceTemplateDesignImageUrl($design, $imageUrl) {
        if (!$design) {
            return null;
        }
        return str_replace('[DESIGN_ID]', $design, $imageUrl);
    }

    public static function getTemplateDesign($productId, $colorId = null) {
        $result = null;
        $designItem = null;
        if ($colorId) {
            $designItem = DB::table('product_template_design_code')
                ->where('product_id', $productId)
                ->where('color_id', $colorId)
                ->first(['design_code']);
        }
        if (!$designItem) {
            $designItem = DB::table('product_template_design_code')
                ->where('product_id', $productId)
                ->first(['design_code']);
        }
        if ($designItem && isset($designItem->design_code)) {
            $result = $designItem->design_code;
        }
        return $result;
    }

    public static function getColorId($skuId) {
        if (array_key_exists($skuId, self::$colorBySku)) {
            return self::$colorBySku[$skuId];
        }
        $result = null;
        $psv = DB::table('product_sku_value')
            ->where('sku_id', $skuId)
            ->where('variant_id', 2)
            ->first();
        if ($psv) {
            $result = $psv->variant_option_id;
            self::$colorBySku[$skuId] = $result;
        }

        return $result;
    }

    public static function hasOverwriteImage($productId) {
        return DB::table('product_template_gallery_overwrite')
            ->where('product_id', $productId)
            ->exists();
    }

    public static function getOverwriteImage($productId, $productSkuId, $isPrimary) {
        return DB::table('product_template_gallery_overwrite')
            ->where('product_id', $productId)
            ->where('product_sku_id', $productSkuId)
            ->where('is_primary', $isPrimary)
            ->first(['image_url']);
    }

    public function getSuggestionCategoryKeywords($keyword, $categories = [])
    {
        $result = [];
        if ($categories) {
            foreach ($categories as $category) {
                $result[] = [
                    'text' => $keyword . ' ' . $category['text'],
                    'link' => '/search?q=' . urlencode($keyword . ' ' . $category['text'])
                ];
            }
        }


        return $result;
    }

    public function getTopCategories()
    {
        $locale = env("APP_LOCALE");
        $orderByCategories = [];

        if ($locale == "") {
            $locale = config('localization::module.default_locale', 'us');
        }
        $cacheKey = "$locale-sale-by-cate-v1";

        if (Redis::exists($cacheKey)) {
            $orderByCategories = json_decode(Redis::get($cacheKey));
        } else {
            $date = new \DateTime();
            $date->modify("-1000 days");
            $orderByCategories = \DB::table('order_item')
                ->join('order', 'order_item.order_id', '=', 'order.id')
                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                ->where('product_n_category.is_parent', '=', 0)
                ->where('order.payment_status', '=', 'PAID')
                ->where('order.created_at', '>', $date->format("Y-m-d H:i:s"))
                ->select('product_n_category.category_id AS categoryId', DB::raw('COUNT(*) AS quantity'))
                ->groupBy('categoryId')
                ->orderBy('quantity', 'DESC')
                ->get();

            Redis::set($cacheKey, json_encode($orderByCategories), 'EX', 60 * 60 * 24 * 7);
        }

        $categoryIds = [];
        for ($i = 0; $i < count($orderByCategories); $i++) { 
            $categoryIds[] = $orderByCategories[$i]->categoryId;
        }

        return DB::table('category')
            ->whereIn('id', $categoryIds)
            ->select(['id', 'name', 'slug'])
            ->get();
    }

    public function buildColorCode($items) {
        $config = config('z-search::color-variant');
        $result = [];
        if ($config && $items) {
            $imgOptions = $this->imageOption($items);
            foreach ($items as $item) {
                $item['color_code'] = isset($item['slug']) && isset($config[$item['slug']]) ? $config[$item['slug']] : '';
                $item['image_url'] = isset($imgOptions[$item['id']]) ? $imgOptions[$item['id']] : '';
                $result[] = $item;
            }
        }
        return $result;
    }

    protected function imageOption($items) {
        $result = [];
        $ids = [];
        foreach ($items as $item) {
            $ids[] = $item['id'];
        }
        if ($ids) {
            $result = DB::table('product_variant_option')
                ->whereIn('id', $ids)
                ->pluck('image_url', 'id');
        }
        return $result;
    }

    public function decorHits(&$items) {
        self::initCacheVariants();
        if ($items) {
            foreach ($items as &$item) {
                $item['_source']['categories'] = [];
                $item['_source']['variant_options'] = [];
                if (isset($item['_source']['category_id'])) {
                    $item['_source']['categories'] = self::getTreeCategory($item['_source']['category_id']);
                }
                if (isset($item['_source']['template_id'])) {
                    $item['_source']['variant_options'] = $this->buildVariantTemplate($item['_source']['template_id']);
                } else if (isset($item['_source']['variant'])) {
                    $item['_source']['variant_options'] = $this->buildVariantOptions($item['_source']['variant']);
                }
                if (!isset($item['_source']['url'])) {
                    $item['_source']['url'] =  getPrefixUrl() . "/" . (!empty($item['_source']['slug']) ? $item['_source']['slug'] : "product") . "-p" . $item['_source']['id'];
                }
            }
        }
    }

    protected function buildVariantTemplate($templateId) {
        if (!array_key_exists($templateId, self::$templates)) {
            $key = decorCacheKey('cache:getVariantTemplate:' . $templateId);
            $result = Cache::get($key);
            if (!$result) {
                $result = $this->getVariantTemplate($templateId);
                Cache::put($key, $result, 1440);
            }
            self::$templates[$templateId] = $result;
        }
        return self::$templates[$templateId];
    }

    protected function getVariantTemplate($templateId) {
        $result = [];
        $skuIds = DB::table('product_template_sku')
            ->where('template_id', $templateId)
            ->pluck('id')
            ->toArray();
        $data = [];
        $options = [];
        if ($skuIds) {
            $options = DB::table('product_template_sku_value')
                ->whereIn('sku_id', $skuIds)
                ->pluck('variant_id', 'variant_option_id')
                ->toArray();
        }
        if ($options) {
            foreach ($options as $optionId => $variantId) {
                if (!array_key_exists($variantId, $data)) {
                    $data[$variantId] = [
                        'variant_id' => $variantId,
                        'variant_name' => $variantId,
                        'options' => [],
                    ];
                }
                $data[$variantId]['options'][] = $optionId;
            }
            $result = $this->buildVariantOptions($data);

        }

        return $result;
    }

    protected function buildVariantOptions($items) {
        $result = [];
        if ($items) {
            foreach ($items as $variantId => $item) {
                $variant = isset(self::$variants[$variantId]) ? self::$variants[$variantId] : [];
                if (!$variant) {
                    continue;
                }
                $options = [];
                if (isset($item['options'])) {
                    foreach ($item['options'] as $optionId) {
                        $options[] = self::getOption($optionId);
                    }
                }
                $result[$variantId] =  [
                    'variant_id' => $variantId,
                    'variant_name' => isset($variant) ? $variant : [],
                    'options' => $options,
                ];

            }
        }
        return $result;
    }



    protected static function getVariantOption() {

    }

    public static function initCacheVariants() {
        $variantIds =  Config::get('z-search::default.filter_variant_id', [1,2,5]);
        self::$variants = ElasticSearchIndex::getProductVariant($variantIds);
    }

    public static function getOption($optionId) {
        if (!array_key_exists($optionId, self::$options)) {
            $item = DB::table('product_variant_option')
                ->where('id', $optionId)
                ->first(['id', 'name', 'slug', 'image_url']);
            self::$options[$optionId] = $item ? (array) $item : [];
        }
        return self::$options[$optionId];
    }

    public static function getTreeCategory($categoryId) {
        if ($categoryId && !array_key_exists($categoryId, self::$categories)) {
            $category = self::getCategory($categoryId);
            $result = [];
            if ($category) {
                $items = DB::table('category')
                    ->where('_lft', '<=', $category->_lft)
                    ->where('_rgt', '>=', $category->_rgt)
                    ->orderBy('_lft', 'asc')
                    ->get(['id', 'slug', 'name', 'image_url']);
                foreach ($items as $item) {
                    $result[] = (array) $item;
                }
            }
            self::$categories[$categoryId] = $result;
            return self::$categories[$categoryId];
        }
        return isset(self::$categories[$categoryId]) ? self::$categories[$categoryId] : [];
    }

    protected static function getCategory($categoryId) {
        return DB::table('category')
            ->where('id', $categoryId)
            ->first(['id', 'slug', 'name', '_lft', '_rgt']);
    }



}

?>