<?php

namespace Modules\ZSearch\Services;

use Illuminate\Support\Facades\DB;
use Modules\ZSearch\Models\Product;

class TrademarkService extends ElasticSearchIndex
{
    protected $users = [];


    public function index($productIds, $filter = [])
    {
        \Log::info('ZSEARCH - TrademarkIndex count', [count($productIds)]);
        foreach (array_chunk($productIds, 200) as $chunkProductIds) {
            try {
                \Log::info('ZSEARCH - TrademarkIndex pids', $chunkProductIds);
                $this->indexProducts($chunkProductIds, $filter);
            } catch (\Exception $e) {
                \Log::error('ZSEARCH - TrademarkIndex ERROR', [$e->getMessage() . ' - ' . $e->getFile() . ' ' . $e->getLine()]);
            }
        }

    }

    protected function indexProducts($productIds, $filter = [])
    {
        $products = Product::getProductQuery(array_merge($filter, [
            'ids' => $productIds,
            'type' => 'init',
        ]))->get(['id', 'name', 'content'])->toArray();
        $this->addCategoryIdProduct($products, $productIds);
        if ($products && count($products) > 0) {
            $params = $this->elasticService->bulkData($products, $this->elasticServiceConfig['index'], self::TYPE_PRODUCTS_TRADEMARK);
            $this->elasticService->bulkAction($params);
        }

    }

    public function search($filter) {
        $params = $this->buildSearchParams($filter);
        return $this->elasticService->searchDocument($params, self::TYPE_PRODUCTS_TRADEMARK, $this->elasticServiceConfig['index']);
    }

    public function buildSearchData($data) {
        $result = [];
        if (isset($data['hits']['hits'])) {
            $searchDecor =  new SearchDataDecor();
            $result = $searchDecor->mergeData([], $data['hits']['hits']);
        }
        return $result;
    }

    protected function buildSearchParams($filter) {
        $elasticParams = [
            'from' => $filter['from'],
            'size' => $filter['page_size'],
            '_source' => ['id']
        ];
        $elasticParams['query']['bool']['should'] = [
            [
                'match_phrase' => [
                    'name' => $filter['keyword']
                ],
            ],
            [
                'match_phrase' => [
                    'content' => $filter['keyword'],
                ],
            ],
        ];
        if (array_key_exists('from_id', $filter)) {
            $elasticParams['query']['bool']['must'][] = [
                'range' => [
                    'id' => ['gt' => $filter['from_id']]
                ]
            ];
        }
        $elasticParams['query']['bool']['minimum_should_match'] = 1;
        if (isset($filter['categories_id'])) {
            $elasticParams['query']['bool']['must'] = [
                [
                    'terms' => [
                        "category_id" =>  $filter['categories_id']
                    ],
                ],
            ];
        }
        return $elasticParams;
    }

    private function addCategoryIdProduct(&$products, $productsId)
    {
        $categories = $this->getCategoryIdOfProduct($productsId);
        foreach ($products as $index => $product) {
            if (isset($product['id']) && isset($categories[$product['id']])) {
                $products[$index]['category_id'] = $categories[$product['id']];
            }
        }
    }

    private function getCategoryIdOfProduct($productsId)
    {
        return DB::table('product_n_category')->whereIn('product_id', $productsId)
            ->where('is_parent', 0)->pluck('category_id', 'product_id')->toArray();
    }
}